/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/property.h"

// Grilo
#include <grilo.h>

// Lucene++
#include <Lucene.h>

#include <BooleanQuery.h>
#include <DateTools.h>
#include <Field.h>
#include <NumericField.h>
#include <NumericRangeQuery.h>
#include <NumericUtils.h>
#include <PhraseQuery.h>
#include <StringReader.h>
#include <StringUtils.h>
#include <Term.h>
#include <TermAttribute.h>
#include <TermRangeQuery.h>
#include <TermQuery.h>
#include <TokenStream.h>
#include <WhitespaceAnalyzer.h>

// Boost C++
#include <boost/algorithm/string/predicate.hpp>
#include <boost/bind.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/function.hpp>
#include <boost/numeric/conversion/bounds.hpp>

// C++ Standard Library
#include <map>
#include <set>
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/dbustypes.h"
#include "mediascanner/glibutils.h"
#include "mediascanner/locale.h"
#include "mediascanner/logging.h"
#include "mediascanner/propertyprivate.h"
#include "mediascanner/settings.h"
#include "mediascanner/utilities.h"

namespace mediascanner {

// Lucene++
using Lucene::newLucene;

// Boost C++
using boost::algorithm::ends_with;
using boost::gregorian::date;
using boost::posix_time::seconds;
using boost::posix_time::microsec;
using boost::posix_time::time_duration;

////////////////////////////////////////////////////////////////////////////////

Property::MetadataKeyCache Property::metadata_key_cache_;
Property::FieldNameCache Property::field_name_cache_;

const GrlKeyID Property::MetadataKey::kPending =
        boost::numeric::bounds<GrlKeyID>::highest();
const GrlKeyID Property::MetadataKey::kInvalid = GRL_METADATA_KEY_INVALID;

static const logging::Domain kError("error/property", logging::error());

const Property::Category Property::Category::Generic(1);
const Property::Category Property::Category::File(Generic.id_ | 2);
const Property::Category Property::Category::Media(File.id_ | 4);
const Property::Category Property::Category::Music(Media.id_ | 8);
const Property::Category Property::Category::Image(Media.id_ | 16);
const Property::Category Property::Category::Photo(Image.id_ | 32);
const Property::Category Property::Category::Movie(Image.id_ | 64);

////////////////////////////////////////////////////////////////////////////////

std::string Property::MetadataKey::name() const {
    return safe_string(grl_metadata_key_get_name(id()));
}

std::string Property::MetadataKey::description() const {
    return grl_metadata_key_get_desc(id());
}

GType Property::MetadataKey::gtype() const {
    return grl_metadata_key_get_type(id());
}

const GList* Property::MetadataKey::relation() const {
    GrlRegistry *const registry = grl_registry_get_default();
    return grl_registry_lookup_metadata_key_relation(registry, id());
}

bool Property::MetadataKey::RegisterCustomKey() {
    g_return_val_if_fail(id_ == kPending, false);

    // Lookup and reuse existing key.
    const Wrapper<GParamSpec> key_spec = take(make_spec_());
    const char *const key_name = g_param_spec_get_name(key_spec.get());

    if (key_spec == null_ptr || key_name == null_ptr) {
        id_ = kInvalid;
        return false;
    }

    GrlRegistry *const registry = grl_registry_get_default();
    id_ = grl_registry_lookup_metadata_key(registry, key_name);

    if (id_ != kInvalid) {
        const GType type = grl_registry_lookup_metadata_key_type(registry, id_);

        if (type != G_PARAM_SPEC_VALUE_TYPE(key_spec.get())) {
            kError("Conflicting type for existing metadata key \"{1}\". "
                   "The existing key is of type {2} instead of {3}.")
                    % key_name % g_type_name(type)
                    % g_type_name(G_PARAM_SPEC_VALUE_TYPE(key_spec.get()));

            return false;
        }

        return true;
    }

    // Register custom key.
    Wrapper<GError> error;

    id_ = grl_registry_register_metadata_key(registry, key_spec.get(),
                                             error.out_param());

    if (error) {
        const std::string error_message = to_string(error);
        kError("Cannot register user metadata key \"{1}\": {2}")
                % key_name % error_message;

        return false;
    }

    BOOST_ASSERT(key_spec->ref_count > 1);

    return true;
}

////////////////////////////////////////////////////////////////////////////////

Property::Property(PrivatePtr impl)
    : d(impl) {
    // Register this property by its field name to make FromFieldName() work.
    field_name_cache_.insert(std::make_pair(d->field_name_, d));

    // Register properties with known Grilo metadata key.
    if (d->metadata_key_.id() != MetadataKey::kInvalid
            && d->metadata_key_.id() != MetadataKey::kPending) {
        metadata_key_cache_.insert(std::make_pair(d->metadata_key_.id(), d));
    }
}

Property::~Property() {
}

////////////////////////////////////////////////////////////////////////////////

String Property::field_name() const {
    return d->field_name_;
}

const Property::MetadataKey &Property::metadata_key() const {
    // Register pending custom metadata keys.
    // See documentation of kPendingMetadataKey for reasoning.
    if (d->metadata_key_.id() == MetadataKey::kPending
            && d->metadata_key_.RegisterCustomKey()) {
            metadata_key_cache_.insert(std::make_pair(d->metadata_key_.id(),
                                                      d));
    }

    return d->metadata_key_;
}

Property::Category Property::category() const {
    return d->category_;
}

std::set<std::string> Property::origins() const {
    std::set<std::string> origins;

    if (category() == Category::File)
        origins.insert("file system");

    if (d->merge_stream_info_
            && d->merge_stream_info_ != merge_nothing)
        origins.insert("GStreamer");

    const Wrapper<GrlRegistry> registry = wrap(grl_registry_get_default());
    const Wrapper<GList> sources = take(grl_registry_get_sources(registry.get(),
                                                                 false));

    for (const GList *l = sources; l; l = l->next) {
        GrlSource *const source = static_cast<GrlSource *>(l->data);
        const GList *const keys = grl_source_supported_keys(source);

        if (g_list_find(const_cast<GList *>(keys),
                        GRLKEYID_TO_POINTER(metadata_key().id())))
            origins.insert(grl_source_get_name(source));
    }

    return origins;
}

bool Property::supports_full_text_search() const {
    return d->supports_full_text_search();
}

Property::MergeStrategy Property::merge_strategy() const {
    return d->merge_strategy_;
}

////////////////////////////////////////////////////////////////////////////////

Property::MetadataKey Property::define_boolean(const char *name,
                                               const char *nick,
                                               const char *blurb,
                                               bool default_value) {
    return MetadataKey(boost::bind(&g_param_spec_boolean,
                                   name, nick, blurb, default_value,
                                   G_PARAM_STATIC_STRINGS | G_PARAM_READWRITE));
}

Property::MetadataKey Property::define_datetime(const char *name,
                                                const char *nick,
                                                const char *blurb) {
    // Defer invokation of G_TYPE_DATE_TIME to avoid warnings
    // about missing g_type_init() calls.
    return MetadataKey(boost::bind(&MakeParamSpecBoxed<GDateTime>,
                                   name, nick, blurb,
                                   G_PARAM_STATIC_STRINGS | G_PARAM_READWRITE));
}

Property::MetadataKey Property::define_string(const char *name,
                                              const char *nick,
                                              const char *blurb,
                                              const char *default_value) {
    return MetadataKey(boost::bind(&g_param_spec_string,
                                   name, nick, blurb, default_value,
                                   G_PARAM_STATIC_STRINGS | G_PARAM_READWRITE));
}

template<> Property::MetadataKey NumericProperty<bool>::
define(const char *name, const char *nick, const char *blurb,
       bool default_value) {
    return define_boolean(name, nick, blurb, default_value);
}

template<> Property::MetadataKey NumericProperty<Fraction>::
define(const char *name, const char *nick, const char *blurb,
       Fraction default_value);

template<typename T> Property::MetadataKey NumericProperty<T>::
define(const char *name, const char *nick, const char *blurb, T default_value) {
    return define(name, nick, blurb, boost::numeric::bounds<T>::lowest(),
                  boost::numeric::bounds<T>::highest(), default_value);
}

template<> Property::MetadataKey NumericProperty<double>::
define(const char *name, const char *nick, const char *blurb,
       double minimum, double maximum, double default_value) {
    return MetadataKey(boost::bind(&g_param_spec_float, name, nick, blurb,
                                   minimum, maximum, default_value,
                                   G_PARAM_STATIC_STRINGS | G_PARAM_READWRITE));
}

template<> Property::MetadataKey NumericProperty<int32_t>::
define(const char *name, const char *nick, const char *blurb,
       int minimum, int maximum, int default_value) {
    return MetadataKey(boost::bind(&g_param_spec_int, name, nick, blurb,
                                   minimum, maximum, default_value,
                                   G_PARAM_STATIC_STRINGS | G_PARAM_READWRITE));
}

template<> Property::MetadataKey NumericProperty<uint64_t>::
define(const char *name, const char *nick, const char *blurb,
       uint64_t minimum, uint64_t maximum, uint64_t default_value) {
    return MetadataKey(boost::bind(&g_param_spec_uint64, name, nick, blurb,
                                   minimum, maximum, default_value,
                                   G_PARAM_STATIC_STRINGS | G_PARAM_READWRITE));
}

////////////////////////////////////////////////////////////////////////////////

bool Property::MergeAny(GstDiscovererInfo *media,
                        GstDiscovererStreamInfo *stream,
                        const StreamInfoFunction &merge_first,
                        const StreamInfoFunction &merge_second,
                        ValueMap *item) {
    return merge_first(media, stream, item)
            || merge_second(media, stream, item);
}

template<typename ValueType>
bool Property::MergeAttribute
                    (GstDiscovererInfo *media,
                     GstDiscovererStreamInfo *,
                     ValueType (*get_attribute)(const GstDiscovererInfo *),
                     Property::ValueMap *item) const {
    item->insert(std::make_pair(*this, get_attribute(media)));
    return true;
}

template<typename ValueType, typename InfoType>
bool Property::MergeAttribute(GstDiscovererInfo *,
                              GstDiscovererStreamInfo *stream,
                              ValueType (*get_attribute)(const InfoType *),
                              Property::ValueMap *item) const {
    if (not G_TYPE_CHECK_INSTANCE_TYPE(stream, internal::GetGType<InfoType>()))
        return false;

    const Value value = get_attribute
            (reinterpret_cast<const InfoType *>(stream));
    item->insert(std::make_pair(*this, value));

    return true;
}

bool Property::MergeStreamInfo(GstDiscovererInfo *media,
                               GstDiscovererStreamInfo *stream,
                               Property::ValueMap *properties) const {
    return d->merge_stream_info_(media, stream, properties);
}

template<typename ValueType>
bool Property::MergeTag(GstDiscovererInfo *,
                        GstDiscovererStreamInfo *stream,
                        const char *tag_id, ValueMap *item) const {
    if (stream == null_ptr)
       return false;

    const GstTagList *const tags = gst_discoverer_stream_info_get_tags(stream);

    if (tags == null_ptr)
       return false;

    const GValue *const tag_value =
            gst_tag_list_get_value_index(tags, tag_id, 0);

    if (tag_value == null_ptr)
        return false;

    Value property_value;

    if (not TransformTagValue<ValueType>(tag_value, &property_value))
        return false;

    item->insert(std::make_pair(*this, property_value));
    return true;
}

////////////////////////////////////////////////////////////////////////////////

template<typename> static GType get_tag_type();

template<> GType get_tag_type<bool>() { return G_TYPE_BOOLEAN; }
template<> GType get_tag_type<DateTime>() { return GST_TYPE_DATE_TIME; }
template<> GType get_tag_type<double>() { return G_TYPE_DOUBLE; }
template<> GType get_tag_type<int32_t>() { return G_TYPE_INT; }
template<> GType get_tag_type<Fraction>() { return GST_TYPE_FRACTION; }
template<> GType get_tag_type<uint64_t>() { return G_TYPE_UINT64; }
template<> GType get_tag_type<String>() { return G_TYPE_STRING; }

template<typename> static bool supports_tag_value(const GValue *const value);

template<> bool supports_tag_value<DateTime>(const GValue *const value) {
    return G_VALUE_HOLDS(value, GST_TYPE_DATE_TIME) ||
        G_VALUE_HOLDS(value, G_TYPE_DATE);
}

template<typename T> bool supports_tag_value(const GValue *const value) {
    return G_VALUE_HOLDS(value, get_tag_type<T>());
}

template<typename T> static T transform_tag_value(const GValue *const value);

template<> bool transform_tag_value(const GValue *const value) {
    return g_value_get_boolean(value);
}

template<> double transform_tag_value(const GValue *const value) {
    return g_value_get_double(value);
}

template<> Fraction transform_tag_value(const GValue *const value) {
    return Fraction(gst_value_get_fraction_numerator(value),
                    gst_value_get_fraction_denominator(value));
}

template<> int transform_tag_value(const GValue *const value) {
    return g_value_get_int(value);
}

template<> DateTime transform_tag_value(const GValue *const value) {
    if (G_VALUE_HOLDS(value, GST_TYPE_DATE_TIME)) {
        const GstDateTime *const dt =
            static_cast<GstDateTime *>(g_value_get_boxed(value));

        if (dt == null_ptr)
            return DateTime();

        time_duration time;
        if (gst_date_time_has_time(dt))
            time += time_duration(gst_date_time_get_hour(dt),
                                  gst_date_time_get_minute(dt), 0);
        if (gst_date_time_has_second(dt))
            time += seconds(gst_date_time_get_second(dt)) +
                microsec(gst_date_time_get_microsecond(dt));

        try {
            return DateTime(date(gst_date_time_get_year(dt),
                                 gst_date_time_has_month(dt) ?
                                 gst_date_time_get_month(dt) : 1,
                                 gst_date_time_has_day(dt) ?
                                 gst_date_time_get_day(dt) : 1),
                            time);
        } catch(const std::out_of_range &ex) {
            const std::string error_message = ex.what();
            kError("Bad date value: {1}") % error_message;
            return DateTime();
        }
    } else if (G_VALUE_HOLDS(value, G_TYPE_DATE)) {
        const GDate *const dt =
            static_cast<GDate *>(g_value_get_boxed(value));

        if (dt == null_ptr)
            return DateTime();

        try {
            return DateTime(date(g_date_get_year(dt),
                                 g_date_get_month(dt),
                                 g_date_get_day(dt)));
        } catch(const std::out_of_range &ex) {
            const std::string error_message = ex.what();
            kError("Bad date value: {1}") % error_message;
            return DateTime();
        }
    } else {
        // Unknown value type
        kError("Unkown date value type: {1}") % G_VALUE_TYPE_NAME(value);
        return DateTime();
    }
}

template<> unsigned transform_tag_value(const GValue *const value) {
    return g_value_get_uint(value);
}

template<> uint64_t transform_tag_value(const GValue *const value) {
    return g_value_get_uint64(value);
}

template<> String transform_tag_value(const GValue *const value) {
    const char *const str = g_value_get_string(value);
    return ToUnicode(str ? str : "");
}

template<typename T>
bool Property::TransformTagValue(const GValue *input, Value *output) const {
    if (supports_tag_value<T>(input)) {
        *output = transform_tag_value<T>(input);
        return true;
    }

    GValue transformed_value = G_VALUE_INIT;
    g_value_init(&transformed_value, get_tag_type<T>());

    if (g_value_transform(input, &transformed_value)) {
        *output = transform_tag_value<T>(&transformed_value);
        return true;
    }

    const String name = field_name();
    kError(L"Unexpected value type {1} in GStreamer tag for \"{2}\" field")
            % G_VALUE_TYPE_NAME(input) % name;

    return false;
}

////////////////////////////////////////////////////////////////////////////////

template<typename> static GType get_grilo_type();

template<> GType get_grilo_type<bool>() { return G_TYPE_BOOLEAN; }
template<> GType get_grilo_type<DateTime>() { return G_TYPE_DATE_TIME; }
template<> GType get_grilo_type<double>() { return G_TYPE_FLOAT; }
template<> GType get_grilo_type<int32_t>() { return G_TYPE_INT; }
template<> GType get_grilo_type<Fraction>() { return G_TYPE_FLOAT; }
template<> GType get_grilo_type<uint64_t>() { return G_TYPE_UINT64; }
template<> GType get_grilo_type<String>() { return G_TYPE_STRING; }

template<typename T> static T transform_grilo_value(const GValue *const value);

template<> bool transform_grilo_value(const GValue *const value) {
    return g_value_get_boolean(value);
}

template<> Fraction transform_grilo_value(const GValue *const value) {
    // FIXME(M5): Is there a better way to get fractions from Grilo values?
    // Maybe we should exponse two metadata key: One with a fraction,
    // and another one with the Grilo compatible double.
    return Fraction(g_value_get_float(value), 1);
}

template<> int transform_grilo_value(const GValue *const value) {
    return g_value_get_int(value);
}

template<> DateTime transform_grilo_value(const GValue *const value) {
    GDateTime *const dt = static_cast<GDateTime *>(g_value_get_boxed(value));

    if (dt == null_ptr)
        return DateTime();

    return DateTime(date(g_date_time_get_year(dt),
                         g_date_time_get_month(dt),
                         g_date_time_get_day_of_month(dt)),
                    time_duration(g_date_time_get_hour(dt),
                                  g_date_time_get_minute(dt),
                                  g_date_time_get_second(dt)) +
                    microsec(g_date_time_get_microsecond(dt)));
}

template<> double transform_grilo_value(const GValue *const value) {
    return g_value_get_float(value);
}

template<> uint64_t transform_grilo_value(const GValue *const value) {
    return g_value_get_uint64(value);
}

template<> String transform_grilo_value(const GValue *const value) {
    const char *const str = g_value_get_string(value);
    return ToUnicode(str ? str : "");
}

template<typename PropertyType, typename ValueType>
bool GenericProperty<PropertyType, ValueType>::
Private::TransformSafeGriloValue(const GValue *input, Value *output) const {
    *output = transform_grilo_value<ValueType>(input);
    return true;
}

template<typename PropertyType, typename ValueType>
bool GenericProperty<PropertyType, ValueType>::
Private::TransformGriloValue(const GValue *input, Value *output) const {
    if (G_TYPE_CHECK_VALUE_TYPE(input, get_grilo_type<ValueType>()))
        return TransformSafeGriloValue(input, output);

    GValue safe_value = G_VALUE_INIT;
    g_value_init(&safe_value, get_grilo_type<ValueType>());

    if (g_value_transform(input, &safe_value))
        return TransformSafeGriloValue(&safe_value, output);

    kError(L"Unexpected value type {1} in metadata value for \"{2}\" field")
              % G_VALUE_TYPE_NAME(input) % field_name_;

    return false;
}

bool Property::TransformGriloValue(const GValue *input, Value *output) const {
    return d ? d->TransformGriloValue(input, output) : false;
}

template<typename PropertyType, typename ValueType>
bool GenericProperty<PropertyType, ValueType>::
Private::TransformDBusVariant(GVariant *input, Value *output) const {
    const Wrapper<GVariant> unboxed_value = take(g_variant_get_variant(input));

    if (not unboxed_value)
        return false;

    if (not g_variant_is_of_type(unboxed_value.get(),
                                 dbus::Type<ValueType>::signature())) {
        kError(L"Unexpected variant type {1} in D-Bus value for \"{2}\" field")
                  % g_variant_get_type_string(input) % field_name_;
        return false;
    }

    *output = dbus::Type<ValueType>::make_value(unboxed_value.get());

    return true;
}

bool Property::TransformDBusVariant(GVariant *input, Value *output) const {
    return d ? d->TransformDBusVariant(input, output) : false;
}

////////////////////////////////////////////////////////////////////////////////

Property::StreamInfoFunction
Property::bind_any(const StreamInfoFunction &first,
                   const StreamInfoFunction &second) const {
    return bind(&Property::MergeAny, _1, _2, first, second, _3);
}

template<typename ValueType>
Property::StreamInfoFunction Property::
bind_attr(ValueType (*get_attribute)(const GstDiscovererInfo *info)) {
    return bind(&Property::MergeAttribute<ValueType>,
                this, _1, _2, get_attribute, _3);
}

template<typename ValueType, typename InfoType>
Property::StreamInfoFunction Property::
bind_attr(ValueType (*get_attribute)(const InfoType *info)) {
    return bind(&Property::MergeAttribute<ValueType, InfoType>,
                this, _1, _2, get_attribute, _3);
}

template Property::StreamInfoFunction
Property::bind_attr(bool (*)(const GstDiscovererInfo *));

template Property::StreamInfoFunction
Property::bind_attr(int32_t (*)(const GstDiscovererInfo *));

template Property::StreamInfoFunction
Property::bind_attr(int32_t (*)(const GstDiscovererAudioInfo *));

template Property::StreamInfoFunction
Property::bind_attr(Fraction (*)(const GstDiscovererVideoInfo *));

template Property::StreamInfoFunction
Property::bind_attr(int32_t (*)(const GstDiscovererVideoInfo *));

template<typename ValueType>
Property::StreamInfoFunction Property::bind_tag(const char *tag_name) const {
    return bind(&Property::MergeTag<ValueType>, this, _1, _2, tag_name, _3);
}

template Property::StreamInfoFunction
Property::bind_tag<bool>(const char *) const;

template Property::StreamInfoFunction
Property::bind_tag<DateTime>(const char *) const;

template Property::StreamInfoFunction
Property::bind_tag<double>(const char *) const;

template Property::StreamInfoFunction
Property::bind_tag<Fraction>(const char *) const;

template Property::StreamInfoFunction
Property::bind_tag<int32_t>(const char *) const;

template Property::StreamInfoFunction
Property::bind_tag<String>(const char *) const;

template Property::StreamInfoFunction
Property::bind_tag<uint64_t>(const char *) const;

////////////////////////////////////////////////////////////////////////////////

Property::LuceneFields Property::MakeFields(const Value &value) const {
    return d->MakeFields(value);
}

Property::Value Property::TransformFields(const LuceneFields &fields) const {
    return d->TransformFields(fields);
}

Property::Value Property::TransformSingleField
                                         (Lucene::FieldablePtr field) const {
    return d->TransformSingleField(field);
}


namespace internal {

class MakeGriloValueVistor : public boost::static_visitor< Wrapper<GValue> > {
private:
    static GArray *make_value_array(unsigned size) {
        GArray *va = g_array_sized_new(false, false, sizeof(GValue), size);
        g_array_set_clear_func(va, (GDestroyNotify) &g_value_unset);
        return va;
    }

public:
    MakeGriloValueVistor() {
        // clang++ requires a user-provided default constructor
    }

    result_type operator()(const boost::blank &) const {
        return result_type();
    }

    result_type operator()(const String &value) const {
        GValue result = G_VALUE_INIT;
        g_value_init(&result, G_TYPE_STRING);
        g_value_set_string(&result, FromUnicode(value).c_str());
        return wrap(&result);
    }

    result_type operator()(const DateTime &value) const {
        const boost::gregorian::date d = value.date();
        const time_duration t = value.time_of_day();

        typedef boost::rational<time_duration::tick_type> FSeconds;
        const FSeconds fsecs(t.fractional_seconds(), t.ticks_per_second());

        GDateTime *const dt = g_date_time_new_utc
                (d.year(), d.month(), d.day(), t.hours(), t.minutes(),
                 t.seconds() + boost::rational_cast<double>(fsecs));

        GValue result = G_VALUE_INIT;
        g_value_init(&result, G_TYPE_DATE_TIME);
        g_value_take_boxed(&result, dt);
        return wrap(&result);
    }

    result_type operator()(const Fraction &value) const {
        GValue result = G_VALUE_INIT;
        g_value_init(&result, G_TYPE_FLOAT);
        g_value_set_float(&result, boost::rational_cast<double>(value));
        return wrap(&result);
    }

    result_type operator()(bool value) const {
        GValue result = G_VALUE_INIT;
        g_value_init(&result, G_TYPE_BOOLEAN);
        g_value_set_boolean(&result, value);
        return wrap(&result);
    }

    result_type operator()(int value) const {
        GValue result = G_VALUE_INIT;
        g_value_init(&result, G_TYPE_INT);
        g_value_set_int(&result, value);
        return wrap(&result);
    }

    result_type operator()(unsigned value) const {
        GValue result = G_VALUE_INIT;
        g_value_init(&result, G_TYPE_UINT);
        g_value_set_uint(&result, value);
        return wrap(&result);
    }

    result_type operator()(uint64_t value) const {
        GValue result = G_VALUE_INIT;
        g_value_init(&result, G_TYPE_UINT64);
        g_value_set_uint64(&result, value);
        return wrap(&result);
    }

    result_type operator()(double value) const {
        GValue result = G_VALUE_INIT;
        g_value_init(&result, G_TYPE_FLOAT);
        g_value_set_float(&result, value);
        return wrap(&result);
    }
};

} // namespace internal

GValue* Property::MakeGriloValue(const Value &value) const {
    return d->MakeGriloValue(value).release();
}

Wrapper<GValue> Property::Private::MakeGriloValue(const Value &value) const {
    static const internal::MakeGriloValueVistor make_grilo_value_visitor;
    return boost::apply_visitor(make_grilo_value_visitor, value);
}

bool DateTimeProperty::Private::
TransformGriloValue(const GValue *input, Value *output) const {
    if (metadata_key_.gtype() != G_TYPE_STRING)
        return inherited::TransformGriloValue(input, output);

    // https://bugzilla.gnome.org/show_bug.cgi?id=686175
    GTimeVal tv;

    if (not g_time_val_from_iso8601(g_value_get_string(input), &tv))
        return false;

    *output = DateTime(boost::gregorian::date(1970, 1, 1))
            + boost::posix_time::seconds(tv.tv_sec)
            + boost::posix_time::microseconds(tv.tv_usec);

    return true;
}

Wrapper<GValue> DateTimeProperty::Private::
MakeGriloValue(const Value &value) const {
    // https://bugzilla.gnome.org/show_bug.cgi?id=686175
    if (metadata_key_.gtype() != G_TYPE_STRING)
        return inherited::MakeGriloValue(value);

    const DateTime dt = boost::get<DateTime>(value);
    const std::wstring str = boost::posix_time::to_iso_extended_wstring(dt);
    return inherited::MakeGriloValue(str + L"Z");
}

////////////////////////////////////////////////////////////////////////////////

Property Property::FromFieldName(const String &name) {
    const FieldNameCache::const_iterator it = field_name_cache_.find(name);

    if (it != field_name_cache_.end())
        return Property(it->second);

    return Property();
}

static bool register_metadata_key(const Property &p) {
    // Querying the metadata key registers it lazily.
    p.metadata_key();
    return false;
}

Property Property::FromMetadataKey(GrlKeyID key) {
    // Metadata keys are registered lazily.
    // So we better make sure we know them all before telling crap.
    if (metadata_key_cache_.size() != field_name_cache_.size())
        VisitAll(boost::bind(&register_metadata_key, _1));

    const MetadataKeyCache::const_iterator it = metadata_key_cache_.find(key);

    if (it != metadata_key_cache_.end())
        return Property(it->second);

    return Property();
}

void Property::VisitAll(const PropertyVisitor &visit) {
    for (const auto &p: field_name_cache_) {
        if (visit(Property(p.second)))
            break;
    }
}

////////////////////////////////////////////////////////////////////////////////

static String date_time_string(const Property::Value &value) {
    return Lucene::DateTools::dateToString
            (boost::get<DateTimeProperty::value_type>(value),
             Lucene::DateTools::RESOLUTION_MILLISECOND);
}

template<> Lucene::FieldablePtr DateTimeProperty::inherited::
Private::MakeSingleField(const Value &value) const {
    return newLucene<Lucene::Field>
            (field_name_, date_time_string(value), Lucene::Field::STORE_YES,
             Lucene::Field::INDEX_ANALYZED_NO_NORMS);
}

template<> Lucene::FieldablePtr NumericProperty<Fraction>::
Private::MakeSingleField(const Value &value) const {
    const Fraction fraction = boost::get<value_type>(value);
    const double double_value = boost::rational_cast<double>(fraction);
    std::wostringstream string_value;

    // Boost C++ only defines a << operator for ostream, but not for wostream.
    // Sadly the stream operator still works, apparently by applying an unsafe
    // typecast. See <https://svn.boost.org/trac/boost/ticket/7187>.
    string_value << Lucene::NumericUtils::doubleToPrefixCoded(double_value)
                 << L' ' << fraction.numerator()
                 << L'/' << fraction.denominator();

    return newLucene<Lucene::Field>
            (field_name_, string_value.str(), Lucene::Field::STORE_YES,
             Lucene::Field::INDEX_ANALYZED_NO_NORMS);
}

template<> Lucene::FieldablePtr NumericProperty<bool>::
Private::MakeSingleField(const Value &value) const {
    return newLucene<Lucene::NumericField>
            (field_name_, Lucene::Field::STORE_YES,
             Lucene::Field::INDEX_ANALYZED_NO_NORMS)->
            setIntValue(boost::get<value_type>(value) ? 1 : 0);
}

template<> Lucene::FieldablePtr NumericProperty<double>::
Private::MakeSingleField(const Value &value) const {
    return newLucene<Lucene::NumericField>
            (field_name_, Lucene::Field::STORE_YES,
             Lucene::Field::INDEX_ANALYZED_NO_NORMS)->
            setDoubleValue(boost::get<value_type>(value));
}

template<> Lucene::FieldablePtr NumericProperty<uint64_t>::
Private::MakeSingleField(const Value &value) const {
    // FIXME(M3): Add unsigned integer support to Lucene++
    return newLucene<Lucene::NumericField>
            (field_name_, Lucene::Field::STORE_YES,
             Lucene::Field::INDEX_ANALYZED_NO_NORMS)->
            setLongValue(boost::get<value_type>(value));
}

template<> Lucene::FieldablePtr NumericProperty<int32_t>::
Private::MakeSingleField(const Value &value) const {
    return newLucene<Lucene::NumericField>
            (field_name_, Lucene::Field::STORE_YES,
             Lucene::Field::INDEX_ANALYZED_NO_NORMS)->
            setIntValue(boost::get<value_type>(value));
}

template<> Lucene::FieldablePtr StringProperty::
Private::MakeSingleField(const Value &value) const {
    return newLucene<Lucene::Field>
            (field_name_, boost::get<value_type>(value),
             Lucene::Field::STORE_YES, Lucene::Field::INDEX_NOT_ANALYZED);
}

template<> Property::LuceneFields TextProperty::
Private::MakeFields(const Value &value) const {
    LuceneFields fields = LuceneFields::newInstance();

    fields.add(newLucene<Lucene::Field>(field_name_ + L"$",
                                        boost::get<value_type>(value),
                                        Lucene::Field::STORE_NO,
                                        Lucene::Field::INDEX_NOT_ANALYZED));
    fields.add(newLucene<Lucene::Field>(field_name_,
                                        boost::get<value_type>(value),
                                        Lucene::Field::STORE_YES,
                                        Lucene::Field::INDEX_ANALYZED));

    return fields;
}

template<typename PropertyType, typename ValueType>
Property::LuceneFields GenericProperty<PropertyType, ValueType>::
Private::MakeFields(const Value &value) const {
    return Lucene::newCollection(MakeSingleField(value));
}

////////////////////////////////////////////////////////////////////////////////

template<> Property::Value DateTimeProperty::inherited::
Private::TransformSingleField(Lucene::FieldablePtr field) const {
    return Lucene::DateTools::stringToDate(field->stringValue());
}

template<> Property::Value NumericProperty<bool>::
Private::TransformSingleField(Lucene::FieldablePtr field) const {
    const String value = Lucene::StringUtils::toLower(field->stringValue());
    return static_cast<bool>(value != L"0" && value != L"false");
}

template<> Property::Value NumericProperty<double>::
Private::TransformSingleField(Lucene::FieldablePtr field) const {
    return Lucene::StringUtils::toDouble(field->stringValue());
}

template<> Property::Value NumericProperty<Fraction>::
Private::TransformSingleField(Lucene::FieldablePtr field) const {
    const String string_value = field->stringValue();
    const String::size_type i = string_value.rfind(L' ');

    if (i == String::npos)
        return value_type();

    std::wistringstream tokenizer(string_value);
    tokenizer.seekg(i + 1);

    // Boost C++ doesn't provide a >> operator for wide streams.
    // See <https://svn.boost.org/trac/boost/ticket/7187>.
    value_type::int_type numerator;
    tokenizer >> numerator;

    if (tokenizer.get() != L'/')
        tokenizer.clear(std::wistringstream::badbit);

    value_type::int_type denominator;
    tokenizer >> denominator;

    if (tokenizer.fail())
        return value_type();

    return value_type(numerator, denominator);
}

template<> Property::Value NumericProperty<int32_t>::
Private::TransformSingleField(Lucene::FieldablePtr field) const {
    return static_cast<value_type>
            (Lucene::StringUtils::toInt(field->stringValue()));
}

template<> Property::Value NumericProperty<uint64_t>::
Private::TransformSingleField(Lucene::FieldablePtr field) const {
    // FIXME(M3): Add unsigned integer support to Lucene++
    return static_cast<value_type>
            (Lucene::StringUtils::toLong(field->stringValue()));
}

template<> Property::Value StringProperty::
Private::TransformSingleField(Lucene::FieldablePtr field) const {
    return field->stringValue();
}

template<> Property::Value TextProperty::
Private::TransformSingleField(Lucene::FieldablePtr field) const {
    return field->stringValue();
}

template<typename PropertyType, typename ValueType>
Property::Value GenericProperty<PropertyType, ValueType>::
Private::TransformFields(const LuceneFields &fields) const {
    if (fields.empty() || *fields.begin() == 0)
        return typename GenericProperty<PropertyType, ValueType>::value_type();

    return TransformSingleField(*fields.begin());
}

////////////////////////////////////////////////////////////////////////////////

DateTimeProperty::
DateTimeProperty(const String &field_name,
                 const MetadataKey &metadata_key,
                 Category category,
                 MergeStrategy merge_strategy,
                 const StreamInfoFunction &stream_info)
    : inherited(new Private(field_name, metadata_key, category,
                            merge_strategy, stream_info)) {
}

DateTimeProperty::DateTimeProperty(Private *impl)
    : inherited(impl) {
}

template<typename T> NumericProperty<T>::
NumericProperty(const String &field_name,
                const Property::MetadataKey &metadata_key,
                Property::Category category,
                Property::MergeStrategy merge_strategy,
                const Property::StreamInfoFunction &stream_info)
    : inherited(new typename inherited::
                Private(field_name, metadata_key, category,
                        merge_strategy, stream_info)) {
}

template<typename T> NumericProperty<T>::
NumericProperty(typename inherited::Private *impl)
    : inherited(impl) {
}

template class NumericProperty<bool>;
template class NumericProperty<double>;
template class NumericProperty<Fraction>;
template class NumericProperty<int32_t>;
template class NumericProperty<uint64_t>;

template<FullTextSearchMode fts> GenericStringProperty<fts>::
GenericStringProperty(const String &field_name,
                      const Property::MetadataKey &metadata_key,
                      Property::Category category,
                      Property::MergeStrategy merge_strategy,
                      const Property::StreamInfoFunction &stream_info)
    : inherited(new typename inherited::
                Private(field_name, metadata_key, category,
                        merge_strategy, stream_info)) {
}

template<FullTextSearchMode fts> GenericStringProperty<fts>::
GenericStringProperty(typename inherited::Private *impl)
    : inherited(impl) {
}

template class GenericStringProperty<DisableFullTextSearch>;
template class GenericStringProperty<EnableFullTextSearch>;

////////////////////////////////////////////////////////////////////////////////

template<> bool TextProperty::Private::supports_full_text_search() const {
    return true;
}

template<typename P, typename V>
bool GenericProperty<P, V>::Private::supports_full_text_search() const {
    return false;
}

////////////////////////////////////////////////////////////////////////////////

template<> Lucene::QueryPtr NumericProperty<bool>::
Private::MakeRangeQuery(const Value &lower_value,
                        Boundary lower_boundary,
                        const Value &upper_value,
                        Boundary upper_boundary) const {
    return Lucene::NumericRangeQuery::newIntRange
            (field_name_,
             boost::get<value_type>(lower_value) ? 1 : 0,
             boost::get<value_type>(upper_value) ? 1 : 0,
             lower_boundary == Property::Inclusive,
             upper_boundary == Property::Inclusive);
}

template<> Lucene::QueryPtr NumericProperty<double>::
Private::MakeRangeQuery(const Value &lower_value,
                        Boundary lower_boundary,
                        const Value &upper_value,
                        Boundary upper_boundary) const {
    return Lucene::NumericRangeQuery::newDoubleRange
            (field_name_,
             boost::get<value_type>(lower_value),
             boost::get<value_type>(upper_value),
             lower_boundary == Property::Inclusive,
             upper_boundary == Property::Inclusive);
}

template<> Lucene::QueryPtr NumericProperty<Fraction>::
Private::MakeRangeQuery(const Value &lower_value,
                        Boundary lower_boundary,
                        const Value &upper_value,
                        Boundary upper_boundary) const {
    return Lucene::NumericRangeQuery::newDoubleRange
            (field_name_,
             boost::rational_cast<double>(boost::get<value_type>(lower_value)),
             boost::rational_cast<double>(boost::get<value_type>(upper_value)),
             lower_boundary == Property::Inclusive,
             upper_boundary == Property::Inclusive);
}

template<> Lucene::QueryPtr NumericProperty<int32_t>::
Private::MakeRangeQuery(const Value &lower_value,
                        Boundary lower_boundary,
                        const Value &upper_value,
                        Boundary upper_boundary) const {
    return Lucene::NumericRangeQuery::newIntRange
            (field_name_,
             boost::get<value_type>(lower_value),
             boost::get<value_type>(upper_value),
             lower_boundary == Property::Inclusive,
             upper_boundary == Property::Inclusive);
}

template<> Lucene::QueryPtr NumericProperty<uint64_t>::
Private::MakeRangeQuery(const Value &lower_value,
                        Boundary lower_boundary,
                        const Value &upper_value,
                        Boundary upper_boundary) const {
    // FIXME(M5): unsigned numbers in Lucene++
    return Lucene::NumericRangeQuery::newLongRange
            (field_name_,
             boost::get<value_type>(lower_value),
             boost::get<value_type>(upper_value),
             lower_boundary == Property::Inclusive,
             upper_boundary == Property::Inclusive);
}

template<> Lucene::QueryPtr DateTimeProperty::inherited::
Private::MakeRangeQuery(const Value &lower_value,
                        Boundary lower_boundary,
                        const Value &upper_value,
                        Boundary upper_boundary) const {
    return newLucene<Lucene::TermRangeQuery>
            (field_name_,
             date_time_string(lower_value),
             date_time_string(upper_value),
             lower_boundary == Property::Inclusive,
             upper_boundary == Property::Inclusive);
}

template<> Lucene::QueryPtr GenericStringProperty<DisableFullTextSearch>::
Private::MakeRangeQuery(const Value &lower_value,
                        Boundary lower_boundary,
                        const Value &upper_value,
                        Boundary upper_boundary) const {
    return newLucene<Lucene::TermRangeQuery>
            (field_name_,
             boost::get<value_type>(lower_value),
             boost::get<value_type>(upper_value),
             lower_boundary == Property::Inclusive,
             upper_boundary == Property::Inclusive);
}

template<> Lucene::QueryPtr GenericStringProperty<EnableFullTextSearch>::
Private::MakeRangeQuery(const Value &lower_value,
                        Boundary lower_boundary,
                        const Value &upper_value,
                        Boundary upper_boundary) const {
    return newLucene<Lucene::TermRangeQuery>
            (field_name_,
             boost::get<value_type>(lower_value),
             boost::get<value_type>(upper_value),
             lower_boundary == Property::Inclusive,
             upper_boundary == Property::Inclusive);
}

Lucene::QueryPtr Property::MakeRangeQuery(const Value &lower_value,
                                          Boundary lower_boundary,
                                          const Value &upper_value,
                                          Boundary upper_boundary) const {
    return d ? d->MakeRangeQuery(lower_value, lower_boundary,
                                 upper_value, upper_boundary)
             : Lucene::QueryPtr();
}

////////////////////////////////////////////////////////////////////////////////

static Lucene::QueryPtr MakePhraseQuery(Lucene::FieldablePtr field,
                                        Lucene::AnalyzerPtr analyzer) {
    Lucene::TokenStreamPtr stream = field->tokenStreamValue();

    if (not stream) {
        const Lucene::StringReaderPtr text =
                newLucene<Lucene::StringReader>(field->stringValue());

        try {
            stream = analyzer->reusableTokenStream(field->name(), text);
            stream->reset();
        } catch(Lucene::IOException &) {
            stream = analyzer->tokenStream(field->name(), text);
        }
    }

    const Lucene::PhraseQueryPtr query = newLucene<Lucene::PhraseQuery>();

    if (const Lucene::TermAttributePtr term_attr =
            stream->getAttribute<Lucene::TermAttribute>()) {
        while (stream->incrementToken()) {
            query->add(newLucene<Lucene::Term>(field->name(),
                                               term_attr->term()));
        }
    }

    if (query->getTerms().empty())
        return Lucene::QueryPtr();

    return query;
}

static Lucene::QueryPtr MakeFieldQuery(Lucene::FieldablePtr field,
                                       Lucene::AnalyzerPtr analyzer) {
    Lucene::QueryPtr query;

    if (field->isTokenized())
        query = MakePhraseQuery(field, analyzer);

    if (not query) {
        const Lucene::TermPtr term =
                newLucene<Lucene::Term>(field->name(), field->stringValue());
        query = newLucene<Lucene::TermQuery>(term);
    }

    return query;
}

template<> Lucene::QueryPtr NumericProperty<bool>::
Private::MakeTermQuery(const Property::Value &value) const {
    return MakeRangeQuery(value, Property::Inclusive,
                          value, Property::Inclusive);
}

template<> Lucene::QueryPtr NumericProperty<double>::
Private::MakeTermQuery(const Property::Value &value) const {
    return MakeRangeQuery(value, Property::Inclusive,
                          value, Property::Inclusive);
}

template<> Lucene::QueryPtr NumericProperty<Fraction>::
Private::MakeTermQuery(const Property::Value &value) const {
    return MakeRangeQuery(value, Property::Inclusive,
                          value, Property::Inclusive);
}

template<> Lucene::QueryPtr NumericProperty<int32_t>::
Private::MakeTermQuery(const Property::Value &value) const {
    return MakeRangeQuery(value, Property::Inclusive,
                          value, Property::Inclusive);
}

template<> Lucene::QueryPtr NumericProperty<uint64_t>::
Private::MakeTermQuery(const Property::Value &value) const {
    return MakeRangeQuery(value, Property::Inclusive,
                          value, Property::Inclusive);
}

template<typename PropertyType, typename ValueType>
Lucene::QueryPtr GenericProperty<PropertyType, ValueType>::
Private::MakeTermQuery(const Value &value) const {
    const Property::LuceneFields fields = MakeFields(value);
    const Lucene::AnalyzerPtr analyzer =
            newLucene<Lucene::WhitespaceAnalyzer>();

    if (fields.empty())
        return Lucene::QueryPtr();

    if (fields.size() == 1 || ends_with((*fields.begin())->name(), L"$"))
        return MakeFieldQuery(*fields.begin(), analyzer);

    const Lucene::BooleanQueryPtr query = newLucene<Lucene::BooleanQuery>();

    for (const Lucene::FieldablePtr &f: fields)
        query->add(MakeFieldQuery(f, analyzer), Lucene::BooleanClause::MUST);

    return query;
}

Lucene::QueryPtr Property::MakeTermQuery(const Value &value) const {
    return d ? d->MakeTermQuery(value) : Lucene::QueryPtr();
}

} // namespace mediascanner
