/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/commitpolicy.h"

// Boost C++
#include <boost/bind.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>

// C++ Standard Library
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/glibutils.h"
#include "mediascanner/writablemediaindex.h"

namespace mediascanner {

CommitPolicy::~CommitPolicy() {
}

CommitPolicyPtr CommitPolicy::default_policy() {
    // TODO(M5): Decide on proper default commit policy
    static const CommitPolicyPtr default_policy(new InstantCommitPolicy);
    return default_policy;
}

bool InstantCommitPolicy::OnCreate(const std::vector<std::wstring> &,
                                   WritableMediaIndex *media_index) {
    media_index->CommitPendingChanges();
    return true;
}

bool InstantCommitPolicy::OnUpdate(const std::vector<std::wstring> &,
                                   WritableMediaIndex *media_index) {
    media_index->CommitPendingChanges();
    return true;
}

bool InstantCommitPolicy::OnRemove(const std::vector<std::wstring> &,
                                   WritableMediaIndex *media_index) {
    media_index->CommitPendingChanges();
    return true;
}

DelayedCommitPolicy::DelayedCommitPolicy()
    : maximum_batch_size_(default_maximum_batch_size())
    , maximum_delay_(default_maximum_delay()) {
}

DelayedCommitPolicy::~DelayedCommitPolicy() {
    DelayedCommitMap::iterator it = delayed_commits_.begin();

    while (it != delayed_commits_.end()) {
        it->second->cancel_timeout();
        it->first->CommitPendingChanges();
        delayed_commits_.erase(it++);
    }
}

unsigned DelayedCommitPolicy::default_maximum_batch_size() {
    return 10;
}

void DelayedCommitPolicy::set_maximum_batch_size(unsigned batch_size) {
    maximum_batch_size_ = batch_size;
    UpdateDelayedCommits();
}

unsigned DelayedCommitPolicy::maximum_batch_size() const {
    return maximum_batch_size_;
}

DelayedCommitPolicy::Duration DelayedCommitPolicy::default_maximum_delay() {
    return boost::posix_time::seconds(5);
}

void DelayedCommitPolicy::set_maximum_delay(Duration delay) {
    maximum_delay_ = delay;
    UpdateDelayedCommits();
}

DelayedCommitPolicy::Duration DelayedCommitPolicy::maximium_delay() const {
    return maximum_delay_;
}

bool DelayedCommitPolicy::OnCreate(const std::vector<std::wstring> &media_urls,
                                   WritableMediaIndex *media_index) {
    return PushDelayedCommit(media_urls, media_index);
}

bool DelayedCommitPolicy::OnUpdate(const std::vector<std::wstring> &media_urls,
                                   WritableMediaIndex *media_index) {
    return PushDelayedCommit(media_urls, media_index);
}

bool DelayedCommitPolicy::OnRemove(const std::vector<std::wstring> &media_urls,
                                   WritableMediaIndex *media_index) {
    return PushDelayedCommit(media_urls, media_index);
}

void DelayedCommitPolicy::UpdateDelayedCommits() {
    const TimeStamp now = Clock::universal_time();
    DelayedCommitMap::iterator it = delayed_commits_.begin();

    while (it != delayed_commits_.end()) {
        WritableMediaIndex *const media_index = it->first;
        const DelayedCommitPtr delayed_commit = it->second;
        delayed_commit->cancel_timeout();

        if (delayed_commit->is_due(this, now)) {
            media_index->CommitPendingChanges();
            delayed_commits_.erase(it++);
        } else {
            delayed_commit->reset_timeout(this);
            ++it;
        }
    }
}

bool DelayedCommitPolicy::PushDelayedCommit
                                (const std::vector<std::wstring> &media_urls,
                                 WritableMediaIndex *media_index) {
    DelayedCommitMap::iterator it = delayed_commits_.find(media_index);

    if (it == delayed_commits_.end()) {
        DelayedCommitPtr commit(new DelayedCommit(this, media_index));
        it = delayed_commits_.insert(std::make_pair(media_index, commit)).first;
    } else {
        it->second->increase_pressure(media_urls.size());
    }

    if (not it->second->is_due(this, Clock::universal_time()))
        return false;

    it->second->cancel_timeout();
    media_index->CommitPendingChanges();
    delayed_commits_.erase(it);

    return true;
}

DelayedCommitPolicy::DelayedCommit::DelayedCommit(DelayedCommitPolicy *policy,
                                                  WritableMediaIndex *index)
    : media_index_(index)
    , due_time_(Clock::universal_time() + policy->maximium_delay())
    , num_commits_(0)
    , timeout_id_(0) {
    reset_timeout(policy);
}

DelayedCommitPolicy::DelayedCommit::~DelayedCommit() {
    cancel_timeout();
}

bool DelayedCommitPolicy::DelayedCommit::is_due(DelayedCommitPolicy *policy,
                                                TimeStamp t) const {
    return due_time_ >= t || num_commits_ >= policy->maximum_batch_size();
}

void DelayedCommitPolicy::DelayedCommit::cancel_timeout() {
    if (timeout_id_) {
        Source::Remove(timeout_id_);
        timeout_id_ = 0;
    }
}

void DelayedCommitPolicy::DelayedCommit::reset_timeout
                                            (DelayedCommitPolicy *policy) {
    cancel_timeout();

    const Source::OneCallFunction commit_function =
            boost::bind(&WritableMediaIndex::CommitPendingChanges,
                        media_index_);

    due_time_ = Clock::universal_time() + policy->maximium_delay();
    timeout_id_ = Timeout::AddOnce(policy->maximium_delay(), commit_function);
}

void DelayedCommitPolicy::DelayedCommit::increase_pressure(size_t amount) {
    num_commits_ += amount;
}

} // namespace mediascanner
