/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: James Henstridge <james.henstridge@canonical.com>
 */

#include <boost/bind.hpp>
#include <grilo.h>
#include <gtest/gtest.h>

#include <string>
#include <vector>

#include "mediascanner/mediautils.h"
#include "mediascanner/metadataresolver.h"
#include "mediascanner/propertyschema.h"

#include "testlib/environments.h"
#include "testlib/testutils.h"

namespace mediascanner {
namespace metadataresolvertest {

class MetadataResolverTest : public testing::Test {
public:
    void StoreMetadata(const std::wstring &url, const MediaInfo &metadata) {
        resolved_metadata = metadata;
        if (loop_) {
            g_main_loop_quit(loop_.get());
            loop_.reset();
        }
    }

    void on_timeout() {
        if (loop_) {
            g_main_loop_quit(loop_.get());
            timeout_reached_ = true;
            loop_.reset();
        }
    }

    void WaitForFinished(Timeout::Duration duration) {
        loop_ = take(g_main_loop_new(null_ptr, false));
        timeout_reached_ = false;
        Timeout::AddOnce(duration,
                         boost::bind(&MetadataResolverTest::on_timeout, this));
        g_main_loop_run(loop_.get());
        ASSERT_FALSE(timeout_reached_);
    }

protected:
    Wrapper<GMainLoop> loop_;
    bool timeout_reached_;
    MediaInfo resolved_metadata;
};

TEST_F(MetadataResolverTest, ResolveVideo) {
    MetadataResolver resolver;
    ASSERT_TRUE(resolver.SetupSources(std::vector<std::string>()
                                      << std::string("grl-tmdb")
                                      << std::string("grl-lastfm-albumart")));

    MediaInfo metadata;
    metadata.add_single(schema::kMimeType.bind_value(L"video/x-awesome"));
    metadata.add_single(schema::kTitle.bind_value(L"BigBuckBunny"));
    resolver.push("http://blah", metadata,
                  boost::bind(&MetadataResolverTest::StoreMetadata,
                              this, _1, _2));

    WaitForFinished(boost::posix_time::seconds(15));

    EXPECT_EQ(L"Big Buck Bunny", resolved_metadata.first(schema::kTitle));
    EXPECT_EQ(L"10378", resolved_metadata.first(schema::kTmdbId));
}

TEST_F(MetadataResolverTest, ResolveAudio) {
    MetadataResolver resolver;
    ASSERT_TRUE(resolver.SetupSources(std::vector<std::string>()
                                      << std::string("grl-tmdb")
                                      << std::string("grl-lastfm-albumart")));

    MediaInfo metadata;
    metadata.add_single(schema::kMimeType.bind_value(L"audio/x-awesome"));
    metadata.add_single(schema::kTitle.bind_value(L"Yo Mama"));
    metadata.add_single(schema::kArtist.bind_value(L"Butterfingers"));
    metadata.add_single(schema::kAlbum.bind_value(L"Breakfast at Fatboys"));
    resolver.push("http://blah", metadata,
                  boost::bind(&MetadataResolverTest::StoreMetadata,
                              this, _1, _2));

    WaitForFinished(boost::posix_time::seconds(15));

    EXPECT_EQ(L"http://userserve-ak.last.fm/serve/500/18717421.jpg",
              resolved_metadata.first(schema::kCover));
}

void SetupTestEnvironments() {
    using testing::AddGlobalTestEnvironment;

    AddGlobalTestEnvironment(new TheMovieDbEnvironment(REQUEST_FILE));
    AddGlobalTestEnvironment(new LastFmEnvironment(REQUEST_FILE));
}
}
}

int main(int argc, char **argv) {
    mediascanner::InitTests(&argc, argv);
    grl_init(&argc, &argv);
    mediascanner::metadataresolvertest::SetupTestEnvironments();
    return RUN_ALL_TESTS();
}
