/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_PROPERTYPRIVATE_H
#define MEDIASCANNER_PROPERTYPRIVATE_H

// Media Scanner Library
#include "mediascanner/property.h"

namespace mediascanner {

class Property::Private {
    friend class Property;

public:
    Private(const String &field_name,
            const MetadataKey &metadata_key,
            Category category,
            MergeStrategy merge_strategy,
            const StreamInfoFunction &merge_stream_info)
        : field_name_(field_name)
        , metadata_key_(metadata_key)
        , category_(category)
        , merge_strategy_(merge_strategy)
        , merge_stream_info_(merge_stream_info) {
    }

    virtual ~Private() {
    }

protected:
    virtual LuceneFields MakeFields(const Value &value) const = 0;
    virtual Value TransformFields(const LuceneFields &fields) const = 0;

    virtual Lucene::QueryPtr MakeTermQuery(const Value &value) const = 0;
    virtual Lucene::QueryPtr MakeRangeQuery(const Value &lower_value,
                                            Boundary lower_boundary,
                                            const Value &upper_value,
                                            Boundary upper_boundary)
                                            const = 0;

    virtual bool TransformGriloValue(const GValue *input,
                                     Value *output) const = 0;

    virtual bool TransformDBusVariant(GVariant *input,
                                      Value *output) const = 0;

    virtual Wrapper<GValue> MakeGriloValue(const Value &value) const;

    virtual bool supports_full_text_search() const = 0;

protected:
    /// Value of Property::field_name().
    const String field_name_;

    /// Value of Property::metadata_key().
    MetadataKey metadata_key_;

    /// Value of Property::category().
    Category category_;

    /// How to resolve merge conflicts for this property.
    MergeStrategy merge_strategy_;

    /// Function for resolving stream information.
    StreamInfoFunction merge_stream_info_;
};

template<typename PropertyType, typename ValueType>
class GenericProperty<PropertyType, ValueType>::Private
        : public Property::Private {
public:
    typedef Property::Private inhertied;

    Private(const String &field_name,
            const MetadataKey &metadata_key,
            Category category,
            MergeStrategy merge_strategy,
            const StreamInfoFunction &merge_stream_info)
        : inhertied(field_name, metadata_key, category,
                    merge_strategy, merge_stream_info) {
    }

protected:
    LuceneFields MakeFields(const Value &value) const;
    Lucene::FieldablePtr MakeSingleField(const Value &value) const;

    Value TransformFields(const LuceneFields &fields) const;
    Value TransformSingleField(Lucene::FieldablePtr field) const;

    Lucene::QueryPtr MakeTermQuery(const Value &value) const;
    Lucene::QueryPtr MakeRangeQuery(const Value &lower_value,
                                    Boundary lower_boundary,
                                    const Value &upper_value,
                                    Boundary upper_boundary) const;

    bool TransformGriloValue(const GValue *input, Value *output) const;
    virtual bool TransformSafeGriloValue(const GValue *input,
                                         Value *output) const;
    bool TransformDBusVariant(GVariant *input, Value *output) const;

    bool supports_full_text_search() const;
};

class DateTimeProperty::Private
        : public DateTimeProperty::inherited::Private {
public:
    typedef DateTimeProperty::inherited::Private inherited;

    Private(const String &field_name,
            const MetadataKey &metadata_key,
            Category category,
            MergeStrategy merge_strategy,
            const StreamInfoFunction &merge_stream_info)
        : inherited(field_name, metadata_key, category,
                    merge_strategy, merge_stream_info) {
    }

    bool TransformGriloValue(const GValue *input, Value *output) const;
    Wrapper<GValue> MakeGriloValue(const Value &value) const;
};

} // namespace mediascanner

#endif // MEDIASCANNER_PROPERTYPRIVATE_H
