/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_MEDIAINDEX_H
#define MEDIASCANNER_MEDIAINDEX_H

// C++ Standard Library
#include <set>
#include <string>

// Media Scanner Library
#include "mediascanner/declarations.h"
#include "mediascanner/mediautils.h"
#include "mediascanner/property.h"
#include "mediascanner/refreshpolicy.h"

namespace mediascanner {

/**
 * @brief A read-only instance of the media index.
 *
 * Use set_refresh_policy() to control when changes are
 * read from the disk.
 *
 * @see WritableMediaIndex for a writable instance.
 */
class MediaIndex {
    NONCOPYABLE(MediaIndex);
    class Collector;
    class Private;

public:
    /**
     * @brief Pass this constant as @p limit argument to methods such as Query(),
     * or VisitAll() to request all available results for those methods.
     */
    static const int32_t kUnlimited;

    static const char kMediaIndexFormat[];
    static const char kMetaIndexFormat[];

    /**
     * @brief Signature of the visitor function passed to methods like
     * Query(), or VisitAll().
     * @param metadata The metadata of the current item.
     * @param remaining_items The number of remaining items.
     */
    typedef boost::function<void(const MediaInfo &metadata,
                                 int32_t remaining_items)> ItemVistor;

    explicit MediaIndex(MediaRootManagerPtr root_manager);
    virtual ~MediaIndex();

    /**
     * @brief The error message of the last failed operation.
     */
    std::string error_message();

    /**
     * @brief The default that is used when path() is empty.
     * @see Open()
     */
    static FileSystemPath default_path();

    /**
     * @brief The file system path of the index, or an empty string to use.
     * the default_path().
     * @see Open()
     */
    FileSystemPath path() const;

    /**
     * @brief Predicate indicating weither the index was opened successfully.
     * @see Open()
     */
    bool is_open() const;

    /**
     * @brief Indicator if the index still reads current data, or if Refresh()
     * should be called.
     */
    bool is_current() const;

    /**
     * @brief Changes the current RefreshPolicy of the media index.
     * @see refresh_policy()
     */
    void set_refresh_policy(RefreshPolicyPtr policy);

    /**
     * @brief The current RefreshPolicy of the media index.
     * @see set_refresh_policy()
     */
    RefreshPolicyPtr refresh_policy() const;

    /**
     * @brief The MediaRootManager of the media index.
     */
    MediaRootManagerPtr root_manager() const;

    /**
     * @brief Opens the media index at @p path. This method must be called
     * before performing any read operations.
     * @param path The file system path of the index, or an empty string to use
     * the default_path().
     * @return @c true on success.
     * @see Close(), is_open), path(), default_path()
     */
    virtual bool Open(const FileSystemPath &path);

    /**
     * @brief Closes the media index. After calling this method no other
     * queries are permitted.
     * @see Open()
     */
    virtual void Close();

    /**
     * @brief Refreshs the media index if it doesn't reflect recent changes
     * anymore. Such a method is needed because, to optimize performance,
     * Lucene++ doesn't automatically reload its index readers upon changes.
     * @see RefreshPolicy, is_current()
     */
    virtual bool Reopen();

    void AddMediaRoot(const MediaRoot &root);
    void RemoveMediaRoot(const MediaRoot &root);

    /**
     * @brief Checks if the media index contains information about the media
     * referenced by @p url.
     * @param url The URL of the media to check
     * @return @c true if the index contains information about @p url
     */
    bool Exists(const std::wstring &url);

    /**
     * @brief Retrieves all information stored about the referenced media.
     * @param url The URL of the media to lookup.
     * @return A key-value map of the stored information.
     */
    MediaInfo Lookup(const std::wstring &url);

    /**
     * @brief Retrieves the information stored in the index for the media
     * referenced by @p url, as requested in @p field_names.
     * @param url The URL of the media to lookup.
     * @param field_names The fields to lookup.
     * @return A key-value map of the stored information.
     */
    MediaInfo Lookup(const std::wstring &url, const Property::Set &fields);

    /**
     * @brief Looks up a single detail about the referenced media.
     * @param url The URL of the media to lookup.
     * @param key The field to lookup.
     * @return The stored information, or a default constructed value.
     */
    template<typename PropertyType, typename ValueType>
    ValueType Lookup(const std::wstring &url,
                     const GenericProperty<PropertyType, ValueType> &key);

    /**
     * @brief Visits all items stored in the media index.
     * @param visit_item The function to call for each item.
     * @param limit The maximum number of items to return, or kUnlimited
     * @param offset Number of items to skip.
     * @see Query
     */
    void VisitAll(const ItemVistor &visit_item,
                  int32_t limit = kUnlimited, int32_t offset = 0);

    /**
     * @brief Visits all items matching the @p filter.
     * @param visit_item The function to call for each item.
     * @param filter The filter to apply to all items.
     * @param limit The maximum number of items to return, or kUnlimited
     * @param offset Number of items to skip.
     * @param @p true on success, check error_message() on error.
     * @see VisitAll, error_message()
     */
    bool Query(const ItemVistor &visit_item, const Filter &filter,
               int32_t limit = kUnlimited, int32_t offset = 0);

protected:
    Lucene::AnalyzerPtr analyzer() const;

    void report_error(const boost::locale::format &error_message);

    static Lucene::TermPtr MakeLookupTerm(const std::wstring &url);
    Lucene::DocumentPtr FindDocument(const std::wstring &url) const;

    static Property::Set GetFields(Lucene::DocumentPtr document);

    static MediaInfo ExtractProperties(Lucene::DocumentPtr document);
    static MediaInfo ExtractProperties(Lucene::DocumentPtr document,
                                       const Property::Set &fields);

    bool Lookup(const std::wstring &url, const Property &key,
                Property::Value *value);

    virtual bool ReadParams();
    virtual bool FlushParams(Wrapper<GKeyFile> params);
    bool FlushParams();

    virtual Lucene::IndexReaderPtr OpenIndex();
    virtual Lucene::IndexReaderPtr OpenChildIndex(const MediaRoot &root);

    class ParamId;

    static const ParamId kParamFormat;
    static const ParamId kParamSegments;
    static const ParamId kParamVolumeUUID;
    static const ParamId kParamRelativePath;

    std::string get_param(const std::string &group,
                          const ParamId &key) const;
    void set_param(const std::string &group,
                   const ParamId &key,
                   const std::string &value);

    FileSystemPath params_path() const;

private:
    Lucene::IndexReaderPtr OpenIndexReader(const FileSystemPath &path);
    Lucene::IndexReaderPtr OpenMetaIndex();

    Private *const d;
};

template<typename P, typename V>
inline V MediaIndex::Lookup(const std::wstring &url,
                            const GenericProperty<P, V> &key) {
    Property::Set fields;
    fields.insert(key);

    return Lookup(url, fields).first<P, V>(key);
}

} // namespace mediascanner

#endif // MEDIASCANNER_MEDIAINDEX_H
