/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/writablemediaindex.h"

// Lucene++
#include <Lucene.h>
#include <Document.h>
#include <Field.h>
#include <FSDirectory.h>
#include <IndexReader.h>
#include <IndexWriter.h>
#include <InfoStream.h>
#include <Lock.h>
#include <NativeFSLockFactory.h>
#include <TermDocs.h>

// Boost C++
#include <boost/algorithm/string.hpp>
#include <boost/filesystem.hpp>
#include <boost/uuid/uuid_generators.hpp>
#include <boost/uuid/uuid_io.hpp>

// C++ Standard Library
#include <map>
#include <string>
#include <vector>

// Media Scanner
#include "mediascanner/commitpolicy.h"
#include "mediascanner/glibutils.h"
#include "mediascanner/locale.h"
#include "mediascanner/logging.h"
#include "mediascanner/mediaroot.h"
#include "mediascanner/mediautils.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/utilities.h"

namespace mediascanner {

// Boost C++
using boost::locale::format;

// Lucene++
using Lucene::LuceneException;
using Lucene::newLucene;

// Context specific logging domains
static const logging::Domain kError("error/index", logging::error());
static const logging::Domain kInfo("info/index", logging::info());
static const logging::Domain kDebug("debug/index", logging::debug());
static const logging::Domain kTrace("trace/index", logging::trace());
static const logging::Domain kTraceLucene("trace/index/lucene",
                                          logging::Domain::Disabled |
                                          logging::Domain::Explicit,
                                          &kTrace);

namespace internal {

class LoggingInfoStream : public Lucene::InfoStream {
public:
    Lucene::InfoStream& operator<<(const String& text) {
        kTraceLucene(L"{1}") % text;
        return *this;
    }
};

} // namespace internal

class WritableMediaIndex::Private {
    friend class WritableMediaIndex;

    Private()
        : write_lock_count_(0)
        , write_lock_timeout_(Lucene::IndexWriter::WRITE_LOCK_TIMEOUT) {
    }

    Lucene::IndexWriterPtr find_writer(const std::wstring &url) const;

    bool obtain_write_lock(const FileSystemPath &path,
                           WritableMediaIndex *index) {
        if (not write_lock_) {
            const std::wstring w_path = ToUnicode(path.string());

            const Lucene::LockFactoryPtr lf =
                    newLucene<Lucene::NativeFSLockFactory>(w_path);

            kDebug("Obtaining write lock for \"{1}\".") % path;
            write_lock_ = lf->makeLock(Lucene::IndexWriter::WRITE_LOCK_NAME);

            if (not write_lock_->obtain(write_lock_timeout_)) {
                index->report_error(format("Cannot obtain write lock for "
                                           "\"{1}\" within permitted time.")
                                    % path);
                return false;
            }
        }

        ++write_lock_count_;
        return true;
    }

    void release_write_lock() {
        if (not write_lock_) {
            kError("Cannot release non-existent write lock.");
            return;
        }

        if (--write_lock_count_ == 0) {
            kDebug("Releasing write lock.");
            write_lock_->release();
            write_lock_.reset();
        }
    }

    typedef std::map<std::wstring, Lucene::IndexWriterPtr> IndexWriterMap;
    IndexWriterMap writers_;

    CommitPolicyPtr commit_policy_;

    unsigned write_lock_count_;
    Lucene::LockPtr write_lock_;
    int64_t write_lock_timeout_;
};

WritableMediaIndex::WritableMediaIndex(MediaRootManagerPtr root_manager)
    : MediaIndex(root_manager)
    , d(new Private) {
    set_commit_policy(CommitPolicy::default_policy());
}

WritableMediaIndex::~WritableMediaIndex() {
    delete d;
}

void WritableMediaIndex::set_commit_policy(CommitPolicyPtr policy) {
    if (not policy)
        policy = CommitPolicy::default_policy();

    if (d->commit_policy_ != policy && d->commit_policy_)
        CommitPendingChanges();

    d->commit_policy_ = policy;
}

CommitPolicyPtr WritableMediaIndex::commit_policy() const {
    return d->commit_policy_;
}

void WritableMediaIndex::set_write_lock_timeout(int64_t timeout) {
    d->write_lock_timeout_ = timeout;

    for (const auto &w: d->writers_) {
        w.second->setWriteLockTimeout(timeout);
    }
}

int64_t WritableMediaIndex::write_lock_timeout() const {
    return d->write_lock_timeout_;
}

bool WritableMediaIndex::Open(const FileSystemPath &path) {
    if (not d->obtain_write_lock(path, this))
        return false;

    const bool success = MediaIndex::Open(path);
    d->release_write_lock();
    return success;
}

// NOTE: About WritableMediaIndex::Reopen(): Currently there is no need for
// reopening the writer since write conflicts are handled internally by Lucene++
// using lock files. Therefore reopening the reader is sufficient.

void WritableMediaIndex::Close() {
    for (const auto &w: d->writers_) {
        const FileSystemPath index_path = path();
        kDebug("Closing media index index for \"{1}\" at \"{2}\".")
                % w.first % index_path;
        w.second->close();
    }

    d->writers_.clear();
    MediaIndex::Close();
}

static std::string make_uuid() {
    boost::uuids::random_generator make_uuid;
    std::ostringstream oss;
    oss << make_uuid();
    return oss.str();
}

bool WritableMediaIndex::ReadParams() {
    const FileSystemPath base_path = params_path().parent_path();

    if (not d->obtain_write_lock(base_path.string(), this))
        return false;

    const bool success = ReadParamsUnlocked();
    d->release_write_lock();
    return success;
}

bool WritableMediaIndex::ReadParamsUnlocked() {
    const FileSystemPath info_path = params_path();
    const FileSystemPath base_path = info_path.parent_path();

    if (not boost::filesystem::is_directory(base_path)
            && not boost::filesystem::create_directories(base_path)) {
        report_error(format("Cannot create index base directory \"{1}\".")
                     % base_path);
        return false;
    }

    if (not boost::filesystem::is_regular_file(info_path)) {
        kDebug("Writing initial settings for \"{1}\".") % base_path;

        std::ostringstream oss;

        oss << "[global]" << std::endl
            << "format = " << kMetaIndexFormat << std::endl;

        Wrapper<GError> error;
        if (not g_file_set_contents(info_path.string().c_str(),
                                    oss.str().data(), oss.str().length(),
                                    error.out_param())) {
            const std::string message = to_string(error);
            report_error(format("Cannot write initial settings: {1}.")
                         % message);
            return false;
        }
    }

    if (not MediaIndex::ReadParams())
        return false;

    const std::string data_dirname = get_param("global", kParamSegments);
    const FileSystemPath data_path = base_path / data_dirname;

    if (not boost::filesystem::is_directory(data_path)
            && not boost::filesystem::create_directories(data_path)) {
        report_error(format("Cannot create index data directory \"{1}\".")
                     % data_path);
        return false;
    }

    return true;
}

bool WritableMediaIndex::FlushParams(Wrapper<GKeyFile> params) {
    const FileSystemPath info_path = params_path();
    const FileSystemPath base_path = info_path.parent_path();
    kDebug("Flushing settings for \"{1}\".") % base_path;

    if (not d->obtain_write_lock(base_path.string(), this))
        return false;

    size_t length;
    Wrapper<GError> error;

    const Wrapper<char> data =
            take(g_key_file_to_data(params.get(), &length, error.out_param()));
    const bool success =
            data && g_file_set_contents(info_path.string().c_str(),
                                        data.get(), length, error.out_param());

    if (not success) {
        const std::string message = to_string(error);
        report_error(format("Cannot write settings for \"{1}\": {2}")
                     % base_path % message);
    }

    d->release_write_lock();
    return success;
}

Lucene::IndexReaderPtr WritableMediaIndex::OpenIndex() {
    return OpenChildIndex(root_manager()->make_root(path().string()));
}

Lucene::IndexReaderPtr WritableMediaIndex::OpenChildIndex
                                                    (const MediaRoot &root) {
    const FileSystemPath root_path = root.path();
    const FileSystemPath index_path = path();
    kInfo("Opening child index for \"{1}\" at \"{2}\"")
            % root_path % index_path;

    if (not root.is_valid()) {
        report_error(format("Cannot open child index for invalid media root"));
        return Lucene::IndexReaderPtr();
    }

    if (not d->obtain_write_lock(path(), this))
        return Lucene::IndexReaderPtr();

    const std::wstring w_media_root = ToUnicode(root.path());
    const Private::IndexWriterMap::const_iterator it =
            d->writers_.find(w_media_root);

    Lucene::IndexReaderPtr reader;

    if (it != d->writers_.end()) {
        reader = it->second->getReader();
    } else {
        std::string index_dirname = get_param(root.group_id(), kParamSegments);

        if (index_dirname.empty()) {
            index_dirname = make_uuid();

            // Store child index parameters. We explicitly store the volume
            // UUID and the relative path, although this information also is
            // used to build the group id, and therefore interested modules
            // could retrieve that information by parsing the group id. Still
            // such an approach would be rather fragile and therefore should
            // be avoided.
            const std::string group_id = root.group_id();
            set_param(group_id, kParamSegments, index_dirname);
            set_param(group_id, kParamRelativePath, root.relative_path());
        }

        const FileSystemPath index_path = path() / index_dirname;
        kDebug("Creating new child index for \"{1}\" at \"{2}\"")
                % w_media_root % index_path;

        try {
            const Lucene::FSDirectoryPtr directory =
                    Lucene::FSDirectory::open(ToUnicode(index_path.string()));
            const Lucene::IndexWriterPtr writer = newLucene<Lucene::IndexWriter>
                    (directory, analyzer(), Lucene::IndexWriter::
                     MaxFieldLengthUNLIMITED);

            writer->setWriteLockTimeout(write_lock_timeout());

            // NOTE: This is a bit static, but Lucene++'s logging is extremely
            // verbose and I don't think we'll have to enable it at runtime.
            if (kTraceLucene.enabled()) {
                using internal::LoggingInfoStream;
                writer->setInfoStream(newLucene<LoggingInfoStream>());
            }

            d->writers_.insert(std::make_pair(w_media_root, writer));
            reader = writer->getReader();
        } catch(const LuceneException &ex) {
            const std::string message = FromUnicode(ex.getError());
            report_error(format("Cannot open index writer: {1}") % message);
        }
    }

    d->release_write_lock();

    return reader;
}

void WritableMediaIndex::CommitPendingChanges() {
    for (const auto &w: d->writers_) {
        const FileSystemPath index_path = path();
        kTrace("Writing changes for \"{1}\" to \"{2}\".")
            % w.first % index_path;
        w.second->commit();
    }
}

bool WritableMediaIndex::Insert(const std::wstring &url,
                                const MediaInfo &metadata) {
    kTrace(L"Inserting for <{1}>") % url;

    const Lucene::IndexWriterPtr writer = find_index_writer(url);

    if (not writer) {
        report_error(format("This media index doesn't cover <{1}>.") % url);
        return false;
    }

    refresh_policy()->OnBeginWriting(this);

    const Lucene::IndexReaderPtr reader = writer->getReader();
    const Lucene::TermPtr url_term = MakeLookupTerm(url);
    const Lucene::TermDocsPtr url_docs = reader->termDocs(url_term);
    const bool document_exists = url_docs->next();

    Lucene::DocumentPtr document;

    if (document_exists)
        document = reader->document(url_docs->doc());

    if (not document) {
        document = newLucene<Lucene::Document>();
        document->add(newLucene<Lucene::Field>
                            (schema::kUrl.field_name(), url,
                             Lucene::Field::STORE_YES,
                             Lucene::Field::INDEX_NOT_ANALYZED));
    }

    // FIXME(M3): Really deal with related properties.
    for (const Property::ValueMap &properties: metadata) {
        for (const auto &p: properties) {
            // FIXME(M3): Check for "writable" attribute instead
            if (p.first == schema::kUrl)
                continue;

            const String field_name = p.first.field_name();
            kTrace(L" - {1}: {2}") % field_name % p.second;
            document->removeFields(field_name);

            for (const Lucene::FieldablePtr field:
                           p.first.MakeFields(p.second)) {
                document->add(field);
            }
        }
    }

    if (document_exists) {
        kTrace(" - updating document");
        writer->updateDocument(url_term, document);
        d->commit_policy_->OnUpdate(std::vector<std::wstring>() << url, this);
    } else {
        kTrace(" - creating document");
        writer->addDocument(document);
        d->commit_policy_->OnCreate(std::vector<std::wstring>() << url, this);
    }

    return true;
}

bool WritableMediaIndex::Delete(const std::wstring &url) {
    kTrace(L"Deleting <{1}>...") % url;

    const Lucene::IndexWriterPtr writer = find_index_writer(url);

    if (not writer) {
        report_error(format("This media index doesn't cover <{1}>.") % url);
        return false;
    }

    refresh_policy()->OnBeginWriting(this);
    writer->deleteDocuments(MakeLookupTerm(url));
    d->commit_policy_->OnRemove(std::vector<std::wstring>() << url, this);

    return true;
}

Lucene::IndexWriterPtr WritableMediaIndex::find_index_writer
                                                    (const std::wstring &url) {
    if (not boost::algorithm::starts_with(url, L"file://")) {
        report_error(format("Unsupported URL: <{1}>") % url);
        return Lucene::IndexWriterPtr();
    }

    boost::filesystem::wpath path = url.substr(strlen("file://"));

    while (not path.empty()) {
        const Private::IndexWriterMap::iterator it =
                d->writers_.find(path.wstring());

        if (it != d->writers_.end()) {
            kTrace(L"Using writer for \"{1}\"") % path;
            return it->second;
        }

        path = path.parent_path();
    }

    return Lucene::IndexWriterPtr();
}

} // namespace mediascanner
