/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/metadataresolver.h"

// Boost C++
#include <boost/bind.hpp>
#include <boost/regex.hpp>

// C++ Standard Library
#include <list>
#include <map>
#include <string>
#include <utility>
#include <vector>
#include <memory>

// Media Scanner Library
#include "mediascanner/glibutils.h"
#include "mediascanner/logging.h"
#include "mediascanner/mediautils.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/utilities.h"

namespace mediascanner {

// Context specific logging domains
static const logging::Domain kDebug("debug/metadata", logging::debug());
static const logging::Domain kWarning("warning/metadata", logging::warning());
static const logging::Domain kInfo("info/metadata", logging::info());
static const logging::Domain kError("error/metadata", logging::error());

static GList *make_media_id_key_list() {
    GList *key_list = NULL;
    GrlKeyID key;

    if ((key = schema::kImdbId.metadata_key().id()) != 0)
        key_list = g_list_prepend(key_list, GRLKEYID_TO_POINTER(key));
    if ((key = schema::kTmdbId.metadata_key().id()) != 0)
        key_list = g_list_prepend(key_list, GRLKEYID_TO_POINTER(key));

    return key_list;
}

static GList *make_title_key_list() {
    return grl_metadata_key_list_new(GRL_METADATA_KEY_TITLE,
                                     GRL_METADATA_KEY_INVALID);
}

class MetadataResolver::Private {
public:
    struct ResolveData {
        ResolveData(Private *d,
                    const std::string &url,
                    const MediaInfo &metadata,
                    const StoreFunction &store_metadata)
            : d(d)
            , url(url)
            , metadata(metadata)
            , store_metadata(store_metadata) {
        }

        Private *const d;
        const std::string url;
        MediaInfo metadata;
        const StoreFunction store_metadata;
    };

    typedef std::shared_ptr<ResolveData> ResolveDataPtr;
    static ResolveDataPtr take_resolve_data(gpointer user_data);

    Private()
        : options_(take(grl_operation_options_new(null_ptr)))
        , media_id_key_list_(take(make_media_id_key_list()))
        , title_key_list_(take(make_title_key_list()))
        , is_canceled_(false)
        , last_push_id_(0)
        , cancel_id_(0) {
        g_mutex_init(&mutex_);
        g_cond_init(&cond_);
    }

    ~Private() {
        cancel();
    }

    bool verify_supported_keys(GrlSource *source);

    bool is_idle() const {
        return pending_pushes_.empty() && current_ops_.empty();
    }

    void push(unsigned push_id, const ResolveDataPtr &data);
    void resolve_media_ids(GrlMedia *media, const ResolveDataPtr &data);
    void resolve_details(GrlMedia *media, const ResolveDataPtr &data);

    void cancel();

    static void OnResolveMediaIds(GrlSource *source, unsigned opid,
                                  GrlMedia *media, void *user_data,
                                  const GError *error);
    static void OnResolveDetails(GrlSource *source, unsigned opid,
                                 GrlMedia *media, void *user_data,
                                 const GError *error);

    std::vector<Wrapper<GrlSource> > available_sources_;
    Wrapper<GrlOperationOptions> options_;
    std::vector<unsigned> current_ops_;
    const Wrapper<GList> media_id_key_list_;
    const Wrapper<GList> title_key_list_;
    volatile bool is_canceled_;

    typedef std::map<unsigned, unsigned> PendingPushesMap;
    PendingPushesMap pending_pushes_;
    unsigned last_push_id_;
    unsigned cancel_id_;

    GMutex mutex_;
    GCond cond_;
};

MetadataResolver::MetadataResolver()
    : d(new Private) {
}

MetadataResolver::~MetadataResolver() {
    delete d;
}

bool MetadataResolver::Private::verify_supported_keys(GrlSource *source) {
    const GList *supported_keys = grl_source_supported_keys(source);

    // Check if any of the media-id keys is supported.
    while (supported_keys) {
        for (const GList *l = media_id_key_list_.get(); l; l = l->next) {
            if (supported_keys->data == l->data)
                return true;
        }

        supported_keys = supported_keys->next;
    }

    return false;
}

bool MetadataResolver::SetupSources(const std::vector<std::string> &sources) {
    const Wrapper<GrlRegistry> registry = wrap(grl_registry_get_default());

    d->available_sources_.clear();

    for (const std::string &id: sources) {
        const Wrapper<GrlSource> source =
                wrap(grl_registry_lookup_source(registry.get(), id.c_str()));

        if (not source) {
            kWarning("Unsupported metadata source \"{1}\".") % id;
            continue;
        }

        if (not d->verify_supported_keys(source.get())) {
            kWarning("Metadata source \"{1}\" doesn't support "
                     "the required metadata keys.") % id;
            continue;
        }

        GrlPlugin *const plugin = grl_source_get_plugin(source.get());

        kInfo("Using \"{1}\" from \"{2}\" to resolve metadata.")
                % id % grl_plugin_get_filename(plugin);

        d->available_sources_.push_back(source);
    }

    return d->available_sources_.size() == sources.size();
}

void MetadataResolver::cancel() {
    g_mutex_lock(&d->mutex_);

    if (d->cancel_id_ == 0) {
        d->cancel_id_ = Idle::AddOnce(boost::bind(&Private::cancel, d),
                                      G_PRIORITY_HIGH_IDLE);
    }

    g_mutex_unlock(&d->mutex_);
}

void MetadataResolver::push(const std::string &url, const MediaInfo &metadata,
                            const StoreFunction &store_metadata) {
    g_mutex_lock(&d->mutex_);

    const Private::ResolveDataPtr data
            (new Private::ResolveData(d, url, metadata, store_metadata));

    const unsigned push_id = ++d->last_push_id_;
    const unsigned idle_id =  Idle::AddOnce(boost::bind(&Private::push,
                                                        d, push_id, data));

    d->pending_pushes_.insert(std::make_pair(push_id, idle_id));

    g_mutex_unlock(&d->mutex_);
}

void MetadataResolver::Private::cancel() {
    g_mutex_lock(&mutex_);

    if (cancel_id_) {
        Source::Remove(cancel_id_);
        cancel_id_ = 0;
    }

    is_canceled_ = true;

    for (const auto p: pending_pushes_) {
        Source::Remove(p.second);
    }

    for (const unsigned opid: current_ops_) {
        kDebug("Canceling resolver task {1}") % opid;
        grl_operation_cancel(opid);
    }

    pending_pushes_.clear();
    current_ops_.clear();

    g_mutex_unlock(&mutex_);
}

void MetadataResolver::Private::push(unsigned push_id,
                                     const ResolveDataPtr &data) {
    const Wrapper<GrlMedia> media =
            take(data->metadata.make_media(title_key_list_.get(), data->url));

    g_mutex_lock(&mutex_);

    resolve_media_ids(media.get(), data);

    const size_t removals = pending_pushes_.erase(push_id);
    BOOST_ASSERT(removals == 1);

    g_mutex_unlock(&mutex_);
}

void MetadataResolver::Private::resolve_media_ids(GrlMedia *media,
                                                  const ResolveDataPtr &data) {
    for (Wrapper<GrlSource> source: available_sources_) {
        const unsigned opid = grl_source_resolve
                (source.get(), media, media_id_key_list_.get(),
                 options_.get(), &Private::OnResolveMediaIds,
                 new ResolveDataPtr(data));

        const std::string media_url = safe_string(grl_media_get_url(media));
        kDebug("Pushing resolver task {1}@{2} for media ids of <{3}>")
                % opid % grl_source_get_id(source.get()) % media_url;

        current_ops_.push_back(opid);
    }
}

void MetadataResolver::Private::resolve_details(GrlMedia *media,
                                                const ResolveDataPtr &data) {
    for (Wrapper<GrlSource> source: available_sources_) {
        const unsigned next_opid = grl_source_resolve
                (source.get(), media,
                 grl_source_supported_keys(source.get()),
                 options_.get(), &Private::OnResolveDetails,
                 new ResolveDataPtr(data));

        const std::string media_url = safe_string(grl_media_get_url(media));
        kDebug("Pushing resolver task {1}@{2} for details of <{3}>")
                % next_opid % grl_source_get_id(source.get()) % media_url;

        current_ops_.push_back(next_opid);
    }
}

MetadataResolver::Private::ResolveDataPtr
MetadataResolver::Private::take_resolve_data(gpointer user_data) {
    ResolveDataPtr *const data = static_cast<ResolveDataPtr *>(user_data);
    std::unique_ptr<ResolveDataPtr> deleter(data);
    return ResolveDataPtr(*deleter);
}

template<typename T>
static bool erase(const T &val, std::vector<T> *vec) {
    typedef typename std::vector<T>::iterator iterator;
    const iterator it = std::find(vec->begin(), vec->end(), val);

    if (it == vec->end())
        return false;

    vec->erase(it);
    return true;
}

void MetadataResolver::Private::OnResolveMediaIds
                                        (GrlSource *source, unsigned opid,
                                         GrlMedia *media, void *user_data,
                                         const GError *error) {
    const ResolveDataPtr data = take_resolve_data(user_data);
    Private *const d = data->d;
    bool media_id_found = false;
    bool retry = false;

    for (const GList *l = d->media_id_key_list_; l; l = l->next) {
        if (grl_data_has_key(GRL_DATA(media), GRLPOINTER_TO_KEYID(l->data))) {
            grl_data_remove(GRL_DATA(media), GRL_METADATA_KEY_TITLE);
            media_id_found = true;
            break;
        }
    }

    if (not media_id_found) {
        static const boost::regex::flag_type
                rx_flags = (boost::regex_constants::normal |
                            boost::regex_constants::optimize);
        static const boost::regex rx_camel_case("(\\l)(\\u)", rx_flags);

        const std::string original_title =
                safe_string(grl_media_get_title(media));
        const std::string modified_title =
                boost::regex_replace(original_title, rx_camel_case, "\\1 \\2");

        if (original_title.length() < modified_title.length()) {
            grl_data_remove(GRL_DATA(media), GRL_METADATA_KEY_TITLE);
            grl_media_set_title(media, modified_title.c_str());
            retry = true;
        }
    }

    g_mutex_lock(&d->mutex_);

    if (not erase(opid, &d->current_ops_))
        kError("Bad operation id {1} in {2}") % opid % __func__;

    if (not d->is_canceled_) {
        if (error) {
            const std::string error_message = to_string(error);
            kWarning("Metadata resolving task {1}@{2} failed: {3}")
                    % opid % grl_source_get_id(source) % error_message;
        } else if (retry) {
            d->resolve_media_ids(media, data);
        } else {
            d->resolve_details(media, data);
        }
    }

    if (d->is_idle())
        g_cond_signal(&d->cond_);

    g_mutex_unlock(&d->mutex_);
}

void MetadataResolver::Private::OnResolveDetails
                                        (GrlSource *source, unsigned opid,
                                         GrlMedia *media, void *user_data,
                                         const GError *error) {
    const ResolveDataPtr data = take_resolve_data(user_data);
    Private *const d = data->d;

    g_mutex_lock(&d->mutex_);

    if (not erase(opid, &d->current_ops_))
        kError("Bad operation id {1} in {2}") % opid % __func__;

    if (not d->is_canceled_) {
        if (error) {
            const std::string error_message = to_string(error);
            kWarning("Metadata resolving task {1}@{2} failed: {3}")
                    % opid % grl_source_get_id(source) % error_message;
        } else {
            kDebug("Metadata resolving task {1}@{2} finished")
                    % opid % grl_source_get_id(source);

            std::string error_message;

            if (data->metadata.fill_from_media
                    (media, grl_data_get_keys(GRL_DATA(media)),
                     null_ptr, &error_message)) {
                const std::wstring url = safe_wstring(grl_media_get_url(media));
                data->store_metadata(url, data->metadata);
            } else {
                kWarning("Failed to collect metadata for {1}@{2} failed: {3}")
                        % opid % grl_source_get_id(source) % error_message;
            }
        }
    }

    if (d->is_idle())
        g_cond_signal(&d->cond_);

    g_mutex_unlock(&d->mutex_);
}

void MetadataResolver::WaitForFinished() {
    g_mutex_lock(&d->mutex_);

    if (not is_idle()) {
        kInfo("Waiting for pending metadata resolvers.");
        g_cond_wait(&d->cond_, &d->mutex_);
        kInfo("Metadata resolving finished.");
    }

    g_mutex_unlock(&d->mutex_);
}

bool MetadataResolver::is_idle() const {
    return d->is_idle();
}

} // namespace mediascanner
