/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_GLIBUTILS_H
#define MEDIASCANNER_GLIBUTILS_H

// GLib Based Libraries
#include <gio/gio.h>
#include <gio/gunixmounts.h>
#include <grilo.h>
#include <gst/pbutils/gstdiscoverer.h>
#include <gudev/gudev.h>

// Boost C++
#include <boost/function.hpp>
#include <boost/date_time/posix_time/posix_time_duration.hpp>

// C++ Standard Library
#include <string>

// Media Scanner Library
#include "mediascanner/declarations.h"

namespace mediascanner {
namespace internal {

// GObject structure to GType mappings /////////////////////////////////////////

template<typename> GType GetGType();

template<> inline GType GetGType<GArray>() {
    return G_TYPE_ARRAY;
}

template<> inline GType GetGType<GByteArray>() {
    return G_TYPE_BYTE_ARRAY;
}

template<> inline GType GetGType<GClosure>() {
    return G_TYPE_CLOSURE;
}

template<> inline GType GetGType<GDate>() {
    return G_TYPE_DATE;
}

template<> inline GType GetGType<GDateTime>() {
    return G_TYPE_DATE_TIME;
}

template<> inline GType GetGType<GDrive>() {
    return G_TYPE_DRIVE;
}

template<> inline GType GetGType<GError>() {
    return G_TYPE_ERROR;
}

template<> inline GType GetGType<GFile>() {
    return G_TYPE_FILE;
}

template<> inline GType GetGType<GFileMonitor>() {
    return G_TYPE_FILE_MONITOR;
}

template<> inline GType GetGType<GFileType>() {
    return G_TYPE_FILE_TYPE;
}

template<> inline GType GetGType<GHashTable>() {
    return G_TYPE_HASH_TABLE;
}

template<> inline GType GetGType<GKeyFile>() {
    return G_TYPE_KEY_FILE;
}

template<> inline GType GetGType<GIcon>() {
    return G_TYPE_ICON;
}

template<> inline GType GetGType<GMainContext>() {
    return G_TYPE_MAIN_CONTEXT;
}

template<> inline GType GetGType<GMainLoop>() {
    return G_TYPE_MAIN_LOOP;
}

template<> inline GType GetGType<GMatchInfo>() {
    return G_TYPE_MATCH_INFO;
}

template<> inline GType GetGType<GMount>() {
    return G_TYPE_MOUNT;
}

template<> inline GType GetGType<GObject>() {
    return G_TYPE_OBJECT;
}

template<> inline GType GetGType<GPtrArray>() {
    return G_TYPE_PTR_ARRAY;
}

template<> inline GType GetGType<GRegex>() {
    return G_TYPE_REGEX;
}

template<> inline GType GetGType<GSettings>() {
    return G_TYPE_SETTINGS;
}

template<> inline GType GetGType<GSource>() {
    return G_TYPE_SOURCE;
}

template<> inline GType GetGType<GString>() {
    return G_TYPE_GSTRING;
}

template<> inline GType GetGType<GValue>() {
    return G_TYPE_VALUE;
}

template<> inline GType GetGType<GVolume>() {
    return G_TYPE_VOLUME;
}

template<> inline GType GetGType<GVolumeMonitor>() {
    return G_TYPE_VOLUME_MONITOR;
}

template<> inline GType GetGType<GVariantBuilder>() {
    return G_TYPE_VARIANT_BUILDER;
}

template<> inline GType GetGType<GVariantType>() {
    return G_TYPE_VARIANT;
}

template<> inline GType GetGType<GrlCaps>() {
    return GRL_CAPS_TYPE;
}

template<> inline GType GetGType<GrlConfig>() {
    return GRL_TYPE_CONFIG;
}

template<> inline GType GetGType<GrlData>() {
    return GRL_TYPE_DATA;
}

template<> inline GType GetGType<GrlMedia>() {
    return GRL_TYPE_MEDIA;
}

template<> inline GType GetGType<GrlPlugin>() {
    return GRL_TYPE_PLUGIN;
}

template<> inline GType GetGType<GrlRegistry>() {
    return GRL_TYPE_REGISTRY;
}

template<> inline GType GetGType<GrlRelatedKeys>() {
    return GRL_TYPE_RELATED_KEYS;
}

template<> inline GType GetGType<GrlSource>() {
    return GRL_TYPE_SOURCE;
}

template<> inline GType GetGType<GstDiscovererInfo>() {
    return GST_TYPE_DISCOVERER_INFO;
}

template<> inline GType GetGType<GstDiscovererStreamInfo>() {
    return GST_TYPE_DISCOVERER_STREAM_INFO;
}

template<> inline GType GetGType<GstDiscovererAudioInfo>() {
    return GST_TYPE_DISCOVERER_AUDIO_INFO;
}

template<> inline GType GetGType<GstDiscovererContainerInfo>() {
    return GST_TYPE_DISCOVERER_CONTAINER_INFO;
}

template<> inline GType GetGType<GstDiscovererVideoInfo>() {
    return GST_TYPE_DISCOVERER_VIDEO_INFO;
}

template<> inline GType GetGType<GUdevClient>() {
    return G_UDEV_TYPE_CLIENT;
}

template<> inline GType GetGType<GUdevDevice>() {
    return G_UDEV_TYPE_DEVICE;
}

// GLib boxing and GObject reference counting //////////////////////////////////

template<typename Type>
struct CopyHelper {
    static Type *Copy(Type *p);
    static void Free(Type *p);
};

template<typename Type>
struct BoxedCopyHelper {
    static Type *Copy(Type *p) {
        return static_cast<Type *>(g_boxed_copy(GetGType<Type>(), p));
    }

    static void Free(Type *p) {
        g_boxed_free(GetGType<Type>(), p);
    }
};

template<typename Type>
struct MiniObjectCopyHelper {
    static Type *Copy(Type *p) {
        return reinterpret_cast<Type *>
                (gst_mini_object_ref(GST_MINI_OBJECT_CAST(p)));
    }

    static void Free(Type *p) {
        gst_mini_object_unref(GST_MINI_OBJECT_CAST(p));
    }
};

template<typename Type>
struct ObjectCopyHelper {
    static Type *Copy(Type *p) {
        return static_cast<Type *>(g_object_ref(p));
    }

    static void Free(Type *p) {
        g_object_unref(p);
    }
};

template<typename Type, typename List = GList>
struct ListCopyHelper {
    static List *Copy(List *const other) {
        List *const copy = CopyHelper<List>::Copy(other);

        for (List *l = copy; l; l = l->next) {
            Type *const data = static_cast<Type *>(l->data);
            l->data = CopyHelper<Type>::Copy(data);
        }

        return copy;
    }

    static void Free(GList *list) {
        for (List *l = list; l; l = l->next) {
            Type *const data = static_cast<Type *>(l->data);
            CopyHelper<Type>::Free(data);
        }

        CopyHelper<List>::Free(list);
    }
};

template<> struct CopyHelper<GMainLoop> :
        public BoxedCopyHelper<GMainLoop> { };
template<> struct CopyHelper<GMainContext> :
        public BoxedCopyHelper<GMainContext> { };
template<> struct CopyHelper<GValue> :
        public BoxedCopyHelper<GValue> { };

template<> struct CopyHelper<GCancellable> :
        public ObjectCopyHelper<GCancellable> { };
template<> struct CopyHelper<GDBusConnection> :
        public ObjectCopyHelper<GDBusConnection> { };
template<> struct CopyHelper<GDrive> :
        public ObjectCopyHelper<GDrive> { };
template<> struct CopyHelper<GFile> :
        public ObjectCopyHelper<GFile> { };
template<> struct CopyHelper<GFileEnumerator> :
        public ObjectCopyHelper<GFileEnumerator> { };
template<> struct CopyHelper<GFileInfo> :
        public ObjectCopyHelper<GFileInfo> { };
template<> struct CopyHelper<GFileMonitor> :
        public ObjectCopyHelper<GFileMonitor> { };
template<> struct CopyHelper<GIcon> :
        public ObjectCopyHelper<GIcon> { };
template<> struct CopyHelper<GMount> :
        public ObjectCopyHelper<GMount> { };
template<> struct CopyHelper<GSettings> :
        public ObjectCopyHelper<GSettings> { };
template<> struct CopyHelper<GVolume> :
        public ObjectCopyHelper<GVolume> { };
template<> struct CopyHelper<GVolumeMonitor> :
        public ObjectCopyHelper<GVolumeMonitor> { };

template<> struct CopyHelper<GDBusMethodInvocation> :
        public ObjectCopyHelper<GDBusMethodInvocation> { };
template<> struct CopyHelper<GDBusProxy> :
        public ObjectCopyHelper<GDBusProxy> { };

template<> struct CopyHelper<GrlCaps> :
        public ObjectCopyHelper<GrlCaps> { };
template<> struct CopyHelper<GrlConfig> :
        public ObjectCopyHelper<GrlConfig> { };
template<> struct CopyHelper<GrlData> :
        public ObjectCopyHelper<GrlData> { };
template<> struct CopyHelper<GrlMedia> :
        public ObjectCopyHelper<GrlMedia> { };
template<> struct CopyHelper<GrlOperationOptions> :
        public ObjectCopyHelper<GrlOperationOptions> { };
template<> struct CopyHelper<GrlPlugin> :
        public ObjectCopyHelper<GrlPlugin> { };
template<> struct CopyHelper<GrlRegistry> :
        public ObjectCopyHelper<GrlRegistry> { };
template<> struct CopyHelper<GrlRelatedKeys> :
        public ObjectCopyHelper<GrlRelatedKeys> { };
template<> struct CopyHelper<GrlSource> :
        public ObjectCopyHelper<GrlSource> { };

template<> struct CopyHelper<GstCaps> :
        public MiniObjectCopyHelper<GstCaps> { };
template<> struct CopyHelper<GstDiscoverer> :
        public ObjectCopyHelper<GstDiscoverer> { };
template<> struct CopyHelper<GstDiscovererInfo> :
        public ObjectCopyHelper<GstDiscovererInfo> { };
template<> struct CopyHelper<GstDiscovererStreamInfo> :
        public ObjectCopyHelper<GstDiscovererStreamInfo> { };

template<> struct CopyHelper<GUdevClient> :
        public ObjectCopyHelper<GUdevClient> { };
template<> struct CopyHelper<GUdevDevice> :
        public ObjectCopyHelper<GUdevDevice> { };

template<> struct CopyHelper<char> {
    static char *Copy(char *p) { return g_strdup(p); }
    static void Free(char *p) { g_free(p); }
};

template<> struct CopyHelper<char *> {
    static char **Copy(char **p) { return g_strdupv(p); }
    static void Free(char **p) { g_strfreev(p); }
};

template<> struct CopyHelper<GDBusArgInfo> {
    static GDBusArgInfo *Copy(GDBusArgInfo *p) {
        return g_dbus_arg_info_ref(p);
    }

    static void Free(GDBusArgInfo *p) {
        g_dbus_arg_info_unref(p);
    }
};

template<> struct CopyHelper<GDBusInterfaceInfo> {
    static GDBusInterfaceInfo *Copy(GDBusInterfaceInfo *p) {
        return g_dbus_interface_info_ref(p);
    }

    static void Free(GDBusInterfaceInfo *p) {
        g_dbus_interface_info_unref(p);
    }
};

template<> struct CopyHelper<GDBusMethodInfo> {
    static GDBusMethodInfo *Copy(GDBusMethodInfo *p) {
        return g_dbus_method_info_ref(p);
    }

    static void Free(GDBusMethodInfo *p) {
        g_dbus_method_info_unref(p);
    }
};

template<> struct CopyHelper<GDBusPropertyInfo> {
    static GDBusPropertyInfo *Copy(GDBusPropertyInfo *p) {
        return g_dbus_property_info_ref(p);
    }

    static void Free(GDBusPropertyInfo *p) {
        g_dbus_property_info_unref(p);
    }
};

template<> struct CopyHelper<GDBusSignalInfo> {
    static GDBusSignalInfo *Copy(GDBusSignalInfo *p) {
        return g_dbus_signal_info_ref(p);
    }

    static void Free(GDBusSignalInfo *p) {
        g_dbus_signal_info_unref(p);
    }
};

template<> struct CopyHelper<GError> {
    static GError *Copy(GError *p) { return g_error_copy(p); }
    static void Free(GError *p) { g_error_free(p); }
};

template<> struct CopyHelper<GKeyFile> {
    static GKeyFile *Copy(GKeyFile *p) {
        return g_key_file_ref(p);
    }

    static void Free(GKeyFile *p) {
        g_key_file_unref(p);
    }
};

template<> struct CopyHelper<GList> {
    static GList *Copy(GList *p) { return g_list_copy(p); }
    static void Free(GList *p) { g_list_free(p); }
};

template<> struct CopyHelper<GParamSpec> {
    static GParamSpec *Copy(GParamSpec *p) { return g_param_spec_ref(p); }
    static void Free(GParamSpec *p) { g_param_spec_unref(p); }
};

template<> struct CopyHelper<GPtrArray> {
    static GPtrArray *Copy(GPtrArray *p) { return g_ptr_array_ref(p); }
    static void Free(GPtrArray *p) { g_ptr_array_unref(p); }
};

template<> struct CopyHelper<GThread> {
    static GThread *Copy(GThread *p) { return g_thread_ref(p); }
    static void Free(GThread *p) { g_thread_unref(p); }
};

template<> struct CopyHelper<GUnixMountEntry> {
    static void Free(GUnixMountEntry *p) { g_unix_mount_free(p); }
};

template<> struct CopyHelper<GVariant> {
    static GVariant *Copy(GVariant *p) { return g_variant_ref(p); }
    static void Free(GVariant *p) { g_variant_unref(p); }
};

template<> struct CopyHelper<GstStructure> {
    static GstStructure *Copy(GstStructure *p) { return gst_structure_copy(p); }
    static void Free(GstStructure *p) { gst_structure_free(p); }
};

// Cast helpers ////////////////////////////////////////////////////////////////

template<typename SourceType, typename TargetType>
struct CastHelper {
    static TargetType *cast(SourceType *p);
};

template<typename SourceType, typename TargetType>
struct TypeInstanceCastHelper {
    static TargetType *Apply(SourceType *p) {
        if (G_TYPE_CHECK_INSTANCE_TYPE(p, GetGType<TargetType>()))
            return reinterpret_cast<TargetType *>(p);

        return 0;
    }
};

template<typename SourceType>
struct TypeInstanceCastHelper<SourceType, GTypeInstance> {
    static GTypeInstance *Apply(SourceType *p) {
        return reinterpret_cast<GTypeInstance *>(p);
    }
};

template<typename TargetType>
struct CastHelper<GrlMedia, TargetType> :
        public TypeInstanceCastHelper<GrlMedia, TargetType> { };

template<typename TargetType>
struct CastHelper<GrlSource, TargetType> :
        public TypeInstanceCastHelper<GrlSource, TargetType> { };

template<typename TargetType>
struct CastHelper<GstDiscovererStreamInfo, TargetType> :
        public TypeInstanceCastHelper<GstDiscovererStreamInfo, TargetType> { };

} // namespace internal

// Smart pointers for GLib boxed types and GObjects ////////////////////////////

/**
 * @brief A shared smart-pointer for GLib related types. Via its helper classes
 * this smart-pointer supports a wide variety of GLib related types, such as
 * GObjects, GstMiniObjects, boxed types, and even simple GLists.
 * @see internal::CopyHelper, internal::CastHelper
 */
template<typename T, typename CopyHelper>
class Wrapper {
public:
    /**
     * @brief Creates an emtpy pointer not holding any object.
     */
    Wrapper()
        : m_ptr(0) {
    }

    /**
     * @brief Copies another pointer instance. For reference counted objects
     * this new pointer will hold a new reference to the object. For boxed
     * types and similiar a new copy of the original object is created.
     * @param other The instance to initialize from.
     * @see take(), wrap(), wrap_static()
     */
    Wrapper(const Wrapper<T> &other)
        : m_ptr(0) {
        reset(other.get());
    }

    /**
     * @brief Destroys the smart-pointer. For reference counted objects
     * this drops the reference hold, for other types this frees the object
     * instance.
     */
    ~Wrapper() {
        if (m_ptr)
            CopyHelper::Free(m_ptr);
    }

    /**
     * @brief Releases the wrapped object. This detaches the wrapped object
     * from this smart-pointer without dropping the reference or freeing it.
     * Use this function to transfer ownership of the wrapped object.
     * @return A pointer to the wrapped object, or @c null if no object
     * was wrapped.
     */
    T *release() {
        T *const p = m_ptr;
        m_ptr = 0;
        return p;
    }

    /**
     * @brief This function gives access to the wrapped object.
     * @return A pointer to the wrapped object.
     */
    T *get() const {
        return m_ptr;
    }

    /**
     * @brief Creates a new reference to, or a new copy of the wrapped object.
     * This function is useful to initialize C structures or arrays.
     * @return A pointer to the new reference, or the new object.
     */
    T *dup() const {
        return CopyHelper::Copy(get());
    }

    /**
     * @brief This function casts the wrapped object to a different type.
     * @return A pointer to the wrapped object.
     */
    template<typename B> B *get() const {
        return internal::CastHelper<T, B>::Apply(get());
    }

    /**
     * @brief This operator gives access to the wrapped object's members.
     * @return A pointer to the wrapped object.
     */
    T *operator->() const {
        return get();
    }

    /**
     * @brief This function requests to wrap a different object.
     * If the wrapped objects are reference-counted, the reference to the
     * old object is dropped, and a reference to the new object is stored.
     * For other types the old object is freed and a copy of the new object
     * is stored.
     * @param p The new object to store, or @c null.
     * @see take(), out_param()
     */
    void reset(T *p = 0) {
        if (p != m_ptr) {
            if (m_ptr)
                CopyHelper::Free(m_ptr);

            m_ptr = 0;

            if (p)
                m_ptr = CopyHelper::Copy(p);
        }
    }

    /**
     * @brief This function requests to take ownership of a different object.
     * If the wrapped objects are reference-counted, the reference to the
     * old object is dropped, for other types the old object is freed.
     * As this function takes ownership of the passed object, no new reference
     * is created, and no copy is created.
     * @param p The new object to store, or @c null.
     * @see reset(), out_param()
     */
    void take(T *p) {
        if (p != m_ptr) {
            if (m_ptr)
                CopyHelper::Free(m_ptr);

            m_ptr = p;
        }
    }

    /**
     * @brief Resets the smart-pointer and returns a pointer to the internal
     * object pointer. This is useful to wrap objects retreived by output
     * parameters.
     * @return A pointer to the internal object pointer.
     * @see reset(), take()
     */
    T **out_param() {
        reset();
        return &m_ptr;
    }

    /**
     * @brief The assignment operator is an alias of the reset() method.
     * @param p The new object to store, or @c null
     * @return A reference to this smart-pointer.
     */
    Wrapper<T> &operator=(T *p) {
        reset(p);
        return *this;
    }

    /**
     * @brief The assignment operator is an alias of the reset() method.
     * @param p The new object to store, or @c null
     * @return A reference to this smart-pointer.
     */
    Wrapper<T> &operator=(const Wrapper<T> &other) {
        return operator=(other.get());
    }

    /**
     * @brief This operator casts the wrapped object to another type.
     * This cast operator also serves as safe bool cast. The more natural
     * operator bool() is problematic because it leads to unwanted implicit
     * casts. Also note that there is no non-const "operatpr T *()" to avoid
     * ambiguity problems for the compiler.
     */
    operator const T *() const {
        return get();
    }

    /**
     * @brief This operator checks if this pointer actually wraps an object.
     * @return @c true if this pointer wraps an object.
     */
    bool operator!() const {
        return get() == 0;
    }

    /**
     * @brief This operator compares two pointers for equality.
     * @param p The other pointer to compare with.
     * @return @c true if both pointers are equal.
     */
    bool operator==(const T *p) const {
        return m_ptr == p;
    }

    /**
     * @brief This operator compares two pointers for inequality.
     * @param p The other pointer to compare with.
     * @return @c true if both pointers are not equal.
     */
    bool operator!=(const T *p) const {
        return !operator==(p);
    }

private:
    T *m_ptr;
};

template<typename T, typename List = GList>
struct ListWrapper : public Wrapper<List, internal::ListCopyHelper<T> > {
    typedef Wrapper<List, internal::ListCopyHelper<T> > inherited;

    ListWrapper(const Wrapper<List> &other) // NOLINT: runtime/explicit
        : inherited(other) {
    }

    /**
     * @brief Constructs a ListWrapper that directly takes ownership of @list.
     * This constructor is useful since creating a deep copy of a list can be
     * pretty expensive.
     */
    explicit ListWrapper(List *list) {
        inherited::take(list);
    }
};

/**
 * @brief A type-safe function for creating shallow copies of structures.
 * @param T The structure's type.
 * @param p A pointer to the structure to copy.
 * @return A new copy of the structure, as returned by @c g_memdup().
 */
template<typename T>
inline T* shallow_copy(const T *p) {
    return static_cast<T *>(g_memdup(p, sizeof(T)));
}

/**
 * @brief A type-safe function for creating shallow copies of pointer arrays.
 * @param T The array's element type.
 * @param N The number of elements in the array.
 * @param p A pointer to the array to copy.
 * @return A new copy of the array, as returned by @c g_memdup().
 */
template<typename T, size_t N>
static T* array_copy(const T (&p)[N]) {
    return static_cast<T*>(g_memdup(p, sizeof p));
}

/**
 * @brief Wraps a pointer to an object by a shared smart-pointer.
 * This function doesn't take ownership of the object. It therefore @b does
 * increase the object's reference count.
 * @param T The type of the object to wrap.
 * @param p The pointer to the object to wrap.
 * @return A new Wrapper instance that holds a new reference to the object.
 */
template<typename T>
inline Wrapper<T> wrap(T *p) {
    Wrapper<T> wrapped;
    wrapped.reset(p);
    return wrapped;
}

/**
 * @brief Wraps a pointer to an object by a shared smart-pointer.
 * This function takes ownership of the object. It therefore @b doesn't
 * increase the object's reference count.
 * @param T The type of the object to wrap.
 * @param p The pointer to the object to wrap.
 * @return A new Wrapper instance that holds now owns the object.
 */
template<typename T>
inline Wrapper<T> take(T *p) {
    Wrapper<T> owned;
    owned.take(p);
    return owned;
}

/**
 * @brief Wraps a statically allocated structure by a shared smart-pointer.
 * This function creates a shallow copy of the structure and then takes
 * ownership of that copy.
 * @param T The type of the structure to wrap.
 * @param p The pointer to the structure to wrap.
 * @return A new Wrapper instance that holds owns a copy of the structure.
 */
template<typename T>
inline Wrapper<T> wrap_static(const T *p) {
    return take(shallow_copy(p));
}

// GLib event loop integration /////////////////////////////////////////////////

/**
 * @brief Type-safe destroy notifier.
 * This function can be used as destroy notifier for GLib functions.
 * It automatically invokes the proper @c delete operator of @p T.
 * @param T The type of the object to destroy.
 * @param user_data A pointer to the object to destroy.
 */
template<typename T>
inline void DestroyNotify(gpointer user_data) {
    delete static_cast<T *>(user_data);
}

template<typename T>
inline void ClosureNotify(gpointer user_data, GClosure *) {
    delete static_cast<T *>(user_data);
}

/**
 * @brief The Source class provides access to the GLib event source mechanism.
 */
class Source {
public:
    /**
     * @brief The signature of a regular idle source.
     * @return If this function returns @c false it is automatically removed
     * from the list of event sources and will not be called again.
     */
    typedef boost::function<bool()> SourceFunction;

    /**
     * @brief The signature of a single-call idle source.
     * This function is called excactly once. After returning it is
     * automatically removed from the list of event sources and will not be
     * called again.
     */
    typedef boost::function<void()> OneCallFunction;

    /**
     * @brief Removes an event source handler.
     * @param id The source handler's identifier.
     * @return @c True if the identifier was valid and the handler got removed.
     */
    static bool Remove(unsigned id) {
        return g_source_remove(id);
    }

protected:
    static gboolean on_source_function(gpointer data) {
        SourceFunction *const function = static_cast<SourceFunction *>(data);

        if (function)
            return (*function)();

        return false;
    }

    static gboolean on_one_call_function(gpointer data) {
        OneCallFunction *const function = static_cast<OneCallFunction *>(data);

        if (function)
            (*function)();

        return false;
    }
};

/**
 * @brief The Idle class provides access to GLib's idle source mechanism.
 * It manages functions which get called whenever there are no higher priority
 * events pending to the default main loop.
 *
 * @see Timeout
 */
class Idle : public Source {
public:
    /**
     * @brief Installs a regular idle source. This source will be invoked
     * until the @p function returns @c false. After that it be removed
     * automatically from the list of event sources.
     * @param function The function to be called on idle.
     * @param priority The priority of the idle source.
     * @return The identifier (greater than 0) of the event source.
     * @see AddOnce(), Source::Remove()
     */
    static unsigned Add(const SourceFunction &function,
                        int priority = G_PRIORITY_DEFAULT) {
        return g_idle_add_full(priority,
                               &Source::on_source_function,
                               new SourceFunction(function),
                               &DestroyNotify<SourceFunction>);
    }

    /**
     * @brief Installs a single-call idle source. This source will be called
     * exactly once. After that it be removed automatically from the list of
     * event sources.
     * @param function The function to be called on idle.
     * @param priority The priority of the idle source.
     * @return The identifier (greater than 0) of the event source.
     * @see Add(), Source::Remove()
     */
    static unsigned AddOnce(const OneCallFunction &function,
                            int priority = G_PRIORITY_DEFAULT) {
        return g_idle_add_full(priority,
                               &Source::on_one_call_function,
                               new OneCallFunction(function),
                               &DestroyNotify<OneCallFunction>);
    }
};

/**
 * @brief The Timeout class provides access to GLib's timeout mechanism.
 * It manages functions which get called whenever a given time interval
 * has passed.
 *
 * Note that timeout functions might get delayed if the event loop gets
 * blocked by other sources.
 *
 * @see Idle
 */
class Timeout : public Source {
public:
    /**
     * @brief This type describes time durations. Note that the duration's
     * resolution impacts which resolution the timeout source will have.
     */
    typedef boost::posix_time::time_duration Duration;

    /**
     * @brief Adds a new timeout function to the event loop. The timeout
     * gets automatically removed from the list of event sources if @p function
     * returns @c false.
     *
     * Note that the exact timeout mechanism is selected upon the @p interval
     * parameter's resolution. If the resolution is in seconds (or even less
     * granular) timeout sources of seconds precision are created. Otherwise
     * the sources have milliseconds resolution.
     *
     * @param interval The time between calls to the function.
     * @param function The function to be called on idle.
     * @param priority The priority of the idle source.
     * @return The identifier (greater than 0) of the event source.
     * @see AddOnce(), Source::Remove()
     */
    static unsigned Add(Duration interval,
                        const SourceFunction &function,
                        int priority = G_PRIORITY_DEFAULT_IDLE) {
        if (interval.resolution() == boost::date_time::sec) {
            return g_timeout_add_seconds_full(priority,
                                              interval.total_seconds(),
                                              &Source::on_source_function,
                                              new SourceFunction(function),
                                              &DestroyNotify<SourceFunction>);
        }

        return g_timeout_add_full(priority,
                                  interval.total_milliseconds(),
                                  &Source::on_source_function,
                                  new SourceFunction(function),
                                  &DestroyNotify<SourceFunction>);
    }

    /**
     * @brief Adds a new timeout function to the event loop that is called
     * exactly once. After its invocation the timeout gets automatically
     * removed from the list of event sources if @p function returns @c false.
     *
     * Note that the exact timeout mechanism is selected upon the @p interval
     * parameter's resolution. If the resolution is in seconds (or even less
     * granular) timeout sources of seconds precision are created. Otherwise
     * the sources have milliseconds resolution.
     *
     * @param interval The time between calls to the function.
     * @param function The function to be called on idle.
     * @param priority The priority of the idle source.
     * @return The identifier (greater than 0) of the event source.
     * @see AddOnce(), Source::Remove()
     */
    static unsigned AddOnce(Duration interval,
                            const OneCallFunction &function,
                            int priority = G_PRIORITY_DEFAULT_IDLE) {
        if (interval.fractional_seconds() == 0) {
            return g_timeout_add_seconds_full(priority,
                                              interval.total_seconds(),
                                              &Source::on_one_call_function,
                                              new OneCallFunction(function),
                                              &DestroyNotify<OneCallFunction>);
        }

        return g_timeout_add_full(priority,
                                  interval.total_milliseconds(),
                                  &Source::on_one_call_function,
                                  new OneCallFunction(function),
                                  &DestroyNotify<OneCallFunction>);
    }
};

// String conversions //////////////////////////////////////////////////////////

/**
 * @brief Describes a @c GError. The returned string contains
 * the error domain, the error code, and of course the error message.
 * @param error The error to describe, or @c null
 * @return A description of the error if appliable.
 */
std::string to_string(const GError *error);

/**
 * @brief Describes GStreamer capabilities
 * @param caps The capabilities to describe, or @c null
 * @return A describption of the capabilities if appliable
 */
std::string to_string(const GstCaps *caps);

// GParamSpec helpers //////////////////////////////////////////////////////////

/**
 * @brief Creates a @c GParamSpec for boxed types. This function is useful in
 * static variable initialization to defer @c G_TYPE_* calls that could lead
 * to warnings about the missing @c g_type_init() invocation.
 * @param BoxedType G_TYPE_BOXED derived type of this property.
 * @param name Canonical name of the property specified.
 * @param nick Nick name for the property specified.
 * @param blurb Description of the property specified.
 * @param flags Flags for the property specified.
 * @return A newly created parameter specification.
 */
template<typename BoxedType>
inline GParamSpec *MakeParamSpecBoxed(const char *name,
                                      const char *nick,
                                      const char *blurb,
                                      GParamFlags flags) {
    return ::g_param_spec_boxed(name, nick, blurb,
                                internal::GetGType<BoxedType>(), flags);
}

} // namespace mediascanner

#endif // MEDIASCANNER_GLIBUTILS_H
