/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */

// Grilo
#include <grilo.h>

// GStreamer
#include <gst/gstdatetime.h>

// Google Tests
#include <gtest/gtest.h>

// Boost C++
#include <boost/date_time/posix_time/posix_time.hpp>

// Lucene++
#include <Lucene.h>
#include <Fieldable.h>

// C++ Standard Library
#include <string>
#include <utility>
#include <vector>

// Media Scanner Library
#include "mediascanner/glibutils.h"
#include "mediascanner/locale.h"
#include "mediascanner/logging.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/utilities.h"

// Test Suite
#include "testlib/environments.h"
#include "testlib/loggingsink.h"
#include "testlib/testutils.h"

using boost::gregorian::date;
using boost::posix_time::time_duration;

namespace mediascanner {
namespace propertytest {

class PropertyTest : public testing::TestWithParam<const Property *> { };

INSTANTIATE_TEST_CASE_P(GenericProperties, PropertyTest,
                        testing::Values(&schema::kUrl,
                                        &schema::kETag,
                                        &schema::kExternalUrl,
                                        &schema::kAuthor,
                                        &schema::kLicense,
                                        &schema::kDescription,
                                        &schema::kRating,
                                        &schema::kTitle,
                                        &schema::kCertification,
                                        &schema::kComment,
                                        &schema::kKeyword,
                                        &schema::kLanguage));

INSTANTIATE_TEST_CASE_P(FileProperties, PropertyTest,
                        testing::Values(&schema::kMimeType,
                                        &schema::kLastModified,
                                        &schema::kFileSize,
                                        &schema::kLastAccessed));

INSTANTIATE_TEST_CASE_P(MediaProperties, PropertyTest,
                        testing::Values(&schema::kDuration,
                                        &schema::kLastPlayed,
                                        &schema::kPlayCount,
                                        &schema::kLastPosition,
                                        &schema::kProducer,
                                        &schema::kPerformer,
                                        &schema::kCover,
                                        &schema::kPoster));

INSTANTIATE_TEST_CASE_P(AudioProperties, PropertyTest,
                        testing::Values(&schema::kAudioBitRate,
                                        &schema::kAudioCodec,
                                        &schema::kSampleRate,
                                        &schema::kChannelCount,
                                        &schema::kAlbumArtist,
                                        &schema::kComposer,
                                        &schema::kTrackCount,
                                        &schema::kDiscNumber));

INSTANTIATE_TEST_CASE_P(MusicProperties, PropertyTest,
                        testing::Values(&schema::kArtist,
                                        &schema::kAlbum,
                                        &schema::kGenre,
                                        &schema::kLyrics,
                                        &schema::kTrackNumber));

INSTANTIATE_TEST_CASE_P(ImageProperties, PropertyTest,
                        testing::Values(&schema::kWidth,
                                        &schema::kHeight,
                                        &schema::kImageOrientation));

INSTANTIATE_TEST_CASE_P(PhotoProperties, PropertyTest,
                        testing::Values(&schema::kDateTaken,
                                        &schema::kDeviceModel,
                                        &schema::kIsoSpeed,
                                        &schema::kExposureTime,
                                        &schema::kFlashUsed,
                                        &schema::kDeviceManufacturer,
                                        &schema::kFocalRatio,
                                        &schema::kFocalLength,
                                        &schema::kMeteringMode,
                                        &schema::kWhiteBalance));

INSTANTIATE_TEST_CASE_P(VideoProperties, PropertyTest,
                        testing::Values(&schema::kWidth,
                                        &schema::kHeight,
                                        &schema::kFrameRate,
                                        &schema::kShowName,
                                        &schema::kEpisode,
                                        &schema::kSeason,
                                        &schema::kBackdrop,
                                        &schema::kVideoCodec,
                                        &schema::kVideoBitRate,
                                        &schema::kDirector));

TEST(PropertyTest, Copy) {
    Property p = schema::kUrl;
    EXPECT_EQ(p, schema::kUrl);
    EXPECT_NE(p, schema::kETag);

    Property q = schema::kETag;
    EXPECT_EQ(q, schema::kETag);
    EXPECT_NE(q, schema::kUrl);
}

TEST(PropertyTest, Equal) {
    EXPECT_EQ(schema::kUrl, schema::kUrl);
    EXPECT_NE(schema::kUrl, schema::kETag);
}

TEST_P(PropertyTest, Definitions) {
    EXPECT_FALSE(GetParam()->field_name().empty());
    EXPECT_NE(Property::MetadataKey::kInvalid, GetParam()->metadata_key().id());
    EXPECT_NE(Property::MetadataKey::kPending, GetParam()->metadata_key().id());
    EXPECT_EQ(*GetParam(), Property::FromFieldName(GetParam()->field_name()));
}

TEST_P(PropertyTest, GriloGType) {
    LoggingSink sink(logging::warning());
    const Property *const p = GetParam();

    GValue input = G_VALUE_INIT;
    g_value_init(&input, p->metadata_key().gtype());

    if (G_VALUE_TYPE(&input) == G_TYPE_DATE_TIME) {
        GDateTime *const dt = g_date_time_new_utc(2012, 10, 15, 13, 32, 42);
        g_value_take_boxed(&input, dt);
    } else if (p->metadata_key().id() == GRL_METADATA_KEY_LAST_PLAYED) {
        g_value_set_string(&input, "2012-10-15T19:54:12Z");
    }

    Property::Value pv;
    const bool grilo_value_transformed = p->TransformGriloValue(&input, &pv);
    EXPECT_TRUE(grilo_value_transformed);

    Wrapper<GValue> output = take(p->MakeGriloValue(pv));

    ASSERT_TRUE(output);
    EXPECT_EQ(safe_string(g_type_name(p->metadata_key().gtype())),
              safe_string(g_type_name(G_VALUE_TYPE(output.get()))));

    if (G_VALUE_TYPE(&input) == G_TYPE_DATE_TIME) {
        EXPECT_TRUE(g_date_time_equal(g_value_get_boxed(&input),
                                      g_value_get_boxed(output)));
    } else {
        GValue input_text = G_VALUE_INIT;
        g_value_init(&input_text, G_TYPE_STRING);
        EXPECT_TRUE(g_value_transform(&input, &input_text));

        GValue output_text = G_VALUE_INIT;
        g_value_init(&output_text, G_TYPE_STRING);
        EXPECT_TRUE(g_value_transform(output, &output_text));

        EXPECT_EQ(safe_string(g_value_get_string(&input_text)),
                  safe_string(g_value_get_string(&output_text)));
    }

    // Check that no warnings got printed.
    EXPECT_TRUE(sink.get().empty()) << sink.get();
}

TEST(PropertyTest, GetFieldName) {
    EXPECT_EQ(L"url", schema::kUrl.field_name());
}

TEST(PropertyTest, FromFieldName) {
    EXPECT_TRUE(Property::FromFieldName(L"url"));
    EXPECT_TRUE(Property::FromFieldName(std::wstring(L"url")));
    EXPECT_FALSE(Property::FromFieldName(L"nonsense"));

    EXPECT_EQ(schema::kUrl, Property::FromFieldName(L"url"));
}

TEST(PropertyTest, GetMetadataKey) {
    EXPECT_EQ(GRL_METADATA_KEY_URL, schema::kUrl.metadata_key().id());

    EXPECT_TRUE(schema::kETag.metadata_key().is_custom());
    EXPECT_TRUE(schema::kETag.metadata_key().id());
}

TEST(PropertyTest, FromMetadataKey) {
    EXPECT_TRUE(Property::FromMetadataKey(GRL_METADATA_KEY_URL));
    EXPECT_FALSE(Property::FromMetadataKey(GRL_METADATA_KEY_INVALID));

    EXPECT_EQ(schema::kUrl, Property::FromMetadataKey(GRL_METADATA_KEY_URL));
}

static bool is_supported_key(Wrapper<GrlSource> source, GrlKeyID key_id) {
    const GList *const keys = grl_source_supported_keys(source.get());
    return g_list_find(const_cast<GList *>(keys), GRLKEYID_TO_POINTER(key_id));
}

TEST(PropertyTest, MetadataSupport) {
    const Wrapper<GrlRegistry> registry = wrap(grl_registry_get_default());

    const Wrapper<GrlSource> media_scanner = wrap(
            grl_registry_lookup_source(registry.get(), "grl-mediascanner"));
    const Wrapper<GrlSource> tmdb =
            wrap(grl_registry_lookup_source(registry.get(), "grl-tmdb"));

    ASSERT_TRUE(media_scanner);
    ASSERT_TRUE(tmdb);

    for (GrlKeyID key_id = 1;; ++key_id) {
        if (key_id == GRL_METADATA_KEY_CHILDCOUNT
                || key_id == GRL_METADATA_KEY_EXTERNAL_PLAYER
                || key_id == GRL_METADATA_KEY_ID
                || key_id == GRL_METADATA_KEY_SOURCE
                || key_id == GRL_METADATA_KEY_START_TIME
                || key_id == GRL_METADATA_KEY_THUMBNAIL
                || key_id == GRL_METADATA_KEY_THUMBNAIL_BINARY)
            continue;

        const char *const key_name =
                grl_registry_lookup_metadata_key_name(registry.get(), key_id);
        const char *const key_desc =
                grl_registry_lookup_metadata_key_desc(registry.get(), key_id);
        const GType key_type =
                grl_registry_lookup_metadata_key_type(registry.get(), key_id);

        if (not key_name)
            break;

        EXPECT_TRUE(Property::FromMetadataKey(key_id))
                << "Metadata: " << key_name << "(" << g_type_name(key_type)
                << "): " << key_desc << std::endl
                << " Sources:"
                << (is_supported_key(media_scanner, key_id) ?
                    " mediascanner" : "")
                << (is_supported_key(tmdb, key_id) ? " tmdb" : "");
    }
}

////////////////////////////////////////////////////////////////////////////////

class PropertyOrientation
        : public testing::TestWithParam<std::pair<std::wstring, int> > {
};

INSTANTIATE_TEST_CASE_P(, PropertyOrientation,
                        testing::Values(std::make_pair(L"rotate-0", 0),
                                        std::make_pair(L"rotate-179", 179),
                                        std::make_pair(L"rotate-360", 0),
                                        std::make_pair(L"rotate-361", 1)));

TEST_P(PropertyOrientation, Test) {
    LoggingSink sink(logging::warning());
    const std::wstring text = GetParam().first;
    const int rotation = GetParam().second;

    std::wostringstream normal_text;
    normal_text << "rotate-" << rotation;

    Wrapper<GValue> value;

    // check simple parsing
    value = schema::kImageOrientation.MakeGriloValue(text);
    ASSERT_TRUE(value);
    EXPECT_EQ(rotation, g_value_get_int(value));

    // check parsing with flipping
    value = schema::kImageOrientation.MakeGriloValue(L"flip-" + text);
    ASSERT_TRUE(value);
    EXPECT_EQ(rotation, g_value_get_int(value));

    // check simple string conversion
    Property::Value pv;
    ASSERT_TRUE(schema::kImageOrientation.TransformGriloValue(value, &pv));
    EXPECT_EQ(normal_text.str(), boost::get<std::wstring>(pv));

    // check string conversion with bad value
    g_value_set_int(value.get(), g_value_get_int(value) + 360);
    ASSERT_TRUE(schema::kImageOrientation.TransformGriloValue(value, &pv));
    EXPECT_EQ(normal_text.str(), boost::get<std::wstring>(pv));

    // Check that no warnings got printed.
    EXPECT_TRUE(sink.get().empty()) << sink.get();
}

////////////////////////////////////////////////////////////////////////////////

class PropertyValueTest
        : public testing::TestWithParam<std::pair<Property, Property::Value> > {
};

INSTANTIATE_TEST_CASE_P
    (All, PropertyValueTest,
     testing::Values(std::make_pair(schema::kLastModified,
                                    boost::posix_time::second_clock::
                                    local_time()),
                     std::make_pair(schema::kExposureTime,
                                    Fraction(22, 7)),
                     std::make_pair(schema::kFlashUsed, true),
                     std::make_pair(schema::kFlashUsed, false),
                     std::make_pair(schema::kFocalLength, 3.1416),
                     std::make_pair(schema::kFileSize,
                                    static_cast<uint64_t>(0x123456789)),
                     std::make_pair(schema::kRating,
                                    static_cast<float>(5)),
                     std::make_pair(schema::kDuration,
                                    static_cast<int>(6360)),
                     std::make_pair(schema::kUrl,
                                    std::wstring(L"http://www.example.com/")),
                     std::make_pair(schema::kTitle,
                                    std::wstring(L"Test Title"))));

TEST_P(PropertyValueTest, LuceneFieldTrip) {
    const Property property = GetParam().first;
    const Property::Value value = GetParam().second;

    const Property::LuceneFields fields = property.MakeFields(value);
    ASSERT_FALSE(fields.empty());

    for (const Lucene::FieldablePtr &f: fields)
        ASSERT_TRUE(f != nullptr);

    EXPECT_EQ(value, property.TransformFields(fields));
}

////////////////////////////////////////////////////////////////////////////////

TEST(TransformTagValueTest, GstDateTime) {
    const Property property = schema::kDateTaken;

    GValue gvalue = G_VALUE_INIT;
    Property::Value value;
    g_value_init(&gvalue, GST_TYPE_DATE_TIME);

    g_value_take_boxed(&gvalue, gst_date_time_new_y(2013));
    ASSERT_TRUE(property.TransformTagValue<DateTime>(&gvalue, &value));
    ASSERT_EQ(value, DateTime(date(2013, 1, 1), time_duration(0, 0, 0)));

    g_value_take_boxed(&gvalue, gst_date_time_new_ym(2013, 8));
    ASSERT_TRUE(property.TransformTagValue<DateTime>(&gvalue, &value));
    ASSERT_EQ(value, DateTime(date(2013, 8, 1), time_duration(0, 0, 0)));

    g_value_take_boxed(&gvalue, gst_date_time_new_ymd(2013, 8, 19));
    ASSERT_TRUE(property.TransformTagValue<DateTime>(&gvalue, &value));
    ASSERT_EQ(value, DateTime(date(2013, 8, 19), time_duration(0, 0, 0)));

    g_value_take_boxed(&gvalue, gst_date_time_new(0, 2013, 8, 19, 12, 42, -1));
    ASSERT_TRUE(property.TransformTagValue<DateTime>(&gvalue, &value));
    ASSERT_EQ(value, DateTime(date(2013, 8, 19), time_duration(12, 42, 0)));

    g_value_take_boxed(&gvalue, gst_date_time_new(0, 2013, 8, 19, 12, 42, 30));
    ASSERT_TRUE(property.TransformTagValue<DateTime>(&gvalue, &value));
    ASSERT_EQ(value, DateTime(date(2013, 8, 19), time_duration(12, 42, 30)));

    g_value_unset(&gvalue);
}

TEST(TransformTagValueTest, GDate) {
    const Property property = schema::kDateTaken;

    GValue gvalue = G_VALUE_INIT;
    g_value_init(&gvalue, G_TYPE_DATE);
    g_value_take_boxed(&gvalue, g_date_new_dmy(19, G_DATE_AUGUST, 2013));

    Property::Value value;
    ASSERT_TRUE(property.TransformTagValue<DateTime>(&gvalue, &value));
    ASSERT_EQ(value, DateTime(date(2013, 8, 19), time_duration(0, 0, 0)));

    g_value_unset(&gvalue);
}

////////////////////////////////////////////////////////////////////////////////

void SetupTestEnvironments() {
    testing::AddGlobalTestEnvironment
            (new MediaScannerEnvironment(FileSystemPath()));
    testing::AddGlobalTestEnvironment
            (new TheMovieDbEnvironment(std::string()));
}

} // namespace propertytest
} // namespace mediascanner

int main(int argc, char *argv[]) {
    mediascanner::InitTests(&argc, argv);
    grl_init(&argc, &argv);

    mediascanner::propertytest::SetupTestEnvironments();

    return RUN_ALL_TESTS();
}
