/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_TASKMANAGER_H
#define MEDIASCANNER_TASKMANAGER_H

// Boost C++
#include <boost/function.hpp>

// Standard Library
#include <string>
#include <memory>

typedef struct _GError GError;

namespace mediascanner {

// TODO(M5): Move this to the internal namespace?

/**
 * @brief The TaskManager schedules tasks for running them in a background
 * thread. The tasks are ordered by an operation ID. This operation ID can
 * also be used to cancel tasks which are still queued, but not started yet.
 *
 * @see MediaIndexFacade for decorating tasks with MediaIndex management
 * routines.
 */
class TaskManager {
    explicit TaskManager(TaskManager const &) = delete;
    TaskManager& operator=(TaskManager const &) = delete;

    class Private;
    class TaskInfo;

public:
    /**
     * @brief The signature of task functions.
     */
    typedef boost::function<void()> TaskFunction;

public:
    static const unsigned kInstantly = 0;

    explicit TaskManager(const std::string &name);
    ~TaskManager();

    unsigned AppendTask(const TaskFunction &task,
                        unsigned priority = kInstantly) {
        return AppendGroupedTask(priority, task, priority);
    }

    unsigned PrependTask(const TaskFunction &task,
                         unsigned priority = kInstantly) {
        return PrependGroupedTask(priority, task, priority);
    }

    void RunTask(const TaskFunction &task,
                 unsigned priority = kInstantly) {
        RunGroupedTask(priority, task, priority);
    }

    unsigned AppendGroupedTask(unsigned group_id,
                               const TaskFunction &task,
                               unsigned priority = kInstantly);

    unsigned PrependGroupedTask(unsigned group_id,
                                const TaskFunction &task,
                                unsigned priority = kInstantly);

    void RunGroupedTask(unsigned group_id,
                        const TaskFunction &task,
                        unsigned priority = kInstantly);

    unsigned CancelByGroupId(unsigned group_id);
    bool CancelTaskByTaskId(unsigned task_id);

    void Shutdown();

private:
    Private *const d;
};

typedef std::shared_ptr<TaskManager> TaskManagerPtr;

} // namespace mediascanner

#endif // MEDIASCANNER_TASKMANAGER_H
