/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */

// Grilo
#include <grilo.h>

// Google Tests
#include <gtest/gtest.h>

// Boost C++
#include <boost/algorithm/string.hpp>
#include <boost/filesystem.hpp>
#include <boost/regex.hpp>

// C++ Standard Library
#include <fstream>
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/filesystemscanner.h"
#include "mediascanner/glibutils.h"
#include "mediascanner/locale.h"
#include "mediascanner/logging.h"
#include "mediascanner/mediaroot.h"
#include "mediascanner/metadataresolver.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/taskfacades.h"
#include "mediascanner/taskmanager.h"
#include "mediascanner/utilities.h"
#include "mediascanner/writablemediaindex.h"

// Test Suite
#include "testlib/environments.h"
#include "testlib/loggingsink.h"
#include "testlib/testutils.h"

namespace mediascanner {
namespace filesystemscannertest {

static const FileSystemPath kDataPath =
        boost::filesystem::current_path() / "run/filesystemscannertest";
static const FileSystemPath kMediaIndexPath = kDataPath / "index";

class FileSystemScannerTest
        : public testing::Test
        , public FileSystemScanner::Listener {
public:
    void OnScanningFinished(const std::string &error_message) {
        EXPECT_EQ(std::string(), error_message);

        if (loop_) {
            g_main_loop_quit(loop_.get());
            loop_.reset();
        }
    }

    void on_timeout() {
        if (loop_) {
            g_main_loop_quit(loop_.get());
            timeout_reached_ = true;
            loop_.reset();
        }
    }

    void WaitForFinished(Timeout::Duration duration) {
        loop_ = take(g_main_loop_new(null_ptr, false));
        timeout_reached_ = false;

        Timeout::AddOnce(duration,
                         boost::bind(&FileSystemScannerTest::on_timeout,
                                     this));

        g_main_loop_run(loop_.get());
        ASSERT_FALSE(timeout_reached_);
    }

protected:
    Wrapper<GMainLoop> loop_;
    bool timeout_reached_;
};

static void verify_scanned_media(const MediaInfo &info,
                                 int32_t remaining,
                                 unsigned *fileCount) {
    EXPECT_LE(remaining, 16);

    using boost::algorithm::starts_with;
    using boost::algorithm::ends_with;

    const std::wstring url = info.first(schema::kUrl);

    const std::wstring::size_type n = url.find_last_of('/');
    ASSERT_NE(std::wstring::npos, n) << url;

    const std::wstring name = url.substr(n + 1);

    if (starts_with(name, L"tears_of_steel_")) {
        EXPECT_EQ(L"Tears of Steel", info.first(schema::kTitle)) << name;
        EXPECT_EQ(L"Ian Hubert", info.first(schema::kDirector)) << name;
        EXPECT_EQ(L"http://www.tearsofsteel.org/",
                  info.first(schema::kHomepageUrl)) << name;
        EXPECT_EQ(L"133701", info.first(schema::kTmdbId)) << name;
        EXPECT_EQ(734, boost::get<int>(info.first(schema::kDuration))) << name;
    } else if (boost::algorithm::starts_with(name, L"big_buck_bunny_")
               || boost::algorithm::starts_with(name, L"BigBuckBunny_")) {
        EXPECT_EQ(L"Big Buck Bunny", info.first(schema::kTitle)) << name;
        EXPECT_EQ(L"Sacha Goedegebure", info.first(schema::kDirector)) << name;
        EXPECT_EQ(L"http://www.bigbuckbunny.org",
                  info.first(schema::kHomepageUrl)) << name;
        EXPECT_EQ(L"10378", info.first(schema::kTmdbId)) << name;
    } else if (boost::algorithm::starts_with(name, L"Sintel.")) {
        EXPECT_EQ(L"Sintel", info.first(schema::kTitle)) << name;
        EXPECT_EQ(L"Colin Levy", info.first(schema::kDirector)) << name;
        EXPECT_EQ(L"http://durian.blender.org/",
                  info.first(schema::kHomepageUrl)) << name;
        EXPECT_EQ(L"45745", info.first(schema::kTmdbId)) << name;
    } else {
        FAIL() << "Unknown media file: " << name;
    }

    if (boost::algorithm::ends_with(name, L".mov")) {
        EXPECT_EQ(L"Quicktime", info.first(schema::kContainerFormat));
    } else if (boost::algorithm::ends_with(name, L".ogg")) {
        EXPECT_EQ(L"Ogg", info.first(schema::kContainerFormat));
    } else if (boost::algorithm::ends_with(name, L".avi")) {
        EXPECT_EQ(L"AVI", info.first(schema::kContainerFormat));
    } else if (boost::algorithm::ends_with(name, L".mkv")) {
        EXPECT_EQ(L"Matroska", info.first(schema::kContainerFormat));
    } else if (not boost::algorithm::ends_with(name, L".mp4")) {
        FAIL() << "Unknown file extension: " << name;
    }

    EXPECT_FALSE(info.first(schema::kBackdrop).empty());
    EXPECT_FALSE(info.first(schema::kDescription).empty());

    ++(*fileCount);
}

TEST_F(FileSystemScannerTest, Test) {
    typedef FileSystemScanner::TaskFacade TaskFacade;
    typedef FileSystemScanner::TaskFacadePtr TaskFacadePtr;

    LoggingSink sink(logging::warning());
    sink.ignore("warning/roots", "Cannot retrieve information about.*");
    sink.ignore("warning/roots", "Cannot identify mount point.*");

    const MediaRootManagerPtr root_manager(new MediaRootManager);
    root_manager->set_enabled(false);

    EXPECT_EQ(1, root_manager.use_count());

    boost::filesystem::remove_all(kMediaIndexPath);

    {
        const MetadataResolverPtr resolver(new MetadataResolver);
        ASSERT_TRUE(resolver->SetupSources(std::vector<std::string>()
                                           << std::string("grl-tmdb")));

        const TaskFacadePtr task_facade
                (new TaskFacade(root_manager, kMediaIndexPath));
        const TaskManagerPtr task_manager
                (new TaskManager(test_info_->test_case_name()));

        EXPECT_EQ(1, resolver.use_count());
        EXPECT_EQ(1, task_manager.use_count());
        EXPECT_EQ(1, task_facade.use_count());

        {
            FileSystemScanner scanner(resolver, task_manager, task_facade);
            scanner.add_directory(MEDIA_DIR);
            scanner.add_listener(this);
            scanner.start_scanning();

            WaitForFinished(boost::posix_time::seconds(15));

            EXPECT_TRUE(resolver->is_idle());
            ASSERT_TRUE(scanner.is_idle());
        }

        EXPECT_EQ(1, resolver.use_count());
        EXPECT_EQ(1, task_manager.use_count());
        EXPECT_EQ(1, task_facade.use_count());
    }

    EXPECT_EQ(1, root_manager.use_count());

    {
        // Verify scanned media
        MediaIndex media_index(root_manager);

        const bool index_opened = media_index.Open(kMediaIndexPath);
        ASSERT_TRUE(index_opened) << media_index.error_message();

        unsigned file_count = 0;

        media_index.VisitAll(boost::bind(verify_scanned_media,
                                         _1, _2, &file_count));

        ASSERT_EQ(16, file_count);
    }

    EXPECT_EQ(1, root_manager.use_count());
    ASSERT_TRUE(sink.get().empty()) << sink.take();
}

static void verify_symlink_media(const MediaInfo &info, int32_t remaining,
                                 const std::wstring &baseUrl,
                                 unsigned *fileCount) {
    EXPECT_LE(remaining, 2);

    const std::wstring file_url = info.first(schema::kUrl);
    EXPECT_TRUE(boost::algorithm::starts_with(file_url, baseUrl)) << file_url;

    const std::wstring file_path = file_url.substr(baseUrl.length());
    EXPECT_TRUE(file_path == L"/mocked.jpg"
                || file_path == L"/subdir/linked.jpg")
            << file_path;

    std::wcout << file_url << std::endl;
    ++(*fileCount);
}

TEST_F(FileSystemScannerTest, Symlinks) {
    const FileSystemPath kMockMediaPath = kDataPath / "media/symlinks";
    const FileSystemPath kSubdirPath = kMockMediaPath / "subdir";
    const FileSystemPath kMockImagePath = kMockMediaPath / "mocked.jpg";
    const FileSystemPath kLinkedImagePath = kSubdirPath / "linked.jpg";

    boost::filesystem::remove_all(kMediaIndexPath);
    boost::filesystem::remove_all(kMockMediaPath);
    boost::filesystem::create_directories(kSubdirPath);

    static const unsigned char kMinimalImage[] = {
        0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01,
        0x01, 0x01, 0x00, 0x48, 0x00, 0x48, 0x00, 0x00, 0xff, 0xdb, 0x00, 0x43,
        0x00, 0x03, 0x02, 0x02, 0x02, 0x02, 0x02, 0x03, 0x02, 0x02, 0x02, 0x03,
        0x03, 0x03, 0x03, 0x04, 0x06, 0x04, 0x04, 0x04, 0x04, 0x04, 0x08, 0x06,
        0x06, 0x05, 0x06, 0x09, 0x08, 0x0a, 0x0a, 0x09, 0x08, 0x09, 0x09, 0x0a,
        0x0c, 0x0f, 0x0c, 0x0a, 0x0b, 0x0e, 0x0b, 0x09, 0x09, 0x0d, 0x11, 0x0d,
        0x0e, 0x0f, 0x10, 0x10, 0x11, 0x10, 0x0a, 0x0c, 0x12, 0x13, 0x12, 0x10,
        0x13, 0x0f, 0x10, 0x10, 0x10, 0xff, 0xc9, 0x00, 0x0b, 0x08, 0x00, 0x01,
        0x00, 0x01, 0x01, 0x01, 0x11, 0x00, 0xff, 0xcc, 0x00, 0x06, 0x00, 0x10,
        0x10, 0x05, 0xff, 0xda, 0x00, 0x08, 0x01, 0x01, 0x00, 0x00, 0x3f, 0x00,
        0xd2, 0xcf, 0x20, 0xff, 0xd9
    };

    std::ofstream(kMockImagePath.string().c_str()).
            write(reinterpret_cast<const char *>(kMinimalImage),
                  sizeof(kMinimalImage));

    boost::filesystem::create_symlink("../mocked.jpg", kLinkedImagePath);
    boost::filesystem::create_symlink("../..", kSubdirPath / "trap");

    typedef FileSystemScanner::TaskFacade TaskFacade;
    typedef FileSystemScanner::TaskFacadePtr TaskFacadePtr;

    LoggingSink sink(logging::warning());
    sink.ignore("warning/roots", "Cannot retrieve information about.*");
    sink.ignore("warning/roots", "Cannot identify mount point.*");

    const MediaRootManagerPtr root_manager(new MediaRootManager);
    root_manager->set_enabled(false);

    EXPECT_EQ(1, root_manager.use_count());

    {
        const TaskFacadePtr task_facade
                (new TaskFacade(root_manager, kMediaIndexPath));
        const TaskManagerPtr task_manager
                (new TaskManager(test_info_->test_case_name()));

        EXPECT_EQ(1, task_manager.use_count());
        EXPECT_EQ(1, task_facade.use_count());

        {
            FileSystemScanner scanner(MetadataResolverPtr(),
                                      task_manager, task_facade);
            scanner.add_directory(kMockMediaPath.string());
            scanner.add_listener(this);
            scanner.start_scanning();

            WaitForFinished(boost::posix_time::seconds(15));

            ASSERT_TRUE(scanner.is_idle());
        }

        EXPECT_EQ(1, task_manager.use_count());
        EXPECT_EQ(1, task_facade.use_count());
    }

    EXPECT_EQ(1, root_manager.use_count());

    {
        // Verify scanned media
        MediaIndex media_index(root_manager);

        const bool index_opened = media_index.Open(kMediaIndexPath);
        ASSERT_TRUE(index_opened) << media_index.error_message();

        Wrapper<GError> error;
        const std::wstring base_url = safe_wstring
                (g_filename_to_uri(kMockMediaPath.string().c_str(),
                                   null_ptr, error.out_param()));

        ASSERT_FALSE(base_url.empty()) << error->message;

        unsigned file_count = 0;

        media_index.VisitAll(boost::bind(verify_symlink_media,
                                         _1, _2, base_url, &file_count));

        ASSERT_EQ(2, file_count);
    }

    EXPECT_EQ(1, root_manager.use_count());
    ASSERT_TRUE(sink.get().empty()) << sink.take();
}

void SetupTestEnvironments() {
    using testing::AddGlobalTestEnvironment;

    AddGlobalTestEnvironment(new LuceneEnvironment(kMediaIndexPath));
    AddGlobalTestEnvironment(new TheMovieDbEnvironment(REQUEST_FILE));
}

} // namespace filesystemscannertest
} // namespace mediascanner

int main(int argc, char *argv[]) {
    mediascanner::InitTests(&argc, argv);

    grl_init(&argc, &argv);
    gst_init(&argc, &argv);

    mediascanner::filesystemscannertest::SetupTestEnvironments();

    return RUN_ALL_TESTS();
}
