/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/filter.h"

// Lucene++
#include <Lucene.h>

#include <Analyzer.h>
#include <BooleanQuery.h>
#include <Fieldable.h>
#include <PrefixQuery.h>
#include <PhraseQuery.h>
#include <QueryParser.h>
#include <StringReader.h>
#include <Term.h>
#include <TermAttribute.h>
#include <TermQuery.h>
#include <TokenStream.h>
#include <WildcardQuery.h>

// Boost C++
#include <boost/bind.hpp>

// C++ Standard Library
#include <vector>

// Media Scanner Library
#include "mediascanner/utilities.h"

namespace mediascanner {

// Lucene++
using Lucene::LuceneException;
using Lucene::StringReader;
using Lucene::Term;
using Lucene::newLucene;

// Standard Libary
using std::wstring;

static const bool kDisableCoord = true;

////////////////////////////////////////////////////////////////////////////////

static Lucene::TokenStreamPtr make_token_stream(Lucene::AnalyzerPtr analyzer,
                                                const wstring &field_name,
                                                Lucene::StringReaderPtr text) {
    Lucene::TokenStreamPtr token_stream;

    try {
        token_stream = analyzer->reusableTokenStream(field_name, text);
        token_stream->reset();
    } catch(Lucene::IOException &) {
        token_stream = analyzer->tokenStream(field_name, text);
    }

    return token_stream;
}

////////////////////////////////////////////////////////////////////////////////

class Filter::Private {
public:
    virtual ~Private() {
    }

    virtual Lucene::QueryPtr BuildQuery(Lucene::QueryParserPtr parser,
                                        wstring *error_message) = 0;
};

Filter::Filter(Private *d)
    : d(d) {
}

Filter::~Filter() {
}

Filter::Filter()
    : d() {
}

Filter::Filter(const Filter &other)
    : d(other.d) {
}

Filter &Filter::operator=(const Filter &other) {
    d = other.d;
    return *this;
}

Lucene::QueryPtr Filter::BuildQuery(Lucene::QueryParserPtr parser,
                                    wstring *error_message) const {
    return d ? d->BuildQuery(parser, error_message) : Lucene::QueryPtr();
}

////////////////////////////////////////////////////////////////////////////////

class QueryStringFilter::Private : public Filter::Private {
public:
    explicit Private(const wstring &text)
        : text_(text) {
    }

    Lucene::QueryPtr BuildQuery(Lucene::QueryParserPtr parser,
                                wstring *error_message) {
        if (text_.empty())
            return Lucene::QueryPtr();

        try {
            return parser->parse(text_);
        } catch(const LuceneException &ex) {
            if (error_message)
                *error_message = ex.getError();

            return Lucene::QueryPtr();
        }
    }

private:
    wstring text_;
};

QueryStringFilter::QueryStringFilter(const wstring &text)
    : Filter(new Private(text)) {
}

////////////////////////////////////////////////////////////////////////////////

// Create Lucene queries from wildcard search string. To match the stored
// terms the string must be pass through the token analyzer. Lucene doesn't
// support spaces in wildcard queries. Therefore queries that contain multiple
// terms are transformed into a phrase query.
static Lucene::QueryPtr wildcard_field_query
                                        (const wstring &field_name,
                                         const Lucene::AnalyzerPtr analyzer,
                                         const Lucene::StringReaderPtr text) {
    text->reset();

    const Lucene::TokenStreamPtr stream =
            make_token_stream(analyzer, field_name, text);

    Lucene::PhraseQueryPtr phrase_query;
    Lucene::TermPtr term;

    if (const Lucene::TermAttributePtr term_attr =
            stream->getAttribute<Lucene::TermAttribute>()) {
        while (stream->incrementToken()) {
            if (term) {
                if (not phrase_query)
                    phrase_query = newLucene<Lucene::PhraseQuery>();

                phrase_query->add(term);
            }

            term = newLucene<Term>(field_name, term_attr->term());
        }
    }

    if (not term)
        return Lucene::QueryPtr();

    if (phrase_query) {
        phrase_query->add(term);
        return phrase_query;
    }

    term->set(term->field(), L"*" + term->text() + L"*");
    return newLucene<Lucene::WildcardQuery>(term);
}

static bool wildcard_search_visitor(const Property &property,
                                    const Lucene::AnalyzerPtr analyzer,
                                    const Lucene::StringReaderPtr text,
                                    const Lucene::BooleanQueryPtr query) {
    if (property.supports_full_text_search()) {
        if (const Lucene::QueryPtr field_query =
                wildcard_field_query(property.field_name(), analyzer, text))
            query->add(field_query, Lucene::BooleanClause::SHOULD);
    }

    return false;
}

class SubStringFilter::Private : public Filter::Private {
public:
    explicit Private(const wstring &text)
        : text_(text) {
    }

    Lucene::QueryPtr BuildQuery(Lucene::QueryParserPtr parser, wstring *) {
        if (text_.empty())
            return Lucene::QueryPtr();

        const Lucene::BooleanQueryPtr query =
                newLucene<Lucene::BooleanQuery>(kDisableCoord);

        query->setMinimumNumberShouldMatch(1);

        Property::VisitAll(boost::bind(wildcard_search_visitor,
                                       _1, parser->getAnalyzer(),
                                       newLucene<StringReader>(text_),
                                       query));

        return query;
    }

private:
    wstring text_;
};

SubStringFilter::SubStringFilter(const wstring &text)
    : Filter(new Private(text)) {
}

////////////////////////////////////////////////////////////////////////////////

// Create Lucene queries from fulltext search string. To match the
// stored terms the string must be pass through the token analyzer.
static Lucene::QueryPtr fulltext_field_query
                                        (const wstring &field_name,
                                         const Lucene::AnalyzerPtr analyzer,
                                         const Lucene::StringReaderPtr text) {
    text->reset();

    const Lucene::TokenStreamPtr stream =
            make_token_stream(analyzer, field_name, text);

    Lucene::PhraseQueryPtr phrase_query;
    Lucene::TermPtr term;

    if (const Lucene::TermAttributePtr term_attr =
            stream->getAttribute<Lucene::TermAttribute>()) {
        while (stream->incrementToken()) {
            if (term) {
                if (not phrase_query)
                    phrase_query = newLucene<Lucene::PhraseQuery>();

                phrase_query->add(term);
            }

            term = newLucene<Term>(field_name, term_attr->term());
        }
    }

    if (not term)
        return Lucene::QueryPtr();

    if (phrase_query) {
        phrase_query->add(term);
        return phrase_query;
    }

    return newLucene<Lucene::TermQuery>(term);
}

static bool fulltext_search_visitor(const Property &property,
                                    const Lucene::AnalyzerPtr analyzer,
                                    const Lucene::StringReaderPtr text,
                                    const Lucene::BooleanQueryPtr query) {
    if (property.supports_full_text_search()) {
        if (const Lucene::QueryPtr field_query =
                fulltext_field_query(property.field_name(), analyzer, text))
            query->add(field_query, Lucene::BooleanClause::SHOULD);
    }

    return false;
}

class FullTextFilter::Private : public Filter::Private {
public:
    explicit Private(const wstring &text)
        : text_(text) {
    }

    Lucene::QueryPtr BuildQuery(Lucene::QueryParserPtr parser, wstring *) {
        if (text_.empty())
            return Lucene::QueryPtr();

        const Lucene::BooleanQueryPtr query =
                newLucene<Lucene::BooleanQuery>(kDisableCoord);

        query->setMinimumNumberShouldMatch(1);

        Property::VisitAll(boost::bind(fulltext_search_visitor, _1,
                                       parser->getAnalyzer(),
                                       newLucene<StringReader>(text_),
                                       query));

        return query;
    }


private:
    wstring text_;
};

FullTextFilter::FullTextFilter(const wstring &text)
    : Filter(new Private(text)) {
}

////////////////////////////////////////////////////////////////////////////////

class PrefixFilter::Private : public Filter::Private {
public:
    explicit Private(const wstring &key, const wstring &value)
        : key_(key)
        , value_(value) {
    }

    Lucene::QueryPtr BuildQuery(Lucene::QueryParserPtr, wstring *) {
        // TODO(M5): Support prefixes that contain whitespace
        return newLucene<Lucene::PrefixQuery>(newLucene<Term>(key_, value_));
    }

private:
    wstring key_;
    wstring value_;
};

PrefixFilter::PrefixFilter(const StringProperty &key, const wstring &value)
    : Filter(new Private(key.field_name(), value)) {
}

////////////////////////////////////////////////////////////////////////////////

class ValueFilter::Private : public Filter::Private {
public:
    explicit Private(const Property &property, const Property::Value &value)
        : property_(property)
        , value_(value) {
    }

    Lucene::QueryPtr BuildQuery(Lucene::QueryParserPtr, wstring *) {
        return property_.MakeTermQuery(value_);
    }

private:
    Property property_;
    Property::Value value_;
};

ValueFilter::ValueFilter(const Property::BoundValue &value)
    : Filter(new Private(value.first, value.second)) {
}

ValueFilter::ValueFilter(const Property &property,
                         const Property::Value &value)
    : Filter(new Private(property, value)) {
}

////////////////////////////////////////////////////////////////////////////////

class RangeFilter::Private : public Filter::Private {
public:
    Private(const Property &property,
            const Property::Value &lower_value,
            const Property::Boundary &lower_boundary,
            const Property::Value &upper_value,
            const Property::Boundary &upper_boundary)
        : property_(property)
        , lower_value_(lower_value)
        , lower_boundary_(lower_boundary)
        , upper_value_(upper_value)
        , upper_boundary_(upper_boundary) {
    }

    Lucene::QueryPtr BuildQuery(Lucene::QueryParserPtr, wstring *) {
        return property_.MakeRangeQuery(lower_value_, lower_boundary_,
                                        upper_value_, upper_boundary_);
    }

private:
    Property property_;
    Property::Value lower_value_;
    Property::Boundary lower_boundary_;
    Property::Value upper_value_;
    Property::Boundary upper_boundary_;
};

RangeFilter::RangeFilter(const Property &property,
                         const Property::Value &lower_value,
                         const Property::Value &upper_value)
    : Filter(new Private(property, lower_value, Property::Inclusive,
                         upper_value, Property::Exclusive)) {
}

RangeFilter::RangeFilter(const Property &property,
                         const Property::Value &lower_value,
                         const Property::Boundary &lower_boundary,
                         const Property::Value &upper_value,
                         const Property::Boundary &upper_boundary)
    : Filter(new Private(property, lower_value, lower_boundary,
                         upper_value, upper_boundary)) {
}

////////////////////////////////////////////////////////////////////////////////

static Lucene::BooleanClause::Occur to_lucene(BooleanFilter::Occur occur) {
    switch (occur) {
    case BooleanFilter::MUST:
        return Lucene::BooleanClause::MUST;
    case BooleanFilter::MUST_NOT:
        return Lucene::BooleanClause::MUST_NOT;
    case BooleanFilter::SHOULD:
        return Lucene::BooleanClause::SHOULD;
    }

    throw std::domain_error("Unknown BooleanFilter::Occur instance");
}

class BooleanFilter::Private : public Filter::Private {
public:
    Lucene::QueryPtr BuildQuery(Lucene::QueryParserPtr parser,
                                wstring *error_message) {
        if (clauses_.empty())
            return Lucene::QueryPtr();
        if (clauses_.size() == 1)
            return clauses_.front().filter().BuildQuery(parser, error_message);

        Lucene::BooleanQueryPtr query = newLucene<Lucene::BooleanQuery>();

        for (const auto &c: clauses_) {
            wstring child_error;

            const Lucene::QueryPtr child_query =
                    c.filter().BuildQuery(parser, &child_error);

            if (not child_query) {
                if (child_error.empty())
                    continue;

                if (error_message)
                    *error_message = child_error;

                return Lucene::QueryPtr();
            }

            query->add(child_query, to_lucene(c.occur()));
        }

        if (query->getClauses().empty())
            return Lucene::QueryPtr();

        return query;
    }

    std::vector<Clause> clauses_;
};

BooleanFilter::BooleanFilter()
    : Filter(new Private) {
}

void BooleanFilter::add_clause(const Clause &clause) {
    data<Private>()->clauses_.push_back(clause);
}

} // namespace mediascanner
