/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/dbusutils.h"

// GLib related libraries
#include <gio/gunixsocketaddress.h>

// Boost C++
#include <boost/locale/format.hpp>

// Standard Library
#include <algorithm>
#include <map>
#include <string>

// Media Scanner Library
#include "mediascanner/logging.h"

namespace mediascanner {
namespace dbus {

// Boost C++
using boost::locale::format;

// Standard Library
using std::string;

static const logging::Domain kError("error/dbus", logging::error());
static const logging::Domain kInfo("info/dbus", logging::info());
static const logging::Domain kTrace("trace/dbus", logging::trace());

////////////////////////////////////////////////////////////////////////////////

void InterfaceInfo::add_method(MethodInfoPtr method) {
    methods_.insert(std::make_pair(method->name(), method));
    info_.reset();
}

void InterfaceInfo::add_property(PropertyInfoPtr property) {
    properties_.insert(std::make_pair(property->name(), property));
    info_.reset();
}

void InterfaceInfo::add_signal(SignalInfoPtr signal) {
    signals_.insert(std::make_pair(signal->name(), signal));
    info_.reset();
}

template<typename T>
static T find_member(const std::map<string, T> &members, const string &name) {
    const typename std::map<string, T>::const_iterator it = members.find(name);
    return it != members.end() ? it->second : T();
}

MethodInfoPtr InterfaceInfo::find_method(const string &name) const {
    return find_member(methods_, name);
}

PropertyInfoPtr InterfaceInfo::find_property(const string &name) const {
    return find_member(properties_, name);
}

SignalInfoPtr InterfaceInfo::find_signal(const string &name) const {
    return find_member(signals_, name);
}

template<typename T>
static typename T::element_type::dbus_info_type**
make_member_info(const std::map<string, T> &members) {
    typedef typename T::element_type::dbus_info_type *dbus_info_type;
    typedef typename std::map<string, T>::value_type element_type;

    dbus_info_type *const info = g_new(dbus_info_type, members.size() + 1);
    dbus_info_type *dst_it = info;

    for (const auto &m: members)
        *(dst_it++) = m.second->info().dup();

    *dst_it = null_ptr;
    return info;
}

Wrapper<GDBusInterfaceInfo> InterfaceInfo::info() const {
    if (not info_) {
        const GDBusInterfaceInfo static_info = {
            1, g_strdup(name().c_str()),
            make_member_info(methods_),
            make_member_info(signals_),
            make_member_info(properties_),
            null_ptr
        };

        info_ = wrap_static(&static_info);
    }

    return info_;
}

////////////////////////////////////////////////////////////////////////////////

bool InterfaceProxy::ConnectAndWait(Wrapper<GDBusConnection> connection,
                                    const string &service_name,
                                    const string &object_path,
                                    Wrapper<GCancellable> cancellable,
                                    GError **error) {
    const string iface_name = interface_name();
    kTrace("Trying connection to interface \"{1}\" at \"{2}\" on \"{3}\"")
            % iface_name % object_path % service_name;

    handle_ = take(g_dbus_proxy_new_sync
            (connection.get(), G_DBUS_PROXY_FLAGS_GET_INVALIDATED_PROPERTIES,
             info().get(), service_name.c_str(), object_path.c_str(),
             interface_name().c_str(), cancellable.get(), error));

    return handle_ != null_ptr;
}

void InterfaceProxy::connect(GBusType bus_type,
                             const string &service_name,
                             const string &object_path,
                             Wrapper<GCancellable> cancellable) {
    const string iface_name = interface_name();
    kTrace("Trying connection to interface \"{1}\" at \"{2}\" on \"{3}\"")
            % iface_name % object_path % service_name;

    handle_.reset();

    g_dbus_proxy_new_for_bus
            (bus_type, G_DBUS_PROXY_FLAGS_GET_INVALIDATED_PROPERTIES,
             info().get(), service_name.c_str(), object_path.c_str(),
             interface_name().c_str(), cancellable.get(),
             &InterfaceProxy::on_connect_for_bus, this);
}

void InterfaceProxy::connect(Wrapper<GDBusConnection> connection,
                             const string &service_name,
                             const string &object_path,
                             Wrapper<GCancellable> cancellable) {
    const string iface_name = interface_name();
    kTrace("Trying connection to interface \"{1}\" at \"{2}\" on \"{3}\"")
            % iface_name % object_path % service_name;

    handle_.reset();

    g_dbus_proxy_new
            (connection.get(), G_DBUS_PROXY_FLAGS_GET_INVALIDATED_PROPERTIES,
             info().get(), service_name.c_str(), object_path.c_str(),
             interface_name().c_str(), cancellable.get(),
             &InterfaceProxy::on_connect, this);
}

void InterfaceProxy::on_connect_for_bus(GObject *,
                                        GAsyncResult *result, void *data) {
    InterfaceProxy *const self = static_cast<InterfaceProxy *>(data);
    Wrapper<GError> error;

    self->handle_ =
            take(g_dbus_proxy_new_for_bus_finish(result, error.out_param()));

    if (self->handle_) {
        self->connected();
    } else {
        self->connect_failed(error);
    }
}

void InterfaceProxy::on_connect(GObject *, GAsyncResult *result, void *data) {
    InterfaceProxy *const self = static_cast<InterfaceProxy *>(data);
    Wrapper<GError> error;

    self->handle_ =
            take(g_dbus_proxy_new_finish(result, error.out_param()));

    if (self->handle_) {
        self->connected();
    } else {
        self->connect_failed(error);
    }
}

Wrapper<GVariant> InterfaceProxy::ReadPropertyValue
                                        (const string &property_name) const {
    Wrapper<GError> error;

    Wrapper<GVariant> params =
            take(g_variant_new("(ss)", interface_name().c_str(),
                               property_name.c_str()));

    Wrapper<GVariant> result =
            take(g_dbus_connection_call_sync(connection().get(),
                                             service_name().c_str(),
                                             object_path().c_str(),
                                             "org.freedesktop.DBus.Properties",
                                             "Get", params.release(),
                                             G_VARIANT_TYPE("(v)"),
                                             G_DBUS_CALL_FLAGS_NONE, -1,
                                             null_ptr, error.out_param()));

    if (not result) {
        kError(to_string(error));
        return Wrapper<GVariant>();
    }

    return take(g_variant_get_child_value(result.get(), 0));
}

Wrapper<GDBusConnection> InterfaceProxy::connection() const {
    if (handle())
        return wrap(g_dbus_proxy_get_connection(handle().get()));

    return Wrapper<GDBusConnection>();
}

string InterfaceProxy::interface_name() const {
    return InterfaceInfo::name();
}

string InterfaceProxy::object_path() const {
    if (handle())
        return g_dbus_proxy_get_object_path(handle().get());

    return string();
}

string InterfaceProxy::service_name() const {
    if (handle())
        return g_dbus_proxy_get_name(handle().get());

    return string();
}

void InterfaceProxy::connected() {
    const string iface_name = interface_name();
    const string path = object_path();
    const string svc_name = service_name();
    kTrace("Connected to interface \"{1}\" at \"{2}\" on \"{3}\"")
            % iface_name % path % svc_name;
}

void InterfaceProxy::connect_failed(Wrapper<GError> error) const {
    const string error_message = to_string(error);
    kError("Connecting to D-Bus service failed: {1}") % error_message;
}

////////////////////////////////////////////////////////////////////////////////

const MethodSkeleton* InterfaceSkeleton::find_method_skeleton
                                                (const string &name) const {
    if (const MethodInfoPtr method = find_method(name))
        return method->skeleton();

    return null_ptr;
}

const PropertySkeleton* InterfaceSkeleton::find_property_skeleton
                                                (const string &name) const {
    if (const PropertyInfoPtr property = find_property(name))
        return property->skeleton();

    return null_ptr;
}

void InterfaceSkeleton::InvokeMethod(MethodInvocation invocation) const {
    if (const MethodSkeleton *const method =
            find_method_skeleton(invocation.method_name()))
        method->Invoke(MethodInvocation(invocation.swap()));
}

GVariant* InterfaceSkeleton::GetProperty(const string &sender,
                                         const string &target,
                                         const string &name,
                                         GError **error) const {
    if (const PropertySkeleton *const property =
            find_property_skeleton(name)) {
        if (property->readable())
            return property->GetValue(sender, target, error);
    }

    return null_ptr;
}

bool InterfaceSkeleton::SetProperty(const string &sender,
                                    const string &target,
                                    const string &name,
                                    GVariant *value,
                                    GError **error) const {
    if (const PropertySkeleton *const property =
            find_property_skeleton(name)) {
        if (property->writable())
            return property->SetValue(sender, target, value, error);
    }

    return false;
}

////////////////////////////////////////////////////////////////////////////////

void MethodInvocation::return_error(GQuark domain, int code,
                                    const string &message) {
    g_dbus_method_invocation_return_error_literal
            (swap(), domain, code, message.c_str());
}

void MethodInvocation::return_dbus_error(const string &error_name,
                                         const string &error_message) {
    g_dbus_method_invocation_return_dbus_error
            (swap(), error_name.c_str(), error_message.c_str());
}

void MethodInvocation::return_error(Wrapper<GError> error) {
    g_dbus_method_invocation_return_gerror(swap(), error.get());
}

void MethodInvocation::emit_signal(const SignalInfo *signal, GVariant *args,
                                   GError **error) const {
    g_dbus_connection_emit_signal(connection(), sender().c_str(),
                                  target().c_str(), interface_name().c_str(),
                                  signal->name().c_str(), args, error);
}

GDBusMethodInvocation* MethodInvocation::dup() const {
    return wrap(invocation_).dup();
}

GDBusMethodInvocation* MethodInvocation::swap(GDBusMethodInvocation *other) {
    assert_called_from_original_thread();
    std::swap(invocation_, other);
    return other;
}

////////////////////////////////////////////////////////////////////////////////

Service::Service()
    : name_id_(0) {
}

Service::~Service() {
    if (name_id_)
        Disconnect();


    while (not objects_.empty()) {
        const ObjectMap::iterator it = objects_.begin();
        g_dbus_connection_unregister_object(connection_.get(),
                                            it->second.object_id);
        objects_.erase(it);
    }
}

Wrapper<GDBusConnection> Service::connection() const {
    return connection_;
}

void Service::Connect(const string &service_name) {
    g_return_if_fail(name_id_ == 0);

    name_id_ = g_bus_own_name(G_BUS_TYPE_SESSION,
                              service_name.c_str(),
                              G_BUS_NAME_OWNER_FLAGS_REPLACE,
                              &Service::OnDBusConnected,
                              &Service::OnDBusNameAcquired,
                              &Service::OnDBusNameLost,
                              this, null_ptr);
}

void Service::Disconnect() {
    g_bus_unown_name(name_id_);
}

void Service::RegisterObject(const string &path,
                             InterfaceSkeletonPtr skeleton) {
    static const GDBusInterfaceVTable kInterfaceVTable = {
        &Service::OnMethodCall, &Service::OnGetProperty,
        &Service::OnSetProperty, { 0 }
    };

    Wrapper<GError> error;

    const unsigned object_id = g_dbus_connection_register_object
            (connection_.get(), path.c_str(), skeleton->info().get(),
             &kInterfaceVTable, this, null_ptr, error.out_param());

    if (object_id == 0) {
        const string error_message = to_string(error);
        kError("Cannot register D-Bus object at {1}: {2}")
                % path % error_message;
    } else {
        const ObjectInfo registration(object_id, skeleton);
        objects_.insert(std::make_pair(skeleton->name(), registration));
    }
}

InterfaceSkeletonPtr Service::find_object(const string &name) const {
    const ObjectMap::const_iterator it = objects_.find(name);

    if (it != objects_.end())
        return it->second.skeleton;

    return InterfaceSkeletonPtr();
}

void Service::Connected() {
    const string connection_name = ConnectionName(connection_);
    kInfo("Published to D-Bus connection at <{1}>") % connection_name;
}

void Service::NameAcquired() {
    kInfo("D-Bus name acquired");
}

void Service::NameLost() {
    kInfo("D-Bus name lost");
}

void Service::InvokeMethod(GDBusMethodInvocation *invocation) {
    const MethodInvocation method_invocation(invocation);

    if (const InterfaceSkeletonPtr object =
            find_object(method_invocation.interface_name()))
        object->InvokeMethod(method_invocation);
}

GVariant* Service::GetProperty(const string &sender, const string &target,
                               const string &interface, const string &property,
                               GError **error) {
    if (const InterfaceSkeletonPtr object = find_object(interface))
        return object->GetProperty(sender, target, property, error);

    return null_ptr;
}

bool Service::SetProperty(const string &sender, const string &target,
                          const string &interface, const string &property,
                          GVariant *value, GError **error) {
    if (const InterfaceSkeletonPtr object = find_object(interface))
        return object->SetProperty(sender, target, property, value, error);

    return false;
}

void Service::OnDBusConnected(GDBusConnection *connection,
                              const char *, void *data) {
    Service *const service = static_cast<Service *>(data);
    service->connection_ = wrap(connection);
    service->Connected();
}

void Service::OnDBusNameAcquired(GDBusConnection *, const char *, void *data) {
    static_cast<Service *>(data)->NameAcquired();
}

void Service::OnDBusNameLost(GDBusConnection *, const char *, void *data) {
    static_cast<Service *>(data)->NameLost();
}

void Service::OnMethodCall(GDBusConnection *, const char *, const char *,
                           const char *, const char *, GVariant *,
                           GDBusMethodInvocation *invocation, void *data) {
    static_cast<Service *>(data)->InvokeMethod(invocation);
}

GVariant* Service::OnGetProperty(GDBusConnection *, const char *sender,
                                 const char *target, const char *interface,
                                 const char *property, GError **error,
                                 void *data) {
    return static_cast<Service *>(data)->GetProperty
            (sender, target, interface, property, error);
}

gboolean Service::OnSetProperty(GDBusConnection *, const char *sender,
                                const char *target, const char *interface,
                                const char *property, GVariant *value,
                                GError **error, void *data) {
    return static_cast<Service *>(data)->SetProperty
            (sender, target, interface, property, value, error);
}

string ConnectionName(Wrapper<GDBusConnection> connection) {
    if (not connection)
        return "none";

    GIOStream *stream = g_dbus_connection_get_stream(connection.get());

    if (G_IS_SOCKET_CONNECTION(stream)) {
        GSocketAddress *address = g_socket_connection_get_remote_address
                (G_SOCKET_CONNECTION(stream), null_ptr);

        if (G_IS_UNIX_SOCKET_ADDRESS(address)) {
            GUnixSocketAddress *unix_address = G_UNIX_SOCKET_ADDRESS(address);

            return (format("unix:path={1},type={2}")
                    % g_unix_socket_address_get_path(unix_address)
                    % g_unix_socket_address_get_address_type(unix_address)).
                    str();
        }

        return (format("unknown:address={1}")
                % G_OBJECT_TYPE_NAME(address)).str();
    }

    return (format("unknown:{1}")
            % G_OBJECT_TYPE_NAME(stream)).str();
}

} // namespace dbus
} // namespace mediascanner
