/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_UTILITIES_H
#define MEDIASCANNER_UTILITIES_H

// Boost C++
#include <boost/type_traits/is_enum.hpp>
#include <boost/utility/enable_if.hpp>

// C++ Standard Libary
#include <iostream>
#include <map>
#include <string>
#include <utility>
#include <vector>

namespace mediascanner {

/**
 * The type of mediascanner::null_ptr - an implementation of C++ 11's
 * @c nullptr based on Scott Meyers' suggestion in his book "Effective C++
 * 2nd Edition" (Item 25).
 */
class null_ptr_t {
  public:
    /// This makes it convertible to any type of null non-member pointer.
    template<class T> inline operator T*() const { return 0; }

    /// This makes it convertible to any type of null member pointer.
    template<class C, class T> inline operator T C::*() const { return 0; }

  private:
    /// This prevents the taking of the address of the null_ptr
    void operator&() const; // NOLINT(runtime/operator)
};

/**
 * An implementation of C++ 11's @c nullptr based on Scott Meyers' suggestion
 * in his book "Effective C++ 2nd Edition" (Item 25).
 *
 * Users of the mediascanner API are not required to use this, but it can be
 * helpful when checking for null pointers or when supplying null pointers to
 * methods, where you might otherwise use @c NULL or @c 0.
 *
 * null_ptr is a const object.
 */
const null_ptr_t null_ptr = {};

template<typename T>
struct flag_operations {
    static T combine(T a, T b) {
        return static_cast<T>(static_cast<unsigned>(a) |
                              static_cast<unsigned>(b));
    }

    static T intersect(T a, T b) {
        return static_cast<T>(static_cast<unsigned>(a) &
                              static_cast<unsigned>(b));
    }

    static T invert(T a) {
        return static_cast<T>(~static_cast<unsigned>(a));
    }
};

template<> struct flag_operations<void> {
};

template<typename T> inline T operator|(T a, T b) {
    typedef boost::enable_if<boost::is_enum<T>, T> enabled;
    return flag_operations<typename enabled::type>::combine(a, b);
}

template<typename T> inline T operator&(T a, T b) {
    typedef boost::enable_if<boost::is_enum<T>, T> enabled;
    return flag_operations<typename enabled::type>::intersect(a, b);
}

template<typename T> inline T operator~(T a) {
    typedef boost::enable_if<boost::is_enum<T>, T> enabled;
    return flag_operations<typename enabled::type>::invert(a);
}

/**
 * @brief abort_with_backtrace
 * Prints a backtrace and aborts then. Useful for std::set_terminate().
 */
void abort_with_backtrace();

/**
 * @brief Safely constructs a std::string from potential null pointers.
 * @param str A nul terminated string, or @c null.
 * @return A new std::string instance.
 */
inline std::string safe_string(const char *str) {
    return str ? std::string(str) : std::string();
}

/**
 * @brief Safely constructs a std::string from potential null pointers
 * and then frees the pointer.
 * @param str A nul terminated string, or @c null.
 * @return A new std::string instance.
 */
inline std::string take_string(char *str) {
    std::string result;

    if (str) {
        result.assign(str);
        ::free(str);
    }

    return result;
}

/**
 * @brief Safely constructs a std::wstring from potential null pointers.
 * @param str A nul terminated string, or @c null.
 * @return A new std::wstring instance.
 */
std::wstring safe_wstring(const char *str);

/**
 * @brief Conveniently fills the std::map @p target with elements.
 * @param target The map to fill.
 * @param element The element to fill in.
 * @return @p map
 */
template<typename KeyType, typename ValueType>
inline std::map<KeyType, ValueType> &operator<<
        (std::map<KeyType, ValueType> &target, // NOLINT: runtime/reference
         const typename std::map<KeyType, ValueType>::value_type &element) {
    target.insert(element);
    return target;
}

/**
 * @brief Conveniently fills the std::vector @p target with elements.
 * @param vector The vector to fill.
 * @param element The element to fill in.
 * @return @p vec
 */
template<typename T>
inline std::vector<T> &operator<<(std::vector<T> &target, const T &element) {
    target.push_back(element);
    return target;
}

/**
 * @brief Conveniently creates a copy of the std::vector @p source,
 * and then pushes @p element to its end.
 * @param source The vector to fill.
 * @param element The element to fill in.
 * @return A new vector
 */
template<typename T>
inline std::vector<T> operator<<(const std::vector<T> &source,
                                 const T &element) {
    std::vector<T> target(source.begin(), source.end());
    target.push_back(element);
    return target;
}

/**
 * @brief Prints a property to a regular I/O stream.
 * This is useful in the context of boost::locale::format and logging.
 * @param os The target stream
 * @param s The property to print
 * @return @p os
 * @group logging
 */
std::ostream &operator<<(std::ostream &os, const class Property &p);

} // namespace mediascanner

// Note: The following stream operators must be defined in the std namespace
// to allow boost::locale::format find them when using Koenig lookup.
namespace std {

/**
 * @brief Prints a wide string to a regular I/O stream.
 * This is useful in the context of boost::locale::format and logging.
 * @param os The target stream
 * @param s The string to print
 * @return @p os
 * @group logging
 */
ostream &operator<<(ostream &os, const wstring &s);

/**
 * @brief Prints a std::vector to a I/O stream.
 * This is useful in the context of boost::locale::format and logging.
 * @param os The target stream
 * @param v The vector to print
 * @return @p os
 * @group logging
 */
template<typename CharType, typename ValueType>
inline basic_ostream<CharType> &operator<<(basic_ostream<CharType> &os,
                                           const vector<ValueType> &v) {
    os << "(vector: size=" << v.size() << ", elements=[";

    typename vector<ValueType>::const_iterator it = v.begin();

    if (it != v.end()) {
        os << *it;

        while (++it != v.end())
            os << ", " << *it;
    }

    return os << "])";
}

/**
 * @brief Prints a std::pair to a I/O stream.
 * This is useful in the context of boost::locale::format and logging.
 * @param os The target stream
 * @param m The pair to print
 * @return @p os
 * @group logging
 */
template<typename T1, typename T2>
ostream& operator<<(ostream &os, const pair<T1, T2> &p) {
    return os << "(" << p.first << ", " << p.second << ")";
}

/**
 * @brief Prints a std::map to a I/O stream.
 * This is useful in the context of boost::locale::format and logging.
 * @param os The target stream
 * @param m The map to print
 * @return @p os
 * @group logging
 */
template<typename CharType, typename KeyType, typename ValueType>
inline basic_ostream<CharType>& operator<<(basic_ostream<CharType> &os,
                                           const map<KeyType, ValueType> &m) {
    os << "(map: size=" << m.size() << ", elements={";

    typename map<KeyType, ValueType>::const_iterator it = m.begin();

    if (it != m.end()) {
        os << *it;

        while (++it != m.end())
            os << ", " << *it;
    }

    return os << "})";
}

} // namespace std

#endif // MEDIASCANNER_UTILITIES_H
