/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/mediaroot.h"

// POSIX Library
#include <sys/types.h>
#include <sys/stat.h>

// Boost C++
#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <boost/locale/format.hpp>

// C++ Standard Library
#include <map>
#include <set>
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/glibutils.h"
#include "mediascanner/logging.h"
#include "mediascanner/utilities.h"

namespace mediascanner {

// Boost C++
using boost::locale::format;

// Context specific logging domains
static const logging::Domain kWarning("warning/roots", logging::warning());

////////////////////////////////////////////////////////////////////////////////

class MediaRoot::Private {
public:
    Private(const Wrapper<GFile> &base,
            const Wrapper<GFile> &root,
            const std::string &volume_uuid)
        : base_(base)
        , root_(root)
        , volume_uuid_(volume_uuid) {
    }

    explicit Private(const std::string &error_message)
        : error_message_(error_message) {
    }

    const Wrapper<GFile> base_;
    const Wrapper<GFile> root_;

    const std::string volume_uuid_;
    const std::string error_message_;
};

////////////////////////////////////////////////////////////////////////////////

class MediaRootManager::Private {
public:
    Private()
        : idle_id_(0)
        , enabled_(true) {
    }

    ~Private() {
        teardown();
    }

    MediaRoot make_root(const MediaRoot &parent,
                        const std::string &path);
    MediaRoot make_root(const std::string &volume_uuid,
                        const std::string &path);
    MediaRoot make_root(const std::string &path);

    void AddRoot(const MediaRoot &root);
    void RemoveRoot(const MediaRoot &root);

    void setup();
    void teardown();

    bool enabled() const {
        return enabled_;
    }

    void initialize() {
        if (enabled_ && not volume_monitor_)
            on_init();
    }

    Wrapper<GUdevClient> udev_client() {
        if (not udev_client_) {
            static const char* const subsystems[] = { "block", null_ptr };
            udev_client_ = take(g_udev_client_new(subsystems));
        }

        return udev_client_;
    }

private:
    void on_init();

    static void OnMountChanged(Private *d, GMount *mount);
    static void OnMountRemoved(Private *d, GMount *mount);

public:
    std::vector<MediaRoot> media_roots_;
    std::vector<std::string> manual_roots_;
    std::vector<Listener *> listeners_;

    typedef std::map<std::string, std::set<std::string> > RelativeRootMap;
    RelativeRootMap relative_roots_;

private:
    Wrapper<GVolumeMonitor> volume_monitor_;
    Wrapper<GUdevClient> udev_client_;
    unsigned idle_id_;
    bool enabled_;
};

////////////////////////////////////////////////////////////////////////////////

static Wrapper<GFile> resolve_path(const Wrapper<GFile> &parent,
                                   const std::string &path) {
    if (g_path_is_absolute(path.c_str()))
        return take(g_file_new_for_path(path.c_str()));

    if (parent)
        return take(g_file_resolve_relative_path(parent.get(), path.c_str()));

    return Wrapper<GFile>();
}

static GUdevDevice* find_volume(const Wrapper<GUdevClient>  &client,
                                uint32_t device_number) {
    return g_udev_client_query_by_device_number
            (client.get(), G_UDEV_DEVICE_TYPE_BLOCK,
             device_number);
}

static std::string get_path(const Wrapper<GFile> &file) {
    if (not file)
        return std::string();

    return take_string(g_file_get_path(file.get()));
}

static uint32_t find_device_number(const Wrapper<GFileInfo> &info) {
    return g_file_info_get_attribute_uint32(info.get(),
                                            G_FILE_ATTRIBUTE_UNIX_DEVICE);
}

static uint32_t find_device_number(const Wrapper<GFile> &file,
                                   std::string *error_message) {
    Wrapper<GError> error;

    const Wrapper<GFileInfo> info =
            take(g_file_query_info(file.get(), G_FILE_ATTRIBUTE_UNIX_DEVICE,
                                   G_FILE_QUERY_INFO_NONE, null_ptr,
                                   error.out_param()));

    if (info)
        return find_device_number(info);

    if (error) {
        *error_message = to_string(error);
    } else {
        const std::string file_path = get_path(file);
        *error_message =
                (format("Cannot resolve UNIX device number of \"{1}\"")
                 % file_path).str();
    }

    return 0;
}

static std::string lookup_device_uuid(const Wrapper<GUdevDevice> &device) {
    if (not device)
        return std::string();

    return safe_string(g_udev_device_get_property(device.get(), "ID_FS_UUID"));
}

static std::string get_relative_path(const Wrapper<GFile> &parent,
                                     const Wrapper<GFile> &descendant) {
    return take_string(g_file_get_relative_path(parent.get(),
                                                descendant.get()));
}

static Wrapper<GFile> find_mount_path(Wrapper<GFile> file,
                                      uint32_t file_device,
                                      std::string *error_message) {
    // Now lets figure out the root path of the enclosing mount.
    //
    // Sadly pretty functions like g_file_find_enclosing_mount() or
    // g_volume_monitor_get_mount_for_uuid() are almost useless since
    // g_unix_mount_is_system_internal() boldly considers almost everything
    // that's not an USB stick a "system internal implementation detail".
    // Even your home folder.
    //
    // So instead of reimplementing GVolumeMonitor, or switching to
    // GUnixMountMonitor and copy-and-pasting the reasonable parts of
    // g_unix_mount_is_system_internal() we do something much more useful:
    //
    // Any mount point should be the top-most folder with a given block device
    // number. So let's traverse the parents of our files and check that number.
    //
    // Let's go!
    //
    // Really.
    //
    Wrapper<GFile> mount_path = file;

    while (mount_path) {
        const Wrapper<GFile> parent = take(g_file_get_parent(mount_path.get()));

        if (not parent)
            break;

        const uint32_t parent_device =
                find_device_number(parent, error_message);

        if (parent_device == 0)
            return Wrapper<GFile>();

        if (parent_device != file_device)
            break;

        mount_path = parent;
    }

    return mount_path;
}

static Wrapper<GFile> find_mount_path(const Wrapper<GUdevDevice> &device,
                                      std::string *error_message) {
    if (not device)
        return Wrapper<GFile>();

    const GUdevDeviceNumber device_number =
            g_udev_device_get_device_number(device.get());

    if (device_number == 0) {
        const std::string message =
            safe_string(g_udev_device_get_name(device.get()));
        *error_message =
                (format("No device number for device \"{1}\".")
                 % message).str();

        return Wrapper<GFile>();
    }

    const ListWrapper<GUnixMountEntry> mounts(g_unix_mounts_get(null_ptr));

    for (GList *l = mounts.get(); l; l = l->next) {
        GUnixMountEntry *const mount = static_cast<GUnixMountEntry*>(l->data);
        const char *device_path = g_unix_mount_get_device_path(mount);

        if (not (device_path && g_path_is_absolute(device_path)))
            continue;

        struct stat device_info;

        if (stat(device_path, &device_info) != 0) {
            kWarning("Cannot retrieve information about \"{1}\": {2}.")
                    % device_path % strerror(errno);
            continue;
        }

        if (S_ISBLK(device_info.st_mode)
                && device_info.st_rdev == device_number) {
            const std::string mount_path =
                safe_string(g_unix_mount_get_mount_path(mount));

            if (mount_path.empty()) {
                const std::string device_name =
                    safe_string(g_udev_device_get_name(device.get()));
                kWarning("No mount path known for device \"{1}\".")
                        % device_name;
                continue;
            }

            return take(g_file_new_for_path(mount_path.c_str()));
        }
    }

    if (g_getenv("MEDIASCANNER_UNDER_TEST")) {
        const std::string device_name =
            safe_string(g_udev_device_get_name(device.get()));
        kWarning("Cannot identify mount point of \"{1}\", but "
                 "MEDIASCANNER_UNDER_TEST is set. Therefore assuming this "
                 "device is the root file system.") % device_name;

        return take(g_file_new_for_path("/"));
    }

    const std::string device_name =
        safe_string(g_udev_device_get_name(device.get()));
    *error_message =
            (format("Cannot identify mount point of \"{1}\"")
             % device_name).str();

    return Wrapper<GFile>();
}

static Wrapper<GUdevDevice> find_device_by_uuid
                                        (const Wrapper<GUdevClient> &client,
                                         const std::string &uuid) {
    const ListWrapper<GUdevDevice> devices =
            take(g_udev_client_query_by_subsystem(client.get(), "block"));

    for (GList *l = devices.get(); l; l = l->next) {
        const Wrapper<GUdevDevice> device =
                wrap(static_cast<GUdevDevice *>(l->data));

        if (lookup_device_uuid(device) == uuid)
            return device;
    }

    return Wrapper<GUdevDevice>();
}

////////////////////////////////////////////////////////////////////////////////

MediaRoot::MediaRoot(Private *d)
    : d(d) {
}

MediaRoot::~MediaRoot() {
}

bool MediaRoot::is_valid() const {
    return d->error_message_.empty()
            && d->root_ && not d->volume_uuid_.empty();
}

std::string MediaRoot::error_message() const {
    return d->error_message_;
}

Wrapper<GFile> MediaRoot::file() const {
    return d->root_;
}

bool MediaRoot::operator==(const MediaRoot &other) const {
    return g_file_equal(file().get(), other.file().get());
}

bool MediaRoot::operator<(const MediaRoot &other) const {
    return path() < other.path();
}

std::string MediaRoot::path() const {
    return get_path(d->root_);
}

std::string MediaRoot::base_path() const {
    return get_path(d->base_);
}

std::string MediaRoot::relative_path() const {
    return get_relative_path(d->base_, d->root_);
}

std::string MediaRoot::volume_uuid() const {
    return d->volume_uuid_;
}

std::string MediaRoot::group_id() const {
    std::string id;

    if (is_valid()) {
        const std::string path = relative_path();
        id += "media:" + volume_uuid();

        if (not path.empty())
            id += ":" + path;
    }

    return id;
}

////////////////////////////////////////////////////////////////////////////////

MediaRoot MediaRootManager::Private::make_root(const MediaRoot &parent,
                                               const std::string &path) {
    BOOST_ASSERT_MSG(not g_path_is_absolute(path.c_str()), path.c_str());

    return MediaRoot(new MediaRoot::Private(parent.d->base_,
                                            resolve_path(parent.d->base_, path),
                                            parent.d->volume_uuid_));
}

MediaRoot MediaRootManager::Private::make_root(const std::string &volume_uuid,
                                               const std::string &path) {
    BOOST_ASSERT_MSG(not g_path_is_absolute(path.c_str()), path.c_str());

    const Wrapper<GUdevDevice> volume =
            find_device_by_uuid(udev_client(), volume_uuid);

    if (not volume) {
        return MediaRoot(new MediaRoot::Private
                ((format("Cannot resolve physical device \"{1}\" : {2}")
                  % volume_uuid % "Lookup of UDev volume failed.").str()));
    }

    std::string error_message;

    const Wrapper<GFile> base = find_mount_path(volume, &error_message);

    if (not base)
        return MediaRoot(new MediaRoot::Private(error_message));

    return MediaRoot(new MediaRoot::Private(base,
                                            resolve_path(base, path),
                                            volume_uuid));
}

MediaRoot MediaRootManager::Private::make_root(const std::string &path) {
    BOOST_ASSERT_MSG(g_path_is_absolute(path.c_str()), path.c_str());

    std::string error_message;
    const Wrapper<GFile> root = take(g_file_new_for_path(path.c_str()));
    const uint32_t device_number = find_device_number(root, &error_message);

    if (device_number == 0) {
        return MediaRoot(new MediaRoot::Private
                ((format("Cannot resolve physical device of \"{1}\": {2}")
                  % path % error_message).str()));
    }

    const Wrapper<GUdevDevice> volume =
            take(find_volume(udev_client(), device_number));

    if (not volume) {
        return MediaRoot(new MediaRoot::Private
                ((format("Cannot resolve physical device of \"{1}\" : {2}")
                  % path % "Lookup of UDev volume failed.").str()));
    }

    const std::string volume_uuid = lookup_device_uuid(volume);

    if (volume_uuid.empty()) {
        return MediaRoot(new MediaRoot::Private
                ((format("Cannot resolve physical device of \"{1}\": {2}")
                  % path % "No UUID or volume serial number found.").str()));
    }

    const Wrapper<GFile> base = find_mount_path(root, device_number,
                                                &error_message);

    if (not base) {
        return MediaRoot(new MediaRoot::Private
                ((format("Cannot resolve mount path for \"{1}\": {2}")
                  % path % error_message).str()));
    }

    return MediaRoot(new MediaRoot::Private(base, root, volume_uuid));
}

void MediaRootManager::Private::AddRoot(const MediaRoot &root) {
    BOOST_FOREACH (const MediaRoot &r, media_roots_) {
        if (r.path() == root.path())
            return;
    }

    media_roots_.push_back(root);

    BOOST_FOREACH (Listener *const l, listeners_) {
        l->OnMediaRootAdded(root);
    }
}

void MediaRootManager::Private::RemoveRoot(const MediaRoot &root) {
    std::vector<MediaRoot>::iterator it = media_roots_.begin();

    while (it != media_roots_.end()) {
        if (it->path() == root.path()) {
            // The media root passed to this function might not have
            // proper UUID and base path because the media got removed
            // already. Therefore use the stored variant for notification.
            const MediaRoot removed = *it;
            media_roots_.erase(it);

            BOOST_FOREACH (Listener *const l, listeners_) {
                l->OnMediaRootRemoved(removed);
            }

            return;
        }

        ++it;
    }
}

void MediaRootManager::Private::setup() {
    enabled_ = true;

    if (not idle_id_ && not volume_monitor_)
        idle_id_ = Idle::AddOnce(boost::bind(&Private::on_init, this));
}

void MediaRootManager::Private::teardown() {
    enabled_ = false;

    if (volume_monitor_)
        g_signal_handlers_disconnect_by_data(volume_monitor_.get(), this);
    if (idle_id_)
        Idle::Remove(idle_id_);
}

void MediaRootManager::Private::on_init() {
    // Initialize in idle handler to ensure all stakeholders are in place.
    volume_monitor_ = take(g_volume_monitor_get());
    idle_id_ = 0;

    g_signal_connect_swapped(volume_monitor_.get(), "mount-added",
                             G_CALLBACK(&Private::OnMountChanged),
                             this);
    g_signal_connect_swapped(volume_monitor_.get(), "mount-changed",
                             G_CALLBACK(&Private::OnMountChanged),
                             this);
    g_signal_connect_swapped(volume_monitor_.get(), "mount-pre-unmount",
                             G_CALLBACK(&Private::OnMountRemoved),
                             this);
    g_signal_connect_swapped(volume_monitor_.get(), "mount-removed",
                             G_CALLBACK(&Private::OnMountRemoved),
                             this);

    const ListWrapper<GMount> mounts
            (g_volume_monitor_get_mounts(volume_monitor_.get()));

    for (GList *l = mounts.get(); l; l = l->next) {
        GMount *const mount = static_cast<GMount *>(l->data);
        OnMountChanged(this, mount);
    }
}

void MediaRootManager::Private::OnMountChanged(Private *d, GMount *mount) {
    // Ignore shadow mounts, they are of no interest to us.
    if (g_mount_is_shadowed(mount))
        return;

    const Wrapper<GFile> root = take(g_mount_get_root(mount));
    d->AddRoot(d->make_root(get_path(root)));
}

void MediaRootManager::Private::OnMountRemoved(Private *d, GMount *mount) {
    // Ignore shadow mounts, they are of no interest to us.
    if (g_mount_is_shadowed(mount))
        return;

    const Wrapper<GFile> root = take(g_mount_get_root(mount));
    d->RemoveRoot(d->make_root(get_path(root)));
}

////////////////////////////////////////////////////////////////////////////////

MediaRootManager::MediaRootManager()
    : d(new Private) {
    d->setup();
}

MediaRootManager::~MediaRootManager() {
    delete d;
}

void MediaRootManager::initialize() {
    d->initialize();
}

void MediaRootManager::add_listener(Listener *listener) {
    d->listeners_.push_back(listener);
}

void MediaRootManager::remove_listener(Listener *listener) {
    std::vector<Listener *>::iterator it = d->listeners_.begin();

    while (it != d->listeners_.end()) {
        if (*it == listener) {
            it = d->listeners_.erase(it);
            continue;
        }

        ++it;
    }
}

std::vector<MediaRoot> MediaRootManager::media_roots() const {
    return d->media_roots_;
}

MediaRoot MediaRootManager::AddRelativeRoot(const std::string &volume_uuid,
                                            const std::string &relative_path) {
    // The root directories of volumes will get reported automatically.
    // if (relative_path.empty())
    //     return;

    // Remember this kind relative media root.
    d->relative_roots_[volume_uuid].insert(relative_path);

    // Check if this relative root's volumne is already know,
    // and report this relative media root if that's the case.
    BOOST_FOREACH (const MediaRoot &known, d->media_roots_) {
        if (known.volume_uuid() == volume_uuid) {
            if (known.relative_path().empty()) {
                // Reuse media root as parent
                const MediaRoot relative = d->make_root(known, relative_path);
                d->AddRoot(relative);
                return relative;
            }

            if (known.relative_path() == relative_path) {
                // Reuse manual media root
                return known;
            }
        }
    }

    // Didn't find a media-root for the volume yet, check device node
    // to see if it belongs to a statically mounted file system.
    const MediaRoot parent = d->make_root(volume_uuid, std::string());

    if (relative_path.empty() || not parent.is_valid())
        return parent;

    return d->make_root(parent, relative_path);
}

void MediaRootManager::AddManualRoot(const std::string &path) {
    d->manual_roots_.push_back(path);
    d->AddRoot(d->make_root(path));
}

std::vector<std::string> MediaRootManager::manual_roots() const {
    return d->manual_roots_;
}

void MediaRootManager::set_enabled(bool enabled) {
    if (enabled) {
        d->setup();
    } else {
        d->teardown();
    }
}

bool MediaRootManager::enabled() const {
    return d->enabled();
}

MediaRoot MediaRootManager::make_root(const std::string &path) const {
    return d->make_root(path);
}

} // namespace mediascanner
