/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */

// Google Tests
#include <gtest/gtest.h>

// Boost C++
#include <boost/filesystem.hpp>
#include <boost/variant/get.hpp>

// Standard Library
#include <string>

// Media Scanner
#include "mediascanner/locale.h"
#include "mediascanner/mediaroot.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/writablemediaindex.h"

// Test Suite
#include "testlib/testutils.h"

namespace mediascanner {

static FileSystemPath PrepareIndexPath() {
    const FileSystemPath index_path =
            FileSystemPath("run/mediaindextest") /
            testing::UnitTest::GetInstance()->current_test_info()->name();

    boost::filesystem::remove_all(index_path);
    boost::filesystem::create_directories(index_path);

    return index_path;
}

TEST(MediaIndexTest, Opening) {
    const MediaRootManagerPtr root_manager(new MediaRootManager);
    const FileSystemPath index_path = PrepareIndexPath();

    // Create a media index:
    // Only existing media indices can be opened for reading.
    WritableMediaIndex writer(root_manager);
    EXPECT_TRUE(writer.error_message().empty());
    ASSERT_TRUE(writer.path().empty());

    const bool writer_opened = writer.Open(index_path);
    EXPECT_EQ(index_path.string(), writer.path());
    EXPECT_EQ(std::string(), writer.error_message());
    ASSERT_TRUE(writer_opened);

    // Try to open a media index.
    MediaIndex reader(root_manager);
    EXPECT_TRUE(reader.error_message().empty());
    ASSERT_TRUE(reader.path().empty());

    const bool reader_opened = reader.Open(index_path);
    EXPECT_EQ(index_path.string(), reader.path());
    EXPECT_EQ(std::string(), reader.error_message());
    ASSERT_TRUE(reader_opened);

    // Try to re-open the media index. This should fail.
    const bool reader_reopened = reader.Open(index_path);
    EXPECT_EQ(index_path.string(), reader.path());
    EXPECT_NE(std::string(), reader.error_message());
    ASSERT_FALSE(reader_reopened);
}

TEST(MediaIndexTest, Lookup) {
    const FileSystemPath index_path = PrepareIndexPath();
    const MediaRootManagerPtr root_manager(new MediaRootManager);
    root_manager->AddManualRoot(MEDIA_DIR);

    WritableMediaIndex writer(root_manager);
    ASSERT_TRUE(writer.Open(index_path));
    ASSERT_EQ(std::string(), writer.error_message());

    const std::wstring test_url = ToUnicode(full_url("lookup"));
    const std::wstring test_tag = L"4711/0815";

    MediaInfo inserted;
    inserted.add_single(schema::kETag.bind_value(test_tag));
    writer.Insert(test_url, inserted);
    ASSERT_EQ(std::string(), writer.error_message());

    MediaIndex reader(root_manager);
    ASSERT_TRUE(reader.Open(index_path));

    const MediaInfo resolved = reader.Lookup(test_url.c_str());

    ASSERT_EQ(1, resolved.count(schema::kETag));
    ASSERT_EQ(1, resolved.count(schema::kUrl));

    EXPECT_EQ(test_tag, resolved.first(schema::kETag));
    EXPECT_EQ(test_url, resolved.first(schema::kUrl));
}

TEST(MediaIndexTest, LookupEmpty) {
    const FileSystemPath index_path = PrepareIndexPath();
    const MediaRootManagerPtr root_manager(new MediaRootManager);
    root_manager->AddManualRoot(MEDIA_DIR);

    WritableMediaIndex writer(root_manager);
    ASSERT_TRUE(writer.Open(index_path));

    MediaIndex reader(root_manager);
    ASSERT_TRUE(reader.Open(index_path));

    const std::wstring test_url = ToUnicode(full_url("lookup"));
    ASSERT_EQ(std::wstring(), reader.Lookup(test_url, schema::kETag));
}

} // namespace mediascanner

int main(int argc, char *argv[]) {
    mediascanner::InitTests(&argc, argv);
    return RUN_ALL_TESTS();
}
