/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/mediautils.h"

// Boost C++
#include <boost/algorithm/string/predicate.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/foreach.hpp>
#include <boost/variant.hpp>

// C++ Standard Library
#include <map>
#include <set>
#include <string>

// Media Scanner Library
#include "mediascanner/glibutils.h"
#include "mediascanner/locale.h"
#include "mediascanner/logging.h"
#include "mediascanner/property.h"
#include "mediascanner/propertyschema.h"
#include "mediascanner/utilities.h"

namespace mediascanner {

// Boost C++
using boost::algorithm::starts_with;

// Context specific logging domains
static const logging::Domain kWarning("warning/media", logging::warning());
static const logging::Domain kTrace("trace/media", logging::trace());

// MIME type constants
const MimeType MimeType::kApplicationOgg(L"application/ogg");
const MimeType MimeType::kAudioPrefix(L"audio/");
const MimeType MimeType::kImagePrefix(L"image/");
const MimeType MimeType::kVideoPrefix(L"video/");

bool MimeType::is_audio() const {
    return starts_with(str(), kAudioPrefix.str())
            || str() == kApplicationOgg.str();
}

bool MimeType::is_image() const {
    return starts_with(str(), kImagePrefix.str());
}

bool MimeType::is_video() const {
    return starts_with(str(), kVideoPrefix.str());
}

GrlMedia* MimeType::make_media() const {
    if (is_video())
        return grl_media_video_new();
    if (is_image())
        return grl_media_image_new();
    if (is_audio())
        return grl_media_audio_new();

    return grl_media_new();
}

MediaInfo::MediaInfo() {
    // Append one initial map for unrelated, single-value properties.
    related_properties_.push_back(Property::ValueMap());
}

void MediaInfo::add_related(const Property::ValueMap &properties) {
    if (not properties.empty())
        related_properties_.push_back(properties);
}

void MediaInfo::add_single(const Property::BoundValue &value) {
    const Property &property = value.first;
    Property::ValueMap &front = related_properties_.front();
    Property::ValueMap::iterator it = front.find(property);

    if (it == front.end()) {
        front.insert(value);
    } else if (property.merge_strategy() == Property::MergeReplace) {
        it->second = value.second;
    } else if (property.merge_strategy() != Property::MergePreserve) {
        related_properties_.push_back(Property::ValueMap());
        related_properties_.back().insert(value);
    }
}

Property::Value MediaInfo::first(Property key) const {
    BOOST_FOREACH (const Property::ValueMap &properties, *this) {
        const Property::ValueMap::const_iterator it = properties.find(key);

        if (it != properties.end())
            return it->second;
    }

    return Property::Value();
}

size_t MediaInfo::count(Property key) const {
    size_t occurrences = 0;

    BOOST_FOREACH (const Property::ValueMap &properties, *this) {
        if (properties.find(key) != properties.end())
            ++occurrences;
    }

    return occurrences;
}

GrlMedia* MediaInfo::make_media(GList *const requested_keys) const {
    const MimeType mime_type(first(schema::kMimeType));
    const std::wstring url = first(schema::kUrl);

    GrlMedia *const media = mime_type.make_media();
    copy_to_media(requested_keys, media);

    if (not url.empty())
        grl_media_set_id(media, FromUnicode(url).c_str());

    return media;
}

GrlMedia* MediaInfo::make_media(GList *const requested_keys,
                                const std::string &url) const {
    GrlMedia *const media = make_media(requested_keys);
    const std::string stored_url = safe_string(grl_media_get_url(media));

    if (url != stored_url)
        grl_media_set_url(media, url.c_str());

    return media;
}

void MediaInfo::copy_to_media(GList *const requested_keys,
                              GrlMedia *const media) const {
    GrlData *const media_data = GRL_DATA(media);

    BOOST_FOREACH (const Property::ValueMap &properties, *this) {
        kTrace(" - copying to media: {1}") % properties;

        typedef std::map<GrlKeyID, Wrapper<GrlRelatedKeys> > RelatedKeyMap;
        RelatedKeyMap related_keys;

        BOOST_FOREACH (const Property::ValueMap::value_type &p, properties) {
            const Property::MetadataKey &key = p.first.metadata_key();

            if (requested_keys &&
                    not g_list_find(requested_keys,
                                    GRLKEYID_TO_POINTER(key.id())))
                continue;

            const GList *const relation = key.relation();
            const GrlKeyID primary_key = GRLPOINTER_TO_KEYID(relation->data);
            RelatedKeyMap::iterator it = related_keys.find(primary_key);

            if (it == related_keys.end()) {
                Wrapper<GrlRelatedKeys> row = take(grl_related_keys_new());
                it = related_keys.insert(std::make_pair(primary_key,
                                                        row)).first;
            }

            grl_related_keys_set(it->second.get(), key.id(),
                                 take(p.first.MakeGriloValue(p.second)).get());
        }

        for (RelatedKeyMap::iterator it = related_keys.begin();
             it != related_keys.end(); ++it) {
            grl_data_add_related_keys(media_data, it->second.release());
        }
    }
}

bool extract_property(GrlRelatedKeys *relation, GrlKeyID key,
                      Property::ValueMap *property_map, GList **failed_keys) {
    const Property property = Property::FromMetadataKey(key);

    if (not property) {
        kWarning("Skipping unknown metadata key \"{1}\"")
                % grl_metadata_key_get_name(key);

        if (failed_keys) {
            *failed_keys = g_list_prepend
                (*failed_keys, GRLKEYID_TO_POINTER(key));
        }

        return false;
    }

    if (const GValue *const grilo_value = grl_related_keys_get(relation, key)) {
        Property::Value property_value;

        if (not property.TransformGriloValue(grilo_value, &property_value)) {
            kWarning("Skipping unsupported value for \"{1}\"")
                    % grl_metadata_key_get_name(key);

            if (failed_keys) {
                *failed_keys = g_list_prepend
                    (*failed_keys, GRLKEYID_TO_POINTER(key));
            }

            return false;
        }

        Property::ValueMap::iterator it;

        if (property.merge_strategy() == Property::MergeReplace) {
            it = property_map->find(property);
        } else {
            it = property_map->end();
        }

        if (it == property_map->end()) {
            property_map->insert(std::make_pair(property, property_value));
        } else if (property.merge_strategy() != Property::MergePreserve) {
            it->second = property_value;
        }
    }

    return true;
}

bool MediaInfo::fill_from_media(GrlMedia *media,
                                const GList *const requested_keys,
                                GList **failed_keys,
                                std::string *error_message) {
    if (failed_keys)
        *failed_keys = null_ptr;

    const Wrapper<GrlRegistry> registry = wrap(grl_registry_get_default());
    GrlData *const media_data = GRL_DATA(media);
    std::set<GrlKeyID> remaining_keys;

    // Copy the requested keys into a more useful data structure.
    for (const GList *l = requested_keys; l; l = l->next) {
        const GrlKeyID key = GRLPOINTER_TO_KEYID(l->data);

        if (key != GRL_METADATA_KEY_ID && key != GRL_METADATA_KEY_SOURCE)
            remaining_keys.insert(GRLPOINTER_TO_KEYID(l->data));
    }

    const std::set<GrlKeyID> requested_key_set = remaining_keys;

    // Retreive all requested keys.
    while (not remaining_keys.empty()) {
        const GList *const keys = grl_registry_lookup_metadata_key_relation
                (registry.get(), *remaining_keys.begin());

        if (not keys) {
            if (error_message) {
                *error_message =
                    "Empty key relation found. This indicates a corrupted "
                    "metadata registry. Aborting property extraction.";
            }

            return false;
        }

        // Remove identified related keys from requested key set.
        for (const GList *l = keys; l; l = l->next)
            remaining_keys.erase(GRLPOINTER_TO_KEYID(l->data));

        const GrlKeyID primary_key = GRLPOINTER_TO_KEYID(keys->data);
        const unsigned num_relations = grl_data_length(media_data, primary_key);

        for (unsigned i = 0; i < num_relations; ++i) {
            GrlRelatedKeys *const relation =
                    grl_data_get_related_keys(media_data, primary_key, i);

            Property::ValueMap property_map;

            for (const GList *l = keys; l; l = l->next) {
                const GrlKeyID key = GRLPOINTER_TO_KEYID(l->data);

                if (key == GRL_METADATA_KEY_THUMBNAIL
                        || key == GRL_METADATA_KEY_THUMBNAIL_BINARY)
                    continue;

                if (requested_key_set.count(key))
                    extract_property(relation, key, &property_map, failed_keys);
            }

            if (not property_map.empty()) {
                if (keys->next) {
                    kTrace(" - related values {1}") % property_map;
                    add_related(property_map);
                } else {
                    Property::ValueMap::iterator iter = property_map.begin();
                    kTrace(" - single value {1}") % *iter;
                    add_single(*iter);
                }
            }
        }
    }

    // Ensure we have a proper URL.
    const std::wstring id = safe_wstring(grl_media_get_id(media));
    std::wstring url = first(schema::kUrl);

    if (url.empty())
        add_single(schema::kUrl.bind_value(url = id));

    if (url != id && not id.empty()) {
        if (error_message)
            *error_message = "Media URL doesn't match the media ID.";

        return false;
    }

    if (url.empty()) {
        // FIXME(M3): Generate URL from UUID if needed?
        if (error_message)
            *error_message = "No media URL provided.";

        return false;
    }

    return true;
}

bool MediaInfo::empty() const {
    switch (related_properties_.size()) {
    case 0:
        return true;
    case 1:
        return related_properties_.front().empty();
    default:
        return false;
    }
}

} // namespace mediascanner
