/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_MEDIAROOT_H
#define MEDIASCANNER_MEDIAROOT_H

// Boost C++
#include <boost/shared_ptr.hpp>

// C++ Standard Library
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/declarations.h"

namespace mediascanner {

class MediaRoot {
    friend class MediaRootManager;

    class Private;

protected:
    explicit MediaRoot(Private *d);

public:
    ~MediaRoot();

    std::string error_message() const;
    bool is_valid() const;

    std::string path() const;
    std::string base_path() const;
    std::string relative_path() const;

    std::string group_id() const;

    Wrapper<GFile> file() const;

    bool operator==(const MediaRoot &other) const;
    bool operator<(const MediaRoot &other) const;

private:
    boost::shared_ptr<Private> d;
};

class MediaRootManager {
    class Private;

public:
    class Listener {
    public:
        virtual ~Listener() {
        }

        virtual void OnMediaRootAdded(const MediaRoot &/*root*/) {
        }

        virtual void OnMediaRootRemoved(const MediaRoot &/*root*/) {
        }
    };

    MediaRootManager();
    ~MediaRootManager();

    /**
     * @brief Explicitly runs the initializer instead of waiting for the main
     * loop to trigger initialization.
     */
    void initialize();

    void add_listener(Listener *listener);
    void remove_listener(Listener *listener);

    std::vector<MediaRoot> media_roots() const;

    MediaRoot AddRelativeRoot(const std::string &relative_path);

    void AddManualRoot(const std::string &path);
    std::vector<std::string> manual_roots() const;

    void set_enabled(bool enabled);
    bool enabled() const;

    MediaRoot make_root(const std::string &path) const;

private:
    Private *const d;
};

} // namespace mediascanner

#endif // MEDIASCANNER_MEDIAROOT_H
