/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_LOGGING_H
#define MEDIASCANNER_LOGGING_H

// Boost C++
#include <boost/locale/format.hpp>
#include <boost/shared_ptr.hpp>

// C++ Standard Libary
#include <iostream>
#include <map>
#include <string>

// Media Scanner Library
#include "mediascanner/utilities.h"

namespace mediascanner {
namespace logging {

class Domain;
class MessageSink;

typedef boost::shared_ptr<MessageSink> MessageSinkPtr;

class MessageSink {
public:
    virtual ~MessageSink();

    virtual void Report(const std::string &domain_name,
                        const std::string &message) = 0;
    virtual void Report(const std::string &domain_name,
                        const std::wstring &message);

    static void set_default_instance(MessageSinkPtr instance);

    static MessageSinkPtr default_instance() {
        return default_instance_;
    }

private:
    static MessageSinkPtr default_instance_;
};

class DefaultMessageSink : public MessageSink {
public:
    enum Color {
        Default, Black, Red, Green, Brown,  Blue, Magenta, Cyan, White, Bold
    };

    explicit DefaultMessageSink(std::ostream *stream = &std::clog,
                                const std::string &prefix = std::string(),
                                Color color = Default);

    void Report(const std::string &domain_name,
                const std::string &message);

private:
    std::ostream *stream_;
    std::string prefix_;
};

class Domain {
public:
    enum Flags {
        Explicit = (1 << 0),
        Enabled = (1 << 1),
        Disabled = 0
    };

    Domain(const std::string &name, Flags flags,
           const Domain *parent = null_ptr,
           MessageSinkPtr sink = MessageSinkPtr());

    explicit Domain(const std::string &name,
                    const Domain *parent = null_ptr,
                    MessageSinkPtr sink = MessageSinkPtr());

    const std::string& name() const {
        return name_;
    }

    Domain* parent() const {
        return parent_;
    }

    void set_enabled(bool enabled = true) {
        if (enabled)
            flags_ = (flags_ | Enabled) | Explicit;
        else
            flags_ = (flags_ & ~Enabled) | Explicit;
    }

    bool enabled() const {
        if (not(flags_ & Enabled))
            return false;
        if (not(flags_ & Explicit) && parent_)
            return parent_->enabled();

        return flags_ & Enabled;
    }

    void reset() {
        flags_ = initial_flags_;
    }

    void set_message_sink(MessageSinkPtr sink) {
        message_sink_ = sink;
    }

    MessageSinkPtr message_sink() const {
        if (message_sink_)
            return message_sink_;
        if (parent_)
            return parent_->message_sink();

        return MessageSink::default_instance();
    }

    MessageSinkPtr default_message_sink() const {
        return default_message_sink_;
    }

    template<typename CharType>
    class Message {
        friend class Domain;

    private:
        typedef boost::locale::basic_format<CharType> format_type;

        Message()
            : format_(null_ptr) {
        }

        Message(const Domain &domain, const std::basic_string<CharType> &text)
            : format_(new format_type(text))
            , domain_name_(domain.name())
            , message_sink_(domain.message_sink()) {
        }

    public:
        ~Message() {
            if (format_ && message_sink_)
                message_sink_->Report(domain_name_, format_->str());

            delete format_;
        }

        template<typename Formattible>
        Message& operator% (Formattible const &object) {
            if (format_)
                *format_ % object;

            return *this;
        }

    private:
        format_type *format_;
        const std::string domain_name_;
        const MessageSinkPtr message_sink_;
    };

    Message<char> print(const std::string &text) const {
        if (enabled())
            return Message<char>(*this, text);

        return Message<char>();
    }

    Message<char> operator()(const std::string &text) const {
        return print(text);
    }

    Message<wchar_t> print(const std::wstring &text) const {
        if (enabled())
            return Message<wchar_t>(*this, text);

        return Message<wchar_t>();
    }

    Message<wchar_t> operator()(const std::wstring &text) const {
        return print(text);
    }

private:
    typedef std::map<std::string, Flags> SettingsMap;

    static const SettingsMap& ReadSettings();
    static Flags LookupFlags(const std::string &name, Flags preset);

    void VerifyGraph() const;
    void VerifyPrefix() const;

    MessageSinkPtr message_sink_;
    Domain *const parent_;
    const std::string name_;

    const Flags initial_flags_;
    Flags flags_;

    const MessageSinkPtr default_message_sink_;
};

Domain* error();
Domain* critical();
Domain* warning();
Domain* message();
Domain* info();
Domain* debug();
Domain* trace();

void capture_glib_messages();
bool is_capturing_glib_messages();

} // namespace logging
} // namespace mediascanner

#endif // MEDIASCANNER_LOGGING_H
