/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_COMMITPOLICY_H
#define MEDIASCANNER_COMMITPOLICY_H

// Boost C++
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/shared_ptr.hpp>

// C++ Standard Library
#include <map>
#include <string>
#include <vector>

// Media Scanner Library
#include "mediascanner/declarations.h"

namespace mediascanner {

/**
 * @brief A commit policy decides when changes to the media index are written
 * back to the disk. Choosing the proper policy has great impact on reliability
 * and performance, but also on hardware wear (for instance, flash disk wear).
 */
class CommitPolicy {
public:
    virtual ~CommitPolicy();

    /**
     * @brief The default policy - currently an instance of InstantCommitPolicy.
     */
    static CommitPolicyPtr default_policy();

    /**
     * @brief This method is called when data has been inserted into the
     * specified @p media_index. Policy implementations can now decide when
     * to call CommitPendingChanges().
     * @param media_urls The URLs of the affected media.
     * @param media_index The affected media index.
     * @return If the change got committed.
     */
    virtual bool OnCreate(const std::vector<std::wstring> &media_urls,
                          WritableMediaIndex *media_index) = 0;

    /**
     * @brief This method is called when data has been updated in the
     * specified @p media_index. Policy implementations can now decide when
     * to call CommitPendingChanges().
     * @param media_urls The URLs of the affected media.
     * @param media_index The affected media index.
     * @return If the change got committed.
     */
    virtual bool OnUpdate(const std::vector<std::wstring> &media_urls,
                          WritableMediaIndex *media_index) = 0;

    /**
     * @brief This method is called when data has been deleted from the
     * specified @p media_index. Policy implementations can now decide when
     * to call CommitPendingChanges().
     * @param media_urls The URLs of the affected media.
     * @param media_index The affected media index.
     * @return If the change got committed.
     */
    virtual bool OnRemove(const std::vector<std::wstring> &media_urls,
                          WritableMediaIndex *media_index) = 0;
};

/**
 * @brief A CommitPolicy which askes for instant commit of any change.
 * This policy gives the most reliable behavior, at the cost of reduced
 * performance and increased hardware wear.
 */
class InstantCommitPolicy : public CommitPolicy {
public:
    bool OnCreate(const std::vector<std::wstring> &media_urls,
                  WritableMediaIndex *media_index);
    bool OnUpdate(const std::vector<std::wstring> &media_urls,
                  WritableMediaIndex *media_index);
    bool OnRemove(const std::vector<std::wstring> &media_urls,
                  WritableMediaIndex *media_index);
};

/**
 * @brief A CommitPolicy which delays commits slightly, trying to batch them.
 * This policy permits weighting reliability versus performance and wear.
 */
class DelayedCommitPolicy : public CommitPolicy {
public:
    typedef boost::posix_time::microsec_clock Clock;
    typedef boost::posix_time::time_duration Duration;
    typedef boost::posix_time::ptime TimeStamp;

    DelayedCommitPolicy();
    ~DelayedCommitPolicy();

    /**
     * @brief The default value of maximum_batch_size() - currently 10.
     */
    static unsigned default_maximum_batch_size();

    /**
     * @brief The default value of maximum_delay() - currently 5 seconds.
     */
    static Duration default_maximum_delay();

    /**
     * @brief Changes the value of maximum_batch_size().
     */
    void set_maximum_batch_size(unsigned batch_size);

    /**
     * @brief The current maximium batch size after which commits @b must
     * be performed.
     */
    unsigned maximum_batch_size() const;

    /**
     * @brief Changes the value of maximium_delay().
     */
    void set_maximum_delay(Duration delay);

    /**
     * @brief The current maximium timely delay after which commits @b must
     * be performed.
     */
    Duration maximium_delay() const;

    bool OnCreate(const std::vector<std::wstring> &media_urls,
                  WritableMediaIndex *media_index);
    bool OnUpdate(const std::vector<std::wstring> &media_urls,
                  WritableMediaIndex *media_index);
    bool OnRemove(const std::vector<std::wstring> &media_urls,
                  WritableMediaIndex *media_index);

protected:
    void UpdateDelayedCommits();
    bool PushDelayedCommit(const std::vector<std::wstring> &media_urls,
                           WritableMediaIndex *media_index);

private:
    class DelayedCommit {
    public:
        DelayedCommit(DelayedCommitPolicy *policy,
                      WritableMediaIndex *media_index);
        ~DelayedCommit();

        bool is_due(DelayedCommitPolicy *policy, TimeStamp t) const;

        void reset_timeout(DelayedCommitPolicy *policy);
        void cancel_timeout();

        void increase_pressure(size_t amount);

    private:
        WritableMediaIndex *const media_index_;

        TimeStamp due_time_;
        size_t num_commits_;
        unsigned timeout_id_;
    };

    typedef boost::shared_ptr<DelayedCommit> DelayedCommitPtr;
    typedef std::map<WritableMediaIndex *, DelayedCommitPtr> DelayedCommitMap;

    DelayedCommitMap delayed_commits_;
    unsigned maximum_batch_size_;
    Duration maximum_delay_;
};

} // namespace mediascanner

#endif // MEDIASCANNER_COMMITPOLICY_H
