// $Id: driver.h 8844 2011-02-10 15:12:33Z jkeiren $
/** \file driver.h Declaration of the example::Driver class. */

#ifndef EXAMPLE_DRIVER_H
#define EXAMPLE_DRIVER_H

#include <fstream>
#include <sstream>
#include <iostream>
#include <string>
#include <vector>

#include "driver.h"
#include "scanner.h"

// forward declaration
class CalcContext;

/** The example namespace is used to encapsulate the three parser classes
 * example::Parser, example::Scanner and example::Driver */
namespace example
{

/** The Driver class brings together all components. It creates an instance of
 * the Parser and Scanner classes and connects them. Then the input stream is
 * fed into the scanner object and the parser gets it's token
 * sequence. Furthermore the driver object is available in the grammar rules as
 * a parameter. Therefore the driver class contains a reference to the
 * structure into which the parsed data is saved. */
class Driver
{
  public:
    /// construct a new parser driver context
    Driver(class CalcContext& _calc)
      : trace_scanning(false),
        trace_parsing(false),
        calc(_calc)
    {}

    /// enable debug output in the flex scanner
    bool trace_scanning;

    /// enable debug output in the bison parser
    bool trace_parsing;

    /// stream name (file or input stream) used for error messages.
    std::string streamname;

    /** Invoke the scanner and parser for a stream.
     * @param in  input stream
     * @param sname stream name for error messages
     * @return    true if successfully parsed
     */
    bool parse_stream(std::istream& in,
                      const std::string& sname = "stream input")
    {
      streamname = sname;

      Scanner scanner(&in);
      scanner.set_debug(trace_scanning);
      this->lexer = &scanner;

      Parser parser(*this);
      parser.set_debug_level(trace_parsing);
      return (parser.parse() == 0);
    }

    /** Invoke the scanner and parser on an input string.
     * @param input input string
     * @param sname stream name for error messages
     * @return    true if successfully parsed
     */
    bool parse_string(const std::string& input,
                      const std::string& sname = "string stream")
    {
      std::istringstream iss(input);
      return parse_stream(iss, sname);
    }

    /** Invoke the scanner and parser on a file. Use parse_stream with a
     * std::ifstream if detection of file reading errors is required.
     * @param filename  input file name
     * @return    true if successfully parsed
     */
    bool parse_file(const std::string& filename)
    {
      std::ifstream in(filename.c_str());
      if (!in.good())
      {
        return false;
      }
      return parse_stream(in, filename);
    }

    // To demonstrate pure handling of parse errors, instead of
    // simply dumping them on the standard error output, we will pass
    // them to the driver using the following two member functions.

    /** Error handling with associated line number. This can be modified to
     * output the error e.g. to a dialog box. */
    void error(const class location& l, const std::string& m)
    {
      std::cerr << l << ": " << m << std::endl;
    }

    /** General error handling. This can be modified to output the error
     * e.g. to a dialog box. */
    void error(const std::string& m)
    {
      std::cerr << m << std::endl;
    }

    /** Pointer to the current lexer instance, this is used to connect the
     * parser to the scanner. It is used in the yylex macro. */
    class Scanner* lexer;

    /** Reference to the calculator context filled during parsing of the
     * expressions. */
    class CalcContext& calc;
};

} // namespace example

#endif // EXAMPLE_DRIVER_H
