#include <core/posix/exec.h>

#include <mcloud/api/client.h>
#include <mcloud/api/diskinfo.h>
#include <mcloud/api/uploadtask.h>
#include <mcloud/api/downloadtask.h>
#include <mcloud/api/outlink.h>
#include <mcloud/api/exceptions.h>
#include <gtest/gtest.h>

#include <iostream>
#include <fstream>
#include <random>
#include <fcntl.h>

using namespace mcloud::api;
using namespace core::posix;
using namespace std;

namespace {
    static const int         time_out = 5;

    static const std::string token  = "valid_token";
    static const std::string TMPDIR = "/tmp";

    std::string tmpdir() {
        const char *v = getenv("TMPDIR");
        return v == NULL ? TMPDIR : std::string(v);
    }

    void create_file(std::string file_path, size_t size) {
        std::ofstream ofs(file_path, std::ios::binary | std::ios::out);
        ofs.seekp(size - 1);
        ofs.write("", 1);
        ofs.close();
    }

    class McloudAPI : public ::testing::Test {
        protected:
            virtual void SetUp() override {
                fake_mcloud_server_ = exec(FAKE_MCLOUD_SERVER, { }, { },
                        StandardStream::stdout);

                ASSERT_GT(fake_mcloud_server_.pid(), 0);
                string port;
                fake_mcloud_server_.cout() >> port;

                string apiroot = "http://127.0.0.1:" + port;
                setenv("MCLOUD_LOCAL_SERVER_URL", apiroot.c_str(), true);

                string download_folder = tmpdir();
                setenv("MCLOUD_DOWNLOAD_FOLDER", download_folder.c_str(), true);
            }

            virtual void TearDown() override {
            }

            ChildProcess fake_mcloud_server_ = ChildProcess::invalid();
    };

    TEST_F(McloudAPI, invalid_token)
    {
        Client c(time_out);
        c.set_access_token("");

        try {
            auto disk_info = c.disk_info();
            EXPECT_EQ(14600, disk_info.free_disk_size());
            EXPECT_EQ(20480, disk_info.disk_size());
        } catch (CredentialException &e) {
            ASSERT_STREQ("invalid_token", e.what());
        }
    }

    TEST_F(McloudAPI, disk_info)
    {
        Client c(time_out);
        c.set_access_token(token);

        auto disk_info = c.disk_info();
        EXPECT_EQ(14600, disk_info.free_disk_size());
        EXPECT_EQ(20480, disk_info.disk_size());
    }

    TEST_F(McloudAPI, share_folder_id)
    {
        Client c(time_out);
        c.set_access_token(token);

        auto root_folder_id = c.cloud_root_folder_id();
        EXPECT_EQ("1811asktx23a00019700101000000001", root_folder_id);
    }

    TEST_F(McloudAPI, cloud_content_list)
    {
        Client c(time_out);
        c.set_access_token(token);

        const int start_index = 0;
        const int count = 50;
        auto root_folder_id = c.cloud_root_folder_id();
        auto content_list = c.cloud_content_list(start_index, 
                count, CloudContent::Type::All, root_folder_id);

        EXPECT_EQ("1811asktx23a00019700101000000001", root_folder_id);
        EXPECT_EQ(50u, content_list.size());
    }

    TEST_F(McloudAPI, content_info)
    {
        Client c(time_out);
        c.set_access_token(token);

        static constexpr const char * content_id = "1811asktx23a05720160531141118i3v";
        auto contentInfo = c.content_info(content_id);

        EXPECT_EQ("image20160527_165701157.jpg", contentInfo->name());
    }

    TEST_F(McloudAPI, create_folder)
    {
        Client c(time_out);
        c.set_access_token(token);

        auto root_folder_id = c.cloud_root_folder_id();
        EXPECT_EQ("1811asktx23a00019700101000000001", root_folder_id);

        string new_folder = "new_folder";
        CloudFolder::Ptr cloud_item = c.create_folder(new_folder, root_folder_id);

        EXPECT_EQ(new_folder, cloud_item->name());
        EXPECT_EQ(root_folder_id, cloud_item->parent_catalog_id());
    }

    TEST_F(McloudAPI, create_folder_extranet_link)
    {
        Client c(time_out);
        c.set_access_token(token);

        Client::Stringlist folderlist{{"1011YfnDD09x03720160403231108kn4"}};
        auto outlink_list = c.create_folder_extranet_link(folderlist);

        EXPECT_EQ(1u, outlink_list.size());
        EXPECT_EQ("http://caiyun.feixin.10086.cn/dl/105CpSe8FosMu", outlink_list[0]->link());
    }

    TEST_F(McloudAPI, create_content_extranet_link)
    {
        Client c(time_out);
        c.set_access_token(token);

        Client::Stringlist contentlist{{"1011YfnDD09x03620151105110237b7o"}};
        auto outlink_list = c.create_content_extranet_link(contentlist);

        EXPECT_EQ(1u, outlink_list.size());
        EXPECT_EQ("http://caiyun.feixin.10086.cn/dl/105CqR96ak9mk", outlink_list[0]->link());
    }

    TEST_F(McloudAPI, update_folder)
    {
        Client c(time_out);
        c.set_access_token(token);

        static constexpr const char* folder_id = "1811asktx23a05420160525143841ng2";
        static constexpr const char* new_name  = "Just 4 fun";
        auto is_updated = c.update_folder(folder_id, new_name);

        EXPECT_TRUE(is_updated);
    }

    TEST_F(McloudAPI, delete_content)
    {
        Client c(time_out);
        c.set_access_token(token);

        Client::Stringlist contentlist{{"1011YfnDD09x03920151110171910rox"}};
        auto is_removed = c.delete_contents(contentlist);

        EXPECT_TRUE(is_removed);
    }

    TEST_F(McloudAPI, move_folders)
    {
        Client c(time_out);
        c.set_access_token(token);

        Client::Stringlist folderlist{{"1811asktx23a05520160525143306t2l"},
                                    {"1811asktx23a05520160525143322t3e"}};
        Client::Stringlist contentlist;
        string parent_folder_id = "1811asktx23a056201605251145104se";
        auto is_moved = c.move_items(folderlist, contentlist, parent_folder_id);

        EXPECT_TRUE(is_moved);
    }

    TEST_F(McloudAPI, move_content)
    {
        Client c(time_out);
        c.set_access_token(token);

        Client::Stringlist folderlist;
        Client::Stringlist contentlist{{"1811asktx23a057201605161541402vd"}, 
                                    {"1811asktx23a057201605161542182ws"}};
        string parent_folder_id = "1811asktx23a056201605251145104se";
        auto is_moved = c.move_items(folderlist, contentlist, parent_folder_id);

        EXPECT_TRUE(is_moved);
    }

    TEST_F(McloudAPI, copy_folders)
    {
        Client c(time_out);
        c.set_access_token(token);

        Client::Stringlist folderlist{{"1811asktx23a05520160525143306t2l"}, 
                                   {"1811asktx23a05520160525143322t3e"}};
   
        string parent_folder_id = "1811asktx23a00019700101000000001";
        auto new_items_list = c.copy_folders(folderlist, parent_folder_id);

        EXPECT_EQ(2u, new_items_list.size());
        EXPECT_EQ("1811asktx23a05820160526174424fx3", new_items_list[0]);
        EXPECT_EQ("1811asktx23a05820160526174424fx2", new_items_list[1]);
    }

    TEST_F(McloudAPI, copy_contents)
    {
        Client c(time_out);
        c.set_access_token(token);

        Client::Stringlist contentlist{{"1811asktx23a057201605161542182ws"},
                                    {"1811asktx23a057201605161541402vd"}};
        string parent_folder_id = "1811asktx23a00019700101000000001";
        auto new_items_list = c.copy_contents(contentlist, parent_folder_id);

        EXPECT_EQ(2u, new_items_list.size());
        EXPECT_EQ("1811asktx23a05520160526174126pqy", new_items_list[0]);
        EXPECT_EQ("1811asktx23a05420160526174126iza", new_items_list[1]);
    }

    TEST_F(McloudAPI, no_need_upload_on_cloud)
    {
        const std::size_t size = 1024;
        static const std::string file_path = tmpdir() + "/file_exists_on_cloud";
        create_file(file_path, size);

        Client c(time_out);
        c.set_access_token(token);
        auto is_exist = c.exist_on_cloud(file_path);

        EXPECT_TRUE(is_exist);
    }

    TEST_F(McloudAPI, time_out)
    {
        Client c(time_out);
        c.set_access_token(token);

        try {
            const std::size_t size = 1024;
            static const std::string file_path = tmpdir() + "/time_out";
            create_file(file_path, size);

            Client c(time_out);
            c.set_access_token(token);
            c.exist_on_cloud(file_path);
        } catch (HttpTimeoutException &e) {
            ASSERT_STREQ("HTTP request timeout", e.what());
        }
    }
}
