/*
 * Copyright (C) 2016 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors: Gary Wang <gary.wang@canonical.com>
 */


#include <McloudProvider.h>

#include <utils/DBusEnvironment.h>
#include <utils/ProviderEnvironment.h>

#include <unity/storage/qt/client-api.h>

#include <QCoreApplication>
#include <QFutureWatcher>
#include <QTemporaryFile>
#include <QFileInfo>
#include <QSignalSpy>
#include <QRegExp>
#include <QDir>

#include <core/posix/exec.h>
#include <gtest/gtest.h>

#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <exception>
#include <memory>
#include <stdexcept>
#include <iostream>
#include <random>
#include <fstream>

using namespace mcloud::api;
using namespace core::posix;
using namespace std;

using namespace unity::storage::qt;
using namespace unity::storage::metadata;
using unity::storage::ItemType;

namespace {
    static constexpr int MCLOUD_ACCOUNT_ID = 99;
	static constexpr int SIGNAL_WAIT_TIME = 30000;
    static constexpr int MB = 1024 * 1024;

    //const QString PROVIDER_ERROR = unity::storage::internal::DBUS_ERROR_PREFIX;
    const QString INVALID_EXCEPTION    = "com.canonical.StorageFramework.InvalidArgumentException";
    const QString REMOTE_EXCEPTION     = "com.canonical.StorageFramework.RemoteCommsException";
    const QString NOTEXISTS_EXCEPTION  = "com.canonical.StorageFramework.NotExistsException";

    static constexpr const char * ITEM_PREFIX = "1811asktx23a0";
    static constexpr const char * ROOT_ID = "1811asktx23a00019700101000000001";
    static constexpr const char * SPRINT_TEST_FILE   = "sprint_file";
    static constexpr const char * SPRINT_TEST_FOLDER = "sprint_folder";
    static constexpr const char * SPRINT_MOVE_FILE   = "sprint_move_file";
    static constexpr const char * SPRINT_COPY_FILE   = "sprint_copy_file";
    static constexpr const char * SPRINT_UPDATE_FILE   = "sprint_update_file";
    static constexpr const char * INVALID_ID = "1811asktx23a05520160718170034xh8_invalid";
    const size_t PNG_LENGTH = 468839;

    int random_size(int min, int max)  {
        return min + qrand() % (max - min);
    }   
  
    QString create_file(size_t size) {
        QTemporaryFile file;
        file.setAutoRemove(false);    
        file.open();
        file.seek(size - 1); 
        file.write("", 1); 
        file.close();
  
        return QFileInfo(file).filePath();
    }
}

class McloudProviderTest : public ::testing::Test
{
protected:
    void SetUp() override {
        dbus_env_.reset(new DBusEnvironment);
        dbus_env_->start_services();
		
		provider_env_.reset(new ProviderEnvironment(
                    unique_ptr<unity::storage::provider::ProviderBase>(new McloudProvider()),
                    MCLOUD_ACCOUNT_ID, *dbus_env_));
		

        fake_mcloud_server_ = exec(FAKE_MCLOUD_SERVER, { }, { }, StandardStream::stdout);

        ASSERT_GT(fake_mcloud_server_.pid(), 0); 
        string port;
        fake_mcloud_server_.cout() >> port;

        string apiroot = "http://127.0.0.1:" + port;
        setenv("MCLOUD_LOCAL_SERVER_URL", apiroot.c_str(), true);

        string download_folder = QDir::tempPath().toStdString();
        setenv("MCLOUD_DOWNLOAD_FOLDER", download_folder.c_str(), true);
    }
  
    void TearDown() override {
    }

    Account get_client() const {
        return provider_env_->get_client();
    }

    Item root_item() const  {
        auto account = get_client();

        unique_ptr<ItemListJob> j(account.roots());
        EXPECT_TRUE(j->isValid());
        EXPECT_EQ(ItemListJob::Status::Loading, j->status());

        QSignalSpy ready_spy(j.get(), &ItemListJob::itemsReady);
        EXPECT_TRUE(ready_spy.wait(SIGNAL_WAIT_TIME));
        auto items = qvariant_cast<QList<Item>>(ready_spy.takeFirst().at(0));
        EXPECT_EQ(1, items.size());
        EXPECT_EQ(ItemListJob::Status::Finished, j->status());

        return items[0];
    }
  
private:
    ChildProcess fake_mcloud_server_ = ChildProcess::invalid();
    std::unique_ptr<DBusEnvironment> dbus_env_;
    std::unique_ptr<ProviderEnvironment> provider_env_;
};

TEST_F(McloudProviderTest, roots)
{
    auto root = root_item();

    EXPECT_EQ(ROOT_ID, root.itemId());
    EXPECT_EQ("root", root.name());
    EXPECT_EQ(Item::Type::Root, root.type());
    EXPECT_TRUE(root.parentIds().isEmpty());
}

TEST_F(McloudProviderTest, list)
{
    auto root = root_item();
    EXPECT_EQ(ROOT_ID, root.itemId());
    unique_ptr<ItemListJob> j(root.list());
    EXPECT_TRUE(j->isValid());
    EXPECT_EQ(ItemListJob::Status::Loading, j->status());

    QSignalSpy ready_spy(j.get(), &ItemListJob::itemsReady);
    ASSERT_TRUE(ready_spy.wait(SIGNAL_WAIT_TIME));
    auto items = qvariant_cast<QList<Item>>(ready_spy.takeFirst().at(0));

    EXPECT_EQ(50, items.size());

    //Check first two items to verify the item metadata.
    auto item = items[0];
    EXPECT_EQ("1811asktx23a00019700101000000043",  item.itemId());
    EXPECT_EQ(Item::Type::Folder, item.type());
    EXPECT_EQ(QStringList{ROOT_ID}, item.parentIds());

    auto item1 = items[1];
    EXPECT_EQ("1811asktx23a00019700101000000044",  item1.itemId());
    EXPECT_EQ(Item::Type::Folder, item1.type());
    EXPECT_EQ(QStringList{ROOT_ID}, item1.parentIds());
}

TEST_F(McloudProviderTest, lookup)
{
    auto root = root_item();

    unique_ptr<ItemListJob> j(root.lookup("animal"));
    EXPECT_TRUE(j->isValid());
    EXPECT_EQ(ItemListJob::Status::Loading, j->status());

    QSignalSpy ready_spy(j.get(), &ItemListJob::itemsReady);
    ASSERT_TRUE(ready_spy.wait(SIGNAL_WAIT_TIME));
    ASSERT_EQ(1, ready_spy.count());
    auto items = qvariant_cast<QList<Item>>(ready_spy.takeFirst().at(0));
    EXPECT_EQ(1, items.size());

    auto item = items[0];
    EXPECT_EQ("1811asktx23a05520160718170034xh1",  item.itemId());
    EXPECT_EQ("animal.png", item.name());
    EXPECT_EQ(QStringList{ROOT_ID}, item.parentIds());
    EXPECT_EQ(Item::Type::File, item.type());
}

TEST_F(McloudProviderTest, metadata)
{

    auto account = get_client();
    {
        unique_ptr<ItemJob> j(account.get("1811asktx23a05520160718170034xh1", {"mcloud:folder_path", SIZE_IN_BYTES}));
        QSignalSpy spy(j.get(), &ItemJob::statusChanged);
        spy.wait(SIGNAL_WAIT_TIME);
        auto item = j->item();
        EXPECT_EQ(Item::Type::File, item.type());
        EXPECT_TRUE(item.metadata().contains(SIZE_IN_BYTES));
        EXPECT_FALSE(item.metadata().contains("mcloud:folder_path")); //we don't have folder_path key
        EXPECT_FALSE(item.metadata().contains("mcloud:owner"));
        EXPECT_FALSE(item.metadata().contains("mcloud:thumbnail_url"));
        EXPECT_FALSE(item.metadata().contains("mcloud:big_thumbnail_url"));
        EXPECT_FALSE(item.metadata().contains("mcloud:present_url"));
    }

    {
        unique_ptr<ItemJob> j(account.get(INVALID_ID));
        QSignalSpy spy(j.get(), &ItemJob::statusChanged);
        spy.wait(SIGNAL_WAIT_TIME);
        EXPECT_FALSE(j->isValid());
        EXPECT_EQ(ItemJob::Status::Error, j->status()); 
        EXPECT_EQ(StorageError::NotExists, j->error().type()); 
        EXPECT_TRUE(j->error().errorString().contains("NotExists: McloudProvider::metadata(): content not exists:")); 
    }
}

TEST_F(McloudProviderTest, upload)
{
    auto root = root_item();

    int file_size = random_size(MB, MB * 5);
    auto filePath = create_file(file_size);

    unique_ptr<Uploader> uploader(root.createFile(SPRINT_TEST_FILE,
                                                  Item::ConflictPolicy::IgnoreConflict,
                                                  file_size,
                                                  ""));

    EXPECT_TRUE(uploader->isValid());
    EXPECT_EQ(Uploader::Status::Loading, uploader->status());
    EXPECT_EQ(Item::ConflictPolicy::IgnoreConflict, uploader->policy());
    EXPECT_EQ(file_size, uploader->sizeInBytes());

    {  
        QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
        ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
        auto arg = spy.takeFirst();
        EXPECT_EQ(Uploader::Status::Ready, qvariant_cast<Uploader::Status>(arg.at(0)));
    }  

    QFile file(filePath);
    file.open(QIODevice::ReadOnly);
    QByteArray ba = file.readAll();
    EXPECT_EQ(file_size, uploader->write(ba));

    QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
    uploader->close();
    ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
    auto status = qvariant_cast<Uploader::Status>(spy.takeFirst().at(0));
    EXPECT_EQ(Uploader::Status::Finished, status);

    auto item = uploader->item();
    EXPECT_EQ(SPRINT_TEST_FILE, item.name());
    EXPECT_EQ(file_size, item.sizeInBytes());
    EXPECT_EQ(QStringList{ROOT_ID}, item.parentIds());
    EXPECT_EQ(Item::Type::File, item.type());
}

TEST_F(McloudProviderTest, create_folder)
{
    auto root = root_item();
  
    unique_ptr<ItemJob> j(root.createFolder(SPRINT_TEST_FOLDER));
    EXPECT_EQ(ItemJob::Status::Loading, j->status());
  
    QSignalSpy status_spy(j.get(), &ItemJob::statusChanged);
    ASSERT_TRUE(status_spy.wait(SIGNAL_WAIT_TIME));
    ASSERT_EQ(1, status_spy.count());
    auto status = qvariant_cast<ItemJob::Status>(status_spy.takeFirst().at(0));
    EXPECT_EQ(ItemJob::Status::Finished, status);
  
    auto item = j->item();
    EXPECT_EQ(SPRINT_TEST_FOLDER,  item.name());
    EXPECT_EQ(QStringList{ROOT_ID}, item.parentIds());
    EXPECT_EQ(Item::Type::Folder, item.type());
}

TEST_F(McloudProviderTest, create_and_move)
{
    auto root = root_item();

    int file_size = random_size(MB, MB * 5);
    auto filePath = create_file(file_size);

    unique_ptr<Uploader> uploader(root.createFile(SPRINT_TEST_FILE,
                                                  Item::ConflictPolicy::IgnoreConflict,
                                                  file_size,
                                                  ""));

    EXPECT_TRUE(uploader->isValid());
    EXPECT_EQ(Uploader::Status::Loading, uploader->status());
    EXPECT_EQ(Item::ConflictPolicy::IgnoreConflict, uploader->policy());
    EXPECT_EQ(file_size, uploader->sizeInBytes());

    {  
        QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
        ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
        auto arg = spy.takeFirst();
        EXPECT_EQ(Uploader::Status::Ready, qvariant_cast<Uploader::Status>(arg.at(0)));
    }  

    QFile file(filePath);
    file.open(QIODevice::ReadOnly);
    QByteArray ba = file.readAll();
    EXPECT_EQ(file_size, uploader->write(ba));

    QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
    uploader->close();
    ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
    auto status = qvariant_cast<Uploader::Status>(spy.takeFirst().at(0));
    EXPECT_EQ(Uploader::Status::Finished, status);

    auto item = uploader->item();
    EXPECT_EQ(SPRINT_TEST_FILE, item.name());

    unique_ptr<ItemJob> j(root.createFolder(SPRINT_TEST_FOLDER));
    EXPECT_EQ(ItemJob::Status::Loading, j->status());
  
    QSignalSpy status_spy(j.get(), &ItemJob::statusChanged);
    ASSERT_TRUE(status_spy.wait(SIGNAL_WAIT_TIME));
    ASSERT_EQ(1, status_spy.count());
    auto jobstatus = qvariant_cast<ItemJob::Status>(status_spy.takeFirst().at(0));
    EXPECT_EQ(ItemJob::Status::Finished, jobstatus);
    auto new_folder = j->item();

    //mcloud doesn't support change file name when moving
    unique_ptr<ItemJob> j1(item.move(new_folder, "new_file"));
    {
        QSignalSpy spy(j1.get(), &ItemJob::statusChanged);
        spy.wait(SIGNAL_WAIT_TIME);
        ASSERT_EQ(1, spy.count());
        auto arg = spy.takeFirst();
        auto status = qvariant_cast<ItemJob::Status>(arg.at(0));
        EXPECT_EQ(ItemJob::Status::Finished, status);
    }
}

TEST_F(McloudProviderTest, create_and_copy)
{
    auto root = root_item();

    int file_size = random_size(MB, MB * 5);
    auto filePath = create_file(file_size);

    unique_ptr<Uploader> uploader(root.createFile(SPRINT_TEST_FILE,
                                                  Item::ConflictPolicy::IgnoreConflict,
                                                  file_size,
                                                  ""));

    EXPECT_TRUE(uploader->isValid());
    EXPECT_EQ(Uploader::Status::Loading, uploader->status());
    EXPECT_EQ(Item::ConflictPolicy::IgnoreConflict, uploader->policy());
    EXPECT_EQ(file_size, uploader->sizeInBytes());

    {  
        QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
        ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
        auto arg = spy.takeFirst();
        EXPECT_EQ(Uploader::Status::Ready, qvariant_cast<Uploader::Status>(arg.at(0)));
    }  

    QFile file(filePath);
    file.open(QIODevice::ReadOnly);
    QByteArray ba = file.readAll();
    EXPECT_EQ(file_size, uploader->write(ba));

    QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
    uploader->close();
    ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
    auto status = qvariant_cast<Uploader::Status>(spy.takeFirst().at(0));
    EXPECT_EQ(Uploader::Status::Finished, status);

    auto item = uploader->item();
    EXPECT_EQ(SPRINT_TEST_FILE, item.name());

    unique_ptr<ItemJob> j(root.createFolder(SPRINT_TEST_FOLDER));
    EXPECT_EQ(ItemJob::Status::Loading, j->status());
  
    QSignalSpy status_spy(j.get(), &ItemJob::statusChanged);
    ASSERT_TRUE(status_spy.wait(SIGNAL_WAIT_TIME));
    ASSERT_EQ(1, status_spy.count());
    auto jobstatus = qvariant_cast<ItemJob::Status>(status_spy.takeFirst().at(0));
    EXPECT_EQ(ItemJob::Status::Finished, jobstatus);
    auto new_folder = j->item();

    //mcloud doesn't support change file name when moving
/*
    unique_ptr<ItemJob> j1(item.move(new_folder, "new_file"));
    {
        QSignalSpy spy(j1.get(), &ItemJob::statusChanged);
        spy.wait(SIGNAL_WAIT_TIME);
        ASSERT_EQ(1, spy.count());
        auto arg = spy.takeFirst();
        auto status = qvariant_cast<ItemJob::Status>(arg.at(0));
        EXPECT_EQ(ItemJob::Status::Finished, status);
    }
*/

    unique_ptr<ItemJob> j1(item.copy(new_folder, "the beautiful world"));
    {
        QSignalSpy spy(j1.get(), &ItemJob::statusChanged);
        spy.wait(SIGNAL_WAIT_TIME);
        ASSERT_EQ(1, spy.count());
        auto arg = spy.takeFirst();
        auto status = qvariant_cast<ItemJob::Status>(arg.at(0));
        EXPECT_EQ(ItemJob::Status::Finished, status);

        EXPECT_TRUE(j1->isValid());
        EXPECT_EQ(ItemJob::Status::Finished, j1->status());
        EXPECT_EQ(StorageError::Type::NoError, j1->error().type());
        auto copied_file = j1->item();
    }
}

TEST_F(McloudProviderTest, update)
{
    auto root = root_item();

    //1.create file firstly.
    Item item;
    {
        int file_size = random_size(MB, MB * 5);
        auto filePath = create_file(file_size);

        unique_ptr<Uploader> uploader(root.createFile(SPRINT_UPDATE_FILE,
                                                      Item::ConflictPolicy::IgnoreConflict,
                                                      file_size,
                                                      ""));

        EXPECT_TRUE(uploader->isValid());
        EXPECT_EQ(Uploader::Status::Loading, uploader->status());
        EXPECT_EQ(file_size, uploader->sizeInBytes());

        {  
            QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
            ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
            auto arg = spy.takeFirst();
            EXPECT_EQ(Uploader::Status::Ready, qvariant_cast<Uploader::Status>(arg.at(0)));
        }  

        QFile file(filePath);
        file.open(QIODevice::ReadOnly);
        QByteArray ba = file.readAll();
        EXPECT_EQ(file_size, uploader->write(ba));

        QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
        uploader->close();
        ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
        auto status = qvariant_cast<Uploader::Status>(spy.takeFirst().at(0));
        EXPECT_EQ(Uploader::Status::Finished, status);

        item = uploader->item();
        EXPECT_EQ(SPRINT_UPDATE_FILE, item.name());
        EXPECT_EQ(file_size, item.sizeInBytes());
    }

    auto old_itemId = item.itemId(); //for later use.
    auto etag = item.etag();

    //2.update the file just created. 
    QByteArray ba("Update the content"); 
    unique_ptr<Uploader> uploader(item.createUploader(Item::ConflictPolicy::IgnoreConflict,
                                                      ba.size()));
    EXPECT_EQ(Uploader::Status::Loading, uploader->status());

    {
        QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
        ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
        auto arg = spy.takeFirst();
        EXPECT_EQ(Uploader::Status::Ready, qvariant_cast<Uploader::Status>(arg.at(0)));
    }
            
    EXPECT_EQ(ba.size(), uploader->write(ba));

    QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
    uploader->close();
    ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
    auto arg = spy.takeFirst();
    EXPECT_EQ(Uploader::Status::Finished, qvariant_cast<Uploader::Status>(arg.at(0)));

    //NOTE: as mcloud doens't have update API, we work around on this
    //However the side-effect for this workaround is content id is chnaged after updating.
    //EXPECT_EQ(QVector<QString>{new_folder_id}, item1.parent_ids);
    auto updated_item = uploader->item();
    EXPECT_NE(old_itemId, updated_item.itemId());
    EXPECT_EQ(SPRINT_UPDATE_FILE, updated_item.name());
    EXPECT_EQ(QStringList{root.itemId()}, updated_item.parentIds());
}

TEST_F(McloudProviderTest, finish_upload_early)
{
    auto root = root_item();

    int file_size = random_size(MB, MB * 5);
    auto filePath = create_file(file_size);

    unique_ptr<Uploader> uploader(root.createFile(SPRINT_TEST_FILE,
                                                  Item::ConflictPolicy::IgnoreConflict,
                                                  file_size,
                                                  ""));

    EXPECT_TRUE(uploader->isValid());
    EXPECT_EQ(Uploader::Status::Loading, uploader->status());
    EXPECT_EQ(file_size, uploader->sizeInBytes());

    {  
        QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
        ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
        auto arg = spy.takeFirst();
        EXPECT_EQ(Uploader::Status::Ready, qvariant_cast<Uploader::Status>(arg.at(0)));
    }  

    QFile file(filePath);
    file.open(QIODevice::ReadOnly);
    QByteArray ba = file.read(512);  
    EXPECT_NE(file_size, uploader->write(ba));  //only send 512 bytes

    QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
    uploader->close();
    ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
    auto status = qvariant_cast<Uploader::Status>(spy.takeFirst().at(0));
    EXPECT_EQ(Uploader::Status::Error, status);
    EXPECT_EQ(StorageError::RemoteCommsError, uploader->error().type());
    EXPECT_TRUE(uploader->error().errorString()
                .contains("Failed to open/read local data from file/application"));
    EXPECT_EQ(Item(), uploader->item());
}

TEST_F(McloudProviderTest, upload_cancel)
{
    auto root = root_item();

    int file_size = random_size(MB, MB * 5);
    auto filePath = create_file(file_size);

    unique_ptr<Uploader> uploader(root.createFile(SPRINT_TEST_FILE,
                                                  Item::ConflictPolicy::IgnoreConflict,
                                                  file_size,
                                                  ""));

    EXPECT_TRUE(uploader->isValid());
    EXPECT_EQ(Uploader::Status::Loading, uploader->status());
    EXPECT_EQ(file_size, uploader->sizeInBytes());

    {  
        QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
        ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
        auto arg = spy.takeFirst();
        EXPECT_EQ(Uploader::Status::Ready, qvariant_cast<Uploader::Status>(arg.at(0)));
    }  

    QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
    uploader->cancel();

    EXPECT_EQ(1, spy.count());
    auto arg = spy.takeFirst();
    EXPECT_EQ(Uploader::Status::Cancelled, qvariant_cast<Uploader::Status>(arg.at(0)));
    EXPECT_EQ("Uploader::cancel(): upload was cancelled", uploader->error().message());
}

TEST_F(McloudProviderTest, upload_cancel_during_upload)
{
    auto root = root_item();

    int file_size = random_size(MB, MB * 5);
    auto filePath = create_file(file_size);

    unique_ptr<Uploader> uploader(root.createFile("upload_test_file",
                                                  Item::ConflictPolicy::IgnoreConflict,
                                                  file_size,
                                                  ""));

    EXPECT_TRUE(uploader->isValid());
    EXPECT_EQ(Uploader::Status::Loading, uploader->status());
    EXPECT_EQ(file_size, uploader->sizeInBytes());

    {
        QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
        ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
        auto arg = spy.takeFirst();
        EXPECT_EQ(Uploader::Status::Ready, qvariant_cast<Uploader::Status>(arg.at(0)));
    }


    //read 1 byte and cancel then.
    QFile file(filePath);
    file.open(QIODevice::ReadOnly);
    QByteArray ba = file.read(1);
    EXPECT_EQ(1, uploader->write(ba));

    QSignalSpy spy(uploader.get(), &Uploader::statusChanged);
    uploader->cancel();

    EXPECT_EQ(1, spy.count());
    auto arg = spy.takeFirst();
    EXPECT_EQ(Uploader::Status::Cancelled, qvariant_cast<Uploader::Status>(arg.at(0)));
    EXPECT_EQ("Uploader::cancel(): upload was cancelled", uploader->error().message());
}

TEST_F(McloudProviderTest, download)
{
    auto root = root_item();

    auto account = get_client();
    Item item;
    {
        unique_ptr<ItemJob> i(account.get("1811asktx23a05520160718170034xh1"));
        QSignalSpy spy(i.get(), &ItemJob::statusChanged);
        spy.wait(SIGNAL_WAIT_TIME);
        item = i->item();
    }

    unique_ptr<Downloader> downloader(item.createDownloader(Item::ConflictPolicy::IgnoreConflict));
    EXPECT_TRUE(downloader->isValid());
    EXPECT_EQ(Downloader::Status::Loading, downloader->status());
    EXPECT_EQ(StorageError::NoError, downloader->error().type());
  
    QByteArray ba; 
    QObject::connect(downloader.get(), &QIODevice::readyRead,
        [&]() {
            auto data = downloader->readAll();
            qDebug() << "read buffer size: " << data.length();
            ba.append(data);
    }); 

    QSignalSpy read_spy(downloader.get(), &Downloader::readChannelFinished);
    ASSERT_TRUE(read_spy.wait(SIGNAL_WAIT_TIME));

    QSignalSpy status_spy(downloader.get(), &Downloader::statusChanged);
    downloader->close();
    while (downloader->status() == Downloader::Ready)
    {
        ASSERT_TRUE(status_spy.wait(SIGNAL_WAIT_TIME));
    }
    EXPECT_EQ(Downloader::Status::Finished, downloader->status());
    EXPECT_GT(ba.length(), 0);
}

TEST_F(McloudProviderTest, download_cancel)
{
    auto root = root_item();

    auto account = get_client();
    Item item;
    {
        unique_ptr<ItemJob> i(account.get("1811asktx23a05520160718170034xh1"));
        QSignalSpy spy(i.get(), &ItemJob::statusChanged);
        spy.wait(SIGNAL_WAIT_TIME);
        item = i->item();
    }

    unique_ptr<Downloader> downloader(item.createDownloader(Item::ConflictPolicy::IgnoreConflict));
    EXPECT_TRUE(downloader->isValid());

    {    
        QSignalSpy spy(downloader.get(), &Downloader::statusChanged);
        ASSERT_TRUE(spy.wait(SIGNAL_WAIT_TIME));
        auto arg = spy.takeFirst();
        EXPECT_EQ(Downloader::Status::Ready, qvariant_cast<Downloader::Status>(arg.at(0)));
    }    

    QSignalSpy spy(downloader.get(), &Downloader::statusChanged);

    downloader->cancel();

    EXPECT_EQ(1, spy.count());
    auto arg = spy.takeFirst();
    EXPECT_EQ(Downloader::Status::Cancelled, qvariant_cast<Downloader::Status>(arg.at(0)));
}

TEST_F(McloudProviderTest, finish_download_early)
{
    auto root = root_item();

    auto account = get_client();
    unique_ptr<ItemJob> i(account.get("1811asktx23a05520160718170034xh1"));
    QSignalSpy spy(i.get(), &ItemJob::statusChanged);
    spy.wait(SIGNAL_WAIT_TIME);
    auto item = i->item();

    unique_ptr<Downloader> downloader(item.createDownloader(Item::ConflictPolicy::IgnoreConflict));
    EXPECT_TRUE(downloader->isValid());
    EXPECT_EQ(Downloader::Status::Loading, downloader->status());
    EXPECT_EQ(StorageError::NoError, downloader->error().type());
  
    QSignalSpy status_spy(downloader.get(), &Downloader::statusChanged);
    {    
        ASSERT_TRUE(status_spy.wait(SIGNAL_WAIT_TIME));
        auto arg = status_spy.takeFirst();
        EXPECT_EQ(Downloader::Status::Ready, qvariant_cast<Downloader::Status>(arg.at(0)));

        QSignalSpy read_spy(downloader.get(), &Downloader::readyRead);
        if (read_spy.count() == 0) {
            read_spy.wait(SIGNAL_WAIT_TIME);
        }
        //read first chunks after signal is triggered, then return 
        auto data = downloader->readAll();
    }

    downloader->cancel();
    auto arg = status_spy.takeFirst();
    EXPECT_EQ(Downloader::Status::Cancelled, qvariant_cast<Downloader::Status>(arg.at(0)));
}

int main(int argc, char **argv)
{
    QCoreApplication app(argc, argv);
    qsrand(QTime::currentTime().msec());

    //Ignore SIGPIPE if it occurs when running finish_download_early test.
    signal(SIGPIPE, SIG_IGN);
    ::testing::InitGoogleTest(&argc, argv);
                            
    return RUN_ALL_TESTS();
}
