/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#ifndef MCLOUD_API_CLOUDFOLDER_H_
#define MCLOUD_API_CLOUDFOLDER_H_

#include <mcloud/api/cloudresource.h>
#include <mcloud/api/visibility.h>

namespace tinyxml2 {
    class XMLElement;
}

namespace mcloud {
namespace api {

/*!
    \class CloudFolder
    \brief CloudFolder represents a folder object on mcloud.
 */

class MCLOUD_API_DLL_PUBLIC CloudFolder : public CloudResource {
public:
    enum class Type {
        Normal = 0,
        Pictures,
        Music,
        Videos,
        Message,
        Docs,
        App,
        Sync,
    };

    typedef std::shared_ptr<CloudFolder> Ptr;

    virtual ~CloudFolder() = default;

    /*!
     * \brief Returns a unique id of cloud folder object.
     */
    const std::string &id() const override;

    /*!
     * \brief Returns the name of cloud folder object.
     */
    const std::string &name() const override;

    /*!
     * \brief Returns the datetime when folder was created.
     */
    const std::time_t &created_date() const override;

    /*!
     * \brief Returns the datetime when folder was updated.
     */
    const std::time_t &updated_date() const override;

    /*!
     * \brief Returns a unique id of the parent folder.
     */
    const std::string &parent_catalog_id() const override;

    /*!
     * \brief Returns etag of a folder object.
     */
    const std::string &etag() const override;

    /*!
     * \brief Returns owner id of a folder object.
     */
    const std::string &owner() const override;

    /*!
     * \brief Returns resource property of an object.
     * \sa CloudResource::Property
     */
    CloudResource::Property property() const override;

    /*!
     * \brief Returns type of this folder object.
     * \sa Type
     */
    Type folder_type() const;

    /*!
     * \brief Returns a current folder's path.
     */
    const std::string &folder_path() const;

private:
    CloudFolder(const tinyxml2::XMLElement *root);

    friend class ClientPriv;

    class Priv;
    std::shared_ptr<Priv>   p_;
};

}
}

#endif // MCLOUD_API_CLOUDFOLDER_H_
