/*
 * Copyright © 2016 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by: Gary Wang <gary.wang@canonical.com>
 */

#ifndef MCLOUD_API_TASKQUEUE_H_
#define MCLOUD_API_TASKQUEUE_H_

#include <memory>
#include <deque>
#include <mutex>

namespace mcloud {
namespace api {

/*!
    \class TaskQueue
    \brief A thread-safe deque template.
 */
template<typename T, typename Container = std::deque<T>>
class TaskQueue {
  public:
    typedef std::shared_ptr<TaskQueue> Ptr;
    typedef typename Container::size_type size_type;
    typedef typename Container::const_iterator const_iterator;

    TaskQueue() = default;

    virtual ~TaskQueue() = default;

    TaskQueue(TaskQueue &&queue) {
        std::lock_guard<std::mutex> lock(mutex_);
        tasks_ = std::move(queue.tasks_);
    }

    TaskQueue(const TaskQueue &queue) {
        std::lock_guard<std::mutex> lock(mutex_);
        tasks_ = queue.tasks_;
    }

    TaskQueue &operator= (const TaskQueue &queue) {
        if (this != &queue) {
            std::lock_guard<std::mutex> lock1(mutex_);
            std::lock_guard<std::mutex> lock2(queue.mutex_);
            tasks_ = queue.tasks_;
        }

        return *this;
    }

    typename Container::iterator begin() {
        std::lock_guard<std::mutex> lock(mutex_);
        return tasks_.begin();
    }

    typename Container::iterator end() {
        std::lock_guard<std::mutex> lock(mutex_);
        return tasks_.end();
    }

    typename Container::iterator cbegin() const {
        std::lock_guard<std::mutex> lock(mutex_);
        return tasks_.cbegin();
    }

    typename Container::iterator cend() const {
        std::lock_guard<std::mutex> lock(mutex_);
        return tasks_.cend();
    }

    size_type size() const {
        std::lock_guard<std::mutex> lock(mutex_);
        return tasks_.size();
    }

    bool empty() const {
        std::lock_guard<std::mutex> lock(mutex_);
        return tasks_.empty();
    }

    void push(const T &task) {
        std::lock_guard<std::mutex> lock(mutex_);
        tasks_.push_back(task);
    }

    void push(const TaskQueue & queue) {
        std::lock_guard<std::mutex> lock(mutex_);
        for (const T & task: queue.tasks_){
            tasks_.push_back(task);
        }
    }

    bool try_pop(T &task) {
        std::lock_guard<std::mutex> lock(mutex_);

        if (tasks_.empty())
          return false;

        task = tasks_.front();
        tasks_.pop_front();
        return true;
    }

    const T & operator[](int index) const {
        std::lock_guard<std::mutex> lock(mutex_);
        return tasks_[index];
    }

  private:
    Container  tasks_;

    mutable std::mutex mutex_;
};

}
}

#endif // MCLOUD_API_TASKQUEUE_H_
