/*
 * Copyright (C) 2016 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors: Gary Wang <gary.wang@canonical.com>
 */

#include <core/posix/exec.h>

#include <unity/storage/provider/ProviderBase.h>
#include <unity/storage/provider/testing/TestServer.h>
#include <unity/storage/provider/metadata_keys.h>
  
#include <McloudProvider.h>

#include <utils/ProviderFixture.h>

#include <gtest/gtest.h>
#include <OnlineAccounts/Account>
#include <OnlineAccounts/Manager>
#include <QCoreApplication>
#include <QDBusConnection>
#include <QDBusConnectionInterface>
#include <QSignalSpy>
#include <QSocketNotifier>

#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <exception>
#include <memory>
#include <mutex>
#include <stdexcept>
#include <iostream>
#include <random>
#include <fstream>

using namespace std;
using namespace unity::storage::provider;
using namespace core::posix;
using unity::storage::ItemType;
using unity::storage::provider::ProviderBase;
using unity::storage::provider::testing::TestServer;

namespace {
    //const QString PROVIDER_ERROR = unity::storage::internal::DBUS_ERROR_PREFIX;
    const auto BUS_PATH = QStringLiteral("/provider");
    const QString INVALID_EXCEPTION    = "com.canonical.StorageFramework.InvalidArgumentException";
    const QString REMOTE_EXCEPTION     = "com.canonical.StorageFramework.RemoteCommsException";
    const QString NOTEXISTS_EXCEPTION  = "com.canonical.StorageFramework.NotExistsException";

    static constexpr const char * ITEM_PREFIX = "1811asktx23a0";
    static constexpr const char * ROOT_ID = "1811asktx23a00019700101000000001";
    static constexpr const char * PNG_ID  = "1811asktx23a05520160718170034xh1";
    static constexpr const char * SPRINT_TEST_FILE   = "sprint_file";
    static constexpr const char * SPRINT_TEST_FOLDER = "sprint_folder";
    static constexpr const char * SPRINT_MOVE_FILE   = "sprint_move_file";
    static constexpr const char * SPRINT_COPY_FILE   = "sprint_copy_file";
    static constexpr const char * SPRINT_UPDATE_FILE   = "sprint_update_file";
    static constexpr const char * INVALID_ID = "1811asktx23a05520160718170034xh8_invalid";
    const size_t PNG_LENGTH = 468839;

    std::string tmpdir() {
        const char *v = getenv("TMPDIR");
        return v == NULL ? "/tmp" : std::string(v);
    } 

    void create_file(std::string file_path, size_t size) {
        std::ofstream ofs(file_path, std::ios::binary | std::ios::out);
        ofs.seekp(size - 1);
        ofs.write("", 1);
        ofs.close();
    }
}

class McloudProviderTest : public ProviderFixture
{
protected:
    void SetUp() override
    {
        client_.reset(new ProviderClient(service_connection_->baseService(), BUS_PATH, connection()));

        fake_mcloud_server_ = exec(FAKE_MCLOUD_SERVER, { }, { }, StandardStream::stdout);

        ASSERT_GT(fake_mcloud_server_.pid(), 0); 
        string port;
        fake_mcloud_server_.cout() >> port;

        string apiroot = "http://127.0.0.1:" + port;
        setenv("MCLOUD_LOCAL_SERVER_URL", apiroot.c_str(), true);

        string download_folder = tmpdir();
        setenv("MCLOUD_DOWNLOAD_FOLDER", download_folder.c_str(), true);
    }
  
    void TearDown() override
    {
        client_.reset();
    }
  
    ChildProcess fake_mcloud_server_ = ChildProcess::invalid();
    std::unique_ptr<ProviderClient> client_;
};

TEST_F(McloudProviderTest, roots)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    auto reply = client_->Roots();
    wait_for(reply);
    ASSERT_TRUE(reply.isValid());
    EXPECT_EQ(1, reply.value().size());
    auto root = reply.value()[0];
    EXPECT_EQ(ROOT_ID, root.item_id);
    EXPECT_EQ("root", root.name);
    EXPECT_EQ(QVector<QString>(), root.parent_ids);
    EXPECT_EQ(ItemType::root, root.type);
}

TEST_F(McloudProviderTest, list)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    auto reply = client_->List(ROOT_ID, "");
    wait_for(reply);
    ASSERT_TRUE(reply.isValid());
    auto items = reply.argumentAt<0>();
    QString page_token = reply.argumentAt<1>();
    ASSERT_EQ(50, items.size());
    EXPECT_EQ("1811asktx23a00019700101000000043", items[0].item_id);
    EXPECT_EQ("1811asktx23a00019700101000000044", items[1].item_id);
    EXPECT_EQ("1", page_token);

    auto reply1 = client_->List(ROOT_ID, page_token);
    wait_for(reply1);
    ASSERT_TRUE(reply1.isValid());
    items = reply1.argumentAt<0>();
    page_token = reply1.argumentAt<1>();
    ASSERT_EQ(50, items.size());
    EXPECT_EQ("2", page_token);

    // Try a bad page token
    auto reply2 = client_->List("root_id", "bad_page_token");
    wait_for(reply2);
    EXPECT_TRUE(reply2.isError());
    EXPECT_EQ(INVALID_EXCEPTION, reply2.error().name());
    EXPECT_TRUE(reply2.error().message().contains("invalid page token"));

    auto reply3 = client_->List("no_such_folder_id", "");
    wait_for(reply3);
    EXPECT_TRUE(reply3.isError());
    EXPECT_EQ(NOTEXISTS_EXCEPTION, reply3.error().name());
    EXPECT_TRUE(reply3.error().message().contains("Failed to invoke the service.catalogID[no_such_folder_id]"));
}

TEST_F(McloudProviderTest, lookup)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    auto reply = client_->Lookup(ROOT_ID, "animal");
    wait_for(reply);
    ASSERT_TRUE(reply.isValid());
    auto items = reply.value();
    ASSERT_EQ(1, items.size());
    auto item = items[0];
    EXPECT_EQ(PNG_ID, item.item_id);
    EXPECT_EQ(QVector<QString>{ROOT_ID}, item.parent_ids);
    EXPECT_EQ("animal.png", item.name);
    EXPECT_EQ(ItemType::file, item.type);

    auto reply1 = client_->Lookup("no_such_folder_id", "animal");
    wait_for(reply1);
    EXPECT_TRUE(reply1.isError());
    EXPECT_EQ(NOTEXISTS_EXCEPTION, reply1.error().name());
    EXPECT_TRUE(reply1.error().message().contains("Failed to invoke the service.catalogID[no_such_folder_id]"));
}

TEST_F(McloudProviderTest, metadata)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    auto reply = client_->Metadata(PNG_ID);
    wait_for(reply);
    ASSERT_TRUE(reply.isValid());
    auto item = reply.value();
    EXPECT_EQ(PNG_ID, item.item_id);
    EXPECT_EQ(QVector<QString>{ROOT_ID}, item.parent_ids);
    EXPECT_EQ("animal.png", item.name);
    EXPECT_EQ(ItemType::file, item.type);

    auto reply1 = client_->Metadata(INVALID_ID);
    wait_for(reply1);
    EXPECT_TRUE(reply1.isError());
    EXPECT_EQ(NOTEXISTS_EXCEPTION, reply1.error().name());
    EXPECT_TRUE(reply1.error().message().contains("The content not exist."));
}

TEST_F(McloudProviderTest, upload)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    std::string upload_file_path = tmpdir() + "/" + SPRINT_TEST_FILE;
    std::default_random_engine engine;
    std::uniform_int_distribution<int> dis(1024 * 512, 1024 * 1024);
    size_t file_size = dis(engine);
    create_file(upload_file_path, file_size);

    QString upload_id;
    QDBusUnixFileDescriptor socket;
    {
        auto reply = client_->CreateFile(ROOT_ID, SPRINT_TEST_FILE, file_size, "", false);
        wait_for(reply);
        ASSERT_TRUE(reply.isValid());
        upload_id = reply.argumentAt<0>();
        socket = reply.argumentAt<1>();
    }

    auto app = QCoreApplication::instance();
    std::ifstream payload(upload_file_path);
    QSocketNotifier notifier(socket.fileDescriptor(), QSocketNotifier::Write);
    QObject::connect(&notifier, &QSocketNotifier::activated,
                       [&payload, app, &notifier](int fd) {
                           char buf[1024];
                           auto length = payload.readsome(buf, 1024);
                           ssize_t n_write = write(fd, buf, length);
                           if (n_write <= 0)
                           {   
                               // Error or end of file
                               notifier.setEnabled(false);
                               app->quit();
                           }   
                       }); 
    notifier.setEnabled(true);
    app->exec();
    auto reply = client_->FinishUpload(upload_id);
    wait_for(reply);
    auto item = reply.value();
    ASSERT_TRUE(reply.isValid());
    auto new_content_id = item.item_id;
    EXPECT_EQ(QVector<QString>{ROOT_ID}, item.parent_ids);
    EXPECT_TRUE(item.name.contains(SPRINT_TEST_FILE));
    EXPECT_EQ(file_size, item.metadata[SIZE_IN_BYTES].toLongLong());
}

TEST_F(McloudProviderTest, create_folder)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    auto reply = client_->CreateFolder(ROOT_ID, SPRINT_TEST_FOLDER);
    wait_for(reply);
    ASSERT_TRUE(reply.isValid());
    auto item = reply.value();
    auto new_folder_id = item.item_id;
    EXPECT_TRUE(new_folder_id.contains(ITEM_PREFIX));
    EXPECT_EQ(QVector<QString>{ROOT_ID}, item.parent_ids);
    //mcloud rename duplicated folder name automatically with "(n)" trailing appending.
    EXPECT_TRUE(item.name.contains(SPRINT_TEST_FOLDER));
    EXPECT_EQ(ItemType::folder, item.type);

    auto reply1 = client_->CreateFolder(INVALID_ID, SPRINT_TEST_FOLDER);
    wait_for(reply1);
    EXPECT_TRUE(reply1.isError());
    EXPECT_EQ(NOTEXISTS_EXCEPTION, reply1.error().name());
    EXPECT_TRUE(reply1.error().message().contains("catalog not exist"));
}

TEST_F(McloudProviderTest, create_and_move)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    std::string upload_file_path = tmpdir() + "/" + SPRINT_MOVE_FILE;   
    std::default_random_engine engine;
    std::uniform_int_distribution<int> dis(1024 * 512, 1024 * 1024);
    size_t file_size = dis(engine);
    create_file(upload_file_path, file_size);

    QString upload_id;
    QDBusUnixFileDescriptor socket;
    {
        auto reply = client_->CreateFile(ROOT_ID, SPRINT_MOVE_FILE, file_size, "", false);
        wait_for(reply);
        ASSERT_TRUE(reply.isValid());
        upload_id = reply.argumentAt<0>();
        socket = reply.argumentAt<1>();
    }

    auto app = QCoreApplication::instance();
    std::ifstream payload(upload_file_path);
    QSocketNotifier notifier(socket.fileDescriptor(), QSocketNotifier::Write);
    QObject::connect(&notifier, &QSocketNotifier::activated,
                       [&payload, app, &notifier](int fd) {
                           char buf[1024];
                           auto length = payload.readsome(buf, 1024);
                           ssize_t n_write = write(fd, buf, length);
                           if (n_write <= 0)
                           {   
                               // Error or end of file
                               notifier.setEnabled(false);
                               app->quit();
                           }   
                       }); 
    notifier.setEnabled(true);
    app->exec();
    auto reply = client_->FinishUpload(upload_id);
    wait_for(reply);
    auto item = reply.value();
    ASSERT_TRUE(reply.isValid());
    auto new_content_id = item.item_id;
    EXPECT_EQ(QVector<QString>{ROOT_ID}, item.parent_ids);
    EXPECT_EQ(SPRINT_MOVE_FILE, item.name);
    EXPECT_EQ(file_size, item.metadata[SIZE_IN_BYTES].toLongLong());

    auto reply1 = client_->CreateFolder(ROOT_ID, SPRINT_TEST_FOLDER);
    wait_for(reply1);
    ASSERT_TRUE(reply1.isValid());
    auto item1 = reply1.value();
    auto new_folder_id = item1.item_id;
    EXPECT_TRUE(new_folder_id.contains(ITEM_PREFIX));
    EXPECT_EQ(QVector<QString>{ROOT_ID}, item1.parent_ids);
    EXPECT_TRUE(item1.name.contains(SPRINT_TEST_FOLDER));
    EXPECT_EQ(ItemType::folder, item1.type);

    //mcloud doesn't support change file name when moving
    auto reply2 = client_->Move(new_content_id, new_folder_id, "");
    wait_for(reply2);
    ASSERT_TRUE(reply2.isValid());
    auto item2 = reply2.value();
    EXPECT_EQ(QVector<QString>{new_folder_id}, item2.parent_ids);
}

TEST_F(McloudProviderTest, create_and_copy)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    std::string upload_file_path = tmpdir() + "/" + SPRINT_COPY_FILE;   
    std::default_random_engine engine;
    std::uniform_int_distribution<int> dis(1024 * 512, 1024 * 1024);
    size_t file_size = dis(engine);
    create_file(upload_file_path, file_size);

    QString upload_id;
    QDBusUnixFileDescriptor socket;
    {
        auto reply = client_->CreateFile(ROOT_ID, SPRINT_COPY_FILE, file_size, "", false);
        wait_for(reply);
        ASSERT_TRUE(reply.isValid());
        upload_id = reply.argumentAt<0>();
        socket = reply.argumentAt<1>();
    }

    auto app = QCoreApplication::instance();
    std::ifstream payload(upload_file_path);
    QSocketNotifier notifier(socket.fileDescriptor(), QSocketNotifier::Write);
    QObject::connect(&notifier, &QSocketNotifier::activated,
                       [&payload, app, &notifier](int fd) {
                           char buf[1024];
                           auto length = payload.readsome(buf, 1024);
                           ssize_t n_write = write(fd, buf, length);
                           if (n_write <= 0)
                           {   
                               // Error or end of file
                               notifier.setEnabled(false);
                               app->quit();
                           }   
                       }); 
    notifier.setEnabled(true);
    app->exec();
    auto reply = client_->FinishUpload(upload_id);
    wait_for(reply);
    auto item = reply.value();
    ASSERT_TRUE(reply.isValid());
    auto new_content_id = item.item_id;
    EXPECT_EQ(QVector<QString>{ROOT_ID}, item.parent_ids);
    EXPECT_EQ(SPRINT_COPY_FILE, item.name);
    EXPECT_EQ(file_size, item.metadata[SIZE_IN_BYTES].toLongLong());


    auto reply1 = client_->CreateFolder(ROOT_ID, SPRINT_TEST_FOLDER);
    wait_for(reply1);
    ASSERT_TRUE(reply1.isValid());
    auto item1 = reply1.value();
    auto new_folder_id = item1.item_id;
    EXPECT_TRUE(new_folder_id.contains(ITEM_PREFIX));
    EXPECT_EQ(QVector<QString>{ROOT_ID}, item1.parent_ids);
    EXPECT_TRUE(item1.name.contains(SPRINT_TEST_FOLDER));
    EXPECT_EQ(ItemType::folder, item1.type);

    //mcloud doesn't support change file name when copying
    auto reply2 = client_->Copy(new_content_id, new_folder_id, "");
    wait_for(reply2);
    ASSERT_TRUE(reply2.isValid());
    auto item2 = reply2.value();
    EXPECT_EQ(QVector<QString>{new_folder_id}, item2.parent_ids);
}

TEST_F(McloudProviderTest, update)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    //1.create a file
    std::string upload_file_path = tmpdir() + "/" + SPRINT_UPDATE_FILE;
    std::default_random_engine engine;
    std::uniform_int_distribution<int> dis(1024 * 512, 1024 * 1024);
    size_t file_size = dis(engine);
    create_file(upload_file_path, file_size);

    QString upload_id;
    QDBusUnixFileDescriptor socket;
    {
        auto reply = client_->CreateFile(ROOT_ID, SPRINT_UPDATE_FILE, file_size, "", false);
        wait_for(reply);
        ASSERT_TRUE(reply.isValid());
        upload_id = reply.argumentAt<0>();
        socket = reply.argumentAt<1>();
    }

    auto app = QCoreApplication::instance();
    std::ifstream payload(upload_file_path);
    QSocketNotifier notifier(socket.fileDescriptor(), QSocketNotifier::Write);
    QObject::connect(&notifier, &QSocketNotifier::activated,
                       [&payload, app, &notifier](int fd) {
                           char buf[1024];
                           auto length = payload.readsome(buf, 1024);
                           ssize_t n_write = write(fd, buf, length);
                           if (n_write <= 0)
                           {   
                               // Error or end of file
                               notifier.setEnabled(false);
                               app->quit();
                           }   
                       }); 
    notifier.setEnabled(true);
    app->exec();
    auto reply = client_->FinishUpload(upload_id);
    wait_for(reply);
    auto item = reply.value();
    ASSERT_TRUE(reply.isValid());
    auto new_content_id = item.item_id;
    EXPECT_EQ(QVector<QString>{ROOT_ID}, item.parent_ids);
  
    //update the file just created.
    {
        std::string update_file_path = tmpdir() + "/" + SPRINT_UPDATE_FILE;   
        std::default_random_engine engine;
        std::uniform_int_distribution<int> dis(1024 * 512, 1024 * 1024);
        size_t file_size = dis(engine);
        create_file(update_file_path, file_size);

        QString upload_id;
        QDBusUnixFileDescriptor socket;
        {
            auto reply = client_->Update(new_content_id, file_size, "");
            wait_for(reply);
            ASSERT_TRUE(reply.isValid());
            upload_id = reply.argumentAt<0>();
            socket = reply.argumentAt<1>();
        }

        auto app = QCoreApplication::instance();
        std::ifstream payload(update_file_path);
        QSocketNotifier notifier(socket.fileDescriptor(), QSocketNotifier::Write);
        QObject::connect(&notifier, &QSocketNotifier::activated,
                [&payload, app, &notifier](int fd) {
                    char buf[1024];
                    auto length = payload.readsome(buf, 1024);
                    ssize_t n_write = write(fd, buf, length);
                    if (n_write <= 0)
                    {   
                        // Error or end of file
                        notifier.setEnabled(false);
                        app->quit();
                    }
                }); 
        notifier.setEnabled(true);
        app->exec();
        auto reply = client_->FinishUpload(upload_id);
        wait_for(reply);
        auto item = reply.value();
        ASSERT_TRUE(reply.isValid());

        //NOTE: as mcloud doens't have update API, we work around on this
        //However the side-effect for this workaround is content id is chnaged after updating.
        //EXPECT_EQ(QVector<QString>{new_folder_id}, item1.parent_ids);
        EXPECT_TRUE(item.name.contains(SPRINT_UPDATE_FILE));
        EXPECT_EQ(file_size, item.metadata[SIZE_IN_BYTES].toLongLong()); 
    }
}

TEST_F(McloudProviderTest, create_bad_name)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));
  
    std::string upload_file_path = tmpdir() + "/" + SPRINT_TEST_FILE;   
    size_t file_size = 1024;
    create_file(upload_file_path, file_size);
  
    QString upload_id;
    QDBusUnixFileDescriptor socket;
    {
        auto reply = client_->CreateFile(ROOT_ID, "||||||?/>|", file_size, "",  false);
        wait_for(reply);
        ASSERT_TRUE(reply.isValid());
        upload_id = reply.argumentAt<0>();
        socket = reply.argumentAt<1>();
    }

    auto reply = client_->FinishUpload(upload_id);
    wait_for(reply);
    ASSERT_TRUE(reply.isError());
    EXPECT_EQ(REMOTE_EXCEPTION, reply.error().name());
    EXPECT_TRUE(reply.error().message().contains("parameter [name] is over max length or illegal"));
}

TEST_F(McloudProviderTest, finish_upload_early)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    //request 1024 bytes and only send 5 bytes then finsih_upload.
    std::string upload_file_path = tmpdir() + "/" + SPRINT_TEST_FILE;   
    size_t file_size = 1024;
    create_file(upload_file_path, file_size);

    QString upload_id;
    QDBusUnixFileDescriptor socket;
    {
        auto reply = client_->CreateFile(ROOT_ID, SPRINT_TEST_FILE, file_size, "",  false);
        wait_for(reply);
        ASSERT_TRUE(reply.isValid());
        upload_id = reply.argumentAt<0>();
        socket = reply.argumentAt<1>();
    }

    auto app = QCoreApplication::instance();
    std::ifstream payload(upload_file_path);
    QSocketNotifier notifier(socket.fileDescriptor(), QSocketNotifier::Write);
    QObject::connect(&notifier, &QSocketNotifier::activated,
             [&payload, app, &notifier](int fd) {
                char buf[5];
                auto length = payload.readsome(buf, 5);
                ssize_t n_write = write(fd, buf, length);
                if (n_write == 5)
                {   
                    // Shut down early
                    notifier.setEnabled(false);
                    app->quit();
                }
             }); 

    notifier.setEnabled(true);
    app->exec();

    //Wait first 5 bytes are written
    QTimer timer;
    timer.setSingleShot(true);
    timer.setInterval(1000);
    timer.start();
    QSignalSpy timer_spy(&timer, &QTimer::timeout);
    ASSERT_TRUE(timer_spy.wait());

    ASSERT_EQ(0, shutdown(socket.fileDescriptor(), SHUT_WR));
    auto reply = client_->FinishUpload(upload_id);
    wait_for(reply);
    auto item = reply.value();
    ASSERT_TRUE(reply.isError());
    EXPECT_TRUE(reply.error().message().contains("Failed to open/read local data from file/application"));
}

TEST_F(McloudProviderTest, cancel_upload)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));
  
    std::string upload_file_path = tmpdir() + "/" + SPRINT_TEST_FILE;
    std::default_random_engine engine;
    std::uniform_int_distribution<int> dis(1024 * 512, 1024 * 1024);
    size_t file_size = dis(engine);
    create_file(upload_file_path, file_size);

    QString upload_id;
    QDBusUnixFileDescriptor socket;
    {
        auto reply = client_->CreateFile(ROOT_ID, SPRINT_UPDATE_FILE, file_size, "", false);
        wait_for(reply);
        ASSERT_TRUE(reply.isValid());
        upload_id = reply.argumentAt<0>();
        socket = reply.argumentAt<1>();
    }

	auto reply = client_->CancelUpload(upload_id);
	wait_for(reply);
	ASSERT_TRUE(reply.isValid());
}

TEST_F(McloudProviderTest, delete_)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    auto reply = client_->CreateFolder(ROOT_ID ,SPRINT_TEST_FOLDER);
    wait_for(reply);
    ASSERT_TRUE(reply.isValid());
    auto item = reply.value();
    auto new_folder_id = item.item_id;
    EXPECT_TRUE(new_folder_id.contains(ITEM_PREFIX));
    EXPECT_EQ(QVector<QString>{ROOT_ID}, item.parent_ids);
    EXPECT_TRUE(item.name.contains(SPRINT_TEST_FOLDER));

    auto reply1 = client_->Delete(new_folder_id);
    wait_for(reply1);
    ASSERT_TRUE(reply1.isValid());
}

TEST_F(McloudProviderTest, download)
{
    set_provider(unique_ptr<ProviderBase>(new McloudProvider()));

    QString download_id;
    QDBusUnixFileDescriptor socket;
    {
        auto reply = client_->Download(PNG_ID);
        wait_for(reply);
        ASSERT_TRUE(reply.isValid());
        download_id = reply.argumentAt<0>();
        socket = reply.argumentAt<1>();
    }

    std::string data;
    auto app = QCoreApplication::instance();
    QSocketNotifier notifier(socket.fileDescriptor(), QSocketNotifier::Read);
    QObject::connect(&notifier, &QSocketNotifier::activated,
                     [&data, app, &notifier](int fd) {
                         char buf[4096];
                         ssize_t n_read = read(fd, buf, sizeof(buf));
                         if (n_read <= 0)
                         {
                             // Error or end of file
                             notifier.setEnabled(false);
                             app->quit();
                         }
                         else
                         {
                             data += string(buf, n_read);
                         }
                     });
    notifier.setEnabled(true);
    app->exec();
    auto reply = client_->FinishDownload(download_id);
    wait_for(reply);
    ASSERT_TRUE(reply.isValid());

    EXPECT_GT(data.length(), 0);
}

int main(int argc, char **argv)
{
    QCoreApplication app(argc, argv);
    qDBusRegisterMetaType<unity::storage::internal::ItemMetadata>();
    qDBusRegisterMetaType<QList<unity::storage::internal::ItemMetadata>>();
    ::testing::InitGoogleTest(&argc, argv);
                            
    return RUN_ALL_TESTS();
}
