"""
Nothing here but dictionaries for generating LinearSegmentedColormaps,
and a dictionary of these dictionaries.

"""

import numpy as np

_binary_data = {
    'red'  :  ((0., 1., 1.), (1., 0., 0.)),
    'green':  ((0., 1., 1.), (1., 0., 0.)),
    'blue' :  ((0., 1., 1.), (1., 0., 0.))
    }

_autumn_data = {'red':   ((0., 1.0, 1.0),(1.0, 1.0, 1.0)),
                'green': ((0., 0., 0.),(1.0, 1.0, 1.0)),
                'blue':  ((0., 0., 0.),(1.0, 0., 0.))}

_bone_data = {'red':   ((0., 0., 0.),(0.746032, 0.652778, 0.652778),(1.0, 1.0, 1.0)),
              'green': ((0., 0., 0.),(0.365079, 0.319444, 0.319444),
                        (0.746032, 0.777778, 0.777778),(1.0, 1.0, 1.0)),
              'blue':  ((0., 0., 0.),(0.365079, 0.444444, 0.444444),(1.0, 1.0, 1.0))}

_cool_data = {'red':   ((0., 0., 0.), (1.0, 1.0, 1.0)),
              'green': ((0., 1., 1.), (1.0, 0.,  0.)),
              'blue':  ((0., 1., 1.), (1.0, 1.,  1.))}

_copper_data = {'red':   ((0., 0., 0.),(0.809524, 1.000000, 1.000000),(1.0, 1.0, 1.0)),
                'green': ((0., 0., 0.),(1.0, 0.7812, 0.7812)),
                'blue':  ((0., 0., 0.),(1.0, 0.4975, 0.4975))}

_flag_data = {
        'red': lambda x: 0.75 * np.sin((x * 31.5 + 0.25) * np.pi) + 0.5,
        'green': lambda x: np.sin(x * 31.5 * np.pi),
        'blue': lambda x: 0.75 * np.sin((x * 31.5 - 0.25) * np.pi) + 0.5,
}

_prism_data = {
        'red': lambda x: 0.75 * np.sin((x * 20.9 + 0.25) * np.pi) + 0.67,
        'green': lambda x: 0.75 * np.sin((x * 20.9 - 0.25) * np.pi) + 0.33,
        'blue': lambda x: -1.1 * np.sin((x * 20.9) * np.pi),
}

def cubehelix(gamma=1.0, s=0.5, r=-1.5, h=1.0):
    """Return custom data dictionary of (r,g,b) conversion functions, which
    can be used with :func:`register_cmap`, for the cubehelix color scheme.

    Unlike most other color schemes cubehelix was designed by D.A. Green to
    be monotonically increasing in terms of perceived brightness.
    Also, when printed on a black and white postscript printer, the scheme
    results in a greyscale with monotonically increasing brightness.
    This color scheme is named cubehelix because the r,g,b values produced
    can be visualised as a squashed helix around the diagonal in the
    r,g,b color cube.

    For a unit color cube (i.e. 3-D coordinates for r,g,b each in the
    range 0 to 1) the color scheme starts at (r,g,b) = (0,0,0), i.e. black,
    and finishes at (r,g,b) = (1,1,1), i.e. white. For some fraction *x*,
    between 0 and 1, the color is the corresponding grey value at that
    fraction along the black to white diagonal (x,x,x) plus a color
    element. This color element is calculated in a plane of constant
    perceived intensity and controlled by the following parameters.

    Optional keyword arguments:

      =========   =======================================================
      Keyword     Description
      =========   =======================================================
      gamma       gamma factor to emphasise either low intensity values
                  (gamma < 1), or high intensity values (gamma > 1);
                  defaults to 1.0.
      s           the start color; defaults to 0.5 (i.e. purple).
      r           the number of r,g,b rotations in color that are made
                  from the start to the end of the color scheme; defaults
                  to -1.5 (i.e. -> B -> G -> R -> B).
      h           the hue parameter which controls how saturated the
                  colors are. If this parameter is zero then the color
                  scheme is purely a greyscale; defaults to 1.0.
      =========   =======================================================

    """

    def get_color_function(p0, p1):
        def color(x):
            # Apply gamma factor to emphasise low or high intensity values
            xg = x**gamma

            # Calculate amplitude and angle of deviation from the black
            # to white diagonal in the plane of constant
            # perceived intensity.
            a = h * xg * (1 - xg) / 2

            phi = 2 * np.pi * (s / 3 + r * x)

            return xg + a * (p0 * np.cos(phi) + p1 * np.sin(phi))
        return color

    return {
            'red': get_color_function(-0.14861, 1.78277),
            'green': get_color_function(-0.29227, -0.90649),
            'blue': get_color_function(1.97294, 0.0),
    }

_cubehelix_data = cubehelix()

_bwr_data = ((0.0, 0.0, 1.0), (1.0, 1.0, 1.0), (1.0, 0.0, 0.0))
_brg_data = ((0.0, 0.0, 1.0), (1.0, 0.0, 0.0), (0.0, 1.0, 0.0))

# Gnuplot palette functions
gfunc = {
        0: lambda x: 0,
        1: lambda x: 0.5,
        2: lambda x: 1,
        3: lambda x: x,
        4: lambda x: x**2,
        5: lambda x: x**3,
        6: lambda x: x**4,
        7: lambda x: np.sqrt(x),
        8: lambda x: np.sqrt(np.sqrt(x)),
        9: lambda x: np.sin(x * np.pi / 2),
        10: lambda x: np.cos(x * np.pi / 2),
        11: lambda x: np.abs(x - 0.5),
        12: lambda x: (2 * x - 1)**2,
        13: lambda x: np.sin(x * np.pi),
        14: lambda x: np.abs(np.cos(x * np.pi)),
        15: lambda x: np.sin(x * 2 * np.pi),
        16: lambda x: np.cos(x * 2 * np.pi),
        17: lambda x: np.abs(np.sin(x * 2 * np.pi)),
        18: lambda x: np.abs(np.cos(x * 2 * np.pi)),
        19: lambda x: np.abs(np.sin(x * 4 * np.pi)),
        20: lambda x: np.abs(np.cos(x * 4 * np.pi)),
        21: lambda x: 3 * x,
        22: lambda x: 3 * x - 1,
        23: lambda x: 3 * x - 2,
        24: lambda x: np.abs(3 * x - 1),
        25: lambda x: np.abs(3 * x - 2),
        26: lambda x: (3 * x - 1) / 2,
        27: lambda x: (3 * x - 2) / 2,
        28: lambda x: np.abs((3 * x  - 1) / 2),
        29: lambda x: np.abs((3 * x  - 2) / 2),
        30: lambda x: x / 0.32 - 0.78125,
        31: lambda x: 2 * x - 0.84,
        32: lambda x: gfunc32(x),
        33: lambda x: np.abs(2 * x - 0.5),
        34: lambda x: 2 * x,
        35: lambda x: 2 * x - 0.5,
        36: lambda x: 2 * x - 1.
}

def gfunc32(x):
    ret = np.zeros(len(x))
    m = (x < 0.25)
    ret[m] = 4 * x[m]
    m = (x >= 0.25) & (x < 0.92)
    ret[m] = -2 * x[m] + 1.84
    m = (x >= 0.92)
    ret[m] = x[m] / 0.08 - 11.5
    return ret

_gnuplot_data = {
        'red': gfunc[7],
        'green': gfunc[5],
        'blue': gfunc[15],
}

_gnuplot2_data = {
        'red': gfunc[30],
        'green': gfunc[31],
        'blue': gfunc[32],
}

_ocean_data = {
        'red': gfunc[23],
        'green': gfunc[28],
        'blue': gfunc[3],
}

_afmhot_data = {
        'red': gfunc[34],
        'green': gfunc[35],
        'blue': gfunc[36],
}

_rainbow_data = {
        'red': gfunc[33],
        'green': gfunc[13],
        'blue': gfunc[10],
}

_seismic_data = (
        (0.0, 0.0, 0.3), (0.0, 0.0, 1.0),
        (1.0, 1.0, 1.0), (1.0, 0.0, 0.0),
        (0.5, 0.0, 0.0))

_terrain_data = (
        (0.00, (0.2, 0.2, 0.6)),
        (0.15, (0.0, 0.6, 1.0)),
        (0.25, (0.0, 0.8, 0.4)),
        (0.50, (1.0, 1.0, 0.6)),
        (0.75, (0.5, 0.36, 0.33)),
        (1.00, (1.0, 1.0, 1.0)))

_gray_data =  {'red':   ((0., 0, 0), (1., 1, 1)),
               'green': ((0., 0, 0), (1., 1, 1)),
               'blue':  ((0., 0, 0), (1., 1, 1))}

_hot_data = {'red':   ((0., 0.0416, 0.0416),(0.365079, 1.000000, 1.000000),(1.0, 1.0, 1.0)),
             'green': ((0., 0., 0.),(0.365079, 0.000000, 0.000000),
                       (0.746032, 1.000000, 1.000000),(1.0, 1.0, 1.0)),
             'blue':  ((0., 0., 0.),(0.746032, 0.000000, 0.000000),(1.0, 1.0, 1.0))}

_hsv_data = {'red':   ((0., 1., 1.),(0.158730, 1.000000, 1.000000),
                       (0.174603, 0.968750, 0.968750),(0.333333, 0.031250, 0.031250),
                       (0.349206, 0.000000, 0.000000),(0.666667, 0.000000, 0.000000),
                       (0.682540, 0.031250, 0.031250),(0.841270, 0.968750, 0.968750),
                       (0.857143, 1.000000, 1.000000),(1.0, 1.0, 1.0)),
             'green': ((0., 0., 0.),(0.158730, 0.937500, 0.937500),
                       (0.174603, 1.000000, 1.000000),(0.507937, 1.000000, 1.000000),
                       (0.666667, 0.062500, 0.062500),(0.682540, 0.000000, 0.000000),
                       (1.0, 0., 0.)),
             'blue':  ((0., 0., 0.),(0.333333, 0.000000, 0.000000),
                       (0.349206, 0.062500, 0.062500),(0.507937, 1.000000, 1.000000),
                       (0.841270, 1.000000, 1.000000),(0.857143, 0.937500, 0.937500),
                       (1.0, 0.09375, 0.09375))}

_jet_data =   {'red':   ((0., 0, 0), (0.35, 0, 0), (0.66, 1, 1), (0.89,1, 1),
                         (1, 0.5, 0.5)),
               'green': ((0., 0, 0), (0.125,0, 0), (0.375,1, 1), (0.64,1, 1),
                         (0.91,0,0), (1, 0, 0)),
               'blue':  ((0., 0.5, 0.5), (0.11, 1, 1), (0.34, 1, 1), (0.65,0, 0),
                         (1, 0, 0))}

_pink_data = {'red':   ((0., 0.1178, 0.1178),(0.015873, 0.195857, 0.195857),
                        (0.031746, 0.250661, 0.250661),(0.047619, 0.295468, 0.295468),
                        (0.063492, 0.334324, 0.334324),(0.079365, 0.369112, 0.369112),
                        (0.095238, 0.400892, 0.400892),(0.111111, 0.430331, 0.430331),
                        (0.126984, 0.457882, 0.457882),(0.142857, 0.483867, 0.483867),
                        (0.158730, 0.508525, 0.508525),(0.174603, 0.532042, 0.532042),
                        (0.190476, 0.554563, 0.554563),(0.206349, 0.576204, 0.576204),
                        (0.222222, 0.597061, 0.597061),(0.238095, 0.617213, 0.617213),
                        (0.253968, 0.636729, 0.636729),(0.269841, 0.655663, 0.655663),
                        (0.285714, 0.674066, 0.674066),(0.301587, 0.691980, 0.691980),
                        (0.317460, 0.709441, 0.709441),(0.333333, 0.726483, 0.726483),
                        (0.349206, 0.743134, 0.743134),(0.365079, 0.759421, 0.759421),
                        (0.380952, 0.766356, 0.766356),(0.396825, 0.773229, 0.773229),
                        (0.412698, 0.780042, 0.780042),(0.428571, 0.786796, 0.786796),
                        (0.444444, 0.793492, 0.793492),(0.460317, 0.800132, 0.800132),
                        (0.476190, 0.806718, 0.806718),(0.492063, 0.813250, 0.813250),
                        (0.507937, 0.819730, 0.819730),(0.523810, 0.826160, 0.826160),
                        (0.539683, 0.832539, 0.832539),(0.555556, 0.838870, 0.838870),
                        (0.571429, 0.845154, 0.845154),(0.587302, 0.851392, 0.851392),
                        (0.603175, 0.857584, 0.857584),(0.619048, 0.863731, 0.863731),
                        (0.634921, 0.869835, 0.869835),(0.650794, 0.875897, 0.875897),
                        (0.666667, 0.881917, 0.881917),(0.682540, 0.887896, 0.887896),
                        (0.698413, 0.893835, 0.893835),(0.714286, 0.899735, 0.899735),
                        (0.730159, 0.905597, 0.905597),(0.746032, 0.911421, 0.911421),
                        (0.761905, 0.917208, 0.917208),(0.777778, 0.922958, 0.922958),
                        (0.793651, 0.928673, 0.928673),(0.809524, 0.934353, 0.934353),
                        (0.825397, 0.939999, 0.939999),(0.841270, 0.945611, 0.945611),
                        (0.857143, 0.951190, 0.951190),(0.873016, 0.956736, 0.956736),
                        (0.888889, 0.962250, 0.962250),(0.904762, 0.967733, 0.967733),
                        (0.920635, 0.973185, 0.973185),(0.936508, 0.978607, 0.978607),
                        (0.952381, 0.983999, 0.983999),(0.968254, 0.989361, 0.989361),
                        (0.984127, 0.994695, 0.994695),(1.0, 1.0, 1.0)),
              'green': ((0., 0., 0.),(0.015873, 0.102869, 0.102869),
                        (0.031746, 0.145479, 0.145479),(0.047619, 0.178174, 0.178174),
                        (0.063492, 0.205738, 0.205738),(0.079365, 0.230022, 0.230022),
                        (0.095238, 0.251976, 0.251976),(0.111111, 0.272166, 0.272166),
                        (0.126984, 0.290957, 0.290957),(0.142857, 0.308607, 0.308607),
                        (0.158730, 0.325300, 0.325300),(0.174603, 0.341178, 0.341178),
                        (0.190476, 0.356348, 0.356348),(0.206349, 0.370899, 0.370899),
                        (0.222222, 0.384900, 0.384900),(0.238095, 0.398410, 0.398410),
                        (0.253968, 0.411476, 0.411476),(0.269841, 0.424139, 0.424139),
                        (0.285714, 0.436436, 0.436436),(0.301587, 0.448395, 0.448395),
                        (0.317460, 0.460044, 0.460044),(0.333333, 0.471405, 0.471405),
                        (0.349206, 0.482498, 0.482498),(0.365079, 0.493342, 0.493342),
                        (0.380952, 0.517549, 0.517549),(0.396825, 0.540674, 0.540674),
                        (0.412698, 0.562849, 0.562849),(0.428571, 0.584183, 0.584183),
                        (0.444444, 0.604765, 0.604765),(0.460317, 0.624669, 0.624669),
                        (0.476190, 0.643958, 0.643958),(0.492063, 0.662687, 0.662687),
                        (0.507937, 0.680900, 0.680900),(0.523810, 0.698638, 0.698638),
                        (0.539683, 0.715937, 0.715937),(0.555556, 0.732828, 0.732828),
                        (0.571429, 0.749338, 0.749338),(0.587302, 0.765493, 0.765493),
                        (0.603175, 0.781313, 0.781313),(0.619048, 0.796819, 0.796819),
                        (0.634921, 0.812029, 0.812029),(0.650794, 0.826960, 0.826960),
                        (0.666667, 0.841625, 0.841625),(0.682540, 0.856040, 0.856040),
                        (0.698413, 0.870216, 0.870216),(0.714286, 0.884164, 0.884164),
                        (0.730159, 0.897896, 0.897896),(0.746032, 0.911421, 0.911421),
                        (0.761905, 0.917208, 0.917208),(0.777778, 0.922958, 0.922958),
                        (0.793651, 0.928673, 0.928673),(0.809524, 0.934353, 0.934353),
                        (0.825397, 0.939999, 0.939999),(0.841270, 0.945611, 0.945611),
                        (0.857143, 0.951190, 0.951190),(0.873016, 0.956736, 0.956736),
                        (0.888889, 0.962250, 0.962250),(0.904762, 0.967733, 0.967733),
                        (0.920635, 0.973185, 0.973185),(0.936508, 0.978607, 0.978607),
                        (0.952381, 0.983999, 0.983999),(0.968254, 0.989361, 0.989361),
                        (0.984127, 0.994695, 0.994695),(1.0, 1.0, 1.0)),
              'blue':  ((0., 0., 0.),(0.015873, 0.102869, 0.102869),
                        (0.031746, 0.145479, 0.145479),(0.047619, 0.178174, 0.178174),
                        (0.063492, 0.205738, 0.205738),(0.079365, 0.230022, 0.230022),
                        (0.095238, 0.251976, 0.251976),(0.111111, 0.272166, 0.272166),
                        (0.126984, 0.290957, 0.290957),(0.142857, 0.308607, 0.308607),
                        (0.158730, 0.325300, 0.325300),(0.174603, 0.341178, 0.341178),
                        (0.190476, 0.356348, 0.356348),(0.206349, 0.370899, 0.370899),
                        (0.222222, 0.384900, 0.384900),(0.238095, 0.398410, 0.398410),
                        (0.253968, 0.411476, 0.411476),(0.269841, 0.424139, 0.424139),
                        (0.285714, 0.436436, 0.436436),(0.301587, 0.448395, 0.448395),
                        (0.317460, 0.460044, 0.460044),(0.333333, 0.471405, 0.471405),
                        (0.349206, 0.482498, 0.482498),(0.365079, 0.493342, 0.493342),
                        (0.380952, 0.503953, 0.503953),(0.396825, 0.514344, 0.514344),
                        (0.412698, 0.524531, 0.524531),(0.428571, 0.534522, 0.534522),
                        (0.444444, 0.544331, 0.544331),(0.460317, 0.553966, 0.553966),
                        (0.476190, 0.563436, 0.563436),(0.492063, 0.572750, 0.572750),
                        (0.507937, 0.581914, 0.581914),(0.523810, 0.590937, 0.590937),
                        (0.539683, 0.599824, 0.599824),(0.555556, 0.608581, 0.608581),
                        (0.571429, 0.617213, 0.617213),(0.587302, 0.625727, 0.625727),
                        (0.603175, 0.634126, 0.634126),(0.619048, 0.642416, 0.642416),
                        (0.634921, 0.650600, 0.650600),(0.650794, 0.658682, 0.658682),
                        (0.666667, 0.666667, 0.666667),(0.682540, 0.674556, 0.674556),
                        (0.698413, 0.682355, 0.682355),(0.714286, 0.690066, 0.690066),
                        (0.730159, 0.697691, 0.697691),(0.746032, 0.705234, 0.705234),
                        (0.761905, 0.727166, 0.727166),(0.777778, 0.748455, 0.748455),
                        (0.793651, 0.769156, 0.769156),(0.809524, 0.789314, 0.789314),
                        (0.825397, 0.808969, 0.808969),(0.841270, 0.828159, 0.828159),
                        (0.857143, 0.846913, 0.846913),(0.873016, 0.865261, 0.865261),
                        (0.888889, 0.883229, 0.883229),(0.904762, 0.900837, 0.900837),
                        (0.920635, 0.918109, 0.918109),(0.936508, 0.935061, 0.935061),
                        (0.952381, 0.951711, 0.951711),(0.968254, 0.968075, 0.968075),
                        (0.984127, 0.984167, 0.984167),(1.0, 1.0, 1.0))}

_spring_data = {'red':   ((0., 1., 1.),(1.0, 1.0, 1.0)),
                'green': ((0., 0., 0.),(1.0, 1.0, 1.0)),
                'blue':  ((0., 1., 1.),(1.0, 0.0, 0.0))}


_summer_data = {'red':   ((0., 0., 0.),(1.0, 1.0, 1.0)),
                'green': ((0., 0.5, 0.5),(1.0, 1.0, 1.0)),
                'blue':  ((0., 0.4, 0.4),(1.0, 0.4, 0.4))}


_winter_data = {'red':   ((0., 0., 0.),(1.0, 0.0, 0.0)),
                'green': ((0., 0., 0.),(1.0, 1.0, 1.0)),
                'blue':  ((0., 1., 1.),(1.0, 0.5, 0.5))}

_spectral_data = {'red': [(0.0, 0.0, 0.0), (0.05, 0.4667, 0.4667),
                          (0.10, 0.5333, 0.5333), (0.15, 0.0, 0.0),
                          (0.20, 0.0, 0.0), (0.25, 0.0, 0.0),
                          (0.30, 0.0, 0.0), (0.35, 0.0, 0.0),
                          (0.40, 0.0, 0.0), (0.45, 0.0, 0.0),
                          (0.50, 0.0, 0.0), (0.55, 0.0, 0.0),
                          (0.60, 0.0, 0.0), (0.65, 0.7333, 0.7333),
                          (0.70, 0.9333, 0.9333), (0.75, 1.0, 1.0),
                          (0.80, 1.0, 1.0), (0.85, 1.0, 1.0),
                          (0.90, 0.8667, 0.8667), (0.95, 0.80, 0.80),
                          (1.0, 0.80, 0.80)],
                  'green': [(0.0, 0.0, 0.0), (0.05, 0.0, 0.0),
                            (0.10, 0.0, 0.0), (0.15, 0.0, 0.0),
                            (0.20, 0.0, 0.0), (0.25, 0.4667, 0.4667),
                            (0.30, 0.6000, 0.6000), (0.35, 0.6667, 0.6667),
                            (0.40, 0.6667, 0.6667), (0.45, 0.6000, 0.6000),
                            (0.50, 0.7333, 0.7333), (0.55, 0.8667, 0.8667),
                            (0.60, 1.0, 1.0), (0.65, 1.0, 1.0),
                            (0.70, 0.9333, 0.9333), (0.75, 0.8000, 0.8000),
                            (0.80, 0.6000, 0.6000), (0.85, 0.0, 0.0),
                            (0.90, 0.0, 0.0), (0.95, 0.0, 0.0),
                            (1.0, 0.80, 0.80)],
                  'blue': [(0.0, 0.0, 0.0), (0.05, 0.5333, 0.5333),
                           (0.10, 0.6000, 0.6000), (0.15, 0.6667, 0.6667),
                           (0.20, 0.8667, 0.8667), (0.25, 0.8667, 0.8667),
                           (0.30, 0.8667, 0.8667), (0.35, 0.6667, 0.6667),
                           (0.40, 0.5333, 0.5333), (0.45, 0.0, 0.0),
                           (0.5, 0.0, 0.0), (0.55, 0.0, 0.0),
                           (0.60, 0.0, 0.0), (0.65, 0.0, 0.0),
                           (0.70, 0.0, 0.0), (0.75, 0.0, 0.0),
                           (0.80, 0.0, 0.0), (0.85, 0.0, 0.0),
                           (0.90, 0.0, 0.0), (0.95, 0.0, 0.0),
                           (1.0, 0.80, 0.80)]}


# 34 colormaps based on color specifications and designs
# developed by Cynthia Brewer (http://colorbrewer.org).
# The ColorBrewer palettes have been included under the terms
# of an Apache-stype license (for details, see the file
# LICENSE_COLORBREWER in the license directory of the matplotlib
# source distribution).

_Accent_data = {'blue': [(0.0, 0.49803921580314636,
0.49803921580314636), (0.14285714285714285, 0.83137255907058716,
0.83137255907058716), (0.2857142857142857, 0.52549022436141968,
0.52549022436141968), (0.42857142857142855, 0.60000002384185791,
0.60000002384185791), (0.5714285714285714, 0.69019609689712524,
0.69019609689712524), (0.7142857142857143, 0.49803921580314636,
0.49803921580314636), (0.8571428571428571, 0.090196080505847931,
0.090196080505847931), (1.0, 0.40000000596046448,
0.40000000596046448)],

    'green': [(0.0, 0.78823530673980713, 0.78823530673980713),
    (0.14285714285714285, 0.68235296010971069, 0.68235296010971069),
    (0.2857142857142857, 0.75294119119644165, 0.75294119119644165),
    (0.42857142857142855, 1.0, 1.0), (0.5714285714285714,
    0.42352941632270813, 0.42352941632270813), (0.7142857142857143,
    0.0078431377187371254, 0.0078431377187371254),
    (0.8571428571428571, 0.35686275362968445, 0.35686275362968445),
    (1.0, 0.40000000596046448, 0.40000000596046448)],

    'red': [(0.0, 0.49803921580314636, 0.49803921580314636),
    (0.14285714285714285, 0.7450980544090271, 0.7450980544090271),
    (0.2857142857142857, 0.99215686321258545, 0.99215686321258545),
    (0.42857142857142855, 1.0, 1.0), (0.5714285714285714,
    0.21960784494876862, 0.21960784494876862), (0.7142857142857143,
    0.94117647409439087, 0.94117647409439087), (0.8571428571428571,
    0.74901962280273438, 0.74901962280273438), (1.0,
    0.40000000596046448, 0.40000000596046448)]}

_Blues_data = {'blue': [(0.0, 1.0, 1.0), (0.125, 0.9686274528503418,
0.9686274528503418), (0.25, 0.93725490570068359, 0.93725490570068359),
(0.375, 0.88235294818878174, 0.88235294818878174), (0.5,
0.83921569585800171, 0.83921569585800171), (0.625, 0.7764706015586853,
0.7764706015586853), (0.75, 0.70980393886566162, 0.70980393886566162),
(0.875, 0.61176472902297974, 0.61176472902297974), (1.0,
0.41960784792900085, 0.41960784792900085)],

    'green': [(0.0, 0.9843137264251709, 0.9843137264251709), (0.125,
    0.92156863212585449, 0.92156863212585449), (0.25,
    0.85882353782653809, 0.85882353782653809), (0.375,
    0.7921568751335144, 0.7921568751335144), (0.5,
    0.68235296010971069, 0.68235296010971069), (0.625,
    0.57254904508590698, 0.57254904508590698), (0.75,
    0.44313725829124451, 0.44313725829124451), (0.875,
    0.31764706969261169, 0.31764706969261169), (1.0,
    0.18823529779911041, 0.18823529779911041)],

    'red': [(0.0, 0.9686274528503418, 0.9686274528503418), (0.125,
    0.87058824300765991, 0.87058824300765991), (0.25,
    0.7764706015586853, 0.7764706015586853), (0.375,
    0.61960786581039429, 0.61960786581039429), (0.5,
    0.41960784792900085, 0.41960784792900085), (0.625,
    0.25882354378700256, 0.25882354378700256), (0.75,
    0.12941177189350128, 0.12941177189350128), (0.875,
    0.031372550874948502, 0.031372550874948502), (1.0,
    0.031372550874948502, 0.031372550874948502)]}

_BrBG_data = {'blue': [(0.0, 0.019607843831181526,
0.019607843831181526), (0.10000000000000001, 0.039215687662363052,
0.039215687662363052), (0.20000000000000001, 0.17647059261798859,
0.17647059261798859), (0.29999999999999999, 0.49019607901573181,
0.49019607901573181), (0.40000000000000002, 0.76470589637756348,
0.76470589637756348), (0.5, 0.96078431606292725, 0.96078431606292725),
(0.59999999999999998, 0.89803922176361084, 0.89803922176361084),
(0.69999999999999996, 0.75686275959014893, 0.75686275959014893),
(0.80000000000000004, 0.56078433990478516, 0.56078433990478516),
(0.90000000000000002, 0.36862745881080627, 0.36862745881080627), (1.0,
0.18823529779911041, 0.18823529779911041)],

    'green': [(0.0, 0.18823529779911041, 0.18823529779911041),
    (0.10000000000000001, 0.31764706969261169, 0.31764706969261169),
    (0.20000000000000001, 0.5058823823928833, 0.5058823823928833),
    (0.29999999999999999, 0.7607843279838562, 0.7607843279838562),
    (0.40000000000000002, 0.90980392694473267, 0.90980392694473267),
    (0.5, 0.96078431606292725, 0.96078431606292725),
    (0.59999999999999998, 0.91764706373214722, 0.91764706373214722),
    (0.69999999999999996, 0.80392158031463623, 0.80392158031463623),
    (0.80000000000000004, 0.59215688705444336, 0.59215688705444336),
    (0.90000000000000002, 0.40000000596046448, 0.40000000596046448),
    (1.0, 0.23529411852359772, 0.23529411852359772)],

    'red': [(0.0, 0.32941177487373352, 0.32941177487373352),
    (0.10000000000000001, 0.54901963472366333, 0.54901963472366333),
    (0.20000000000000001, 0.74901962280273438, 0.74901962280273438),
    (0.29999999999999999, 0.87450981140136719, 0.87450981140136719),
    (0.40000000000000002, 0.96470588445663452, 0.96470588445663452),
    (0.5, 0.96078431606292725, 0.96078431606292725),
    (0.59999999999999998, 0.78039216995239258, 0.78039216995239258),
    (0.69999999999999996, 0.50196081399917603, 0.50196081399917603),
    (0.80000000000000004, 0.20784313976764679, 0.20784313976764679),
    (0.90000000000000002, 0.0039215688593685627,
    0.0039215688593685627), (1.0, 0.0, 0.0)]}

_BuGn_data = {'blue': [(0.0, 0.99215686321258545,
0.99215686321258545), (0.125, 0.97647058963775635,
0.97647058963775635), (0.25, 0.90196079015731812,
0.90196079015731812), (0.375, 0.78823530673980713,
0.78823530673980713), (0.5, 0.64313727617263794, 0.64313727617263794),
(0.625, 0.46274510025978088, 0.46274510025978088), (0.75,
0.27058824896812439, 0.27058824896812439), (0.875,
0.17254902422428131, 0.17254902422428131), (1.0, 0.10588235408067703,
0.10588235408067703)],

    'green': [(0.0, 0.98823529481887817, 0.98823529481887817), (0.125,
    0.96078431606292725, 0.96078431606292725), (0.25,
    0.92549020051956177, 0.92549020051956177), (0.375,
    0.84705883264541626, 0.84705883264541626), (0.5,
    0.7607843279838562, 0.7607843279838562), (0.625,
    0.68235296010971069, 0.68235296010971069), (0.75,
    0.54509806632995605, 0.54509806632995605), (0.875,
    0.42745098471641541, 0.42745098471641541), (1.0,
    0.26666668057441711, 0.26666668057441711)], 'red': [(0.0,
    0.9686274528503418, 0.9686274528503418), (0.125,
    0.89803922176361084, 0.89803922176361084), (0.25,
    0.80000001192092896, 0.80000001192092896), (0.375,
    0.60000002384185791, 0.60000002384185791), (0.5,
    0.40000000596046448, 0.40000000596046448), (0.625,
    0.25490197539329529, 0.25490197539329529), (0.75,
    0.13725490868091583, 0.13725490868091583), (0.875, 0.0, 0.0),
    (1.0, 0.0, 0.0)]}

_BuPu_data = {'blue': [(0.0, 0.99215686321258545,
0.99215686321258545), (0.125, 0.95686274766921997,
0.95686274766921997), (0.25, 0.90196079015731812,
0.90196079015731812), (0.375, 0.85490196943283081,
0.85490196943283081), (0.5, 0.7764706015586853, 0.7764706015586853),
(0.625, 0.69411766529083252, 0.69411766529083252), (0.75,
0.61568629741668701, 0.61568629741668701), (0.875,
0.48627451062202454, 0.48627451062202454), (1.0, 0.29411765933036804,
0.29411765933036804)],

    'green': [(0.0, 0.98823529481887817, 0.98823529481887817), (0.125,
    0.92549020051956177, 0.92549020051956177), (0.25,
    0.82745099067687988, 0.82745099067687988), (0.375,
    0.73725491762161255, 0.73725491762161255), (0.5,
    0.58823531866073608, 0.58823531866073608), (0.625,
    0.41960784792900085, 0.41960784792900085), (0.75,
    0.25490197539329529, 0.25490197539329529), (0.875,
    0.058823529630899429, 0.058823529630899429), (1.0, 0.0, 0.0)],

    'red': [(0.0, 0.9686274528503418, 0.9686274528503418), (0.125,
    0.87843137979507446, 0.87843137979507446), (0.25,
    0.74901962280273438, 0.74901962280273438), (0.375,
    0.61960786581039429, 0.61960786581039429), (0.5,
    0.54901963472366333, 0.54901963472366333), (0.625,
    0.54901963472366333, 0.54901963472366333), (0.75,
    0.53333336114883423, 0.53333336114883423), (0.875,
    0.5058823823928833, 0.5058823823928833), (1.0,
    0.30196079611778259, 0.30196079611778259)]}

_Dark2_data = {'blue': [(0.0, 0.46666666865348816,
0.46666666865348816), (0.14285714285714285, 0.0078431377187371254,
0.0078431377187371254), (0.2857142857142857, 0.70196080207824707,
0.70196080207824707), (0.42857142857142855, 0.54117649793624878,
0.54117649793624878), (0.5714285714285714, 0.11764705926179886,
0.11764705926179886), (0.7142857142857143, 0.0078431377187371254,
0.0078431377187371254), (0.8571428571428571, 0.11372549086809158,
0.11372549086809158), (1.0, 0.40000000596046448,
0.40000000596046448)],

    'green': [(0.0, 0.61960786581039429, 0.61960786581039429),
    (0.14285714285714285, 0.37254902720451355, 0.37254902720451355),
    (0.2857142857142857, 0.43921568989753723, 0.43921568989753723),
    (0.42857142857142855, 0.16078431904315948, 0.16078431904315948),
    (0.5714285714285714, 0.65098041296005249, 0.65098041296005249),
    (0.7142857142857143, 0.67058825492858887, 0.67058825492858887),
    (0.8571428571428571, 0.46274510025978088, 0.46274510025978088),
    (1.0, 0.40000000596046448, 0.40000000596046448)],

    'red': [(0.0, 0.10588235408067703, 0.10588235408067703),
    (0.14285714285714285, 0.85098040103912354, 0.85098040103912354),
    (0.2857142857142857, 0.45882353186607361, 0.45882353186607361),
    (0.42857142857142855, 0.90588235855102539, 0.90588235855102539),
    (0.5714285714285714, 0.40000000596046448, 0.40000000596046448),
    (0.7142857142857143, 0.90196079015731812, 0.90196079015731812),
    (0.8571428571428571, 0.65098041296005249, 0.65098041296005249),
    (1.0, 0.40000000596046448, 0.40000000596046448)]}

_GnBu_data = {'blue': [(0.0, 0.94117647409439087,
0.94117647409439087), (0.125, 0.85882353782653809,
0.85882353782653809), (0.25, 0.77254903316497803,
0.77254903316497803), (0.375, 0.70980393886566162,
0.70980393886566162), (0.5, 0.76862746477127075, 0.76862746477127075),
(0.625, 0.82745099067687988, 0.82745099067687988), (0.75,
0.7450980544090271, 0.7450980544090271), (0.875, 0.67450982332229614,
0.67450982332229614), (1.0, 0.5058823823928833, 0.5058823823928833)],

    'green': [(0.0, 0.98823529481887817, 0.98823529481887817), (0.125,
    0.9529411792755127, 0.9529411792755127), (0.25,
    0.92156863212585449, 0.92156863212585449), (0.375,
    0.86666667461395264, 0.86666667461395264), (0.5,
    0.80000001192092896, 0.80000001192092896), (0.625,
    0.70196080207824707, 0.70196080207824707), (0.75,
    0.54901963472366333, 0.54901963472366333), (0.875,
    0.40784314274787903, 0.40784314274787903), (1.0,
    0.25098040699958801, 0.25098040699958801)],

    'red': [(0.0, 0.9686274528503418, 0.9686274528503418), (0.125,
    0.87843137979507446, 0.87843137979507446), (0.25,
    0.80000001192092896, 0.80000001192092896), (0.375,
    0.65882354974746704, 0.65882354974746704), (0.5,
    0.48235294222831726, 0.48235294222831726), (0.625,
    0.30588236451148987, 0.30588236451148987), (0.75,
    0.16862745583057404, 0.16862745583057404), (0.875,
    0.031372550874948502, 0.031372550874948502), (1.0,
    0.031372550874948502, 0.031372550874948502)]}

_Greens_data = {'blue': [(0.0, 0.96078431606292725,
0.96078431606292725), (0.125, 0.87843137979507446,
0.87843137979507446), (0.25, 0.75294119119644165,
0.75294119119644165), (0.375, 0.60784316062927246,
0.60784316062927246), (0.5, 0.46274510025978088, 0.46274510025978088),
(0.625, 0.364705890417099, 0.364705890417099), (0.75,
0.27058824896812439, 0.27058824896812439), (0.875,
0.17254902422428131, 0.17254902422428131), (1.0, 0.10588235408067703,
0.10588235408067703)],

    'green': [(0.0, 0.98823529481887817, 0.98823529481887817), (0.125,
    0.96078431606292725, 0.96078431606292725), (0.25,
    0.91372549533843994, 0.91372549533843994), (0.375,
    0.85098040103912354, 0.85098040103912354), (0.5,
    0.76862746477127075, 0.76862746477127075), (0.625,
    0.67058825492858887, 0.67058825492858887), (0.75,
    0.54509806632995605, 0.54509806632995605), (0.875,
    0.42745098471641541, 0.42745098471641541), (1.0,
    0.26666668057441711, 0.26666668057441711)],

    'red': [(0.0, 0.9686274528503418, 0.9686274528503418), (0.125,
    0.89803922176361084, 0.89803922176361084), (0.25,
    0.78039216995239258, 0.78039216995239258), (0.375,
    0.63137257099151611, 0.63137257099151611), (0.5,
    0.45490196347236633, 0.45490196347236633), (0.625,
    0.25490197539329529, 0.25490197539329529), (0.75,
    0.13725490868091583, 0.13725490868091583), (0.875, 0.0, 0.0),
    (1.0, 0.0, 0.0)]}

_Greys_data = {'blue': [(0.0, 1.0, 1.0), (0.125, 0.94117647409439087,
0.94117647409439087), (0.25, 0.85098040103912354,
0.85098040103912354), (0.375, 0.74117648601531982,
0.74117648601531982), (0.5, 0.58823531866073608, 0.58823531866073608),
(0.625, 0.45098039507865906, 0.45098039507865906), (0.75,
0.32156863808631897, 0.32156863808631897), (0.875,
0.14509804546833038, 0.14509804546833038), (1.0, 0.0, 0.0)],

    'green': [(0.0, 1.0, 1.0), (0.125, 0.94117647409439087,
    0.94117647409439087), (0.25, 0.85098040103912354,
    0.85098040103912354), (0.375, 0.74117648601531982,
    0.74117648601531982), (0.5, 0.58823531866073608,
    0.58823531866073608), (0.625, 0.45098039507865906,
    0.45098039507865906), (0.75, 0.32156863808631897,
    0.32156863808631897), (0.875, 0.14509804546833038,
    0.14509804546833038), (1.0, 0.0, 0.0)],

    'red': [(0.0, 1.0, 1.0), (0.125, 0.94117647409439087,
    0.94117647409439087), (0.25, 0.85098040103912354,
    0.85098040103912354), (0.375, 0.74117648601531982,
    0.74117648601531982), (0.5, 0.58823531866073608,
    0.58823531866073608), (0.625, 0.45098039507865906,
    0.45098039507865906), (0.75, 0.32156863808631897,
    0.32156863808631897), (0.875, 0.14509804546833038,
    0.14509804546833038), (1.0, 0.0, 0.0)]}

_Oranges_data = {'blue': [(0.0, 0.92156863212585449,
0.92156863212585449), (0.125, 0.80784314870834351,
0.80784314870834351), (0.25, 0.63529413938522339,
0.63529413938522339), (0.375, 0.41960784792900085,
0.41960784792900085), (0.5, 0.23529411852359772, 0.23529411852359772),
(0.625, 0.074509806931018829, 0.074509806931018829), (0.75,
0.0039215688593685627, 0.0039215688593685627), (0.875,
0.011764706112444401, 0.011764706112444401), (1.0,
0.015686275437474251, 0.015686275437474251)],

    'green': [(0.0, 0.96078431606292725, 0.96078431606292725), (0.125,
    0.90196079015731812, 0.90196079015731812), (0.25,
    0.81568628549575806, 0.81568628549575806), (0.375,
    0.68235296010971069, 0.68235296010971069), (0.5,
    0.55294120311737061, 0.55294120311737061), (0.625,
    0.4117647111415863, 0.4117647111415863), (0.75,
    0.28235295414924622, 0.28235295414924622), (0.875,
    0.21176470816135406, 0.21176470816135406), (1.0,
    0.15294118225574493, 0.15294118225574493)],

    'red': [(0.0, 1.0, 1.0), (0.125, 0.99607843160629272,
    0.99607843160629272), (0.25, 0.99215686321258545,
    0.99215686321258545), (0.375, 0.99215686321258545,
    0.99215686321258545), (0.5, 0.99215686321258545,
    0.99215686321258545), (0.625, 0.94509804248809814,
    0.94509804248809814), (0.75, 0.85098040103912354,
    0.85098040103912354), (0.875, 0.65098041296005249,
    0.65098041296005249), (1.0, 0.49803921580314636,
    0.49803921580314636)]}

_OrRd_data = {'blue': [(0.0, 0.92549020051956177,
0.92549020051956177), (0.125, 0.78431373834609985,
0.78431373834609985), (0.25, 0.61960786581039429,
0.61960786581039429), (0.375, 0.51764708757400513,
0.51764708757400513), (0.5, 0.3490196168422699, 0.3490196168422699),
(0.625, 0.28235295414924622, 0.28235295414924622), (0.75,
0.12156862765550613, 0.12156862765550613), (0.875, 0.0, 0.0), (1.0,
0.0, 0.0)],

    'green': [(0.0, 0.9686274528503418, 0.9686274528503418), (0.125,
    0.90980392694473267, 0.90980392694473267), (0.25,
    0.83137255907058716, 0.83137255907058716), (0.375,
    0.73333334922790527, 0.73333334922790527), (0.5,
    0.55294120311737061, 0.55294120311737061), (0.625,
    0.3960784375667572, 0.3960784375667572), (0.75,
    0.18823529779911041, 0.18823529779911041), (0.875, 0.0, 0.0),
    (1.0, 0.0, 0.0)],

    'red': [(0.0, 1.0, 1.0), (0.125, 0.99607843160629272,
    0.99607843160629272), (0.25, 0.99215686321258545,
    0.99215686321258545), (0.375, 0.99215686321258545,
    0.99215686321258545), (0.5, 0.98823529481887817,
    0.98823529481887817), (0.625, 0.93725490570068359,
    0.93725490570068359), (0.75, 0.84313726425170898,
    0.84313726425170898), (0.875, 0.70196080207824707,
    0.70196080207824707), (1.0, 0.49803921580314636,
    0.49803921580314636)]}

_Paired_data = {'blue': [(0.0, 0.89019608497619629,
0.89019608497619629), (0.090909090909090912, 0.70588237047195435,
0.70588237047195435), (0.18181818181818182, 0.54117649793624878,
0.54117649793624878), (0.27272727272727271, 0.17254902422428131,
0.17254902422428131), (0.36363636363636365, 0.60000002384185791,
0.60000002384185791), (0.45454545454545453, 0.10980392247438431,
0.10980392247438431), (0.54545454545454541, 0.43529412150382996,
0.43529412150382996), (0.63636363636363635, 0.0, 0.0),
(0.72727272727272729, 0.83921569585800171, 0.83921569585800171),
(0.81818181818181823, 0.60392159223556519, 0.60392159223556519),
(0.90909090909090906, 0.60000002384185791, 0.60000002384185791), (1.0,
0.15686275064945221, 0.15686275064945221)],

    'green': [(0.0, 0.80784314870834351, 0.80784314870834351),
    (0.090909090909090912, 0.47058823704719543, 0.47058823704719543),
    (0.18181818181818182, 0.87450981140136719, 0.87450981140136719),
    (0.27272727272727271, 0.62745100259780884, 0.62745100259780884),
    (0.36363636363636365, 0.60392159223556519, 0.60392159223556519),
    (0.45454545454545453, 0.10196078568696976, 0.10196078568696976),
    (0.54545454545454541, 0.74901962280273438, 0.74901962280273438),
    (0.63636363636363635, 0.49803921580314636, 0.49803921580314636),
    (0.72727272727272729, 0.69803923368453979, 0.69803923368453979),
    (0.81818181818181823, 0.23921568691730499, 0.23921568691730499),
    (0.90909090909090906, 1.0, 1.0), (1.0, 0.3490196168422699,
    0.3490196168422699)],

    'red': [(0.0, 0.65098041296005249, 0.65098041296005249),
    (0.090909090909090912, 0.12156862765550613, 0.12156862765550613),
    (0.18181818181818182, 0.69803923368453979, 0.69803923368453979),
    (0.27272727272727271, 0.20000000298023224, 0.20000000298023224),
    (0.36363636363636365, 0.9843137264251709, 0.9843137264251709),
    (0.45454545454545453, 0.89019608497619629, 0.89019608497619629),
    (0.54545454545454541, 0.99215686321258545, 0.99215686321258545),
    (0.63636363636363635, 1.0, 1.0), (0.72727272727272729,
    0.7921568751335144, 0.7921568751335144), (0.81818181818181823,
    0.41568627953529358, 0.41568627953529358), (0.90909090909090906,
    1.0, 1.0), (1.0, 0.69411766529083252, 0.69411766529083252)]}

_Pastel1_data = {'blue': [(0.0, 0.68235296010971069,
0.68235296010971069), (0.125, 0.89019608497619629,
0.89019608497619629), (0.25, 0.77254903316497803,
0.77254903316497803), (0.375, 0.89411765336990356,
0.89411765336990356), (0.5, 0.65098041296005249, 0.65098041296005249),
(0.625, 0.80000001192092896, 0.80000001192092896), (0.75,
0.74117648601531982, 0.74117648601531982), (0.875,
0.92549020051956177, 0.92549020051956177), (1.0, 0.94901961088180542,
0.94901961088180542)],

    'green': [(0.0, 0.70588237047195435, 0.70588237047195435), (0.125,
    0.80392158031463623, 0.80392158031463623), (0.25,
    0.92156863212585449, 0.92156863212585449), (0.375,
    0.79607844352722168, 0.79607844352722168), (0.5,
    0.85098040103912354, 0.85098040103912354), (0.625, 1.0, 1.0),
    (0.75, 0.84705883264541626, 0.84705883264541626), (0.875,
    0.85490196943283081, 0.85490196943283081), (1.0,
    0.94901961088180542, 0.94901961088180542)],

    'red': [(0.0, 0.9843137264251709, 0.9843137264251709), (0.125,
    0.70196080207824707, 0.70196080207824707), (0.25,
    0.80000001192092896, 0.80000001192092896), (0.375,
    0.87058824300765991, 0.87058824300765991), (0.5,
    0.99607843160629272, 0.99607843160629272), (0.625, 1.0, 1.0),
    (0.75, 0.89803922176361084, 0.89803922176361084), (0.875,
    0.99215686321258545, 0.99215686321258545), (1.0,
    0.94901961088180542, 0.94901961088180542)]}

_Pastel2_data = {'blue': [(0.0, 0.80392158031463623,
0.80392158031463623), (0.14285714285714285, 0.67450982332229614,
0.67450982332229614), (0.2857142857142857, 0.90980392694473267,
0.90980392694473267), (0.42857142857142855, 0.89411765336990356,
0.89411765336990356), (0.5714285714285714, 0.78823530673980713,
0.78823530673980713), (0.7142857142857143, 0.68235296010971069,
0.68235296010971069), (0.8571428571428571, 0.80000001192092896,
0.80000001192092896), (1.0, 0.80000001192092896,
0.80000001192092896)],

    'green': [(0.0, 0.88627451658248901, 0.88627451658248901),
    (0.14285714285714285, 0.80392158031463623, 0.80392158031463623),
    (0.2857142857142857, 0.83529412746429443, 0.83529412746429443),
    (0.42857142857142855, 0.7921568751335144, 0.7921568751335144),
    (0.5714285714285714, 0.96078431606292725, 0.96078431606292725),
    (0.7142857142857143, 0.94901961088180542, 0.94901961088180542),
    (0.8571428571428571, 0.88627451658248901, 0.88627451658248901),
    (1.0, 0.80000001192092896, 0.80000001192092896)],

    'red': [(0.0, 0.70196080207824707, 0.70196080207824707),
    (0.14285714285714285, 0.99215686321258545, 0.99215686321258545),
    (0.2857142857142857, 0.79607844352722168, 0.79607844352722168),
    (0.42857142857142855, 0.95686274766921997, 0.95686274766921997),
    (0.5714285714285714, 0.90196079015731812, 0.90196079015731812),
    (0.7142857142857143, 1.0, 1.0), (0.8571428571428571,
    0.94509804248809814, 0.94509804248809814), (1.0,
    0.80000001192092896, 0.80000001192092896)]}

_PiYG_data = {'blue': [(0.0, 0.32156863808631897,
0.32156863808631897), (0.10000000000000001, 0.49019607901573181,
0.49019607901573181), (0.20000000000000001, 0.68235296010971069,
0.68235296010971069), (0.29999999999999999, 0.85490196943283081,
0.85490196943283081), (0.40000000000000002, 0.93725490570068359,
0.93725490570068359), (0.5, 0.9686274528503418, 0.9686274528503418),
(0.59999999999999998, 0.81568628549575806, 0.81568628549575806),
(0.69999999999999996, 0.52549022436141968, 0.52549022436141968),
(0.80000000000000004, 0.25490197539329529, 0.25490197539329529),
(0.90000000000000002, 0.12941177189350128, 0.12941177189350128), (1.0,
0.098039217293262482, 0.098039217293262482)],

    'green': [(0.0, 0.0039215688593685627, 0.0039215688593685627),
    (0.10000000000000001, 0.10588235408067703, 0.10588235408067703),
    (0.20000000000000001, 0.46666666865348816, 0.46666666865348816),
    (0.29999999999999999, 0.7137255072593689, 0.7137255072593689),
    (0.40000000000000002, 0.87843137979507446, 0.87843137979507446),
    (0.5, 0.9686274528503418, 0.9686274528503418),
    (0.59999999999999998, 0.96078431606292725, 0.96078431606292725),
    (0.69999999999999996, 0.88235294818878174, 0.88235294818878174),
    (0.80000000000000004, 0.73725491762161255, 0.73725491762161255),
    (0.90000000000000002, 0.57254904508590698, 0.57254904508590698),
    (1.0, 0.39215686917304993, 0.39215686917304993)],

    'red': [(0.0, 0.55686277151107788, 0.55686277151107788),
    (0.10000000000000001, 0.77254903316497803, 0.77254903316497803),
    (0.20000000000000001, 0.87058824300765991, 0.87058824300765991),
    (0.29999999999999999, 0.94509804248809814, 0.94509804248809814),
    (0.40000000000000002, 0.99215686321258545, 0.99215686321258545),
    (0.5, 0.9686274528503418, 0.9686274528503418),
    (0.59999999999999998, 0.90196079015731812, 0.90196079015731812),
    (0.69999999999999996, 0.72156864404678345, 0.72156864404678345),
    (0.80000000000000004, 0.49803921580314636, 0.49803921580314636),
    (0.90000000000000002, 0.30196079611778259, 0.30196079611778259),
    (1.0, 0.15294118225574493, 0.15294118225574493)]}

_PRGn_data = {'blue': [(0.0, 0.29411765933036804,
0.29411765933036804), (0.10000000000000001, 0.51372551918029785,
0.51372551918029785), (0.20000000000000001, 0.67058825492858887,
0.67058825492858887), (0.29999999999999999, 0.81176471710205078,
0.81176471710205078), (0.40000000000000002, 0.90980392694473267,
0.90980392694473267), (0.5, 0.9686274528503418, 0.9686274528503418),
(0.59999999999999998, 0.82745099067687988, 0.82745099067687988),
(0.69999999999999996, 0.62745100259780884, 0.62745100259780884),
(0.80000000000000004, 0.3803921639919281, 0.3803921639919281),
(0.90000000000000002, 0.21568627655506134, 0.21568627655506134), (1.0,
0.10588235408067703, 0.10588235408067703)],

    'green': [(0.0, 0.0, 0.0), (0.10000000000000001,
    0.16470588743686676, 0.16470588743686676), (0.20000000000000001,
    0.43921568989753723, 0.43921568989753723), (0.29999999999999999,
    0.64705884456634521, 0.64705884456634521), (0.40000000000000002,
    0.83137255907058716, 0.83137255907058716), (0.5,
    0.9686274528503418, 0.9686274528503418), (0.59999999999999998,
    0.94117647409439087, 0.94117647409439087), (0.69999999999999996,
    0.85882353782653809, 0.85882353782653809), (0.80000000000000004,
    0.68235296010971069, 0.68235296010971069), (0.90000000000000002,
    0.47058823704719543, 0.47058823704719543), (1.0,
    0.26666668057441711, 0.26666668057441711)],

    'red': [(0.0, 0.25098040699958801, 0.25098040699958801),
    (0.10000000000000001, 0.46274510025978088, 0.46274510025978088),
    (0.20000000000000001, 0.60000002384185791, 0.60000002384185791),
    (0.29999999999999999, 0.7607843279838562, 0.7607843279838562),
    (0.40000000000000002, 0.90588235855102539, 0.90588235855102539),
    (0.5, 0.9686274528503418, 0.9686274528503418),
    (0.59999999999999998, 0.85098040103912354, 0.85098040103912354),
    (0.69999999999999996, 0.65098041296005249, 0.65098041296005249),
    (0.80000000000000004, 0.35294118523597717, 0.35294118523597717),
    (0.90000000000000002, 0.10588235408067703, 0.10588235408067703),
    (1.0, 0.0, 0.0)]}

_PuBu_data = {'blue': [(0.0, 0.9843137264251709, 0.9843137264251709),
(0.125, 0.94901961088180542, 0.94901961088180542), (0.25,
0.90196079015731812, 0.90196079015731812), (0.375,
0.85882353782653809, 0.85882353782653809), (0.5, 0.81176471710205078,
0.81176471710205078), (0.625, 0.75294119119644165,
0.75294119119644165), (0.75, 0.69019609689712524,
0.69019609689712524), (0.875, 0.55294120311737061,
0.55294120311737061), (1.0, 0.34509804844856262,
0.34509804844856262)],

    'green': [(0.0, 0.9686274528503418, 0.9686274528503418), (0.125,
    0.90588235855102539, 0.90588235855102539), (0.25,
    0.81960785388946533, 0.81960785388946533), (0.375,
    0.74117648601531982, 0.74117648601531982), (0.5,
    0.66274511814117432, 0.66274511814117432), (0.625,
    0.56470590829849243, 0.56470590829849243), (0.75,
    0.43921568989753723, 0.43921568989753723), (0.875,
    0.35294118523597717, 0.35294118523597717), (1.0,
    0.21960784494876862, 0.21960784494876862)],

    'red': [(0.0, 1.0, 1.0), (0.125, 0.92549020051956177,
    0.92549020051956177), (0.25, 0.81568628549575806,
    0.81568628549575806), (0.375, 0.65098041296005249,
    0.65098041296005249), (0.5, 0.45490196347236633,
    0.45490196347236633), (0.625, 0.21176470816135406,
    0.21176470816135406), (0.75, 0.019607843831181526,
    0.019607843831181526), (0.875, 0.015686275437474251,
    0.015686275437474251), (1.0, 0.0078431377187371254,
    0.0078431377187371254)]}

_PuBuGn_data = {'blue': [(0.0, 0.9843137264251709,
0.9843137264251709), (0.125, 0.94117647409439087,
0.94117647409439087), (0.25, 0.90196079015731812,
0.90196079015731812), (0.375, 0.85882353782653809,
0.85882353782653809), (0.5, 0.81176471710205078, 0.81176471710205078),
(0.625, 0.75294119119644165, 0.75294119119644165), (0.75,
0.54117649793624878, 0.54117649793624878), (0.875, 0.3490196168422699,
0.3490196168422699), (1.0, 0.21176470816135406, 0.21176470816135406)],

    'green': [(0.0, 0.9686274528503418, 0.9686274528503418), (0.125,
    0.88627451658248901, 0.88627451658248901), (0.25,
    0.81960785388946533, 0.81960785388946533), (0.375,
    0.74117648601531982, 0.74117648601531982), (0.5,
    0.66274511814117432, 0.66274511814117432), (0.625,
    0.56470590829849243, 0.56470590829849243), (0.75,
    0.5058823823928833, 0.5058823823928833), (0.875,
    0.42352941632270813, 0.42352941632270813), (1.0,
    0.27450981736183167, 0.27450981736183167)],

    'red': [(0.0, 1.0, 1.0), (0.125, 0.92549020051956177,
    0.92549020051956177), (0.25, 0.81568628549575806,
    0.81568628549575806), (0.375, 0.65098041296005249,
    0.65098041296005249), (0.5, 0.40392157435417175,
    0.40392157435417175), (0.625, 0.21176470816135406,
    0.21176470816135406), (0.75, 0.0078431377187371254,
    0.0078431377187371254), (0.875, 0.0039215688593685627,
    0.0039215688593685627), (1.0, 0.0039215688593685627,
    0.0039215688593685627)]}

_PuOr_data = {'blue': [(0.0, 0.031372550874948502,
0.031372550874948502), (0.10000000000000001, 0.023529412224888802,
0.023529412224888802), (0.20000000000000001, 0.078431375324726105,
0.078431375324726105), (0.29999999999999999, 0.38823530077934265,
0.38823530077934265), (0.40000000000000002, 0.7137255072593689,
0.7137255072593689), (0.5, 0.9686274528503418, 0.9686274528503418),
(0.59999999999999998, 0.92156863212585449, 0.92156863212585449),
(0.69999999999999996, 0.82352942228317261, 0.82352942228317261),
(0.80000000000000004, 0.67450982332229614, 0.67450982332229614),
(0.90000000000000002, 0.53333336114883423, 0.53333336114883423), (1.0,
0.29411765933036804, 0.29411765933036804)],

    'green': [(0.0, 0.23137255012989044, 0.23137255012989044),
    (0.10000000000000001, 0.34509804844856262, 0.34509804844856262),
    (0.20000000000000001, 0.50980395078659058, 0.50980395078659058),
    (0.29999999999999999, 0.72156864404678345, 0.72156864404678345),
    (0.40000000000000002, 0.87843137979507446, 0.87843137979507446),
    (0.5, 0.9686274528503418, 0.9686274528503418),
    (0.59999999999999998, 0.85490196943283081, 0.85490196943283081),
    (0.69999999999999996, 0.67058825492858887, 0.67058825492858887),
    (0.80000000000000004, 0.45098039507865906, 0.45098039507865906),
    (0.90000000000000002, 0.15294118225574493, 0.15294118225574493),
    (1.0, 0.0, 0.0)],

    'red': [(0.0, 0.49803921580314636, 0.49803921580314636),
    (0.10000000000000001, 0.70196080207824707, 0.70196080207824707),
    (0.20000000000000001, 0.87843137979507446, 0.87843137979507446),
    (0.29999999999999999, 0.99215686321258545, 0.99215686321258545),
    (0.40000000000000002, 0.99607843160629272, 0.99607843160629272),
    (0.5, 0.9686274528503418, 0.9686274528503418),
    (0.59999999999999998, 0.84705883264541626, 0.84705883264541626),
    (0.69999999999999996, 0.69803923368453979, 0.69803923368453979),
    (0.80000000000000004, 0.50196081399917603, 0.50196081399917603),
    (0.90000000000000002, 0.32941177487373352, 0.32941177487373352),
    (1.0, 0.17647059261798859, 0.17647059261798859)]}

_PuRd_data = {'blue': [(0.0, 0.97647058963775635,
0.97647058963775635), (0.125, 0.93725490570068359,
0.93725490570068359), (0.25, 0.85490196943283081,
0.85490196943283081), (0.375, 0.78039216995239258,
0.78039216995239258), (0.5, 0.69019609689712524, 0.69019609689712524),
(0.625, 0.54117649793624878, 0.54117649793624878), (0.75,
0.33725491166114807, 0.33725491166114807), (0.875,
0.26274511218070984, 0.26274511218070984), (1.0, 0.12156862765550613,
0.12156862765550613)],

    'green': [(0.0, 0.95686274766921997, 0.95686274766921997), (0.125,
    0.88235294818878174, 0.88235294818878174), (0.25,
    0.72549021244049072, 0.72549021244049072), (0.375,
    0.58039218187332153, 0.58039218187332153), (0.5,
    0.3960784375667572, 0.3960784375667572), (0.625,
    0.16078431904315948, 0.16078431904315948), (0.75,
    0.070588238537311554, 0.070588238537311554), (0.875, 0.0, 0.0),
    (1.0, 0.0, 0.0)],

    'red': [(0.0, 0.9686274528503418, 0.9686274528503418), (0.125,
    0.90588235855102539, 0.90588235855102539), (0.25,
    0.83137255907058716, 0.83137255907058716), (0.375,
    0.78823530673980713, 0.78823530673980713), (0.5,
    0.87450981140136719, 0.87450981140136719), (0.625,
    0.90588235855102539, 0.90588235855102539), (0.75,
    0.80784314870834351, 0.80784314870834351), (0.875,
    0.59607845544815063, 0.59607845544815063), (1.0,
    0.40392157435417175, 0.40392157435417175)]}

_Purples_data = {'blue': [(0.0, 0.99215686321258545,
0.99215686321258545), (0.125, 0.96078431606292725,
0.96078431606292725), (0.25, 0.92156863212585449,
0.92156863212585449), (0.375, 0.86274510622024536,
0.86274510622024536), (0.5, 0.78431373834609985, 0.78431373834609985),
(0.625, 0.729411780834198, 0.729411780834198), (0.75,
0.63921570777893066, 0.63921570777893066), (0.875,
0.56078433990478516, 0.56078433990478516), (1.0, 0.49019607901573181,
0.49019607901573181)],

    'green': [(0.0, 0.9843137264251709, 0.9843137264251709), (0.125,
    0.92941176891326904, 0.92941176891326904), (0.25,
    0.85490196943283081, 0.85490196943283081), (0.375,
    0.74117648601531982, 0.74117648601531982), (0.5,
    0.60392159223556519, 0.60392159223556519), (0.625,
    0.49019607901573181, 0.49019607901573181), (0.75,
    0.31764706969261169, 0.31764706969261169), (0.875,
    0.15294118225574493, 0.15294118225574493), (1.0, 0.0, 0.0)],

    'red': [(0.0, 0.98823529481887817, 0.98823529481887817), (0.125,
    0.93725490570068359, 0.93725490570068359), (0.25,
    0.85490196943283081, 0.85490196943283081), (0.375,
    0.73725491762161255, 0.73725491762161255), (0.5,
    0.61960786581039429, 0.61960786581039429), (0.625,
    0.50196081399917603, 0.50196081399917603), (0.75,
    0.41568627953529358, 0.41568627953529358), (0.875,
    0.32941177487373352, 0.32941177487373352), (1.0,
    0.24705882370471954, 0.24705882370471954)]}

_RdBu_data = {'blue': [(0.0, 0.12156862765550613,
0.12156862765550613), (0.10000000000000001, 0.16862745583057404,
0.16862745583057404), (0.20000000000000001, 0.30196079611778259,
0.30196079611778259), (0.29999999999999999, 0.50980395078659058,
0.50980395078659058), (0.40000000000000002, 0.78039216995239258,
0.78039216995239258), (0.5, 0.9686274528503418, 0.9686274528503418),
(0.59999999999999998, 0.94117647409439087, 0.94117647409439087),
(0.69999999999999996, 0.87058824300765991, 0.87058824300765991),
(0.80000000000000004, 0.76470589637756348, 0.76470589637756348),
(0.90000000000000002, 0.67450982332229614, 0.67450982332229614), (1.0,
0.3803921639919281, 0.3803921639919281)],

    'green': [(0.0, 0.0, 0.0), (0.10000000000000001,
    0.094117648899555206, 0.094117648899555206), (0.20000000000000001,
    0.37647059559822083, 0.37647059559822083), (0.29999999999999999,
    0.64705884456634521, 0.64705884456634521), (0.40000000000000002,
    0.85882353782653809, 0.85882353782653809), (0.5,
    0.9686274528503418, 0.9686274528503418), (0.59999999999999998,
    0.89803922176361084, 0.89803922176361084), (0.69999999999999996,
    0.77254903316497803, 0.77254903316497803), (0.80000000000000004,
    0.57647061347961426, 0.57647061347961426), (0.90000000000000002,
    0.40000000596046448, 0.40000000596046448), (1.0,
    0.18823529779911041, 0.18823529779911041)],

    'red': [(0.0, 0.40392157435417175, 0.40392157435417175),
    (0.10000000000000001, 0.69803923368453979, 0.69803923368453979),
    (0.20000000000000001, 0.83921569585800171, 0.83921569585800171),
    (0.29999999999999999, 0.95686274766921997, 0.95686274766921997),
    (0.40000000000000002, 0.99215686321258545, 0.99215686321258545),
    (0.5, 0.9686274528503418, 0.9686274528503418),
    (0.59999999999999998, 0.81960785388946533, 0.81960785388946533),
    (0.69999999999999996, 0.57254904508590698, 0.57254904508590698),
    (0.80000000000000004, 0.26274511218070984, 0.26274511218070984),
    (0.90000000000000002, 0.12941177189350128, 0.12941177189350128),
    (1.0, 0.019607843831181526, 0.019607843831181526)]}

_RdGy_data = {'blue': [(0.0, 0.12156862765550613,
0.12156862765550613), (0.10000000000000001, 0.16862745583057404,
0.16862745583057404), (0.20000000000000001, 0.30196079611778259,
0.30196079611778259), (0.29999999999999999, 0.50980395078659058,
0.50980395078659058), (0.40000000000000002, 0.78039216995239258,
0.78039216995239258), (0.5, 1.0, 1.0), (0.59999999999999998,
0.87843137979507446, 0.87843137979507446), (0.69999999999999996,
0.729411780834198, 0.729411780834198), (0.80000000000000004,
0.52941179275512695, 0.52941179275512695), (0.90000000000000002,
0.30196079611778259, 0.30196079611778259), (1.0, 0.10196078568696976,
0.10196078568696976)],

    'green': [(0.0, 0.0, 0.0), (0.10000000000000001,
    0.094117648899555206, 0.094117648899555206), (0.20000000000000001,
    0.37647059559822083, 0.37647059559822083), (0.29999999999999999,
    0.64705884456634521, 0.64705884456634521), (0.40000000000000002,
    0.85882353782653809, 0.85882353782653809), (0.5, 1.0, 1.0),
    (0.59999999999999998, 0.87843137979507446, 0.87843137979507446),
    (0.69999999999999996, 0.729411780834198, 0.729411780834198),
    (0.80000000000000004, 0.52941179275512695, 0.52941179275512695),
    (0.90000000000000002, 0.30196079611778259, 0.30196079611778259),
    (1.0, 0.10196078568696976, 0.10196078568696976)],

    'red': [(0.0, 0.40392157435417175, 0.40392157435417175),
    (0.10000000000000001, 0.69803923368453979, 0.69803923368453979),
    (0.20000000000000001, 0.83921569585800171, 0.83921569585800171),
    (0.29999999999999999, 0.95686274766921997, 0.95686274766921997),
    (0.40000000000000002, 0.99215686321258545, 0.99215686321258545),
    (0.5, 1.0, 1.0), (0.59999999999999998, 0.87843137979507446,
    0.87843137979507446), (0.69999999999999996, 0.729411780834198,
    0.729411780834198), (0.80000000000000004, 0.52941179275512695,
    0.52941179275512695), (0.90000000000000002, 0.30196079611778259,
    0.30196079611778259), (1.0, 0.10196078568696976,
    0.10196078568696976)]}

_RdPu_data = {'blue': [(0.0, 0.9529411792755127, 0.9529411792755127),
(0.125, 0.86666667461395264, 0.86666667461395264), (0.25,
0.75294119119644165, 0.75294119119644165), (0.375,
0.70980393886566162, 0.70980393886566162), (0.5, 0.63137257099151611,
0.63137257099151611), (0.625, 0.59215688705444336,
0.59215688705444336), (0.75, 0.49411764740943909,
0.49411764740943909), (0.875, 0.46666666865348816,
0.46666666865348816), (1.0, 0.41568627953529358,
0.41568627953529358)],

    'green': [(0.0, 0.9686274528503418, 0.9686274528503418), (0.125,
    0.87843137979507446, 0.87843137979507446), (0.25,
    0.77254903316497803, 0.77254903316497803), (0.375,
    0.62352943420410156, 0.62352943420410156), (0.5,
    0.40784314274787903, 0.40784314274787903), (0.625,
    0.20392157137393951, 0.20392157137393951), (0.75,
    0.0039215688593685627, 0.0039215688593685627), (0.875,
    0.0039215688593685627, 0.0039215688593685627), (1.0, 0.0, 0.0)],

    'red': [(0.0, 1.0, 1.0), (0.125, 0.99215686321258545,
    0.99215686321258545), (0.25, 0.98823529481887817,
    0.98823529481887817), (0.375, 0.98039215803146362,
    0.98039215803146362), (0.5, 0.9686274528503418,
    0.9686274528503418), (0.625, 0.86666667461395264,
    0.86666667461395264), (0.75, 0.68235296010971069,
    0.68235296010971069), (0.875, 0.47843137383460999,
    0.47843137383460999), (1.0, 0.28627452254295349,
    0.28627452254295349)]}

_RdYlBu_data = {'blue': [(0.0, 0.14901961386203766,
                 0.14901961386203766), (0.10000000149011612,
                 0.15294118225574493, 0.15294118225574493),
                 (0.20000000298023224, 0.26274511218070984,
                 0.26274511218070984), (0.30000001192092896,
                 0.3803921639919281, 0.3803921639919281),
                 (0.40000000596046448, 0.56470590829849243,
                 0.56470590829849243), (0.5, 0.74901962280273438,
                 0.74901962280273438), (0.60000002384185791,
                 0.97254902124404907, 0.97254902124404907),
                 (0.69999998807907104, 0.91372549533843994,
                 0.91372549533843994), (0.80000001192092896,
                 0.81960785388946533, 0.81960785388946533),
                 (0.89999997615814209, 0.70588237047195435,
                 0.70588237047195435), (1.0, 0.58431375026702881,
                 0.58431375026702881)], 'green': [(0.0, 0.0, 0.0),
                 (0.10000000149011612, 0.18823529779911041,
                 0.18823529779911041), (0.20000000298023224,
                 0.42745098471641541, 0.42745098471641541),
                 (0.30000001192092896, 0.68235296010971069,
                 0.68235296010971069), (0.40000000596046448,
                 0.87843137979507446, 0.87843137979507446), (0.5, 1.0,
                 1.0), (0.60000002384185791, 0.9529411792755127,
                 0.9529411792755127), (0.69999998807907104,
                 0.85098040103912354, 0.85098040103912354),
                 (0.80000001192092896, 0.67843139171600342,
                 0.67843139171600342), (0.89999997615814209,
                 0.45882353186607361, 0.45882353186607361), (1.0,
                 0.21176470816135406, 0.21176470816135406)], 'red':
                 [(0.0, 0.64705884456634521, 0.64705884456634521),
                 (0.10000000149011612, 0.84313726425170898,
                 0.84313726425170898), (0.20000000298023224,
                 0.95686274766921997, 0.95686274766921997),
                 (0.30000001192092896, 0.99215686321258545,
                 0.99215686321258545), (0.40000000596046448,
                 0.99607843160629272, 0.99607843160629272), (0.5, 1.0,
                 1.0), (0.60000002384185791, 0.87843137979507446,
                 0.87843137979507446), (0.69999998807907104,
                 0.67058825492858887, 0.67058825492858887),
                 (0.80000001192092896, 0.45490196347236633,
                 0.45490196347236633), (0.89999997615814209,
                 0.27058824896812439, 0.27058824896812439), (1.0,
                 0.19215686619281769, 0.19215686619281769)]}

_RdYlGn_data = {'blue': [(0.0, 0.14901961386203766,
0.14901961386203766), (0.10000000000000001, 0.15294118225574493,
0.15294118225574493), (0.20000000000000001, 0.26274511218070984,
0.26274511218070984), (0.29999999999999999, 0.3803921639919281,
0.3803921639919281), (0.40000000000000002, 0.54509806632995605,
0.54509806632995605), (0.5, 0.74901962280273438, 0.74901962280273438),
(0.59999999999999998, 0.54509806632995605, 0.54509806632995605),
(0.69999999999999996, 0.41568627953529358, 0.41568627953529358),
(0.80000000000000004, 0.38823530077934265, 0.38823530077934265),
(0.90000000000000002, 0.31372550129890442, 0.31372550129890442), (1.0,
0.21568627655506134, 0.21568627655506134)],

    'green': [(0.0, 0.0, 0.0), (0.10000000000000001,
    0.18823529779911041, 0.18823529779911041), (0.20000000000000001,
    0.42745098471641541, 0.42745098471641541), (0.29999999999999999,
    0.68235296010971069, 0.68235296010971069), (0.40000000000000002,
    0.87843137979507446, 0.87843137979507446), (0.5, 1.0, 1.0),
    (0.59999999999999998, 0.93725490570068359, 0.93725490570068359),
    (0.69999999999999996, 0.85098040103912354, 0.85098040103912354),
    (0.80000000000000004, 0.74117648601531982, 0.74117648601531982),
    (0.90000000000000002, 0.59607845544815063, 0.59607845544815063),
    (1.0, 0.40784314274787903, 0.40784314274787903)],

    'red': [(0.0, 0.64705884456634521, 0.64705884456634521),
    (0.10000000000000001, 0.84313726425170898, 0.84313726425170898),
    (0.20000000000000001, 0.95686274766921997, 0.95686274766921997),
    (0.29999999999999999, 0.99215686321258545, 0.99215686321258545),
    (0.40000000000000002, 0.99607843160629272, 0.99607843160629272),
    (0.5, 1.0, 1.0), (0.59999999999999998, 0.85098040103912354,
    0.85098040103912354), (0.69999999999999996, 0.65098041296005249,
    0.65098041296005249), (0.80000000000000004, 0.40000000596046448,
    0.40000000596046448), (0.90000000000000002, 0.10196078568696976,
    0.10196078568696976), (1.0, 0.0, 0.0)]}

_Reds_data = {'blue': [(0.0, 0.94117647409439087,
0.94117647409439087), (0.125, 0.82352942228317261,
0.82352942228317261), (0.25, 0.63137257099151611,
0.63137257099151611), (0.375, 0.44705882668495178,
0.44705882668495178), (0.5, 0.29019609093666077, 0.29019609093666077),
(0.625, 0.17254902422428131, 0.17254902422428131), (0.75,
0.11372549086809158, 0.11372549086809158), (0.875,
0.08235294371843338, 0.08235294371843338), (1.0, 0.050980392843484879,
0.050980392843484879)],

    'green': [(0.0, 0.96078431606292725, 0.96078431606292725), (0.125,
    0.87843137979507446, 0.87843137979507446), (0.25,
    0.73333334922790527, 0.73333334922790527), (0.375,
    0.57254904508590698, 0.57254904508590698), (0.5,
    0.41568627953529358, 0.41568627953529358), (0.625,
    0.23137255012989044, 0.23137255012989044), (0.75,
    0.094117648899555206, 0.094117648899555206), (0.875,
    0.058823529630899429, 0.058823529630899429), (1.0, 0.0, 0.0)],

    'red': [(0.0, 1.0, 1.0), (0.125, 0.99607843160629272,
    0.99607843160629272), (0.25, 0.98823529481887817,
    0.98823529481887817), (0.375, 0.98823529481887817,
    0.98823529481887817), (0.5, 0.9843137264251709,
    0.9843137264251709), (0.625, 0.93725490570068359,
    0.93725490570068359), (0.75, 0.79607844352722168,
    0.79607844352722168), (0.875, 0.64705884456634521,
    0.64705884456634521), (1.0, 0.40392157435417175,
    0.40392157435417175)]}

_Set1_data = {'blue': [(0.0, 0.10980392247438431,
0.10980392247438431), (0.125, 0.72156864404678345,
0.72156864404678345), (0.25, 0.29019609093666077,
0.29019609093666077), (0.375, 0.63921570777893066,
0.63921570777893066), (0.5, 0.0, 0.0), (0.625, 0.20000000298023224,
0.20000000298023224), (0.75, 0.15686275064945221,
0.15686275064945221), (0.875, 0.74901962280273438,
0.74901962280273438), (1.0, 0.60000002384185791,
0.60000002384185791)],

    'green': [(0.0, 0.10196078568696976, 0.10196078568696976), (0.125,
    0.49411764740943909, 0.49411764740943909), (0.25,
    0.68627452850341797, 0.68627452850341797), (0.375,
    0.30588236451148987, 0.30588236451148987), (0.5,
    0.49803921580314636, 0.49803921580314636), (0.625, 1.0, 1.0),
    (0.75, 0.33725491166114807, 0.33725491166114807), (0.875,
    0.5058823823928833, 0.5058823823928833), (1.0,
    0.60000002384185791, 0.60000002384185791)],

    'red': [(0.0, 0.89411765336990356, 0.89411765336990356), (0.125,
    0.21568627655506134, 0.21568627655506134), (0.25,
    0.30196079611778259, 0.30196079611778259), (0.375,
    0.59607845544815063, 0.59607845544815063), (0.5, 1.0, 1.0),
    (0.625, 1.0, 1.0), (0.75, 0.65098041296005249,
    0.65098041296005249), (0.875, 0.9686274528503418,
    0.9686274528503418), (1.0, 0.60000002384185791,
    0.60000002384185791)]}

_Set2_data = {'blue': [(0.0, 0.64705884456634521,
0.64705884456634521), (0.14285714285714285, 0.38431373238563538,
0.38431373238563538), (0.2857142857142857, 0.79607844352722168,
0.79607844352722168), (0.42857142857142855, 0.76470589637756348,
0.76470589637756348), (0.5714285714285714, 0.32941177487373352,
0.32941177487373352), (0.7142857142857143, 0.18431372940540314,
0.18431372940540314), (0.8571428571428571, 0.58039218187332153,
0.58039218187332153), (1.0, 0.70196080207824707,
0.70196080207824707)],

    'green': [(0.0, 0.7607843279838562, 0.7607843279838562),
    (0.14285714285714285, 0.55294120311737061, 0.55294120311737061),
    (0.2857142857142857, 0.62745100259780884, 0.62745100259780884),
    (0.42857142857142855, 0.54117649793624878, 0.54117649793624878),
    (0.5714285714285714, 0.84705883264541626, 0.84705883264541626),
    (0.7142857142857143, 0.85098040103912354, 0.85098040103912354),
    (0.8571428571428571, 0.76862746477127075, 0.76862746477127075),
    (1.0, 0.70196080207824707, 0.70196080207824707)],

    'red': [(0.0, 0.40000000596046448, 0.40000000596046448),
    (0.14285714285714285, 0.98823529481887817, 0.98823529481887817),
    (0.2857142857142857, 0.55294120311737061, 0.55294120311737061),
    (0.42857142857142855, 0.90588235855102539, 0.90588235855102539),
    (0.5714285714285714, 0.65098041296005249, 0.65098041296005249),
    (0.7142857142857143, 1.0, 1.0), (0.8571428571428571,
    0.89803922176361084, 0.89803922176361084), (1.0,
    0.70196080207824707, 0.70196080207824707)]}

_Set3_data = {'blue': [(0.0, 0.78039216995239258,
0.78039216995239258), (0.090909090909090912, 0.70196080207824707,
0.70196080207824707), (0.18181818181818182, 0.85490196943283081,
0.85490196943283081), (0.27272727272727271, 0.44705882668495178,
0.44705882668495178), (0.36363636363636365, 0.82745099067687988,
0.82745099067687988), (0.45454545454545453, 0.38431373238563538,
0.38431373238563538), (0.54545454545454541, 0.4117647111415863,
0.4117647111415863), (0.63636363636363635, 0.89803922176361084,
0.89803922176361084), (0.72727272727272729, 0.85098040103912354,
0.85098040103912354), (0.81818181818181823, 0.74117648601531982,
0.74117648601531982), (0.90909090909090906, 0.77254903316497803,
0.77254903316497803), (1.0, 0.43529412150382996,
0.43529412150382996)],

    'green': [(0.0, 0.82745099067687988, 0.82745099067687988),
    (0.090909090909090912, 1.0, 1.0), (0.18181818181818182,
    0.729411780834198, 0.729411780834198), (0.27272727272727271,
    0.50196081399917603, 0.50196081399917603), (0.36363636363636365,
    0.69411766529083252, 0.69411766529083252), (0.45454545454545453,
    0.70588237047195435, 0.70588237047195435), (0.54545454545454541,
    0.87058824300765991, 0.87058824300765991), (0.63636363636363635,
    0.80392158031463623, 0.80392158031463623), (0.72727272727272729,
    0.85098040103912354, 0.85098040103912354), (0.81818181818181823,
    0.50196081399917603, 0.50196081399917603), (0.90909090909090906,
    0.92156863212585449, 0.92156863212585449), (1.0,
    0.92941176891326904, 0.92941176891326904)],

    'red': [(0.0, 0.55294120311737061, 0.55294120311737061),
    (0.090909090909090912, 1.0, 1.0), (0.18181818181818182,
    0.7450980544090271, 0.7450980544090271), (0.27272727272727271,
    0.9843137264251709, 0.9843137264251709), (0.36363636363636365,
    0.50196081399917603, 0.50196081399917603), (0.45454545454545453,
    0.99215686321258545, 0.99215686321258545), (0.54545454545454541,
    0.70196080207824707, 0.70196080207824707), (0.63636363636363635,
    0.98823529481887817, 0.98823529481887817), (0.72727272727272729,
    0.85098040103912354, 0.85098040103912354), (0.81818181818181823,
    0.73725491762161255, 0.73725491762161255), (0.90909090909090906,
    0.80000001192092896, 0.80000001192092896), (1.0, 1.0, 1.0)]}

_Spectral_data = {'blue': [(0.0, 0.25882354378700256,
0.25882354378700256), (0.10000000000000001, 0.30980393290519714,
0.30980393290519714), (0.20000000000000001, 0.26274511218070984,
0.26274511218070984), (0.29999999999999999, 0.3803921639919281,
0.3803921639919281), (0.40000000000000002, 0.54509806632995605,
0.54509806632995605), (0.5, 0.74901962280273438, 0.74901962280273438),
(0.59999999999999998, 0.59607845544815063, 0.59607845544815063),
(0.69999999999999996, 0.64313727617263794, 0.64313727617263794),
(0.80000000000000004, 0.64705884456634521, 0.64705884456634521),
(0.90000000000000002, 0.74117648601531982, 0.74117648601531982), (1.0,
0.63529413938522339, 0.63529413938522339)],

    'green': [(0.0, 0.0039215688593685627, 0.0039215688593685627),
    (0.10000000000000001, 0.24313725531101227, 0.24313725531101227),
    (0.20000000000000001, 0.42745098471641541, 0.42745098471641541),
    (0.29999999999999999, 0.68235296010971069, 0.68235296010971069),
    (0.40000000000000002, 0.87843137979507446, 0.87843137979507446),
    (0.5, 1.0, 1.0), (0.59999999999999998, 0.96078431606292725,
    0.96078431606292725), (0.69999999999999996, 0.86666667461395264,
    0.86666667461395264), (0.80000000000000004, 0.7607843279838562,
    0.7607843279838562), (0.90000000000000002, 0.53333336114883423,
    0.53333336114883423), (1.0, 0.30980393290519714,
    0.30980393290519714)],

    'red': [(0.0, 0.61960786581039429, 0.61960786581039429),
    (0.10000000000000001, 0.83529412746429443, 0.83529412746429443),
    (0.20000000000000001, 0.95686274766921997, 0.95686274766921997),
    (0.29999999999999999, 0.99215686321258545, 0.99215686321258545),
    (0.40000000000000002, 0.99607843160629272, 0.99607843160629272),
    (0.5, 1.0, 1.0), (0.59999999999999998, 0.90196079015731812,
    0.90196079015731812), (0.69999999999999996, 0.67058825492858887,
    0.67058825492858887), (0.80000000000000004, 0.40000000596046448,
    0.40000000596046448), (0.90000000000000002, 0.19607843458652496,
    0.19607843458652496), (1.0, 0.36862745881080627,
    0.36862745881080627)]}

_YlGn_data = {'blue': [(0.0, 0.89803922176361084,
0.89803922176361084), (0.125, 0.72549021244049072,
0.72549021244049072), (0.25, 0.63921570777893066,
0.63921570777893066), (0.375, 0.55686277151107788,
0.55686277151107788), (0.5, 0.47450980544090271, 0.47450980544090271),
(0.625, 0.364705890417099, 0.364705890417099), (0.75,
0.26274511218070984, 0.26274511218070984), (0.875,
0.21568627655506134, 0.21568627655506134), (1.0, 0.16078431904315948,
0.16078431904315948)],

    'green': [(0.0, 1.0, 1.0), (0.125, 0.98823529481887817,
    0.98823529481887817), (0.25, 0.94117647409439087,
    0.94117647409439087), (0.375, 0.86666667461395264,
    0.86666667461395264), (0.5, 0.7764706015586853,
    0.7764706015586853), (0.625, 0.67058825492858887,
    0.67058825492858887), (0.75, 0.51764708757400513,
    0.51764708757400513), (0.875, 0.40784314274787903,
    0.40784314274787903), (1.0, 0.27058824896812439,
    0.27058824896812439)],

     'red': [(0.0, 1.0, 1.0), (0.125, 0.9686274528503418,
     0.9686274528503418), (0.25, 0.85098040103912354,
     0.85098040103912354), (0.375, 0.67843139171600342,
     0.67843139171600342), (0.5, 0.47058823704719543,
     0.47058823704719543), (0.625, 0.25490197539329529,
     0.25490197539329529), (0.75, 0.13725490868091583,
     0.13725490868091583), (0.875, 0.0, 0.0), (1.0, 0.0, 0.0)]}

_YlGnBu_data = {'blue': [(0.0, 0.85098040103912354,
0.85098040103912354), (0.125, 0.69411766529083252,
0.69411766529083252), (0.25, 0.70588237047195435,
0.70588237047195435), (0.375, 0.73333334922790527,
0.73333334922790527), (0.5, 0.76862746477127075, 0.76862746477127075),
(0.625, 0.75294119119644165, 0.75294119119644165), (0.75,
0.65882354974746704, 0.65882354974746704), (0.875,
0.58039218187332153, 0.58039218187332153), (1.0, 0.34509804844856262,
0.34509804844856262)],

    'green': [(0.0, 1.0, 1.0), (0.125, 0.97254902124404907,
    0.97254902124404907), (0.25, 0.91372549533843994,
    0.91372549533843994), (0.375, 0.80392158031463623,
    0.80392158031463623), (0.5, 0.7137255072593689,
    0.7137255072593689), (0.625, 0.56862747669219971,
    0.56862747669219971), (0.75, 0.36862745881080627,
    0.36862745881080627), (0.875, 0.20392157137393951,
    0.20392157137393951), (1.0, 0.11372549086809158,
    0.11372549086809158)],

    'red': [(0.0, 1.0, 1.0), (0.125, 0.92941176891326904,
    0.92941176891326904), (0.25, 0.78039216995239258,
    0.78039216995239258), (0.375, 0.49803921580314636,
    0.49803921580314636), (0.5, 0.25490197539329529,
    0.25490197539329529), (0.625, 0.11372549086809158,
    0.11372549086809158), (0.75, 0.13333334028720856,
    0.13333334028720856), (0.875, 0.14509804546833038,
    0.14509804546833038), (1.0, 0.031372550874948502,
    0.031372550874948502)]}

_YlOrBr_data = {'blue': [(0.0, 0.89803922176361084,
0.89803922176361084), (0.125, 0.73725491762161255,
0.73725491762161255), (0.25, 0.56862747669219971,
0.56862747669219971), (0.375, 0.30980393290519714,
0.30980393290519714), (0.5, 0.16078431904315948, 0.16078431904315948),
(0.625, 0.078431375324726105, 0.078431375324726105), (0.75,
0.0078431377187371254, 0.0078431377187371254), (0.875,
0.015686275437474251, 0.015686275437474251), (1.0,
0.023529412224888802, 0.023529412224888802)],

    'green': [(0.0, 1.0, 1.0), (0.125, 0.9686274528503418,
    0.9686274528503418), (0.25, 0.89019608497619629,
    0.89019608497619629), (0.375, 0.76862746477127075,
    0.76862746477127075), (0.5, 0.60000002384185791,
    0.60000002384185791), (0.625, 0.43921568989753723,
    0.43921568989753723), (0.75, 0.29803922772407532,
    0.29803922772407532), (0.875, 0.20392157137393951,
    0.20392157137393951), (1.0, 0.14509804546833038,
    0.14509804546833038)],

    'red': [(0.0, 1.0, 1.0), (0.125, 1.0, 1.0), (0.25,
    0.99607843160629272, 0.99607843160629272), (0.375,
    0.99607843160629272, 0.99607843160629272), (0.5,
    0.99607843160629272, 0.99607843160629272), (0.625,
    0.92549020051956177, 0.92549020051956177), (0.75,
    0.80000001192092896, 0.80000001192092896), (0.875,
    0.60000002384185791, 0.60000002384185791), (1.0,
    0.40000000596046448, 0.40000000596046448)]}

_YlOrRd_data = {'blue': [(0.0, 0.80000001192092896,
0.80000001192092896), (0.125, 0.62745100259780884,
0.62745100259780884), (0.25, 0.46274510025978088,
0.46274510025978088), (0.375, 0.29803922772407532,
0.29803922772407532), (0.5, 0.23529411852359772, 0.23529411852359772),
(0.625, 0.16470588743686676, 0.16470588743686676), (0.75,
0.10980392247438431, 0.10980392247438431), (0.875,
0.14901961386203766, 0.14901961386203766), (1.0, 0.14901961386203766,
0.14901961386203766)],

    'green': [(0.0, 1.0, 1.0), (0.125, 0.92941176891326904,
    0.92941176891326904), (0.25, 0.85098040103912354,
    0.85098040103912354), (0.375, 0.69803923368453979,
    0.69803923368453979), (0.5, 0.55294120311737061,
    0.55294120311737061), (0.625, 0.30588236451148987,
    0.30588236451148987), (0.75, 0.10196078568696976,
    0.10196078568696976), (0.875, 0.0, 0.0), (1.0, 0.0, 0.0)],

    'red': [(0.0, 1.0, 1.0), (0.125, 1.0, 1.0), (0.25,
    0.99607843160629272, 0.99607843160629272), (0.375,
    0.99607843160629272, 0.99607843160629272), (0.5,
    0.99215686321258545, 0.99215686321258545), (0.625,
    0.98823529481887817, 0.98823529481887817), (0.75,
    0.89019608497619629, 0.89019608497619629), (0.875,
    0.74117648601531982, 0.74117648601531982), (1.0,
    0.50196081399917603, 0.50196081399917603)]}

# The next 7 palettes are from the Yorick scientific visalisation package,
# an evolution of the GIST package, both by David H. Munro.
# They are released under a BSD-like license (see LICENSE_YORICK in
# the license directory of the matplotlib source distribution).
#
# Most palette functions have been reduced to simple function descriptions
# by Reinier Heeres, since the rgb components were mostly straight lines.
# gist_earth_data and gist_ncar_data were simplified by a script and some
# manual effort.

_gist_earth_data = \
{'red': (
(0.0, 0.0, 0.0000),
(0.2824, 0.1882, 0.1882),
(0.4588, 0.2714, 0.2714),
(0.5490, 0.4719, 0.4719),
(0.6980, 0.7176, 0.7176),
(0.7882, 0.7553, 0.7553),
(1.0000, 0.9922, 0.9922),
), 'green': (
(0.0, 0.0, 0.0000),
(0.0275, 0.0000, 0.0000),
(0.1098, 0.1893, 0.1893),
(0.1647, 0.3035, 0.3035),
(0.2078, 0.3841, 0.3841),
(0.2824, 0.5020, 0.5020),
(0.5216, 0.6397, 0.6397),
(0.6980, 0.7171, 0.7171),
(0.7882, 0.6392, 0.6392),
(0.7922, 0.6413, 0.6413),
(0.8000, 0.6447, 0.6447),
(0.8078, 0.6481, 0.6481),
(0.8157, 0.6549, 0.6549),
(0.8667, 0.6991, 0.6991),
(0.8745, 0.7103, 0.7103),
(0.8824, 0.7216, 0.7216),
(0.8902, 0.7323, 0.7323),
(0.8980, 0.7430, 0.7430),
(0.9412, 0.8275, 0.8275),
(0.9569, 0.8635, 0.8635),
(0.9647, 0.8816, 0.8816),
(0.9961, 0.9733, 0.9733),
(1.0000, 0.9843, 0.9843),
), 'blue': (
(0.0, 0.0, 0.0000),
(0.0039, 0.1684, 0.1684),
(0.0078, 0.2212, 0.2212),
(0.0275, 0.4329, 0.4329),
(0.0314, 0.4549, 0.4549),
(0.2824, 0.5004, 0.5004),
(0.4667, 0.2748, 0.2748),
(0.5451, 0.3205, 0.3205),
(0.7843, 0.3961, 0.3961),
(0.8941, 0.6651, 0.6651),
(1.0000, 0.9843, 0.9843),
)}

_gist_gray_data = {
        'red': gfunc[3],
        'green': gfunc[3],
        'blue': gfunc[3],
}

_gist_heat_data = {
        'red': lambda x: 1.5 * x,
        'green': lambda x: 2 * x - 1,
        'blue': lambda x: 4 * x - 3,
}

_gist_ncar_data = \
{'red': (
(0.0, 0.0, 0.0000),
(0.3098, 0.0000, 0.0000),
(0.3725, 0.3993, 0.3993),
(0.4235, 0.5003, 0.5003),
(0.5333, 1.0000, 1.0000),
(0.7922, 1.0000, 1.0000),
(0.8471, 0.6218, 0.6218),
(0.8980, 0.9235, 0.9235),
(1.0000, 0.9961, 0.9961),
), 'green': (
(0.0, 0.0, 0.0000),
(0.0510, 0.3722, 0.3722),
(0.1059, 0.0000, 0.0000),
(0.1569, 0.7202, 0.7202),
(0.1608, 0.7537, 0.7537),
(0.1647, 0.7752, 0.7752),
(0.2157, 1.0000, 1.0000),
(0.2588, 0.9804, 0.9804),
(0.2706, 0.9804, 0.9804),
(0.3176, 1.0000, 1.0000),
(0.3686, 0.8081, 0.8081),
(0.4275, 1.0000, 1.0000),
(0.5216, 1.0000, 1.0000),
(0.6314, 0.7292, 0.7292),
(0.6863, 0.2796, 0.2796),
(0.7451, 0.0000, 0.0000),
(0.7922, 0.0000, 0.0000),
(0.8431, 0.1753, 0.1753),
(0.8980, 0.5000, 0.5000),
(1.0000, 0.9725, 0.9725),
), 'blue': (
(0.0, 0.5020, 0.5020),
(0.0510, 0.0222, 0.0222),
(0.1098, 1.0000, 1.0000),
(0.2039, 1.0000, 1.0000),
(0.2627, 0.6145, 0.6145),
(0.3216, 0.0000, 0.0000),
(0.4157, 0.0000, 0.0000),
(0.4745, 0.2342, 0.2342),
(0.5333, 0.0000, 0.0000),
(0.5804, 0.0000, 0.0000),
(0.6314, 0.0549, 0.0549),
(0.6902, 0.0000, 0.0000),
(0.7373, 0.0000, 0.0000),
(0.7922, 0.9738, 0.9738),
(0.8000, 1.0000, 1.0000),
(0.8431, 1.0000, 1.0000),
(0.8980, 0.9341, 0.9341),
(1.0000, 0.9961, 0.9961),
)}

_gist_rainbow_data = (
        (0.000, (1.00, 0.00, 0.16)),
        (0.030, (1.00, 0.00, 0.00)),
        (0.215, (1.00, 1.00, 0.00)),
        (0.400, (0.00, 1.00, 0.00)),
        (0.586, (0.00, 1.00, 1.00)),
        (0.770, (0.00, 0.00, 1.00)),
        (0.954, (1.00, 0.00, 1.00)),
        (1.000, (1.00, 0.00, 0.75))
)

_gist_stern_data = {
        'red': (
            (0.000, 0.000, 0.000), (0.0547, 1.000, 1.000),
            (0.250, 0.027, 0.250), #(0.2500, 0.250, 0.250),
            (1.000, 1.000, 1.000)),
        'green': ((0, 0, 0), (1, 1, 1)),
        'blue': (
            (0.000, 0.000, 0.000), (0.500, 1.000, 1.000),
            (0.735, 0.000, 0.000), (1.000, 1.000, 1.000))
}

_gist_yarg_data = {
        'red': lambda x: 1 - x,
        'green': lambda x: 1 - x,
        'blue': lambda x: 1 - x,
}

# This bipolar color map was generated from CoolWarmFloat33.csv of
# "Diverging Color Maps for Scientific Visualization" by Kenneth Moreland. 
# <http://www.cs.unm.edu/~kmorel/documents/ColorMaps/>
_coolwarm_data = {
    'red': [
        (0.0, 0.2298057, 0.2298057),
        (0.03125, 0.26623388, 0.26623388),
        (0.0625, 0.30386891, 0.30386891),
        (0.09375, 0.342804478, 0.342804478),
        (0.125, 0.38301334, 0.38301334),
        (0.15625, 0.424369608, 0.424369608),
        (0.1875, 0.46666708, 0.46666708),
        (0.21875, 0.509635204, 0.509635204),
        (0.25, 0.552953156, 0.552953156),
        (0.28125, 0.596262162, 0.596262162),
        (0.3125, 0.639176211, 0.639176211),
        (0.34375, 0.681291281, 0.681291281),
        (0.375, 0.722193294, 0.722193294),
        (0.40625, 0.761464949, 0.761464949),
        (0.4375, 0.798691636, 0.798691636),
        (0.46875, 0.833466556, 0.833466556),
        (0.5, 0.865395197, 0.865395197),
        (0.53125, 0.897787179, 0.897787179),
        (0.5625, 0.924127593, 0.924127593),
        (0.59375, 0.944468518, 0.944468518),
        (0.625, 0.958852946, 0.958852946),
        (0.65625, 0.96732803, 0.96732803),
        (0.6875, 0.969954137, 0.969954137),
        (0.71875, 0.966811177, 0.966811177),
        (0.75, 0.958003065, 0.958003065),
        (0.78125, 0.943660866, 0.943660866),
        (0.8125, 0.923944917, 0.923944917),
        (0.84375, 0.89904617, 0.89904617),
        (0.875, 0.869186849, 0.869186849),
        (0.90625, 0.834620542, 0.834620542),
        (0.9375, 0.795631745, 0.795631745),
        (0.96875, 0.752534934, 0.752534934),
        (1.0, 0.705673158, 0.705673158)],
    'green': [
        (0.0, 0.298717966, 0.298717966),
        (0.03125, 0.353094838, 0.353094838),
        (0.0625, 0.406535296, 0.406535296),
        (0.09375, 0.458757618, 0.458757618),
        (0.125, 0.50941904, 0.50941904),
        (0.15625, 0.558148092, 0.558148092),
        (0.1875, 0.604562568, 0.604562568),
        (0.21875, 0.648280772, 0.648280772),
        (0.25, 0.688929332, 0.688929332),
        (0.28125, 0.726149107, 0.726149107),
        (0.3125, 0.759599947, 0.759599947),
        (0.34375, 0.788964712, 0.788964712),
        (0.375, 0.813952739, 0.813952739),
        (0.40625, 0.834302879, 0.834302879),
        (0.4375, 0.849786142, 0.849786142),
        (0.46875, 0.860207984, 0.860207984),
        (0.5, 0.86541021, 0.86541021),
        (0.53125, 0.848937047, 0.848937047),
        (0.5625, 0.827384882, 0.827384882),
        (0.59375, 0.800927443, 0.800927443),
        (0.625, 0.769767752, 0.769767752),
        (0.65625, 0.734132809, 0.734132809),
        (0.6875, 0.694266682, 0.694266682),
        (0.71875, 0.650421156, 0.650421156),
        (0.75, 0.602842431, 0.602842431),
        (0.78125, 0.551750968, 0.551750968),
        (0.8125, 0.49730856, 0.49730856),
        (0.84375, 0.439559467, 0.439559467),
        (0.875, 0.378313092, 0.378313092),
        (0.90625, 0.312874446, 0.312874446),
        (0.9375, 0.24128379, 0.24128379),
        (0.96875, 0.157246067, 0.157246067),
        (1.0, 0.01555616, 0.01555616)],
    'blue': [
        (0.0, 0.753683153, 0.753683153),
        (0.03125, 0.801466763, 0.801466763),
        (0.0625, 0.84495867, 0.84495867),
        (0.09375, 0.883725899, 0.883725899),
        (0.125, 0.917387822, 0.917387822),
        (0.15625, 0.945619588, 0.945619588),
        (0.1875, 0.968154911, 0.968154911),
        (0.21875, 0.98478814, 0.98478814),
        (0.25, 0.995375608, 0.995375608),
        (0.28125, 0.999836203, 0.999836203),
        (0.3125, 0.998151185, 0.998151185),
        (0.34375, 0.990363227, 0.990363227),
        (0.375, 0.976574709, 0.976574709),
        (0.40625, 0.956945269, 0.956945269),
        (0.4375, 0.931688648, 0.931688648),
        (0.46875, 0.901068838, 0.901068838),
        (0.5, 0.865395561, 0.865395561),
        (0.53125, 0.820880546, 0.820880546),
        (0.5625, 0.774508472, 0.774508472),
        (0.59375, 0.726736146, 0.726736146),
        (0.625, 0.678007945, 0.678007945),
        (0.65625, 0.628751763, 0.628751763),
        (0.6875, 0.579375448, 0.579375448),
        (0.71875, 0.530263762, 0.530263762),
        (0.75, 0.481775914, 0.481775914),
        (0.78125, 0.434243684, 0.434243684),
        (0.8125, 0.387970225, 0.387970225),
        (0.84375, 0.343229596, 0.343229596),
        (0.875, 0.300267182, 0.300267182),
        (0.90625, 0.259301199, 0.259301199),
        (0.9375, 0.220525627, 0.220525627),
        (0.96875, 0.184115123, 0.184115123),
        (1.0, 0.150232812, 0.150232812)]
    }


datad = {
    'afmhot': _afmhot_data,
    'autumn': _autumn_data,
    'bone':   _bone_data,
    'binary': _binary_data,
    'bwr':    _bwr_data,
    'brg':    _brg_data,
    'cool':   _cool_data,
    'copper': _copper_data,
    'cubehelix': _cubehelix_data,
    'flag':   _flag_data,
    'gnuplot': _gnuplot_data,
    'gnuplot2': _gnuplot2_data,
    'gray' :  _gray_data,
    'hot':    _hot_data,
    'hsv':    _hsv_data,
    'jet' :   _jet_data,
    'ocean':  _ocean_data,
    'pink':   _pink_data,
    'prism':  _prism_data,
    'rainbow': _rainbow_data,
    'seismic': _seismic_data,
    'spring': _spring_data,
    'summer': _summer_data,
    'terrain': _terrain_data,
    'winter': _winter_data,
    'spectral': _spectral_data
    }


datad['Accent']=_Accent_data
datad['Blues']=_Blues_data
datad['BrBG']=_BrBG_data
datad['BuGn']=_BuGn_data
datad['BuPu']=_BuPu_data
datad['Dark2']=_Dark2_data
datad['GnBu']=_GnBu_data
datad['Greens']=_Greens_data
datad['Greys']=_Greys_data
datad['Oranges']=_Oranges_data
datad['OrRd']=_OrRd_data
datad['Paired']=_Paired_data
datad['Pastel1']=_Pastel1_data
datad['Pastel2']=_Pastel2_data
datad['PiYG']=_PiYG_data
datad['PRGn']=_PRGn_data
datad['PuBu']=_PuBu_data
datad['PuBuGn']=_PuBuGn_data
datad['PuOr']=_PuOr_data
datad['PuRd']=_PuRd_data
datad['Purples']=_Purples_data
datad['RdBu']=_RdBu_data
datad['RdGy']=_RdGy_data
datad['RdPu']=_RdPu_data
datad['RdYlBu']=_RdYlBu_data
datad['RdYlGn']=_RdYlGn_data
datad['Reds']=_Reds_data
datad['Set1']=_Set1_data
datad['Set2']=_Set2_data
datad['Set3']=_Set3_data
datad['Spectral']=_Spectral_data
datad['YlGn']=_YlGn_data
datad['YlGnBu']=_YlGnBu_data
datad['YlOrBr']=_YlOrBr_data
datad['YlOrRd']=_YlOrRd_data
datad['gist_earth']=_gist_earth_data
datad['gist_gray']=_gist_gray_data
datad['gist_heat']=_gist_heat_data
datad['gist_ncar']=_gist_ncar_data
datad['gist_rainbow']=_gist_rainbow_data
datad['gist_stern']=_gist_stern_data
datad['gist_yarg']=_gist_yarg_data
datad['coolwarm']=_coolwarm_data
