(* mathcomp analysis (c) 2017 Inria and AIST. License: CeCILL-C.              *)
From mathcomp Require Import all_ssreflect all_algebra finmap.
Require Import boolp classical_sets mathcomp_extra reals ereal signed.
Require Import functions topology normedtype sequences cardinality esum fsbigop.
From HB Require Import structures.

(******************************************************************************)
(*                            Measure Theory                                  *)
(*                                                                            *)
(* This files provides a formalization of the basics of measure theory. This  *)
(* includes the formalization of mathematical structures for measure theory   *)
(* and of measures, as well as standard theorems such as the Boole            *)
(* inequality, Caratheodory's theorem, the Hahn extension, etc.               *)
(*                                                                            *)
(* Main references:                                                           *)
(* - Daniel Li, Intégration et applications, 2016                             *)
(* - Achim Klenke, Probability Theory 2nd edition, 2014                       *)
(*                                                                            *)
(*      setI_closed G == the set of sets G is closed under finite             *)
(*                       intersection                                         *)
(*      setU_closed G == the set of sets G is closed under finite union       *)
(*      setC_closed G == the set of sets G is closed under complement         *)
(*      setD_closed G == the set of sets G is closed under difference         *)
(*     ndseq_closed G == the set of sets G is closed under non-decreasing     *)
(*                       countable union                                      *)
(*  trivIset_closed G == the set of sets G is closed under pairwise-disjoint  *)
(*                       countable union                                      *)
(*                                                                            *)
(*       sigma_algebra D G == the set of sets G forms a sigma algebra on D    *)
(*             <<s D, G >> == sigma-algebra generated by G on D               *)
(*                         := smallest (sigma_algebra D) G                    *)
(*                <<s G >> := <<s setT, G >>                                  *)
(*      monotone_class D G == G is a monotone class of subsets of D           *)
(*             <<m D, G >> == monotone class generated by G on D              *)
(*                <<m G >> := <<m setT, G >>                                  *)
(*                dynkin G == G is a set of sets that form a Dynkin           *)
(*                            (or a lambda) system                            *)
(*                <<d G >> == Dynkin system generated by G, i.e.,             *)
(*                            smallest dynkin G                               *)
(*                                                                            *)
(* * Mathematical structures for measure theory:                              *)
(*    semiRingOfSetsType == the type of semirings of sets                     *)
(*        ringOfSetsType == the type of rings of sets                         *)
(*     algebraOfSetsType == the type of algebras of sets                      *)
(*        measurableType == the type of sigma-algebras                        *)
(*                                                                            *)
(*    g_measurableType G == the measurableType corresponding to s<< G >>      *)
(*                                                                            *)
(*    measurable_fun D f == the function f with domain D is measurable        *)
(*                                                                            *)
(*  preimage_class D f G == class generated by the preimages of G by f        *)
(*     image_class D f G == class generated by the images of G by f           *)
(*                                                                            *)
(* * Measures:                                                                *)
(*　　{additive_measure set T -> {ereal R}} == type of a function over sets of *)
(*                   elements of type T where R is expected to be a           *)
(*                   numFieldType such that this function maps set0 to 0, is  *)
(*                   non-negative over measurable sets, and is semi-additive  *)
(* 　{measure set T -> {ereal R}} == type of a function over sets of elements  *)
(*                   of type T where R is expected to be a numFieldType such  *)
(*                   that this function maps set0 to 0, is non-negative over  *)
(*                   measurable sets and is semi-sigma-additive               *)
(*   sigma_finite A f == the measure f is sigma-finite on A : set T with      *)
(*                   T : ringOfSetsType.                                      *)
(*   mu.-negligible A == A is mu negligible                                   *)
(* 　{ae mu, forall x, P x} == P holds almost everywhere for the measure mu   *)
(*                                                                            *)
(* 　{outer_measure set T -> {ereal R}} == type of an outer measure over sets  *)
(*                   of elements of type T where R is expected to be a        *)
(*                   numFieldType                                             *)
(*                                                                            *)
(*   mu.-measurable A == A is Caratheodory measurable for the outer measure   *)
(*                   mu, i.e.,                                                *)
(*                   forall B, mu A = mu (A `&` B) + mu (A `&` ~` B)          *)
(*   measure_is_complete mu == the measure mu is complete                     *)
(*                                                                            *)
(* * Caratheodory theorem (from an outer measure to a measure):               *)
(* caratheodory_type mu := T, where mu : {outer_measure set T -> {ereal R}}   *)
(*                         it is a canonical mesurableType copy of T.         *)
(* caratheodory_measure mu == the restriction of the outer measure mu to the  *)
(*                         sigma algebra of Caratheodory measurable sets is a *)
(*                         measure                                            *)
(*                         Remark: sets that are negligible for               *)
(*                         caratheodory_measure are Caratheodory measurable   *)
(*                                                                            *)
(* * Caratheodory theorem (from a premeasure to an outer measure):            *)
(* measurable_cover X == the set of sequences F such that                     *)
(*                       - forall k, F k is measurable                        *)
(*                       - X `<=` \bigcup_k (F k)                             *)
(*          mu_ext mu == extension of the measure mu over a semiring of sets; *)
(*                       it is an outer measure, declared as; we have the     *)
(*                       notation [outer_measure of mu_ext mu])               *)
(*                                                                            *)
(* * Hahn Extension:                                                          *)
(*  Hahn_ext_measure mu == extension of the additive measure mu over a        *)
(*                         semiring of sets to a measure over the generated   *)
(*                         sigma algebra (requires a proof of                 *)
(*                         sigma-sub-additivity)                              *)
(*                                                                            *)
(* * Product of measurable spaces:                                            *)
(*   preimage_classes f1 f2 := <<s preimage_class setT f1 measurable `|`      *)
(*                                 preimage_class setT f2 measurable >>.      *)
(*   prod_measurableType == product of measurableType                         *)
(******************************************************************************)

Set Implicit Arguments.
Unset Strict Implicit.
Unset Printing Implicit Defensive.
Import Order.TTheory GRing.Theory Num.Def Num.Theory.

Reserved Notation "'s<|' D , G '|>'" (at level 40, G, D at next level).
Reserved Notation "'s<<' A '>>'".
Reserved Notation "'d<<' D '>>'".
Reserved Notation "mu .-negligible" (at level 2, format "mu .-negligible").
Reserved Notation "{ 'ae' m , P }" (at level 0, format "{ 'ae'  m ,  P }").
Reserved Notation "mu .-measurable" (at level 2, format "mu .-measurable").

Local Open Scope classical_set_scope.
Local Open Scope ring_scope.

Section classes.
Context {T} (C : set (set T) -> Prop) (D : set T) (G : set (set T)).

Definition setC_closed := forall A, G A -> G (~` A).
Definition setI_closed := forall A B, G A -> G B -> G (A `&` B).
Definition setU_closed := forall A B, G A -> G B -> G (A `|` B).
Definition setD_closed := forall A B, B `<=` A -> G A -> G B -> G (A `\` B).
Definition setDI_closed := forall A B, G A -> G B -> G (A `\` B).

Definition fin_bigcap_closed :=
    forall I (D : set I) A_, finite_set D -> (forall i, D i -> G (A_ i)) ->
  G (\bigcap_(i in D) (A_ i)).

Definition finN0_bigcap_closed :=
    forall I (D : set I) A_, finite_set D -> D !=set0 ->
    (forall i, D i -> G (A_ i)) ->
  G (\bigcap_(i in D) (A_ i)).

Definition fin_bigcup_closed :=
    forall I (D : set I) A_, finite_set D -> (forall i, D i -> G (A_ i)) ->
  G (\bigcup_(i in D) (A_ i)).

Definition semi_setD_closed := forall A B, G A -> G B -> exists D,
  [/\ finite_set D, D `<=` G, A `\` B = \bigcup_(X in D) X & trivIset D id].

Definition ndseq_closed :=
 forall F, nondecreasing_seq F -> (forall i, G (F i)) -> G (\bigcup_i (F i)).

Definition trivIset_closed :=
  forall F : (set T)^nat, trivIset setT F -> (forall n, G (F n)) ->
                    G (\bigcup_k F k).

Definition fin_trivIset_closed :=
  forall I (D : set I) (F : I -> set T), finite_set D -> trivIset D F ->
   (forall i, D i -> G (F i)) -> G (\bigcup_(k in D) F k).

Definition setring := [/\ G set0, setU_closed & setDI_closed].

Definition sigma_algebra :=
  [/\ G set0, (forall A, G A -> G (D `\` A)) &
     (forall A : (set T)^nat, (forall n, G (A n)) -> G (\bigcup_k A k))].

Definition dynkin := [/\ G setT, setC_closed & trivIset_closed].

Definition monotone_class :=
  [/\ forall A, G A -> A `<=` D, G D, setD_closed & ndseq_closed].

End classes.

Notation "'<<m' D , G '>>'" := (smallest (monotone_class D) G)
  (at level 2, format "'<<m'  D ,  G  '>>'") : classical_set_scope.
Notation "'<<m' G '>>'" := (<<m setT, G>>)
  (at level 2, format "'<<m'  G  '>>'") : classical_set_scope.
Notation "'<<s' D , G '>>'" := (smallest (sigma_algebra D) G)
  (at level 2, format "'<<s'  D ,  G  '>>'") : classical_set_scope.
Notation "'<<s' G '>>'" := (<<s setT, G>>)
  (at level 2, format "'<<s'  G  '>>'") : classical_set_scope.
Notation "'<<d' G '>>'" := (smallest dynkin G)
  (at level 2, format "'<<d'  G '>>'") : classical_set_scope.
Notation "'<<r' G '>>'" := (smallest setring G)
  (at level 2, format "'<<r'  G '>>'") : classical_set_scope.
Notation "'<<fu' G '>>'" := (smallest fin_trivIset_closed G)
  (at level 2, format "'<<fu'  G '>>'") : classical_set_scope.

Lemma fin_bigcup_closedP T (G : set (set T)) :
  (G set0 /\ setU_closed G) <-> fin_bigcup_closed G.
Proof.
split=> [[G0 GU] I D A DF GA|GU]; last first.
  have G0 : G set0 by have := GU void set0 point; rewrite bigcup0//; apply.
  by split=> // A B GA GB; rewrite -bigcup2inE; apply: GU => // -[|[|[]]].
elim/Pchoice: I => I in D DF A GA *; rewrite bigcup_fset_set// big_seq.
by elim/big_ind: _ => // i; rewrite in_fset_set// inE => /GA.
Qed.

Lemma finN0_bigcap_closedP T (G : set (set T)) :
  setI_closed G <-> finN0_bigcap_closed G.
Proof.
split=> [GI I D A DF [i Di] GA|GI A B GA GB]; last first.
  by rewrite -bigcap2inE; apply: GI => // [|[|[|[]]]]; first by exists 0%N.
elim/Pchoice: I => I in D DF i Di A GA *.
have finDDi : finite_set (D `\ i) by exact: finite_setD.
rewrite (bigcap_setD1 i)// bigcap_fset_set// big_seq.
elim/big_rec: _ => // [|j B]; first by rewrite setIT; apply: GA.
rewrite in_fset_set// inE => -[Dj /eqP nij] GAB.
by rewrite setICA; apply: GI => //; apply: GA.
Qed.

Lemma sedDI_closedP T (G : set (set T)) :
  setDI_closed G <-> (setI_closed G /\ setD_closed G).
Proof.
split=> [GDI|[GI GD]].
  by split=> A B => [|AB] => GA GB; rewrite -?setDD//; do ?apply: (GDI).
move=> A B GA GB; suff <- : A `\` (A `&` B) = A `\` B.
  by apply: GD => //; apply: GI.
by rewrite setDE setCI setIUr -setDE setDv set0U.
Qed.

Lemma sigma_algebra_bigcap T (I : choiceType) (D : set T)
    (F : I -> set (set T)) (J : set I) :
  (forall n, J n -> sigma_algebra D (F n)) ->
  sigma_algebra D (\bigcap_(i in J) F i).
Proof.
move=> mG; split=> [i Ji|A AJ i Ji|H GH i Ji]; first by have [] := mG i.
- by have [_ mGiC _] := mG i Ji; exact/mGiC/AJ.
- by have [_ _ mGiU] := mG i Ji; apply mGiU => j; exact: GH.
Qed.

Lemma sigma_algebraP T U (C : set (set T)) :
  (forall X, C X -> X `<=` U) ->
  sigma_algebra U C <->
  [/\ C U, setD_closed C, ndseq_closed C & setI_closed C].
Proof.
move=> C_subU; split => [[C0 CD CU]|[DT DC DU DI]]; split.
- by rewrite -(setD0 U); apply: CD.
- move=> A B BA CA CB; rewrite (_ : A `\` B = U `\` ((U `\` A) `|` B)).
    by apply CD; rewrite -bigcup2E; apply: CU => -[|[|[|]]] //=; exact: CD.
  rewrite setDUr setDD [in RHS]setDE setIACA setIid -setDE setIidr//.
  by rewrite setDE; apply: subIset; left; apply: C_subU.
- by move=> F ndF DF; exact: CU.
- move=> A B DA DB; rewrite (_ : A `&` B = U `\` ((U `\` A) `|` (U `\` B))).
    by apply CD; rewrite -bigcup2E; apply: CU => -[|[|[|]]] //; exact: CD.
  rewrite setDUr !setDD setIACA setIid (@setIidr _ U)//.
  by apply: subIset; left; exact: C_subU.
- by rewrite -(setDv U); exact: DC.
- by move=> A CA; apply: DC => //; exact: C_subU.
- move=> F DF.
  rewrite [X in C X](_ : _ = \bigcup_i \big[setU/set0]_(j < i.+1) F j).
    apply: DU; first by move=> *; exact/subsetPset/subset_bigsetU.
    elim=> [|n ih]; first by rewrite big_ord_recr /= big_ord0 set0U; exact: DF.
    have CU : setU_closed C.
      move=> A B DA DB; rewrite (_ : A `|` B = U `\` ((U `\` A) `&` (U `\` B))).
        apply DC => //; last by apply: DI; apply: DC => //; exact: C_subU.
        by apply: subIset; left; apply: subIset; left.
      by rewrite setDIr// !setDD (setIidr (C_subU _ DA)) (setIidr (C_subU _ _)).
    by rewrite big_ord_recr; exact: CU.
  rewrite predeqE => x; split => [[n _ Fnx]|[n _]].
    by exists n => //; rewrite big_ord_recr /=; right.
  by rewrite -bigcup_mkord => -[m /=]; rewrite ltnS => _ Fmx; exists m.
Qed.

Section generated_sigma_algebra.
Context {T : Type} (D : set T) (G : set (set T)).
Implicit Types (M : set (set T)).

Lemma smallest_sigma_algebra : sigma_algebra D <<s D, G >>.
Proof.
split=> [|A GA|A GA] P [[P0 PD PU]] GP //.
  by apply: (PD); apply: GA.
by apply: (PU) => n; apply: GA.
Qed.
Hint Resolve smallest_sigma_algebra : core.

Lemma sub_sigma_algebra2 M : M `<=` G -> <<s D, M >> `<=` <<s D, G >>.
Proof. exact: sub_smallest2r. Qed.

Lemma sigma_algebra_id : sigma_algebra D G -> <<s D, G >> = G.
Proof. by move=> /smallest_id->. Qed.

Lemma sub_sigma_algebra : G `<=` <<s D, G >>. Proof. exact: sub_smallest. Qed.

Lemma sigma_algebra0 : <<s D, G >> set0.
Proof. by case: smallest_sigma_algebra. Qed.

Lemma sigma_algebraD : forall A, <<s D, G >> A -> <<s D, G >> (D `\` A).
Proof. by case: smallest_sigma_algebra. Qed.

Lemma sigma_algebra_bigcup (A : (set T)^nat) :
  (forall i, <<s D, G >> (A i)) ->  <<s D, G >> (\bigcup_i (A i)).
Proof. by case: smallest_sigma_algebra A. Qed.

End generated_sigma_algebra.
#[global] Hint Resolve smallest_sigma_algebra : core.

Section generated_setring.
Context {T : Type} (G : set (set T)).
Implicit Types (M : set (set T)).

Lemma smallest_setring : setring <<r G >>.
Proof.
split=> [|A B GA GB|A B GA GB] P [[P0 PU PDI]] GP //.
  by apply: (PU); [apply: GA|apply: GB].
by apply: (PDI); [apply: GA|apply: GB].
Qed.
Hint Resolve smallest_setring : core.

Lemma sub_setring2 M : M `<=` G -> <<r M >> `<=` <<r G >>.
Proof. exact: sub_smallest2r. Qed.

Lemma setring_id : setring G -> <<r G >> = G.
Proof. by move=> /smallest_id->. Qed.

Lemma sub_setring : G `<=` <<r G >>. Proof. exact: sub_smallest. Qed.

Lemma setring0 : <<r G >> set0.
Proof. by case: smallest_setring. Qed.

Lemma setringDI : setDI_closed <<r G>>.
Proof. by case: smallest_setring. Qed.

Lemma setringU : setU_closed <<r G>>.
Proof. by case: smallest_setring. Qed.

Lemma setring_fin_bigcup : fin_bigcup_closed <<r G>>.
Proof.
by apply/fin_bigcup_closedP; split; [apply: setring0|apply: setringU].
Qed.

End generated_setring.
#[global] Hint Resolve smallest_setring setring0 : core.

Lemma monotone_class_g_salgebra T (G : set (set T)) (D : set T) :
  (forall X, <<s D, G >> X -> X `<=` D) -> G D ->
  monotone_class D (<<s D, G >>).
Proof.
move=> sDGD GD; have := smallest_sigma_algebra D G.
by move=> /(sigma_algebraP sDGD) [sT sD snd sI]; split.
Qed.

Section smallest_monotone_classE.
Variables (T : Type) (G : set (set T)) (setIG : setI_closed G).
Variables (D : set T) (GD : G D).
Variables (H : set (set T)) (monoH : monotone_class D H) (GH : G `<=` H).

Lemma smallest_monotone_classE : (forall X, <<s D, G >> X -> X `<=` D) ->
  (forall E, monotone_class D E -> G `<=` E -> H `<=` E) ->
  H = <<s D, G >>.
Proof.
move=> sDGD smallestH; rewrite eqEsubset; split.
  apply: (smallestH _ _ (@sub_sigma_algebra _ D G)).
  exact: monotone_class_g_salgebra.
suff: setI_closed H.
  move=> IH; apply: smallest_sub => //.
  by apply/sigma_algebraP; by case: monoH.
pose H_ A := [set X | H X /\ H (X `&` A)].
have setDH_ A : setD_closed (H_ A).
  move=> X Y XY [HX HXA] [HY HYA]; case: monoH => h _ setDH _; split.
    exact: setDH.
  rewrite (_ : _ `&` _ = (X `&` A) `\` (Y `&` A)); last first.
    rewrite predeqE => x; split=> [[[? ?] ?]|]; first by split => // -[].
    by move=> [[? ?] YAx]; split => //; split => //; apply: contra_not YAx.
  by apply setDH => //; exact: setSI.
have ndH_ A : ndseq_closed (H_ A).
  move=> F ndF H_AF; split.
    by case: monoH => h _ _; apply => // => n; have [] := H_AF n.
  rewrite setI_bigcupl; case: monoH => h _ _; apply => //.
    by move=> m n mn; apply/subsetPset; apply: setSI; apply/subsetPset/ndF.
  by move=> n; have [] := H_AF n.
have GGH_ X : G X -> G `<=` H_ X.
  by move=> *; split; [exact: GH | apply: GH; exact: setIG].
have GHH_ X : G X -> H `<=` H_ X.
  move=> CX; apply: smallestH; [split => //; last exact: GGH_|exact: GGH_].
  by move=> ? [? ?]; case: monoH => + _ _ _; exact.
have HGH_ X : H X -> G `<=` H_ X.
  by move=> *; split; [exact: GH|rewrite setIC; apply GHH_].
have HHH_ X : H X -> H `<=` H_ X.
  move=> HX; apply: (smallestH _ _ (HGH_ _ HX)); split=> //.
  - by move=> ? [? ?]; case: monoH => + _ _ _; exact.
  - exact: HGH_.
by move=> *; apply HHH_.
Qed.

End smallest_monotone_classE.

Section monotone_class_subset.
Variables (T : Type) (G : set (set T)) (setIG : setI_closed G).
Variables (D : set T) (GD : G D).
Variables (H : set (set T)) (monoH : monotone_class D H) (GH : G `<=` H).

Lemma monotone_class_subset : (forall X, (<<s D, G >>) X -> X `<=` D) ->
  <<s D, G >> `<=` H.
Proof.
move=> sDGD; set M := <<m D, G >>.
rewrite -(@smallest_monotone_classE _ _ setIG _ _ M) //.
- exact: smallest_sub.
- split => [A MA | E [monoE] | A B BA MA MB E [[EsubD ED setDE ndE] GE] |].
  + by case: monoH => + _ _ _; apply; exact: MA.
  + exact.
  + by apply setDE => //; [exact: MA|exact: MB].
  + by move=> F ndF MF E [[EsubD ED setDE ndE] CE]; apply ndE=> // n; exact: MF.
- exact: sub_smallest.
- by move=> ? ? ?; exact: smallest_sub.
Qed.

End monotone_class_subset.

Section dynkin.
Variable T : Type.
Implicit Types G D : set (set T).

Lemma dynkinT G : dynkin G -> G setT. Proof. by case. Qed.

Lemma dynkinC G : dynkin G -> setC_closed G. Proof. by case. Qed.

Lemma dynkinU G : dynkin G -> (forall F : (set T)^nat, trivIset setT F ->
  (forall n, G (F n)) -> G (\bigcup_k F k)). Proof. by case. Qed.

End dynkin.

Section dynkin_lemmas.
Variable T : Type.
Implicit Types D G : set (set T).

Lemma dynkin_monotone G : dynkin G <-> monotone_class setT G.
Proof.
split => [[GT setCG trG]|[_ GT setDG ndG]]; split => //.
- move=> A B BA GA GB; rewrite setDE -(setCK (_ `&` _)) setCI; apply setCG.
  rewrite setCK -bigcup2E; apply: trG.
  + by rewrite -trivIset_bigcup2 setIC; apply subsets_disjoint.
  + by move=> [|[//|n]]; [exact: setCG|rewrite /bigcup2 -setCT; apply: setCG].
- move=> F ndF GF; rewrite eq_bigcup_seqD; apply trG.
    exact: trivIset_seqD.
  move=> [/=|n]; first exact: GF.
  rewrite /seqD setDE -(setCK (_ `&` _)) setCI; apply setCG.
  rewrite setCK -bigcup2E; apply trG.
  + rewrite -trivIset_bigcup2 setIC; apply subsets_disjoint.
    exact/subsetPset/ndF/ltnW.
  + move=> [|[|]]; rewrite /bigcup2 /=; [exact/setCG/GF|exact/GF|].
    by move=> _; rewrite -setCT; apply: setCG.
- by move=> A B; rewrite -setTD; apply: setDG.
- move=> F tF GF; pose A i := \big[setU/set0]_(k < i.+1) F k.
  rewrite (_ : bigcup _ _ = \bigcup_i A i); last first.
    rewrite predeqE => t; split => [[n _ Fn]|[n _]].
      by exists n => //; rewrite /A -bigcup_mkord; exists n=> //=; rewrite ltnS.
    by rewrite /A -bigcup_mkord => -[m /=]; rewrite ltnS => mn Fmt; exists m.
  apply ndG; first by move=> a b ab; exact/subsetPset/subset_bigsetU.
  elim=> /= => [|n ih].
    by rewrite /A big_ord_recr /= big_ord0 set0U; exact: GF.
  rewrite /A /= big_ord_recr /= -/(A n).
  rewrite (_ : _ `|` _ = ~` (~` A n `\` F n.+1)); last first.
    by rewrite setDE setCI !setCK.
  rewrite -setTD; apply setDG => //; apply setDG => //; last first.
    by rewrite -setTD; apply: setDG.
  apply/disjoints_subset; rewrite setIC.
  by apply: (@trivIset_bigsetUI _ predT) => //; rewrite /predT /= trueE.
Qed.

Lemma dynkin_g_dynkin G : dynkin (<<d G >>).
Proof.
split=> [D /= [] []//| | ].
- by move=> Y sGY D /= [dD GD]; exact/(dynkinC dD)/(sGY D).
- by move=> F tF gGF D /= [dD GD]; apply dD => // k; exact: gGF.
Qed.

Lemma sigma_algebra_dynkin G : sigma_algebra setT G -> dynkin G.
Proof.
case=> ? DC DU; split => [| |? ? ?]; last exact: DU.
- by rewrite -setC0 -setTD; exact: DC.
- by move=> A GA; rewrite -setTD; apply: DC.
Qed.

Lemma dynkin_setI_bigsetI G (F : (set T)^nat) : dynkin G -> setI_closed G ->
  (forall n, G (F n)) -> forall n, G (\big[setI/setT]_(i < n) F i).
Proof.
move=> dG GI GF; elim => [|n ih]; last by rewrite big_ord_recr /=; apply: GI.
by rewrite big_ord0; exact: (dynkinT dG).
Qed.

Lemma dynkin_setI_sigma_algebra G : dynkin G -> setI_closed G ->
  sigma_algebra setT G.
Proof.
move=> dG GI; split => [|//|F DF].
- by rewrite -setCT; exact/(dynkinC dG)/(dynkinT dG).
- by move=> A GA; rewrite setTD; exact: (dynkinC dG).
- rewrite seqDU_bigcup_eq; apply/(dynkinU dG) => //.
  move=> n; rewrite /seqDU setDE; apply GI => //.
  rewrite -bigcup_mkord setC_bigcup bigcap_mkord.
  by apply: (@dynkin_setI_bigsetI _ (fun x => ~` F x)) => // ?; exact/(dynkinC dG).
Qed.

Lemma setI_closed_gdynkin_salgebra G : setI_closed G -> <<d G >> = <<s G >>.
Proof.
move=> GI; rewrite eqEsubset; split.
  by apply: sub_smallest2l; apply: sigma_algebra_dynkin.
pose delta (D : set T) := [set E | <<d G >> (E `&` D)].
have ddelta (D : set T) : <<d G >> D -> dynkin (delta D).
  move=> dGD; split; first by rewrite /delta /= setTI.
  - move=> E DE; rewrite /delta /=.
    have -> : (~` E) `&` D = ~` ((E `&` D) `|` (~` D)).
      by rewrite -[LHS]setU0 -(setICl D) -setIUl -setCI -{2}(setCK D) -setCU.
    have : <<d G >> ((E `&` D) `|` ~` D).
      rewrite -bigcup2E => S [dS GS]; apply (dynkinU dS).
        move=> [|[|i]] [|[|j]] => // _ _; rewrite /bigcup2 /=.
        + by rewrite -setIA setICr setI0 => /set0P; rewrite eqxx.
        + by rewrite setI0 => /set0P; rewrite eqxx.
        + by rewrite setICA setICl setI0 => /set0P; rewrite eqxx.
        + by rewrite setI0 => /set0P; rewrite eqxx.
        + by rewrite set0I => /set0P; rewrite eqxx.
        + by rewrite set0I => /set0P; rewrite eqxx.
        + by rewrite set0I => /set0P; rewrite eqxx.
      move=> [|[|n]] //; rewrite /bigcup2 /=; [exact: DE| |].
      + suff: <<d G >> (~` D) by exact.
        by move=> F [dF GF]; apply (dynkinC dF) => //; exact: dGD.
      + by rewrite -setCT; apply/(dynkinC dS)/(dynkinT dS).
    by move=> dGEDD S /= [+ GS] => dS; apply/(dynkinC dS); exact: dGEDD.
  - move=> F tF deltaDF; rewrite /delta /= => S /= [dS GS].
    rewrite setI_bigcupl; apply (dynkinU dS) => //.
      by under eq_fun do rewrite setIC; exact: trivIset_setI.
    by move=> n; exact: deltaDF.
have GdG : G `<=` <<d G >> by move=> ? ? ? [_]; apply.
have Gdelta A : G A -> G `<=` delta A.
  by move=> ? ? ?; rewrite /delta /= => ? [?]; apply; exact/GI.
have GdGdelta A : G A -> <<d G >> `<=` delta A.
  move=> ?; apply: smallest_sub => //; last exact: Gdelta.
  by apply/ddelta; exact: GdG.
have dGGI A B : <<d G >> A -> G B -> <<d G >> (A `&` B).
  by move=> ? ?; apply: GdGdelta.
have dGGdelta A : <<d G >> A -> G `<=` delta A.
  by move=> ? ? ?; rewrite /delta /= setIC; exact: dGGI.
have dGdGdelta A : <<d G >> A -> <<d G >> `<=` delta A.
  by move=> ?; exact: smallest_sub (ddelta _ _) (dGGdelta _ _).
have dGdGdG A B : <<d G >> A -> <<d G >> B -> <<d G >> (A `&` B).
  by move=> ? ?; exact: dGdGdelta.
apply: smallest_sub => //; apply: dynkin_setI_sigma_algebra => //.
exact: dynkin_g_dynkin.
Qed.

End dynkin_lemmas.

HB.mixin Record isSemiRingOfSets T := {
  ptclass : Pointed.class_of T;
  measurable : set (set T) ;
  measurable0 : measurable set0 ;
  measurableI : setI_closed measurable;
  semi_measurableD : semi_setD_closed measurable;
}.

HB.structure Definition SemiRingOfSets := {T of isSemiRingOfSets T}.
Notation semiRingOfSetsType := SemiRingOfSets.type.

Canonical semiRingOfSets_eqType (T : semiRingOfSetsType) := EqType T ptclass.
Canonical semiRingOfSets_choiceType (T : semiRingOfSetsType) :=
  ChoiceType T ptclass.
Canonical semiRingOfSets_ptType (T : semiRingOfSetsType) :=
  PointedType T ptclass.

HB.mixin Record RingOfSets_from_semiRingOfSets T of isSemiRingOfSets T := {
  measurableU : forall A B : set T,
    measurable A -> measurable B -> measurable (A `|` B) }.

HB.structure Definition RingOfSets := {T of RingOfSets_from_semiRingOfSets T &}.
Notation ringOfSetsType := RingOfSets.type.

Canonical ringOfSets_eqType (T : ringOfSetsType) := EqType T ptclass.
Canonical ringOfSets_choiceType (T : ringOfSetsType) := ChoiceType T ptclass.
Canonical ringOfSets_ptType (T : ringOfSetsType) := PointedType T ptclass.

HB.mixin Record AlgebraOfSets_from_RingOfSets T of RingOfSets T := {
  measurableT : measurable (@setT T)
}.

HB.structure Definition AlgebraOfSets :=
  {T of AlgebraOfSets_from_RingOfSets T &}.
Notation algebraOfSetsType := AlgebraOfSets.type.

Canonical algebraOfSets_eqType (T : algebraOfSetsType) := EqType T ptclass.
Canonical algebraOfSets_choiceType (T : algebraOfSetsType) :=
  ChoiceType T ptclass.
Canonical algebraOfSets_ptType (T : algebraOfSetsType) :=
  PointedType T ptclass.

HB.mixin Record Measurable_from_algebraOfSets T of AlgebraOfSets T := {
  bigcupT_measurable : forall F : (set T)^nat, (forall i, measurable (F i)) ->
    measurable (\bigcup_i (F i))
}.

HB.structure Definition Measurable := {T of Measurable_from_algebraOfSets T &}.
Notation measurableType := Measurable.type.

Canonical measurable_eqType (T : measurableType) := EqType T ptclass.
Canonical measurable_choiceType (T : measurableType) := ChoiceType T ptclass.
Canonical measurable_ptType (T : measurableType) := PointedType T ptclass.

HB.factory Record isRingOfSets T := {
  ptclass : Pointed.class_of T;
  measurable : set (set T) ;
  measurable0 : measurable set0 ;
  measurableU : setU_closed measurable;
  measurableD : setDI_closed measurable;
}.

HB.builders Context T of isRingOfSets T.
Implicit Types (A B C D : set T).

Lemma mI : setI_closed measurable.
Proof. by move=> A B mA mB; rewrite -setDD; do ?apply: measurableD. Qed.

Lemma mD : semi_setD_closed measurable.
Proof.
move=> A B Am Bm; exists [set A `\` B]; split; rewrite ?bigcup_set1//.
  by move=> C ->; apply: measurableD.
by move=> X Y -> ->.
Qed.

HB.instance Definition T_isSemiRingOfSets :=
  @isSemiRingOfSets.Build T ptclass measurable measurable0 mI mD.

HB.instance Definition T_isRingOfSets :=
  RingOfSets_from_semiRingOfSets.Build T measurableU.

HB.end.

HB.factory Record isAlgebraOfSets T := {
  ptclass : Pointed.class_of T;
  measurable : set (set T) ;
  measurable0 : measurable set0 ;
  measurableU : setU_closed measurable;
  measurableC : setC_closed measurable
}.

HB.builders Context T of isAlgebraOfSets T.

Lemma mD : setDI_closed measurable.
Proof.
move=> A B mA mB; rewrite setDE -[A]setCK -setCU.
by do ?[apply: measurableU | apply: measurableC].
Qed.

HB.instance Definition T_isRingOfSets := @isRingOfSets.Build T ptclass
  measurable measurable0 measurableU mD.

Lemma measurableT : measurable (@setT T).
Proof. by rewrite -setC0; apply: measurableC; exact: measurable0. Qed.

HB.instance Definition T_isAlgebraOfSets : AlgebraOfSets_from_RingOfSets T :=
  AlgebraOfSets_from_RingOfSets.Build T measurableT.

HB.end.

HB.factory Record isMeasurable T := {
  ptclass : Pointed.class_of T;
  measurable : set (set T) ;
  measurable0 : measurable set0 ;
  measurableC : forall A, measurable A -> measurable (~` A) ;
  measurable_bigcup : forall F : (set T)^nat, (forall i, measurable (F i)) ->
    measurable (\bigcup_i (F i))
}.

HB.builders Context T of isMeasurable T.

Obligation Tactic := idtac.

Lemma mU : setU_closed measurable.
Proof.
move=> A B mA mB; rewrite -bigcup2E.
by apply measurable_bigcup => -[//|[//|i]]; exact: measurable0.
Qed.

Lemma mC : setC_closed measurable. Proof. by move=> *; apply: measurableC. Qed.

HB.instance Definition T_isAlgebraOfSets :=
  @isAlgebraOfSets.Build T ptclass measurable measurable0 mU mC.

HB.instance Definition T_isMeasurable :=
  @Measurable_from_algebraOfSets.Build T measurable_bigcup.

HB.end.

#[global] Hint Extern 0 (measurable set0) => solve [apply: measurable0] : core.
#[global] Hint Extern 0 (measurable setT) => solve [apply: measurableT] : core.

Section ringofsets_lemmas.
Variables T : ringOfSetsType.
Implicit Types A B : set T.

Lemma bigsetU_measurable I r (P : pred I) (F : I -> set T) :
  (forall i, P i -> measurable (F i)) ->
  measurable (\big[setU/set0]_(i <- r | P i) F i).
Proof. by move=> mF; elim/big_ind : _ => //; exact: measurableU. Qed.

Lemma fin_bigcup_measurable I (D : set I) (F : I -> set T) :
    finite_set D ->
    (forall i, D i -> measurable (F i)) ->
  measurable (\bigcup_(i in D) F i).
Proof.
elim/Pchoice: I => I in D F * => Dfin Fm.
rewrite bigcup_fset_set// big_seq; apply: bigsetU_measurable => i.
by rewrite in_fset_set ?inE// => *; apply: Fm.
Qed.

Lemma measurableD : setDI_closed (@measurable T).
Proof.
move=> A B mA mB; case: (semi_measurableD A B) => // [D [Dfin Dl -> _]].
by apply: fin_bigcup_measurable.
Qed.

End ringofsets_lemmas.

Section algebraofsets_lemmas.
Variables T : algebraOfSetsType.
Implicit Types A B : set T.

Lemma measurableC A : measurable A -> measurable (~` A).
Proof. by move=> mA; rewrite -setTD; exact: measurableD. Qed.

Lemma bigsetI_measurable I r (P : pred I) (F : I -> set T) :
  (forall i, P i -> measurable (F i)) ->
  measurable (\big[setI/setT]_(i <- r | P i) F i).
Proof.
move=> mF; rewrite -[X in measurable X]setCK setC_bigsetI; apply: measurableC.
by apply: bigsetU_measurable => i Pi; apply/measurableC/mF.
Qed.

Lemma fin_bigcap_measurable I (D : set I) (F : I -> set T) :
    finite_set D ->
    (forall i, D i -> measurable (F i)) ->
  measurable (\bigcap_(i in D) F i).
Proof.
elim/Pchoice: I => I in D F * => Dfin Fm.
rewrite bigcap_fset_set// big_seq; apply: bigsetI_measurable => i.
by rewrite in_fset_set ?inE// => *; apply: Fm.
Qed.

End algebraofsets_lemmas.

Section measurable_lemmas.
Variables T : measurableType.
Implicit Types (A B : set T) (F : (set T)^nat) (P : set nat).

Lemma sigma_algebra_measurable : sigma_algebra setT (@measurable T).
Proof. by split=> // [A|]; [exact: measurableD|exact: bigcupT_measurable]. Qed.

Lemma bigcup_measurable F P :
  (forall k, P k -> measurable (F k)) -> measurable (\bigcup_(i in P) F i).
Proof.
move=> PF; rewrite bigcup_mkcond; apply: bigcupT_measurable => k.
by case: ifP => //; rewrite inE; exact: PF.
Qed.

Lemma bigcap_measurable F P :
  (forall k, P k -> measurable (F k)) -> measurable (\bigcap_(i in P) F i).
Proof.
move=> PF; rewrite -[X in measurable X]setCK setC_bigcap; apply: measurableC.
by apply: bigcup_measurable => k Pk; exact/measurableC/PF.
Qed.

Lemma bigcapT_measurable F : (forall i, measurable (F i)) ->
  measurable (\bigcap_i (F i)).
Proof. by move=> ?; exact: bigcap_measurable. Qed.

End measurable_lemmas.

Lemma bigcupT_measurable_rat (T : measurableType) (F : rat -> set T) :
  (forall i, measurable (F i)) -> measurable (\bigcup_i F i).
Proof.
move=> Fm; have /ppcard_eqP[f] := card_rat.
by rewrite (reindex_bigcup f^-1%FUN setT)//=; exact: bigcupT_measurable.
Qed.

Definition g_measurable {T} (G : set (set T)) := T.

Section g_salgebra_instance.
Variables (T : pointedType) (G : set (set T)).

Lemma sigma_algebraC (A : set T) : <<s G >> A -> <<s G >> (~` A).
Proof. by move=> sGA; rewrite -setTD; exact: sigma_algebraD. Qed.

Canonical g_measurable_eqType := EqType (g_measurable G) (Equality.class T).
Canonical g_measurable_choiceType :=
  ChoiceType (g_measurable G) (Choice.class T).
Canonical g_measurable_ptType := PointedType (g_measurable G) (Pointed.class T).
HB.instance Definition _ := @isMeasurable.Build (g_measurable G)
  (Pointed.class T) <<s G >> (@sigma_algebra0 _ setT G) (@sigma_algebraC)
  (@sigma_algebra_bigcup _ setT G).

Definition g_measurableType := [the measurableType of g_measurable G].

End g_salgebra_instance.

Lemma measurable_g_measurableTypeE (T : pointedType) (G : set (set T)) :
  sigma_algebra setT G -> @measurable (g_measurableType G) = G.
Proof. exact: sigma_algebra_id. Qed.

Definition measurable_fun (T U : measurableType) (D : set T) (f : T -> U) :=
  measurable D -> forall Y, measurable Y -> measurable (D `&` f @^-1` Y).

Section measurable_fun.
Implicit Types T : measurableType.

Lemma measurable_fun_id T (D : set T) : measurable_fun D id.
Proof. by move=> mD A mA; apply: measurableI. Qed.

Lemma measurable_fun_comp T1 T2 T3 (f : T2 -> T3) E (g : T1 -> T2) :
  measurable_fun setT f -> measurable_fun E g -> measurable_fun E (f \o g).
Proof.
move=> mf mg /= mE A mA; rewrite comp_preimage; apply/mg => //.
by rewrite -[X in measurable X]setTI; apply/mf.
Qed.

Lemma eq_measurable_fun T1 T2 (D : set T1) (f g : T1 -> T2) :
  {in D, f =1 g} -> measurable_fun D f -> measurable_fun D g.
Proof.
move=> Dfg Df mD A mA; rewrite (_ : D `&` _ = D `&` f @^-1` A); first exact: Df.
apply/seteqP; rewrite /preimage; split => [x /= [Dx Agx]|x /= [Dx Afx]].
  by split=> //; rewrite Dfg// inE.
by split=> //; rewrite -Dfg// inE.
Qed.

Lemma measurable_fun_cst T1 T2 (D : set T1) (r : T2) :
  measurable_fun D (cst r : T1 -> _).
Proof.
move=> mD A mA; have [rA|rA] := boolP (r \in A).
- rewrite [X in measurable X](_ : _ = D) // predeqE => t; split=> [[]//|Dt].
  by split => //; rewrite inE in rA.
- rewrite [X in measurable X](_ : _ = set0)// predeqE.
  by move=> t; split=> // -[]; rewrite notin_set in rA.
Qed.

Lemma measurable_funU T1 T2 (D E : set T1) (f : T1 -> T2) :
  measurable D -> measurable E ->
  measurable_fun (D `|` E) f <-> measurable_fun D f /\ measurable_fun E f.
Proof.
move=> mD mE; split=> [mDEf|[mDf mEf] mDE A mA]; last first.
  by rewrite setIUl; apply: measurableU; [exact: mDf|exact: mEf].
split.
- move=> {}mD A /mDEf => /(_ (measurableU _ _ mD mE))/(measurableI D)-/(_ mD).
  by rewrite setICA setIA setUK.
- move=> {}mE A /mDEf => /(_ (measurableU _ _ mD mE))/(measurableI E)-/(_ mE).
 by rewrite setICA setIA setUC setUK.
Qed.

Lemma measurable_funS T1 T2 (E D : set T1) (f : T1 -> T2) :
     measurable E -> D `<=` E -> measurable_fun E f ->
  measurable_fun D f.
Proof.
move=> mE DE mf mD; have mC : measurable (E `\` D) by exact: measurableD.
move: (mD).
have := measurable_funU f mD mC.
suff -> : (D `|` (E `\` D)) = E by move=> [[]] //.
by rewrite setDUK.
Qed.

Lemma measurable_fun_ext T1 T2 D (f g : T1 -> T2) :
  {in D, f =1 g} -> measurable_fun D f -> measurable_fun D g.
Proof.
by move=> fg mf mD A mA; rewrite [X in measurable X](_ : _ = D `&` f @^-1` A);
  [exact: mf|exact/esym/eq_preimage].
Qed.

Lemma measurable_restrict {aT rT : measurableType} D E (f : aT -> rT) :
  measurable D -> measurable E -> D `<=` E ->
  measurable_fun D f <-> measurable_fun E (f \_ D).
Proof.
move=> mD mE DE; split => mf _ /= X mX.
- rewrite preimage_restrict; apply/measurableI => //.
  by apply/measurableU/mf => //; case: ifP => // _; apply: measurableC.
- have := mf mE _ mX; rewrite preimage_restrict.
  case: ifP => ptX; last first.
    rewrite set0U => /(measurableI _ _ mD).
    by rewrite (setIA D) (setIidl DE) setIA setIid.
  rewrite setUIr setvU setTI => /(measurableI _ _ mD).
  by rewrite setIA (setIidl DE) setIUr setIv set0U.
Qed.

End measurable_fun.
Arguments measurable_fun_ext {T1 T2 D} f {g}.

Section measurability.

Definition preimage_class (aT rT : Type) (D : set aT) (f : aT -> rT)
    (G : set (set rT)) : set (set aT) :=
  [set D `&` f @^-1` B | B in G].

(* f1 is measurable on the sigma-algebra generated by itself *)
Lemma preimage_class_measurable_fun (aT : pointedType) (rT : measurableType)
  (D : set aT) (f : aT -> rT) :
  @measurable_fun (g_measurableType (preimage_class D f measurable))
    rT D f.
Proof. by move=> mD A mA; apply: sub_sigma_algebra; exists A. Qed.

Lemma sigma_algebra_preimage_class (aT rT : Type) (G : set (set rT))
    (D : set aT) (f : aT -> rT) :
  sigma_algebra setT G -> sigma_algebra D (preimage_class D f G).
Proof.
case=> h0 hC hU; split; first by exists set0 => //; rewrite preimage_set0 setI0.
- move=> A; rewrite /preimage_class /= => -[B mB <-{A}].
  exists (~` B); first by rewrite -setTD; exact: hC.
  rewrite predeqE => x; split=> [[Dx Bfx]|[Dx]]; first by split => // -[] _ /Bfx.
  by move=> /not_andP[].
- move=> F; rewrite /preimage_class /= => mF.
  have {}mF n : exists x, G x /\ D `&` f @^-1` x = F n.
    by have := mF n => -[B mB <-]; exists B.
  have [F' mF'] := @choice _ _ (fun x y => G y /\ D `&` f @^-1` y = F x) mF.
  exists (\bigcup_k (F' k)); first by apply: hU => n; exact: (mF' n).1.
  rewrite preimage_bigcup setI_bigcupr; apply eq_bigcupr => i _.
  exact: (mF' i).2.
Qed.

Definition image_class (aT rT : Type) (D : set aT) (f : aT -> rT)
    (G : set (set aT)) : set (set rT) :=
  [set B : set rT | G (D `&` f @^-1` B)].

Lemma sigma_algebra_image_class (aT rT : Type) (D : set aT) (f : aT -> rT)
    (G : set (set aT)) :
  sigma_algebra D G -> sigma_algebra setT (image_class D f G).
Proof.
move=> [G0 GC GU]; split; rewrite /image_class.
- by rewrite /= preimage_set0 setI0.
- move=> A /= GfAD; rewrite setTD -preimage_setC -setDE.
  rewrite (_ : _ `\` _ = D `\` (D `&` f @^-1` A)); first exact: GC.
  rewrite predeqE => x; split=> [[Dx fAx]|[Dx fADx]].
    by split => // -[] _ /fAx.
  by split => //; exact: contra_not fADx.
- by move=> F /= mF; rewrite preimage_bigcup setI_bigcupr; exact: GU.
Qed.

Lemma sigma_algebra_preimage_classE aT (rT : pointedType) (D : set aT)
    (f : aT -> rT) (G' : set (set rT)) :
  <<s D, preimage_class D f G' >> =
    preimage_class D f (@measurable (g_measurableType G')).
Proof.
rewrite eqEsubset; split.
  have mG : sigma_algebra D
      (preimage_class D f (@measurable (g_measurableType G'))).
    exact/sigma_algebra_preimage_class/sigma_algebra_measurable.
  have subset_preimage : preimage_class D f G' `<=`
                         preimage_class D f (@measurable (g_measurableType G')).
    by move=> A [B CCB <-{A}]; exists B => //; apply sub_sigma_algebra.
  exact: smallest_sub.
have G'pre A' : G' A' -> (preimage_class D f G') (D `&` f @^-1` A').
  by move=> ?; exists A'.
pose I : set (set aT) := <<s D, preimage_class D f G' >>.
have G'sfun : G' `<=` image_class D f I.
  by move=> A' /G'pre[B G'B h]; apply: sub_sigma_algebra; exists B.
have sG'sfun : <<s G' >> `<=` image_class D f I.
  apply: smallest_sub => //; apply: sigma_algebra_image_class.
  exact: smallest_sigma_algebra.
by move=> _ [B mB <-]; exact: sG'sfun.
Qed.

Lemma measurability (aT rT : measurableType) (D : set aT) (f : aT -> rT)
    (G' : set (set rT)) :
  @measurable rT = <<s G' >> -> preimage_class D f G' `<=` @measurable aT ->
  measurable_fun D f.
Proof.
move=> sG_rT fG_aT mD.
suff h : preimage_class D f (@measurable rT) `<=` @measurable aT.
  by move=> A mA; apply: h; exists A.
have -> : preimage_class D f (@measurable rT) =
         <<s D , (preimage_class D f G')>>.
  by rewrite [in LHS]sG_rT [in RHS]sigma_algebra_preimage_classE.
apply: smallest_sub => //; split => //.
- by move=> A mA; exact: measurableD.
- by move=> F h; exact: bigcupT_measurable.
Qed.

End measurability.

Local Open Scope ereal_scope.

Section additivity.
Variables (R : numFieldType) (T : semiRingOfSetsType) (mu : set T -> \bar R).

Definition semi_additive2 := forall A B, measurable A -> measurable B ->
  measurable (A `|` B) ->
  A `&` B = set0 -> mu (A `|` B) = mu A + mu B.

Definition semi_additive := forall F n,
 (forall k : nat, measurable (F k)) -> trivIset setT F ->
  measurable (\big[setU/set0]_(k < n) F k) ->
  mu (\big[setU/set0]_(i < n) F i) = \sum_(i < n) mu (F i).

Definition semi_sigma_additive :=
  forall F, (forall i : nat, measurable (F i)) -> trivIset setT F ->
  measurable (\bigcup_n F n) ->
  (fun n => \sum_(0 <= i < n) mu (F i)) --> mu (\bigcup_n F n).

Definition additive2 := forall A B, measurable A -> measurable B ->
  A `&` B = set0 -> mu (A `|` B) = mu A + mu B.

Definition additive :=
  forall F, (forall i : nat, measurable (F i)) -> trivIset setT F ->
  forall n, mu (\big[setU/set0]_(i < n) F i) = \sum_(i < n) mu (F i).

Definition sigma_additive :=
  forall F, (forall i : nat, measurable (F i)) -> trivIset setT F ->
  (fun n => \sum_(0 <= i < n) mu (F i)) --> mu (\bigcup_n F n).

Definition sub_additive := forall (A : set T) (F : nat -> set T) n,
 (forall k, `I_n k -> measurable (F k)) -> measurable A ->
  A `<=` \big[setU/set0]_(k < n) F k ->
  mu A <= \sum_(k < n) mu (F k).

Definition sigma_sub_additive := forall (A : set T) (F : nat -> set T),
 (forall n, measurable (F n)) -> measurable A ->
  A `<=` \bigcup_n F n ->
  mu A <= \sum_(n <oo) mu (F n).

Definition sigma_finite (A : set T) (mu : set T -> \bar R) :=
  exists2 F : (set T)^nat,  A = \bigcup_(i : nat) F i &
    forall i, measurable (F i)/\ mu (F i) < +oo.

Lemma semi_additiveW : mu set0 = 0 -> semi_additive -> semi_additive2.
Proof.
move=> mu0 amx A B mA mB + AB; rewrite -bigcup2inE bigcup_mkord.
move=> /(amx (bigcup2 A B))->.
- by rewrite !(big_ord_recl, big_ord0)/= adde0.
- by move=> [|[|[]]]//=.
- by move=> [|[|i]] [|[|j]]/= _ _; rewrite ?(AB, setI0, set0I, setIC) => -[].
Qed.

End additivity.

Section ring_additivity.
Variables (R : numFieldType) (T : ringOfSetsType) (mu : set T -> \bar R).

Lemma semi_additiveE : semi_additive mu = additive mu.
Proof.
rewrite propeqE; split=> [sa A mA tA n|+ A m mA tA UAm]; last by move->.
by rewrite sa //; exact: bigsetU_measurable.
Qed.

Lemma semi_additive2E : semi_additive2 mu = additive2 mu.
Proof.
rewrite propeqE; split=> [amu A B ? ? ?|amu A B ? ? _ ?]; last by rewrite amu.
by rewrite amu //; exact: measurableU.
Qed.

Lemma additive2P : mu set0 = 0 -> semi_additive mu <-> additive2 mu.
Proof.
move=> mu0; rewrite -semi_additive2E; split; first exact: semi_additiveW.
rewrite semi_additiveE semi_additive2E => muU A Am Atriv n.
elim: n => [|n IHn]; rewrite ?(big_ord_recr, big_ord0) ?mu0//=.
rewrite muU ?IHn//=; first by apply: bigsetU_measurable.
rewrite -bigcup_mkord -subset0 => x [[/= m + Amx] Anx].
by rewrite (Atriv m n) ?ltnn//=; exists x.
Qed.

End ring_additivity.

(*NB: generalize to numFieldType when Rhausdorff is generalized *)
Lemma semi_sigma_additive_is_additive (R : realFieldType)
    (X : semiRingOfSetsType) (mu : set X -> \bar R) :
  mu set0 = 0 -> semi_sigma_additive mu -> semi_additive mu.
Proof.
move=> mu0 samu A n Am Atriv UAm.
have := samu (fun i => if (i < n)%N then A i else set0).
rewrite (bigcup_splitn n) bigcup0 ?setU0; last first.
  by move=> i _; rewrite -ltn_subRL subnn.
under eq_bigr do rewrite ltn_ord.
move=> /(_ _ _ UAm)/(@cvg_lim _) <-//; last 2 first.
- by move=> i; case: ifP.
- move=> i j _ _; do 2![case: ifP] => ? ?; do ?by rewrite (setI0, set0I) => -[].
  by move=> /Atriv; apply.
apply: lim_near_cst => //=; near=> i.
have /subnKC<- : (n <= i)%N by near: i; exists n.
transitivity (\sum_(j < n + (i - n)) mu (if (j < n)%N then A j else set0)).
  by rewrite big_mkord.
rewrite big_split_ord/=; under eq_bigr do rewrite ltn_ord.
by rewrite [X in _ + X]big1 ?adde0// => ?; rewrite -ltn_subRL subnn.
Unshelve. all: by end_near. Qed.

Lemma semi_sigma_additiveE
  (R : numFieldType) (X : measurableType) (mu : set X -> \bar R) :
  semi_sigma_additive mu = sigma_additive mu.
Proof.
rewrite propeqE; split=> [amu A mA tA|amu A mA tA mbigcupA]; last exact: amu.
by apply: amu => //; exact: bigcupT_measurable.
Qed.

Lemma sigma_additive_is_additive
  (R : realFieldType) (X : measurableType) (mu : set X -> \bar R) :
  mu set0 = 0 -> sigma_additive mu -> additive mu.
Proof.
move=> mu0; rewrite -semi_sigma_additiveE -semi_additiveE.
exact: semi_sigma_additive_is_additive.
Qed.

Lemma fset_set_image (T1 T2 : choiceType) (D : set T1) (f : T1 -> T2) :
  finite_set D -> fset_set (f @` D) = (f @` (fset_set D))%fset.
Proof.
move=> Dfin; apply/fsetP => z; rewrite in_fset_set; last exact: finite_image.
apply/idP/idP => [|/imfsetP[r /=]].
  rewrite inE /= => -[x Dx <-{z}]; apply/imfsetP => /=; exists x => //.
  by rewrite in_fset_set ?inE/=.
by rewrite in_fset_set// inE => Dr ->; rewrite inE; exists r.
Qed.

Module AdditiveMeasure.

Section ClassDef.

Variables (R : numFieldType) (T : semiRingOfSetsType).
Record axioms (mu : set T -> \bar R) := Axioms {
  _ : mu set0 = 0 ;
  _ : forall x, 0 <= mu x ;
  _ : semi_additive mu }.

Structure map (phUV : phant (set T -> \bar R)) :=
  Pack {apply : set T -> \bar R ; _ : axioms apply}.
Local Coercion apply : map >-> Funclass.

Variables (phUV : phant (set T -> \bar R)) (f g : set T -> \bar R).
Variable (cF : map phUV).
Definition class := let: Pack _ c as cF' := cF return axioms cF' in c.
Definition clone fA of phant_id g (apply cF) & phant_id fA class :=
  @Pack phUV f fA.

End ClassDef.

Module Exports.
Notation additive_measure f := (axioms f).
Coercion apply : map >-> Funclass.
Notation AdditiveMeasure fA := (Pack (Phant _) fA).
Notation "{ 'additive_measure' fUV }" := (map (Phant fUV))
  (at level 0, format "{ 'additive_measure'  fUV }") : ring_scope.
Notation "[ 'additive_measure' 'of' f 'as' g ]" := (@clone _ _ _ f g _ _ idfun id)
  (at level 0, format "[ 'additive_measure'  'of'  f  'as'  g ]") : form_scope.
Notation "[ 'additive_measure' 'of' f ]" := (@clone _ _ _ f f _ _ id id)
  (at level 0, format "[ 'additive_measure'  'of'  f ]") : form_scope.
End Exports.

End AdditiveMeasure.
Include AdditiveMeasure.Exports.

Section additive_measure_on_semiring_of_sets.
Variables (R : realFieldType) (T : semiRingOfSetsType)
  (mu : {additive_measure set T -> \bar R}).

Lemma measure0 : mu set0 = 0.
Proof. by case: mu => ? []. Qed.
Hint Resolve measure0 : core.

Lemma measure_ge0 : forall x, 0 <= mu x.
Proof. by case: mu => ? []. Qed.
Hint Resolve measure_ge0 : core.

Lemma measure_semi_additive : semi_additive mu.
Proof. by case: mu => ? []. Qed.
Hint Resolve measure_semi_additive : core.

Lemma content_fin_bigcup (I : choiceType) (D : set I) (F : I -> set T) :
    finite_set D ->
    trivIset D F ->
    (forall i, D i -> measurable (F i)) ->
    measurable (\bigcup_(i in D) F i) ->
  mu (\bigcup_(i in D) F i) = \sum_(i <- fset_set D) mu (F i).
Proof.
move=> Dfin Ftriv Fm UFm; pose G (i : option I) : set T := oapp (F \_ D) set0 i.
have UFG : \bigcup_(i in D) F i = \bigcup_(i in [set Some x | x in D]) G i.
  apply/predeqP => x; split=> [[i Di Fix]|[[]// _ [i Di [<- /=]]]].
    by exists (Some i) => //=; rewrite ?patchT ?inE//; exists i.
  by rewrite ?patchT ?inE// => Fix; exists i.
suff muG : mu (\bigcup_(i in some @` D) G i) = \sum_(i <- fset_set (some @` D)) mu (G i).
  transitivity (mu (\bigcup_(i in some @` D) G i)); first by rewrite UFG.
    rewrite muG fset_set_image// (big_imfset _ _ _ (in2W Some_inj))/=.
    by apply: eq_big_seq => X; rewrite in_fset_set// => XD; rewrite patchT.
have oDfin: finite_set (Some @` D) by exact: finite_image.
rewrite bigcup_fset_set//= !(big_nth None) !big_mkord.
set s : seq (option I) := fset_set (some @` D).
rewrite (@measure_semi_additive (fun i => G (nth None s i)))//.
- move=> i; case: (ltnP i (size s)) => [/(mem_nth None)|/(nth_default _)->//=].
  rewrite in_fset_set// inE/= => -[j Dj <-]//=.
  by rewrite patchT ?inE//; apply: Fm.
- move=> m n _ _.
  case: (ltnP m (size s)) => [lt_ms|/(nth_default _)->]; last first.
    by rewrite set0I => -[].
  case: (ltnP n (size s)) => [lt_ns|/(nth_default _)->]; last first.
    by rewrite setI0 => -[].
  have := mem_nth None lt_ns; have := mem_nth None lt_ms.
  rewrite !in_fset_set// !inE/= => -[i Di ieq]/= -[j Dj jeq].
  rewrite -ieq -jeq/= !patchT ?inE// => /(Ftriv _ _ Di Dj) eqij.
  by have := jeq; rewrite -eqij ieq => /uniqP; apply => //; rewrite fset_uniq.
- rewrite -(big_mkord predT (fun i => G (nth None s i))) -(big_nth None xpredT).
  by rewrite -bigcup_fset_set// -UFG.
Qed.

Lemma measure_semi_additive2 : semi_additive2 mu.
Proof. exact/semi_additiveW. Qed.
Hint Resolve measure_semi_additive2 : core.

End additive_measure_on_semiring_of_sets.

#[global] Hint Extern 0
  (is_true (0 <= (_ : {additive_measure set _ -> \bar _}) _)%E) =>
  solve [apply: measure_ge0] : core.

#[global]
Hint Resolve measure0 measure_semi_additive2 measure_semi_additive : core.

Section additive_measure_on_ring_of_sets.
Variables (R : realFieldType) (T : ringOfSetsType)
  (mu : {additive_measure set T -> \bar R}).

Lemma measureU : additive2 mu.
Proof. by rewrite -semi_additive2E. Qed.

Lemma measure_bigsetU : additive mu.
Proof. by rewrite -semi_additiveE. Qed.

Lemma measure_fin_bigcup (I : choiceType) (D : set I) (F : I -> set T) :
    finite_set D ->
    trivIset D F ->
    (forall i, D i -> measurable (F i)) ->
  mu (\bigcup_(i in D) F i) = \sum_(i <- fset_set D) mu (F i).
Proof.
move=> Dfin Ftriv Fm; rewrite content_fin_bigcup//.
exact: fin_bigcup_measurable.
Qed.

Lemma measure_bigsetU_ord_cond n (P : {pred 'I_n}) (F : 'I_n -> set T) :
  (forall i : 'I_n, P i -> measurable (F i)) -> trivIset P F ->
  mu (\big[setU/set0]_(i < n | P i) F i) = (\sum_(i < n | P i) mu (F i))%E.
Proof.
move=> mF tF; rewrite -bigcup_set_cond measure_fin_bigcup//=; last first.
- by move=> i /andP[_ /mF].
- by apply: sub_trivIset tF => i /andP[].
rewrite -[RHS]big_filter; apply: perm_big.
rewrite uniq_perm ?fset_uniq ?filter_uniq ?index_enum_uniq//=.
move=> i; rewrite in_fset_set// ?mem_filter.
  by apply/idP/idP; rewrite ?inE/= andbC.
by rewrite set_andb/=; apply: finite_setIr.
Qed.

Lemma measure_bigsetU_ord n (P : {pred 'I_n}) (F : 'I_n -> set T) :
  (forall i : 'I_n, measurable (F i)) -> trivIset setT F ->
  mu (\big[setU/set0]_(i < n | P i) F i) = (\sum_(i < n | P i) mu (F i))%E.
Proof.
by move=> mF tF; rewrite measure_bigsetU_ord_cond//; apply: sub_trivIset tF.
Qed.

Lemma measure_fbigsetU (I : choiceType) (A : {fset I}) (F : I -> set T) :
  (forall i, i \in A -> measurable (F i)) -> trivIset [set` A] F ->
  mu (\big[setU/set0]_(i <- A) F i) = (\sum_(i <- A) mu (F i))%E.
Proof. by move=> mF tF; rewrite -bigcup_fset measure_fin_bigcup// set_fsetK. Qed.

End additive_measure_on_ring_of_sets.

#[global]
Hint Resolve measureU measure_bigsetU : core.

Module Measure.

Section ClassDef.

Variables (R : numFieldType) (T : semiRingOfSetsType).
Record axioms (mu : set T -> \bar R) := Axioms {
  _ : mu set0 = 0 ;
  _ : forall x, 0 <= mu x ;
  _ : semi_sigma_additive mu }.

Structure map (phUV : phant (set T -> \bar R)) :=
  Pack {apply : set T -> \bar R ; _ : axioms apply}.
Local Coercion apply : map >-> Funclass.

Variables (phUV : phant (set T -> \bar R)) (f g : set T -> \bar R).
Variable (cF : map phUV).
Definition class := let: Pack _ c as cF' := cF return axioms cF' in c.
Definition clone fA of phant_id g (apply cF) & phant_id fA class :=
  @Pack phUV f fA.

End ClassDef.

Module Exports.
Notation is_measure f := (axioms f).
Coercion apply : map >-> Funclass.
Notation Measure fA := (Pack (Phant _) fA).
Notation "{ 'measure' fUV }" := (map (Phant fUV))
  (at level 0, format "{ 'measure'  fUV }") : ring_scope.
Notation "[ 'measure' 'of' f 'as' g ]" := (@clone _ _ _ f g _ _ idfun id)
  (at level 0, format "[ 'measure'  'of'  f  'as'  g ]") : form_scope.
Notation "[ 'measure' 'of' f ]" := (@clone _ _ _ f f _ _ id id)
  (at level 0, format "[ 'measure'  'of'  f ]") : form_scope.
End Exports.

End Measure.
Include Measure.Exports.

Section measure_lemmas.
Variables (R : realFieldType) (T : semiRingOfSetsType).

Coercion measure_to_nadditive_measure (mu : set T -> \bar R) :
  is_measure mu -> additive_measure mu.
Proof.
by move=> [mu0 muge0 /(semi_sigma_additive_is_additive mu0)]; split.
Qed.

Variable mu : {measure set T -> \bar R}.

Lemma measure_semi_sigma_additive : semi_sigma_additive mu.
Proof. by case: mu => ? []. Qed.

Lemma measure_semi_bigcup A :
    (forall i : nat, measurable (A i)) ->
    trivIset setT A ->
    measurable (\bigcup_n A n) ->
  mu (\bigcup_n A n) = \sum_(i <oo) mu (A i).
Proof. by move=> Am Atriv /measure_semi_sigma_additive/cvg_lim<-//. Qed.

End measure_lemmas.

Section measure_lemmas.
Variables (R : realFieldType) (T : measurableType).
Variable mu : {measure set T -> \bar R}.

Lemma measure_sigma_additive : sigma_additive mu.
Proof.
by rewrite -semi_sigma_additiveE //; apply: measure_semi_sigma_additive.
Qed.

Lemma measure_bigcup A : (forall i : nat, measurable (A i)) ->
  trivIset setT A ->
  mu (\bigcup_n A n) = \sum_(i <oo) mu (A i).
Proof.
by move=> Am Atriv; rewrite measure_semi_bigcup//; apply: bigcupT_measurable.
Qed.

End measure_lemmas.
Arguments measure_bigcup {R T} mu A.

#[global] Hint Extern 0 (_ set0 = 0) => solve [apply: measure0] : core.
#[global] Hint Extern 0 (sigma_additive _) =>
  solve [apply: measure_sigma_additive] : core.
#[global] Hint Extern 0 (is_true (0 <= _)) => solve [apply: measure_ge0] : core.

Section measure_is_additive_measure.
Variables (R : realFieldType) (T : semiRingOfSetsType)
  (mu : {measure set T -> \bar R}).

Lemma measure_is_additive_measure : additive_measure mu.
Proof.
case: mu => f [f0 fg0 fsa]; split => //=.
exact/semi_sigma_additive_is_additive.
Qed.

Canonical measure_additive_measure :=
  AdditiveMeasure measure_is_additive_measure.

End measure_is_additive_measure.

Coercion measure_additive_measure : Measure.map >-> AdditiveMeasure.map.


Section measure_restr.
Variables (T : measurableType) (R : realType) (mu : {measure set T -> \bar R}).
Variables (D : set T) (mD : measurable D).

Let measure_restr' (X : set _) : \bar R := mu (X `&` D).

Let measure_restr'0 : measure_restr' set0 = 0%E.
Proof. by rewrite /measure_restr' set0I measure0. Qed.

Let measure_restr'_ge0 (A : set _) : (0 <= measure_restr' A)%E.
Proof.
by rewrite /measure_restr'; apply: measure_ge0; exact: measurableI.
Qed.

Let measure_restr'_sigma_additive : semi_sigma_additive measure_restr'.
Proof.
move=> F mF tF mU; pose F' i := F i `&` D.
have mF' i : measurable (F' i) by apply: measurableI.
have tF' : trivIset setT F'.
  apply/trivIsetP => i j _ _ ij; move/trivIsetP : tF => /(_ i j Logic.I Logic.I ij).
  by rewrite /F' setIACA => ->; rewrite set0I.
have h := @measure_sigma_additive _ _ mu _ mF' tF'.
by rewrite /measure_restr' setI_bigcupl.
Qed.

Definition measure_restr : {measure set _ -> \bar R} :=
  Measure.Pack _ (Measure.Axioms
    measure_restr'0 measure_restr'_ge0 measure_restr'_sigma_additive).

Lemma measure_restrE A : measure_restr A = mu (A `&` D).
Proof. by []. Qed.

End measure_restr.

Lemma big_trivIset (I : choiceType) D T (R : Type) (idx : R)
   (op : Monoid.com_law idx) (A : I -> set T) (F : set T -> R) :
    finite_set D -> trivIset D A -> F set0 = idx ->
  \big[op/idx]_(i <- fset_set D) F (A i) =
  \big[op/idx]_(X <- (A @` fset_set D)%fset) F X.
Proof.
elim/Pchoice: R => R in idx op F *.
move=> Dfin Atriv F0; symmetry.
pose D' := [fset i in fset_set D | A i != set0]%fset.
transitivity (\big[op/idx]_(X <- (A @` D')%fset) F X).
  apply: perm_big_supp; rewrite uniq_perm ?filter_uniq//=.
  move=> X; rewrite !mem_filter; case: (eqVneq (F X) idx) => //= FXNidx.
  apply/imfsetP/imfsetP=> -[i/=]; rewrite ?(inE, in_fset_set)//=.
    move=> Di XAi; exists i; rewrite // !(inE, in_fset_set)//=.
    by rewrite (mem_set Di)/= -XAi; apply: contra_neq FXNidx => ->.
  by move=> /andP[Di AiN0] XAi; exists i; rewrite ?in_fset_set.
rewrite big_imfset//=; last first.
  move=> i j; rewrite !(inE, in_fset_set)//= => /andP[+ +] /andP[+ +].
  rewrite !inE => Di /set0P[x Aix] Dj _ Aij.
  by apply: (Atriv _ _ Di Dj); exists x; split=> //; rewrite -Aij.
apply: perm_big_supp; rewrite uniq_perm ?filter_uniq//= => i.
rewrite !mem_filter; case: (eqVneq (F (A i)) idx) => //= FAiidx.
rewrite !(in_fset_set, inE)//=; case: (boolP (i \in D)) => //= Di.
by apply: contra_neq FAiidx => ->.
Qed.

Section covering.
Context {T : Type}.
Implicit Type (C : forall I, set (set I)).
Implicit Type (P : forall I, set I -> set (I -> set T)).

Definition covered_by C P :=
  [set X : set T | exists I D A, [/\ C I D, P I D A & X = \bigcup_(i in D) A i]].

Lemma covered_bySr C P P' : (forall I D A, P I D A -> P' I D A) ->
  covered_by C P `<=` covered_by C P'.
Proof.
by move=> PP' X [I [D [A [CX PX ->]]]]; exists I, D, A; split=> //; apply: PP'.
Qed.

Lemma covered_byP C P I D A : C I D -> P I D A ->
  covered_by C P (\bigcup_(i  in D) A i).
Proof. by move=> CID PIDA; exists I, D, A. Qed.

Lemma covered_by_finite P :
    (forall I (D : set I) A, (forall i, D i -> A i = set0) -> P I D A) ->
    (forall (I : pointedType) D A, finite_set D -> P I D A ->
       P nat `I_#|` fset_set D| (A \o nth point (fset_set D))) ->
  covered_by (@finite_set) P =
    [set X : set T | exists n A, [/\ P nat `I_n A & X = \bigcup_(i in `I_n) A i]].
Proof.
move=> P0 Pc; apply/predeqP=> X; rewrite /covered_by /cover/=; split; last first.
  by move=> [n [A [Am ->]]]; exists nat, `I_n, A; split.
case; elim/Ppointed=> I [D [A [Dfin Am ->]]].
  exists 0%N, (fun=> set0); split; first by rewrite II0; apply: P0.
  by rewrite //= emptyE II0 !bigcup0.
exists #|`fset_set D|, (A \o nth point (fset_set D)).
split; first exact: Pc.
by rewrite bigcup_fset_set// (big_nth point) big_mkord bigcup_mkord.
Qed.

Lemma covered_by_countable P :
    (forall I (D : set I) A, (forall i, D i -> A i = set0) -> P I D A) ->
    (forall (I : pointedType) (D : set I) (A : I -> set T) (f : nat -> I),
       set_surj [set: nat] D f ->
       P I D A -> P nat [set: nat] (A \o f)) ->
  covered_by (@countable) P =
    [set X : set T | exists A, [/\ P nat [set: nat] A & X = \bigcup_i A i]].
Proof.
move=> P0 Pc; apply/predeqP=> X; rewrite /covered_by /cover/=; split; last first.
  by move=> [A [Am ->]]; exists nat, [set: nat], A; split.
case; elim/Ppointed=> I [D [A [Dcnt Am ->]]].
  exists (fun=> set0); split; first exact: P0.
  by rewrite emptyE bigcup_set0 bigcup0.
have /pfcard_geP[->|[f]] := Dcnt.
  exists (fun=> set0); split; first exact: P0.
  by rewrite !bigcup_set0 bigcup0.
pose g := [splitsurjfun of split f].
exists (A \o g); split=> /=; first exact: Pc Am.
apply/predeqP=> x; split=> [[i Di Aix]|[n _ Afnx]].
  by exists (g^-1%FUN i) => //=; rewrite invK// inE.
by exists (g n) => //; apply: funS.
Qed.

End covering.

Module SetRing.
Definition type (T : Type) := T.

Section SetRing.
Context {T : semiRingOfSetsType}.

Notation rT := (type T).
Canonical ring_eqType := EqType rT ptclass.
Canonical ring_choiceType := ChoiceType rT ptclass.
Canonical ring_ptType := PointedType rT ptclass.
#[export]
HB.instance Definition _ := isRingOfSets.Build rT ptclass
  (@setring0 T measurable) (@setringU T measurable) (@setringDI T measurable).
Definition ring := [the ringOfSetsType of rT].

Local Definition measurable_fin_trivIset : set (set T) :=
  [set A | exists B : set (set T),
      [/\  A = \bigcup_(X in B) X, forall X : set T, B X -> measurable X,
           finite_set B & trivIset B id]].

Lemma ring_measurableE : @measurable ring = measurable_fin_trivIset.
Proof.
apply/seteqP; split; last first.
  move=> _ [B [-> Bm Bfin Btriv]]; apply: fin_bigcup_measurable => //.
  by move=> i Di; apply: sub_gen_smallest; apply: Bm.
have mdW A : measurable A -> measurable_fin_trivIset A.
  move=> Am; exists [set A]; split; do ?by [rewrite bigcup_set1|move=> ? ->|].
  by move=> ? ? -> ->.
have mdI : setI_closed measurable_fin_trivIset.
  move=> _ _ [A [-> Am Afin Atriv]] [B [-> Bm Bfin Btriv]].
  rewrite setI_bigcupl; under eq_bigcupr do rewrite setI_bigcupr.
  rewrite bigcup_bigcup -(bigcup_image _ _ id).
  eexists; split; [reflexivity | | exact/finite_image/finite_setM |].
    by move=> _ [X [? ?] <-]; apply: measurableI; [apply: Am|apply: Bm].
  apply: trivIset_sets => -[a b] [a' b']/= [Xa Xb] [Xa' Xb']; rewrite setIACA.
  by move=> [x [Ax Bx]]; rewrite (Atriv a a') 1?(Btriv b b')//; exists x.
have mdisj_bigcap : finN0_bigcap_closed measurable_fin_trivIset.
   exact/finN0_bigcap_closedP/mdI.
have mDbigcup I (D : set I) (A : set T) (B : I -> set T) : finite_set D ->
    measurable A -> (forall i, D i -> measurable (B i)) ->
    measurable_fin_trivIset (A `\` \bigcup_(i in D) B i).
  have [->|/set0P D0] := eqVneq D set0.
    by rewrite bigcup0// setD0 => *; apply: mdW.
  move=> Dfin Am Bm; rewrite -bigcupDr//; apply: mdisj_bigcap=> // i Di.
  by have [F [Ffin Fm -> ?]] := semi_measurableD A (B i) Am (Bm _ Di); exists F.
have mdU : fin_trivIset_closed measurable_fin_trivIset.
  elim/Pchoice=> I D F Dfin Ftriv Fm.
  have /(_ _ (set_mem _))/cid-/(all_sig_cond_dep (fun=> set0))
       [G /(_ _ (mem_set _))GP] := Fm _ _.
  under eq_bigcupr => i Di do case: (GP i Di) => ->.
  rewrite bigcup_bigcup_dep -(bigcup_image _ _ id); eexists; split=> //.
  - by move=> _ [i [Di Gi] <-]; have [_ + _ _] := GP i.1 Di; apply.
  - by apply: finite_image; apply: finite_setMR=> // i Di; have [] := GP i Di.
  apply: trivIset_sets => -[i X] [j Y] /= [Di Gi] [Dj Gj] XYN0.
  suff eqij : i = j.
    by rewrite {i}eqij in Di Gi *; have [_ _ _ /(_ _ _ _ _ XYN0)->] := GP j Dj.
  apply: Ftriv => //; have [-> _ _ _] := GP j Dj; have [-> _ _ _] := GP i Di.
  by case: XYN0 => [x [Xx Yx]]; exists x; split; [exists X|exists Y].
have mdDI : setDI_closed measurable_fin_trivIset.
  move=> A B mA mB; have [F [-> Fm Ffin Ftriv]] := mA.
  have [F' [-> F'm F'fin F'triv]] := mB.
  have [->|/set0P F'N0] := eqVneq F' set0.
    by rewrite bigcup_set0 setD0; exists F.
  rewrite setD_bigcupl; apply: mdU => //; first by apply: trivIset_setIr.
  move=> X DX; rewrite -bigcupDr//; apply: mdisj_bigcap => //.
  move=> Y DY; case: (semi_measurableD X Y); [exact: Fm|exact: F'm|].
  by move=> G [Gfin Gm -> Gtriv]; exists G.
apply: smallest_sub => //; split=> //; first by apply: mdW.
move=> A B mA mB; rewrite -(setUIDK B A) setUA [X in X `|` _]setUidl//.
rewrite -bigcup2inE; apply: mdU => //; last by move=> [|[]]// _; apply: mdDI.
by move=> [|[]]// [|[]]//= _ _ []; rewrite setDE ?setIA => X [] []//.
Qed.

Lemma ring_fsets (A : set rT) : measurable A -> exists B : {fset set T},
  [/\ all [pred X | X != set0] B,
      trivIset [set` B] id,
      (forall X : set T, X \in B -> measurable X) &
      A = \bigcup_(X in [set` B]) X].
Proof.
rewrite ring_measurableE => -[B [-> Bm Bfin Btriv]].
exists [fset X in fset_set B | X != set0]%fset; split => //.
- by apply/allP => _ /imfsetP[/= X /andP[_ +] ->].
- move=> _ _ /imfsetP[X /andP[+ _] ->] /imfsetP[Y/= /andP[+ _] ->].
  by rewrite !in_fset_set ?inE//; apply: Btriv.
- by move=> _ /imfsetP[X /andP[+ _] ->]; rewrite in_fset_set// inE; apply: Bm.
rewrite bigcup_fset big_imfset//= big_filter big_rmcond; last first.
  by move=> X; rewrite negbK => /eqP.
by rewrite -bigcup_fset_set.
Qed.

Definition decomp (A : set rT) : {fset set T} :=
  if A == set0 then [fset set0]%fset else
  if pselect (measurable A) is left mA then projT1 (cid (ring_fsets mA)) else [fset A]%fset.

Lemma decomp_triv (A : set rT) : trivIset [set` decomp A] id.
Proof.
rewrite /decomp; case: ifP => _.
  by move=> i j; rewrite /= !inE => -> ->.
case: pselect => // Am; first by case: cid => //= ? [].
by move=> i j /=; rewrite !inE => -> ->.
Qed.
Hint Resolve decomp_triv : core.

Lemma all_decomp_neq0 (A : set rT) : A !=set0 -> all [pred X | X != set0] (decomp A).
Proof.
move=> /set0P AN0; rewrite /decomp/= (negPf AN0).
case: pselect => //= Am; first by case: cid => //= ? [].
by apply/allP => X; rewrite !inE => ->.
Qed.

Lemma decomp_neq0 (A : set rT) X : A !=set0 -> X \in decomp A -> X !=set0.
Proof. by move=> /all_decomp_neq0/allP/[apply]/set0P. Qed.

Lemma decomp_measurable (A : set rT) (X : set T) :
  measurable A -> X \in decomp A -> measurable X.
Proof.
rewrite /decomp; case: ifP => _; first by rewrite inE => _ /eqP->.
by case: pselect => // Am _; case: cid => //= ? [_ _ + _]; apply.
Qed.

Lemma cover_decomp (A : set rT) : \bigcup_(X in [set` decomp A]) X = A.
Proof.
rewrite /decomp; case: ifP => [/eqP->|_].
  by rewrite set_fset1 bigcup_set1.
case: pselect => // Am; first by case: cid => //= ? [].
by rewrite set_fset1 bigcup_set1.
Qed.

Lemma decomp_sub (A : set rT) (X : set T) : X \in decomp A -> X `<=` A.
Proof.
rewrite /decomp; case: ifP => _; first by rewrite !inE => ->.
case: pselect => //= Am; last by rewrite inE => /eqP->.
by case: cid => //= D [_ _ _ ->] XD; exact: bigcup_sup.
Qed.

Lemma decomp_set0 : decomp set0 = [fset set0]%fset.
Proof. by rewrite /decomp eqxx. Qed.

Lemma decompN0 (A : set rT) : decomp A != fset0.
Proof.
rewrite /decomp; case: ifPn => [_|AN0].
  by apply/fset0Pn; exists set0; rewrite inE.
case: pselect=> //= Am; last by apply/fset0Pn; exists A; rewrite inE.
case: cid=> //= D [_ _ _ Aeq]; apply: contra_neq AN0; rewrite Aeq => ->.
by rewrite set_fset0 bigcup_set0.
Qed.

Definition measure (R : numDomainType) (mu : set T -> \bar R)
  (A : set rT) : \bar R := \sum_(X <- decomp A) mu X.

Section additive_measure.
Context {R : realFieldType} (mu : {additive_measure set T -> \bar R}).
Local Notation Rmu := (measure mu).
Arguments big_trivIset {I D T R idx op} A F.

Lemma Rmu_fin_bigcup (I : choiceType) (D : set I) (F : I -> set T) :
    finite_set D -> trivIset D F -> (forall i, D i -> measurable (F i)) ->
  Rmu (\bigcup_(i in D) F i) = \sum_(i <- fset_set D) mu (F i).
Proof.
move=> Dfin Ftriv Fm; rewrite /measure.
have mUD : measurable (\bigcup_(i in D) F i : set rT).
  apply: fin_bigcup_measurable => // i Di.
  by apply: sub_gen_smallest; apply: Fm.
have [->|/set0P[i0 Di0]] := eqVneq D set0.
  by rewrite bigcup_set0 decomp_set0 big_seq_fset1 fset_set0 big_seq_fset0.
set E := (decomp _); have Em X := decomp_measurable mUD X.
transitivity (\sum_(X <- E) \sum_(i <- fset_set D) mu (X `&` F i)).
  apply: eq_big_seq => /= X XE.
  have Xeq : X = \bigcup_(i in D) (X `&` F i).
    by rewrite -setI_bigcupr setIidl//; apply: decomp_sub.
  rewrite -content_fin_bigcup// -?Xeq//; last exact: Em.
    exact: trivIset_setI.
  by move=> i Di; apply: measurableI; [apply: Em| apply: Fm].
rewrite exchange_big; apply: eq_big_seq => i; rewrite in_fset_set ?inE//= => Di.
have Feq : F i = \bigcup_(X in [set` E]) (X `&` F i).
  by rewrite -setI_bigcupl setIidr// cover_decomp; apply: bigcup_sup.
rewrite -[E]set_fsetK -content_fin_bigcup -?Feq//; last exact: Fm.
  exact/trivIset_setIr/decomp_triv.
by move=> X /= XE; apply: measurableI; [apply: Em | apply: Fm].
Qed.

Lemma RmuE (A : set T) : measurable A -> Rmu A = mu A.
Proof.
move=> Am; rewrite -[A in LHS](@bigcup_set1 _ unit _ tt).
by rewrite Rmu_fin_bigcup// ?fset_set1// ?big_seq_fset1// => -[].
Qed.

Lemma Rmu0 : Rmu set0 = 0.
Proof.
rewrite -(bigcup_set0 (fun _ : void => set0)).
by rewrite Rmu_fin_bigcup// fset_set0 big_seq_fset0.
Qed.

Lemma Rmu_ge0 A : Rmu A >= 0.
Proof. by rewrite sume_ge0//= => *; rewrite ?measure_ge0. Qed.

Lemma Rmu_additive : semi_additive Rmu.
Proof.
apply/(additive2P Rmu0) => // A B.
move=> /ring_fsets[/= {}A [_ Atriv Am ->]] /ring_fsets[/= {}B [_ Btriv Bm ->]].
rewrite -subset0 => coverAB0.
have AUBfin : finite_set ([set` A] `|` [set` B]) by rewrite finite_setU.
have AUBtriv : trivIset ([set` A] `|` [set` B]) id.
  move=> X Y [] ABX [] ABY; do ?by [apply: Atriv|apply: Btriv].
    by move=> [u [Xu Yu]]; case: (coverAB0 u); split; [exists X|exists Y].
  by move=> [u [Xu Yu]]; case: (coverAB0 u); split; [exists Y|exists X].
rewrite -bigcup_setU !Rmu_fin_bigcup//=; last by move=> X [/Am|/Bm].
rewrite -!fsbig_finite//= fsbigU//= => [X /= [XA XB]].
have [->//|/set0P[x Xx]] := eqVneq X set0.
by case: (coverAB0 x); split; [exists X|exists X].
Qed.

Lemma Rmu_additive_measure : additive_measure Rmu.
Proof. by split; [apply: Rmu0|apply: Rmu_ge0|apply: Rmu_additive]. Qed.

Canonical measure_additive_measure :=
  AdditiveMeasure Rmu_additive_measure.

End additive_measure.

End SetRing.
Module Exports.
Canonical ring_eqType.
Canonical ring_choiceType.
Canonical ring_ptType.
Canonical measure_additive_measure.
HB.reexport.
End Exports.
End SetRing.
Export SetRing.Exports.

Lemma le_measure (R : realFieldType) (T : semiRingOfSetsType)
    (mu : {additive_measure set T -> \bar R}) :
  {in measurable &, {homo mu : A B / A `<=` B >-> (A <= B)%E}}.
Proof.
move=> A B; rewrite ?inE => mA mB AB; have [|muBfin] := leP +oo%E (mu B).
  by rewrite lee_pinfty_eq => /eqP ->; rewrite lee_pinfty.
rewrite -[leRHS]SetRing.RmuE// -[B](setDUK AB) measureU/= ?setDIK//.
- by rewrite SetRing.RmuE ?lee_addl// ?measure_ge0.
- exact: sub_gen_smallest.
- by apply: measurableD; exact: sub_gen_smallest.
Qed.

Lemma measure_le0  (T : semiRingOfSetsType) (R : realFieldType)
    (mu : {additive_measure set T -> \bar R}) (A : set T) :
    (mu A <= 0)%E = (mu A == 0)%E.
Proof. by case: ltgtP (measure_ge0 mu A). Qed.

Section more_content_semiring_lemmas.
Variables (R : realFieldType) (T : semiRingOfSetsType).
Variable mu : {additive_measure set T -> \bar R}.

Lemma content_sub_additive : sub_additive mu.
Proof.
move=> X A n Am Xm XA; pose B i := A\_`I_n i `&` X.
have XE : X = \big[setU/set0]_(i < n) B i.
  rewrite -big_distrl/= setIidr// => x /XA/=.
  by rewrite -!bigcup_mkord => -[k nk Ax]; exists k; rewrite // patchT ?inE.
have Bm i : measurable (B i).
  case: (ltnP i n) => ltin; last by rewrite /B patchC ?inE ?set0I//= leq_gtF.
  by rewrite /B ?patchT ?inE//; apply: measurableI => //; apply: Am.
have subBA i : B i `<=` A i.
  by rewrite /B/patch; case: ifP; rewrite // set0I//= => _ ?.
have subDUB i : seqDU B i `<=` A i by  move=> x [/subBA].
have DUBm i : measurable (seqDU B i : set (SetRing.type T)).
  apply: measurableD; first exact: sub_gen_smallest.
  by apply: bigsetU_measurable => ? _; apply: sub_gen_smallest.
have DU0 i : (i >= n)%N -> seqDU B i = set0.
  move=> leni; rewrite -subset0 => x []; rewrite /B patchC ?inE/= ?leq_gtF//.
  by case.
rewrite -SetRing.RmuE// XE bigsetU_seqDU measure_bigsetU//.
rewrite [leRHS](big_ord_widen n (mu \o A))//= [leRHS]big_mkcond/=.
rewrite lee_sum => // i _; case: ltnP => ltin; last by rewrite DU0 ?measure0.
rewrite -[leRHS]SetRing.RmuE; last exact: Am.
by rewrite le_measure ?inE//=; last by apply: sub_gen_smallest; apply: Am.
Qed.

Lemma content_sub_fsum (I : choiceType) D (A : set T) (A_ : I -> set T) :
  finite_set D ->
  (forall i, D i -> measurable (A_ i)) ->
  measurable A ->
  A `<=` \bigcup_(i in D) A_ i -> mu A <= \sum_(i \in D) mu (A_ i).
Proof.
elim/choicePpointed: I => I in A_ D *.
  rewrite !emptyE bigcup_set0// subset0 => _ _ _ ->.
  by rewrite measure0 fsbig_set0.
move=> Dfin A_m Am Asub; have [n /ppcard_eqP[f]] := Dfin.
rewrite (reindex_fsbig f^-1%FUN `I_n)//= -fsbig_ord.
rewrite (@content_sub_additive A (A_ \o f^-1%FUN))//=.
  by move=> i ltin; apply: A_m; apply: funS.
rewrite (fsbig_ord _ _ (A_ \o f^-1%FUN))/= -(reindex_fsbig _ _ D)//=.
by rewrite fsbig_setU.
Qed.

(* (* alternative proof *) *)
(* Theorem semi_Boole_inequality : sub_additive mu. *)
(* Proof. *)
(* move=> X A n Am Xm Xsub; rewrite -SetRing.RmuE//. *)
(* under eq_bigr => i do [rewrite -SetRing.RmuE; do ?by apply: Am=> /=]. *)
(* pose rT := SetRing.type T. *)
(* have {}Am i : `I_n i -> measurable (A i : set rT). *)
(*   by move=> *; apply/SetRing.measurableW/Am => /=. *)
(* have {}Xm : measurable (X : set rT) by exact: SetRing.measurableW. *)
(* pose ammu := [additive_measure of SetRing.measure mu]. *)
(* rewrite (le_trans (le_measure ammu  _ _ Xsub)) ?inE// {Xsub}. *)
(*   by rewrite -bigcup_mkord; apply: fin_bigcup_measurable. *)
(* elim: n Am Xm => [|n IHn] Am Xm; first by rewrite !big_ord0 measure0. *)
(* have Anm : measurable (A n : set rT) by apply: Am => /=. *)
(* set B := \big[setU/set0]_(i < n) A i. *)
(* set C := \big[setU/set0]_(i < n.+1) A i. *)
(* have -> : C = B `|` (A n `\` B). *)
(*   suff -> : A n `\` B = C `\` B by rewrite setDUK// /C big_ord_recr/=; left. *)
(*   by rewrite /C big_ord_recr/= !setDE setIUl -!setDE setDv set0U. *)
(* have Bm : measurable (B : set rT). *)
(*   by rewrite -[B]bigcup_mkord; apply: fin_bigcup_measurable => //= i /ltnW/Am. *)
(* rewrite measureU // ?setDIK//; last exact: measurableD. *)
(* rewrite (@le_trans _ _ (ammu B + ammu (A n))) // ?lee_add2l //; last first. *)
(*   by rewrite big_ord_recr /= lee_add2r// IHn// => i /ltnW/Am. *)
(* by rewrite le_measure // ?inE// ?setDE//; exact: measurableD. *)
(* Qed. *)

End more_content_semiring_lemmas.

Section content_ring_lemmas.
Variables (R : realType) (T : ringOfSetsType).
Variable mu : {additive_measure set T -> \bar R}.

Lemma content_ring_sup_sigma_additive (A : nat -> set T) :
  (forall i, measurable (A i)) -> measurable (\bigcup_i A i) ->
  trivIset [set: nat] A -> \sum_(i <oo) mu (A i) <= mu (\bigcup_i A i).
Proof.
move=> Am UAm At; rewrite ereal_lim_le//; first exact: is_cvg_ereal_nneg_series.
near=> n; rewrite big_mkord -measure_bigsetU//= le_measure ?inE//=.
- exact: bigsetU_measurable.
- by rewrite -bigcup_mkord; apply: bigcup_sub => i lein; apply: bigcup_sup.
Unshelve. all: by end_near. Qed.

Lemma content_ring_sigma_additive : sigma_sub_additive mu -> semi_sigma_additive mu.
Proof.
move=> mu_sub A Am Atriv UAm.
suff <- : \sum_(i <oo) mu (A i) = mu (\bigcup_n A n).
  exact: is_cvg_ereal_nneg_series.
by apply/eqP; rewrite eq_le mu_sub// ?content_ring_sup_sigma_additive.
Qed.

End content_ring_lemmas.

Section ring_sigma_sub_additive_content.
Context (R : realType) (T : semiRingOfSetsType)
        (mu : {additive_measure set T -> \bar R}).
Local Notation Rmu := (SetRing.measure mu).
Import SetRing.

Lemma ring_sigma_sub_additive : sigma_sub_additive mu -> sigma_sub_additive Rmu.
Proof.
move=> muS; move=> /= D A Am Dm Dsub.
rewrite /Rmu -(eq_ereal_pseries (fun=> esum_fset _))//.
rewrite ereal_pseries_esum ?esum_esum//=; last by move=> *; rewrite esum_ge0.
set K := _ `*`` _.
have /ppcard_eqP[f] : (K #= [set: nat])%card.
  apply: cardMR_eq_nat => [|i].
    by rewrite (_ : [set _ | true] = setT)//; exact/predeqP.
  by split=> //; apply/set0P; rewrite set_fset_eq0 decompN0.
have {Dsub} : D `<=` \bigcup_(k in K) k.2.
  apply: (subset_trans Dsub); apply: bigcup_sub => i _.
  rewrite -[A i]cover_decomp; apply: bigcup_sub => X/= XAi.
  by move=> x Xx; exists (i, X).
rewrite -(image_eq [bij of f^-1%FUN])/=.
rewrite (esum_set_image _ f^-1)//= bigcup_image => Dsub.
have DXsub X : X \in decomp D -> X `<=` \bigcup_i ((f^-1%FUN i).2 `&` X).
  move=> XD; rewrite -setI_bigcupl -[Y in Y `<=` _](setIidr (decomp_sub XD)).
  by apply: setSI.
have mf i : measurable (((f^-1)%function i).2).
  by have [_ /decomp_measurable] := 'invS_f (I : setT i); apply; apply: Am.
have mfD i X : X \in decomp D -> measurable (((f^-1)%FUN i).2 `&` X : set T).
  by move=> XD; apply: measurableI; [apply: mf|apply: (decomp_measurable _ XD)].
apply: (@le_trans _ _
    (\sum_(i <oo) \sum_(X <- decomp D) mu ((f^-1%FUN i).2 `&` X))).
  rewrite ereal_pseries_sum// [leLHS]big_seq_cond [leRHS]big_seq_cond.
  rewrite lee_sum// => X; rewrite andbT => XD.
  have Xm : measurable X by exact: decomp_measurable XD.
  by apply: muS => // [i|]; [apply: mfD | apply: DXsub].
apply: lee_lim => /=; do ?apply: is_cvg_ereal_nneg_series=> //.
  by move=> n _; apply: sume_ge0.
near=> n; rewrite [n in _ <= n]big_mkcond; apply: lee_sum => i _.
rewrite ifT ?inE//.
under eq_big_seq do [rewrite -RmuE//; last exact: mfD].
rewrite  -[decomp _]set_fsetK -measure_fin_bigcup//=; last 2 first.
- by apply: trivIset_setI; apply: decomp_triv.
- by move=> X /= XD; apply: sub_gen_smallest; apply: mfD.
rewrite -setI_bigcupr (cover_decomp D) -[leRHS]RmuE// ?le_measure ?inE//.
  by apply: measurableI => //; apply: sub_gen_smallest; apply: mf.
by apply: sub_gen_smallest; apply: mf.
Unshelve. all: by end_near. Qed.

Lemma ring_sigma_additive : sigma_sub_additive mu -> semi_sigma_additive Rmu.
Proof.
move=> mu_sub; apply: content_ring_sigma_additive.
by apply: ring_sigma_sub_additive.
Qed.

End ring_sigma_sub_additive_content.

Section more_premeasure_ring_lemmas.
Variables (R : realType) (T : semiRingOfSetsType).
Variable mu : {measure set T -> \bar R}.
Import SetRing.

Lemma measure_sigma_sub_additive : sigma_sub_additive mu.
Proof.
move=> X A Am Xm XA; pose B i := A i `&` X.
have XE : X = \bigcup_i B i by rewrite -setI_bigcupl setIidr.
have Bm i : measurable (B i) by rewrite /B; apply: measurableI.
have subBA i : B i `<=` A i by rewrite /B.
have subDUB i : seqDU B i `<=` A i by  move=> x [/subBA].
have DUBm i : measurable (seqDU B i : set (SetRing.type T)).
  by apply: measurableD => //;
     do 1?apply: bigsetU_measurable => *; apply: sub_gen_smallest.
rewrite XE; move: (XE); rewrite seqDU_bigcup_eq.
under eq_bigcupr do rewrite -[seqDU B _]cover_decomp//.
rewrite bigcup_bigcup_dep; set K := _ `*`` _.
have /ppcard_eqP[f] : (K #= [set: nat])%card.
  apply: cardMR_eq_nat=> // i; split=> //.
  by apply/set0P; rewrite set_fset_eq0 decompN0.
pose f' := f^-1%FUN.
rewrite -(image_eq [bij of f'])/= bigcup_image/=.
pose g n := (f' n).2.
have fVtriv : trivIset [set: nat] g.
  move=> i j _ _; rewrite /g.
  have [/= _ f'iB] : K (f' i) by apply: funS.
  have [/= _ f'jB] : K (f' j) by apply: funS.
  have [f'ij|f'ij] := eqVneq (f' i).1 (f' j).1.
    move=> /(decomp_triv f'iB)/=; rewrite f'ij => /(_ f'jB) f'ij2.
    apply: 'inj_f'; rewrite ?inE//= -!/(f' _); move: f'ij f'ij2.
    by case: (f' i) (f' j) => [? ?] [? ?]//= -> ->.
  move=> [x [f'ix f'jx]]; have Bij := @trivIset_seqDU _ B (f' i).1 (f' j).1 I I.
  rewrite Bij ?eqxx// in f'ij; exists x; split.
    exact: (decomp_sub f'iB).
  exact: (decomp_sub f'jB).
have g_inj : set_inj [set i | g i != set0] g.
  apply: trivIset_inj=> [i /set0P//|].
  by apply: sub_trivIset fVtriv.
move=> XEbig; rewrite measure_semi_bigcup//= -?XEbig//; last first.
  move=> i; have [/= _ f'iB] : K (f' i) by apply: funS.
  by apply: decomp_measurable f'iB.
rewrite [leLHS](_ : _ = \sum_(i <oo | g i != set0) mu (g i)); last first.
  rewrite !ereal_pseries_esum// esum_mkcond [RHS]esum_mkcond; apply: eq_esum.
  move=> i _; rewrite ifT ?inE//=; case: ifPn => //.
  by rewrite notin_set /= -/(g _) => /negP/negPn/eqP ->.
rewrite -(esum_pred_image mu g)//.
rewrite [esum _ _](_ : _ = \esum_(X in range g) mu X); last first.
  rewrite esum_mkcond [RHS]esum_mkcond; apply: eq_esum.
  move=> Y _; case: ifPn; rewrite ?(inE, notin_set)/=.
    by move=> [i giN0 giY]; rewrite ifT// ?inE//=; exists i.
  move=> Ngx; case: ifPn; rewrite ?(inE, notin_set)//=.
  move=> [i _ giY]; apply: contra_not_eq Ngx; rewrite -giY => mugi.
  by exists i => //; apply: contra_neq mugi => ->; rewrite measure0.
have -> : range g = \bigcup_i [set` decomp (seqDU B i)].
  apply/predeqP => /= Y; split => [[n _ gnY]|[n _ /= YBn]].
  have [/= _ f'nB] : K (f' n) by apply: funS.
    by exists (f' n).1 => //=; rewrite -gnY.
  by exists (f (n, Y)) => //; rewrite /g /f' funK//= inE; split.
rewrite esum_bigcup//; last first.
   move=> i j /=.
   have [->|/set0P DUBiN0] := eqVneq (seqDU B i) set0.
     rewrite decomp_set0 ?set_fset1 => /negP[].
     apply/eqP/predeqP=> x; split=> [[Y ->]|->]//=; first by rewrite measure0.
     by exists set0.
   have [->|/set0P DUBjN0] := eqVneq (seqDU B j) set0.
     rewrite decomp_set0 ?set_fset1 => _ /negP[].
     apply/eqP/predeqP=> x; split=> [[Y ->]|->]//=; first by rewrite measure0.
     by exists set0.
   move=> _ _ [Y /=  [/[dup]+ /decomp_sub YBi /decomp_sub YBj]].
   move=> /(decomp_neq0 DUBiN0) [y Yy].
   apply: (@trivIset_seqDU _ B) => //; exists y.
   by split => //; [exact: YBi|exact: YBj].
rewrite ereal_pseries_esum// set_true le_esum// => i _.
rewrite esum_fset// -[decomp _]set_fsetK.
rewrite -SetRing.Rmu_fin_bigcup//=; last 2 first.
  exact: decomp_triv.
  by move=> ?; apply: decomp_measurable.
rewrite -[leRHS]SetRing.RmuE// le_measure ?inE//=.
- apply: fin_bigcup_measurable=> // j /= jdec.
  by apply: sub_gen_smallest; exact: decomp_measurable jdec.
- exact: sub_gen_smallest.
- by rewrite cover_decomp.
Qed.

End more_premeasure_ring_lemmas.

Section ring_sigma_additive_measure.
Context (R : realType) (T : semiRingOfSetsType)
        (mu : {measure set T -> \bar R}).
Local Notation Rmu := (SetRing.measure mu).
Import SetRing.

Local Lemma ring_sigma_additive_measure : semi_sigma_additive Rmu.
Proof.
by apply: ring_sigma_additive; apply: measure_sigma_sub_additive.
Qed.

Canonical set_ring_measure : {measure set (SetRing.type T) -> \bar R} :=
  @Measure.Pack _ _ (Phant _) Rmu
    (Measure.Axioms (measure0 _) (measure_ge0 _) ring_sigma_additive_measure).

End ring_sigma_additive_measure.

Lemma measureIl (R : realFieldType) (T : semiRingOfSetsType)
    (mu : {additive_measure set T -> \bar R}) (A B : set T) :
  measurable A -> measurable B -> (mu (A `&` B) <= mu A)%E.
Proof. by move=> mA mB; rewrite le_measure ?inE//; apply: measurableI. Qed.

Lemma measureIr (R : realFieldType) (T : semiRingOfSetsType)
    (mu : {additive_measure set T -> \bar R}) (A B : set T) :
  measurable A -> measurable B -> (mu (A `&` B) <= mu B)%E.
Proof. by move=> mA mB; rewrite le_measure ?inE//; apply: measurableI. Qed.

Lemma subset_measure0 (T : semiRingOfSetsType) (R : realType)
  (mu : {additive_measure set T -> \bar R}) (A B : set T) :
  measurable A -> measurable B -> A `<=` B ->
  mu B = 0%E -> mu A = 0%E.
Proof.
move=> mA mB AB B0; apply/eqP; rewrite eq_le measure_ge0// ?andbT -?B0.
by apply: le_measure; rewrite ?inE.
Qed.

Section measureD.
Variables (R : realFieldType) (T : ringOfSetsType).
Variable mu : {measure set T -> \bar R}.

Lemma measureDI A B : measurable A -> measurable B ->
  mu A = mu (A `\` B) + mu (A `&` B).
Proof.
move=> mA mB; rewrite -measure_semi_additive2.
- by rewrite -setDDr setDv setD0.
- exact: measurableD.
- exact: measurableI.
- by apply: measurableU; [exact: measurableD |exact: measurableI].
- by rewrite setDE setIACA setICl setI0.
Qed.

Lemma measureD A B : measurable A -> measurable B ->
  mu A < +oo -> mu (A `\` B) = mu A - mu (A `&` B).
Proof.
move=> mA mB mAoo.
rewrite (measureDI mA mB) addeK// fin_numE 1?gt_eqF 1?lt_eqF//.
- rewrite (le_lt_trans _ mAoo)// le_measure // ?inE//.
  + exact: measurableI.
- by rewrite (lt_le_trans _ (measure_ge0 _ _))// ?lte_ninfty.
Qed.

End measureD.

Lemma measureUfinr (T : ringOfSetsType) (R : realFieldType) (A B : set T)
   (mu : {measure set T -> \bar R}):
    measurable A -> measurable B -> (mu B < +oo)%E ->
  mu (A `|` B) = (mu A + mu B - mu (A `&` B))%E.
Proof.
move=> Am Bm mBfin; rewrite -[B in LHS](setDUK (@subIsetl _ _ A)) setUA.
rewrite [A `|` _]setUidl; last exact: subIsetr.
rewrite measureU//=; do ?by apply:measurableD; do ?apply: measurableI.
  rewrite measureD//; do ?exact: measurableI.
  by rewrite addeA setIA setIid setIC.
by rewrite setDE setCI setIUr -!setDE setDv set0U setDIK.
Qed.

Lemma measureUfinl (T : ringOfSetsType) (R : realFieldType) (A B : set T)
   (mu : {measure set T -> \bar R}):
    measurable A -> measurable B -> (mu A < +oo)%E ->
  mu (A `|` B) = (mu A + mu B - mu (A `&` B))%E.
Proof. by move=> *; rewrite setUC measureUfinr// setIC [(mu B + _)%E]addeC. Qed.

Lemma eq_measureU (T : ringOfSetsType) (R : realFieldType) (A B : set T)
   (mu mu' : {measure set T -> \bar R}):
    measurable A -> measurable B ->
  mu A = mu' A -> mu B = mu' B -> mu (A `&` B) = mu' (A `&` B) ->
  mu (A `|` B) = mu' (A `|` B).
Proof.
move=> mA mB muA muB muAB; have [mu'ANoo|] := ltP (mu' A) +oo.
  by rewrite !measureUfinl ?muA ?muB ?muAB.
rewrite lee_pinfty_eq => /eqP mu'A; transitivity (+oo : \bar R); apply/eqP.
  by rewrite -lee_pinfty_eq -mu'A -muA le_measure ?inE//=; apply: measurableU.
by rewrite eq_sym -lee_pinfty_eq -mu'A le_measure ?inE//=; apply: measurableU.
Qed.

Lemma null_set_setU (R : realFieldType) (T : ringOfSetsType)
  (mu : {measure set T -> \bar R}) (A B : set T) :
  measurable A -> measurable B -> mu A = 0%E -> mu B = 0%E -> mu (A `|` B) = 0%E.
Proof.
move=> mA mB A0 B0; rewrite measureUfinl ?A0 ?lte_pinfty//= ?B0 ?add0e.
apply/eqP; rewrite oppe_eq0 -measure_le0/=; do ?exact: measurableI.
by rewrite -A0 measureIl.
Qed.

Lemma cvg_mu_inc (R : realFieldType) (T : ringOfSetsType)
  (mu : {measure set T -> \bar R}) (F : (set T) ^nat) :
  (forall i, measurable (F i)) -> measurable (\bigcup_n F n) ->
  nondecreasing_seq F ->
  mu \o F --> mu (\bigcup_n F n).
Proof.
move=> mF mbigcupF ndF.
have Binter : trivIset setT (seqD F) := trivIset_seqD ndF.
have FBE : forall n, F n.+1 = F n `|` seqD F n.+1 := setU_seqD ndF.
have FE n : F n = \big[setU/set0]_(i < n.+1) (seqD F) i := eq_bigsetU_seqD n ndF.
rewrite eq_bigcup_seqD.
have mB i : measurable (seqD F i) by elim: i => * //=; apply: measurableD.
apply: cvg_trans (measure_semi_sigma_additive mB Binter _); last first.
  by rewrite -eq_bigcup_seqD.
apply: (@cvg_trans _ [filter of (fun n => \sum_(i < n.+1) mu (seqD F i))]).
  rewrite [X in _ --> X](_ : _ = mu \o F) // funeqE => n.
  by rewrite -measure_semi_additive // -?FE// => -[|k].
move=> S [n _] nS; exists n => // m nm.
under eq_fun do rewrite -(big_mkord predT (mu \o seqD F)).
exact/(nS m.+1)/(leq_trans nm).
Qed.

Section boole_inequality.
Variables (R : realFieldType) (T : ringOfSetsType).
Variables (mu : {additive_measure set T -> \bar R}).

Theorem Boole_inequality (A : (set T) ^nat) n :
    (forall i, `I_n i -> measurable (A i)) ->
  mu (\big[setU/set0]_(i < n) A i) <= \sum_(i < n) mu (A i).
Proof.
move=> Am; rewrite content_sub_additive// -bigcup_mkord.
exact: fin_bigcup_measurable.
Qed.

End boole_inequality.
Notation le_mu_bigsetU := Boole_inequality.

Section sigma_finite_lemma.
Variables (R : realFieldType) (T : ringOfSetsType) (A : set T)
  (mu : {additive_measure set T -> \bar R}).

Lemma sigma_finiteP : sigma_finite A mu ->
  exists2 F, A = \bigcup_i F i &
    nondecreasing_seq F /\ forall i, measurable (F i) /\ mu (F i) < +oo.
Proof.
move=> [S AS moo]; exists (fun n => \big[setU/set0]_(i < n.+1) S i).
  rewrite AS predeqE => t; split => [[i _ Sit]|[i _]].
    by exists i => //; rewrite big_ord_recr /=; right.
  by rewrite -bigcup_mkord => -[j /= ji Sjt]; exists j.
split=> [|i].
- apply/nondecreasing_seqP => i; rewrite [in leRHS]big_ord_recr /=.
  by apply/subsetPset; left.
- split; first by apply: bigsetU_measurable => j _; exact: (moo j).1.
  rewrite (@le_lt_trans _ _ (\sum_(j < i.+1) mu (S j)))//.
    by apply: Boole_inequality => j _; exact: (moo j).1.
  by apply/lte_sum_pinfty => j _; exact: (moo j).2.
Qed.

End sigma_finite_lemma.

Section generalized_boole_inequality.
Variables (R : realType) (T : ringOfSetsType).
Variable (mu : {measure set T -> \bar R}).

Theorem generalized_Boole_inequality (A : (set T) ^nat) :
  (forall i, measurable (A i)) -> measurable (\bigcup_n A n) ->
  mu (\bigcup_n A n) <= \sum_(i <oo) mu (A i).
Proof. by move=> Am UAm; rewrite measure_sigma_sub_additive. Qed.

End generalized_boole_inequality.
Notation le_mu_bigcup := generalized_Boole_inequality.

Section negligible.
Variables (R : realFieldType) (T : ringOfSetsType).

Definition negligible (mu : set T -> \bar R) (N : set T) :=
  exists A : set T, [/\ measurable A, mu A = 0 & N `<=` A].

Local Notation "mu .-negligible" := (negligible mu).

Lemma negligibleP (mu : {additive_measure _ -> _}) A :
  measurable A -> mu.-negligible A <-> mu A = 0.
Proof.
move=> mA; split => [[B [mB mB0 AB]]|mA0]; last by exists A; split.
apply/eqP; rewrite eq_le measure_ge0 // andbT -mB0.
by apply: (le_measure mu) => //; rewrite in_setE.
Qed.

Lemma negligible_set0 (mu : {additive_measure _ -> _}) : mu.-negligible set0.
Proof. exact/negligibleP. Qed.

Lemma measure_negligible (mu : {additive_measure set T -> \bar R}) (A : set T) :
  measurable A -> mu.-negligible A -> mu A = 0%E.
Proof. by move=> mA /negligibleP ->. Qed.

Definition almost_everywhere (mu : set T -> \bar R) (P : T -> Prop)
     & (phantom Prop (forall x, P x)) :=
   mu.-negligible (~` [set x | P x]).
Local Notation "{ 'ae' m , P }" :=
  (almost_everywhere m (inPhantom P)) : type_scope.

Lemma aeW (mu : {measure _ -> _}) (P : T -> Prop) :
  (forall x, P x) -> {ae mu, forall x, P x}.
Proof.
move=> aP; have -> : P = setT by rewrite predeqE => t; split.
by apply/negligibleP; [rewrite setCT|rewrite setCT measure0].
Qed.

End negligible.

Notation "mu .-negligible" := (negligible mu) : type_scope.

Notation "{ 'ae' m , P }" := (almost_everywhere m (inPhantom P)) : type_scope.

Definition sigma_subadditive (R : numFieldType) (T : Type)
  (mu : set T -> \bar R) := forall (F : (set T) ^nat),
  mu (\bigcup_n (F n)) <= \sum_(i <oo) mu (F i).

Module OuterMeasure.

Section ClassDef.

Variables (R : numFieldType) (T : Type).
Record axioms (mu : set T -> \bar R) := Axioms {
  _ : mu set0 = 0 ;
  _ : forall x, 0 <= mu x ;
  _ : {homo mu : A B / A `<=` B >-> A <= B} ;
  _ : sigma_subadditive mu }.

Structure map (phUV : phant (set T -> \bar R)) :=
  Pack {apply : set T -> \bar R ; _ : axioms apply}.
Local Coercion apply : map >-> Funclass.

Variables (phUV : phant (set T -> \bar R)) (f g : set T -> \bar R).
Variable (cF : map phUV).
Definition class := let: Pack _ c as cF' := cF return axioms cF' in c.
Definition clone fA of phant_id g (apply cF) & phant_id fA class :=
  @Pack phUV f fA.

End ClassDef.

Module Exports.
Notation outer_measure f := (axioms f).
Coercion apply : map >-> Funclass.
Notation OuterMeasure fA := (Pack (Phant _) fA).
Notation "{ 'outer_measure' fUV }" := (map (Phant fUV))
  (at level 0, format "{ 'outer_measure'  fUV }") : ring_scope.
Notation "[ 'outer_measure' 'of' f 'as' g ]" := (@clone _ _ _ f g _ _ idfun id)
  (at level 0, format "[ 'outer_measure'  'of'  f  'as'  g ]") : form_scope.
Notation "[ 'outer_measure' 'of' f ]" := (@clone _ _ _ f f _ _ id id)
  (at level 0, format "[ 'outer_measure'  'of'  f ]") : form_scope.
End Exports.

End OuterMeasure.
Include OuterMeasure.Exports.

Section outer_measure_lemmas.
Variables (R : numFieldType) (T : Type).
Variable mu : {outer_measure set T -> \bar R}.

Lemma outer_measure0 : mu set0 = 0.
Proof. by case: mu => ? []. Qed.

Lemma outer_measure_ge0 : forall x, 0 <= mu x.
Proof. by case: mu => ? []. Qed.

Lemma le_outer_measure : {homo mu : A B / A `<=` B >-> A <= B}.
Proof. by case: mu => ? []. Qed.

Lemma outer_measure_sigma_subadditive : sigma_subadditive mu.
Proof. by case: mu => ? []. Qed.

End outer_measure_lemmas.

#[global] Hint Extern 0 (_ set0 = 0) => solve [apply: outer_measure0] : core.
#[global] Hint Extern 0 (sigma_subadditive _) =>
  solve [apply: outer_measure_sigma_subadditive] : core.

Lemma le_outer_measureIC (R : realFieldType) T
  (mu : {outer_measure set T -> \bar R}) (A X : set T) :
  mu X <= mu (X `&` A) + mu (X `&` ~` A).
Proof.
pose B : (set T) ^nat := bigcup2 (X `&` A) (X `&` ~` A).
have cvg_mu : (fun n => \sum_(i < n) mu (B i)) --> mu (B 0%N) + mu (B 1%N).
  rewrite -2!cvg_shiftS /=.
  rewrite [X in X --> _](_ : _ = (fun=> mu (B 0%N) + mu (B 1%N))); last first.
    rewrite funeqE => i; rewrite 2!big_ord_recl /= big1 ?adde0 // => j _.
    by rewrite /B /bigcup2 /=.
  exact: cvg_cst.
have := outer_measure_sigma_subadditive mu B.
suff : \bigcup_n B n = X.
  move=> -> /le_trans; apply; under eq_fun do rewrite big_mkord.
  by rewrite (cvg_lim _ cvg_mu).
transitivity (\big[setU/set0]_(i < 2) B i).
  by rewrite (bigcup_splitn 2) // -bigcup_mkord setUidl// => t -[].
by rewrite 2!big_ord_recl big_ord0 setU0 /= -setIUr setUCr setIT.
Unshelve. all: by end_near. Qed.

Definition caratheodory_measurable (R : realType) (T : Type)
  (mu : set T -> \bar R) (A : set T) := forall X,
  mu X = mu (X `&` A) + mu (X `&` ~` A).

Notation "mu .-measurable" := (caratheodory_measurable mu)
  (at level 2, format "mu .-measurable") : type_scope.

Lemma le_caratheodory_measurable (R : realType) T
  (mu : {outer_measure set T -> \bar R}) (A : set T) :
  (forall X, mu (X `&` A) + mu (X `&` ~` A) <= mu X) ->
  mu.-measurable A.
Proof.
move=> suf X; apply/eqP; rewrite eq_le; apply/andP; split;
  [exact: le_outer_measureIC | exact: suf].
Qed.

Section caratheodory_theorem_sigma_algebra.

Variables (R : realType) (T : Type) (mu : {outer_measure set T -> \bar R}).

Lemma outer_measure_bigcup_lim (A : (set T) ^nat) X :
  mu (X `&` \bigcup_k A k) <= \sum_(k <oo) mu (X `&` A k).
Proof.
apply: (le_trans _ (outer_measure_sigma_subadditive mu (fun n => X `&` A n))).
by apply/le_outer_measure; rewrite setI_bigcupr.
Qed.

Let M := mu.-measurable.

Lemma caratheodory_measurable_set0 : M set0.
Proof. by move=> X /=; rewrite setI0 outer_measure0 add0e setC0 setIT. Qed.

Lemma caratheodory_measurable_setC A : M A -> M (~` A).
Proof. by move=> MA X; rewrite setCK addeC -MA. Qed.

Lemma caratheodory_measurable_setU_le (X A B : set T) :
  mu.-measurable A -> mu.-measurable B ->
  mu (X `&` (A `|` B)) + mu (X `&` ~` (A `|` B)) <= mu X.
Proof.
move=> mA mB; pose Y := X `&` A `|` X `&` B `&` ~` A.
have /(lee_add2r (mu (X `&` ~` (A `|` B)))) :
    mu Y <= mu (X `&` A) + mu (X `&` B `&` ~` A).
  pose Z := bigcup2 (X `&` A) (X `&` B `&` ~` A).
  have -> : Y = \bigcup_k Z k.
    rewrite predeqE => t; split=> [[?|?]|[]]; [by exists O|by exists 1%N|].
    by move=> [_ ?|[_ ?|//]]; [left|right].
  rewrite (le_trans (outer_measure_sigma_subadditive mu Z)) //.
  suff : ((fun n => \sum_(i < n) mu (Z i)) -->
      mu (X `&` A) + mu (X `&` B `&` ~` A)).
    move/cvg_lim => /=; under [in leLHS]eq_fun do rewrite big_mkord.
    by move=> ->.
  rewrite -(cvg_shiftn 2) /=; set l := (X in _ --> X).
  rewrite [X in X --> _](_ : _ = cst l); first exact: cvg_cst.
  rewrite funeqE => i; rewrite addn2 2!big_ord_recl big1 ?adde0 //.
  by move=> ? _; exact: outer_measure0.
have /le_trans : mu (X `&` (A `|` B)) + mu (X `&` ~` (A `|` B)) <=
    mu Y + mu (X `&` ~` (A `|` B)).
  rewrite setIUr (_ : X `&` A `|` X `&` B = Y) //.
  rewrite /Y -[in LHS](setIT B) -(setUCr A) 2!setIUr setUC -[in RHS]setIA.
  rewrite setUC setUA; congr (_ `|` _).
  by rewrite setUidPl setICA; apply subIset; right.
suff -> : mu (X `&` A) + mu (X `&` B `&` ~` A) +
    mu (X `&` (~` (A `|` B))) = mu X by exact.
by rewrite setCU setIA -(setIA X) setICA (setIC B) -addeA -mB -mA.
Qed.

Lemma caratheodory_measurable_setU A B : M A -> M B -> M (A `|` B).
Proof.
move=> mA mB X; apply/eqP; rewrite eq_le.
by rewrite le_outer_measureIC andTb caratheodory_measurable_setU_le.
Qed.

Lemma caratheodory_measurable_bigsetU (A : (set T) ^nat) : (forall n, M (A n)) ->
  forall n, M (\big[setU/set0]_(i < n) A i).
Proof.
move=> MA; elim=> [|n ih]; first by rewrite big_ord0; exact: caratheodory_measurable_set0.
by rewrite big_ord_recr; apply caratheodory_measurable_setU.
Qed.

Lemma caratheodory_measurable_setI A B : M A -> M B -> M (A `&` B).
Proof.
move=> mA mB; rewrite -(setCK A) -(setCK B) -setCU.
by apply/caratheodory_measurable_setC/caratheodory_measurable_setU;
  exact/caratheodory_measurable_setC.
Qed.

Lemma caratheodory_measurable_setD A B : M A -> M B -> M (A `\` B).
Proof.
move=> mA mB; rewrite setDE; apply caratheodory_measurable_setI => //.
exact: caratheodory_measurable_setC.
Qed.

Section additive_ext_lemmas.
Variable A B : set T.
Hypothesis (mA : M A) (mB : M B).

Let caratheodory_decomp X :
  mu X = mu (X `&` A `&` B) + mu (X `&` A `&` ~` B) +
         mu (X `&` ~` A `&` B) + mu (X `&` ~` A `&` ~` B).
Proof. by rewrite mA mB [X in _ + _ + X = _]mB addeA. Qed.

Let caratheorody_decompIU X : mu (X `&` (A `|` B)) =
  mu (X `&` A `&` B) + mu (X `&` ~` A `&` B) + mu (X `&` A `&` ~` B).
Proof.
rewrite caratheodory_decomp -!addeA; congr (mu _ + _).
  rewrite -!setIA; congr (_ `&` _).
  by rewrite setIC; apply/setIidPl; apply subIset; left; left.
rewrite addeA addeC [X in mu X + _](_ : _ = set0); last first.
  by rewrite -setIA -setCU -setIA setICr setI0.
rewrite outer_measure0 add0e addeC -!setIA; congr (mu (X `&` _) + mu (X `&` _)).
by rewrite setIC; apply/setIidPl; apply subIset; right; right.
by rewrite setIC; apply/setIidPl; apply subIset; left; left.
Qed.

Lemma disjoint_caratheodoryIU X : [disjoint A & B] ->
  mu (X `&` (A `|` B)) = mu (X `&` A) + mu (X `&` B).
Proof.
move=> /eqP AB; rewrite caratheodory_decomp -setIA AB setI0 outer_measure0.
rewrite add0e addeC -setIA -setCU -setIA setICr setI0 outer_measure0 add0e.
rewrite -!setIA; congr (mu (X `&` _ ) + mu (X `&` _)).
rewrite (setIC A) setIA setIC; apply/setIidPl.
- by rewrite setIUl setICr setU0 subsetI; move/disjoints_subset in AB; split.
- rewrite setIA setIC; apply/setIidPl; rewrite setIUl setICr set0U.
  by move: AB; rewrite setIC => /disjoints_subset => AB; rewrite subsetI; split.
Qed.
End additive_ext_lemmas.

Lemma caratheodory_additive (A : (set T) ^nat) : (forall n, M (A n)) ->
  trivIset setT A -> forall n X,
    mu (X `&` \big[setU/set0]_(i < n) A i) = \sum_(i < n) mu (X `&` A i).
Proof.
move=> MA ta; elim=> [|n ih] X; first by rewrite !big_ord0 setI0 outer_measure0.
rewrite big_ord_recr /= disjoint_caratheodoryIU // ?ih ?big_ord_recr //.
- exact: caratheodory_measurable_bigsetU.
- by apply/eqP/(@trivIset_bigsetUI _ predT) => //; rewrite /predT /= trueE.
Qed.

Lemma caratheodory_lim_lee (A : (set T) ^nat) : (forall n, M (A n)) ->
  trivIset setT A -> forall X,
  \sum_(k <oo) mu (X `&` A k) + mu (X `&` ~` \bigcup_k A k) <= mu X.
Proof.
move=> MA tA X.
set A' := \bigcup_k A k; set B := fun n => \big[setU/set0]_(k < n) (A k).
suff : forall n, \sum_(k < n) mu (X `&` A k) + mu (X `&` ~` A') <= mu X.
  move=> XA; rewrite (_ : lim _ = ereal_sup
      ((fun n => \sum_(k < n) mu (X `&` A k)) @` setT)); last first.
    under eq_fun do rewrite big_mkord.
    apply/cvg_lim => //; apply/ereal_nondecreasing_cvg.
    apply: (lee_sum_nneg_ord (fun n => mu (X `&` A n)) xpredT) => n _.
    exact: outer_measure_ge0.
  move XAx : (mu (X `&` ~` A')) => [x| |].
  - rewrite -lee_subr_addr //; apply ub_ereal_sup => /= _ [n _] <-.
    by rewrite EFinN lee_subr_addr // -XAx XA.
  - suff : mu X = +oo by move=> ->; rewrite lee_pinfty.
    by apply/eqP; rewrite -lee_pinfty_eq -XAx le_outer_measure.
  - by rewrite addeC /= lee_ninfty.
move=> n.
apply (@le_trans _ _ (\sum_(k < n) mu (X `&` A k) + mu (X `&` ~` B n))).
  apply/lee_add2l/le_outer_measure; apply: setIS; apply: subsetC => t.
  by rewrite /B -bigcup_mkord => -[i ? ?]; exists i.
rewrite [in leRHS](caratheodory_measurable_bigsetU MA n) lee_add2r //.
by rewrite caratheodory_additive.
Qed.

Lemma caratheodory_measurable_trivIset_bigcup (A : (set T) ^nat) :
  (forall n, M (A n)) -> trivIset setT A -> M (\bigcup_k (A k)).
Proof.
move=> MA tA; apply le_caratheodory_measurable => X /=.
have /(lee_add2r (mu (X `&` ~` \bigcup_k A k))) := outer_measure_bigcup_lim A X.
by move/le_trans; apply; exact: caratheodory_lim_lee.
Qed.

Lemma caratheodory_measurable_bigcup (A : (set T) ^nat) : (forall n, M (A n)) ->
  M (\bigcup_k (A k)).
Proof.
move=> MA; rewrite -eq_bigcup_seqD_bigsetU.
apply/caratheodory_measurable_trivIset_bigcup; last first.
  apply: (@trivIset_seqD _ (fun n => \big[setU/set0]_(i < n.+1) A i)).
  by move=> n m nm; exact/subsetPset/subset_bigsetU.
by case=> [|n /=]; [| apply/caratheodory_measurable_setD => //];
  exact/caratheodory_measurable_bigsetU.
Qed.

End caratheodory_theorem_sigma_algebra.

Definition caratheodory_type (R : realType) (T : Type)
  (mu : set T -> \bar R) := T.

Section caratheodory_sigma_algebra.
Variables (R : realType) (T : pointedType) (mu : {outer_measure set T -> \bar R}).

HB.instance Definition caratheodory_mixin := @isMeasurable.Build
  (caratheodory_type mu) (Pointed.class T) mu.-measurable
    (caratheodory_measurable_set0 mu)
    (@caratheodory_measurable_setC _ _ mu)
    (@caratheodory_measurable_bigcup _ _ mu).

End caratheodory_sigma_algebra.

Definition measure_is_complete (R : realType) (T : measurableType)
    (mu : set T -> \bar R) :=
  forall X, mu.-negligible X -> measurable X.

Section caratheodory_measure.
Variables (R : realType) (T : pointedType) (mu : {outer_measure set T -> \bar R}).
Local Notation U := (caratheodory_type mu).

Lemma caratheodory_measure0 : mu (set0 : set U) = 0.
Proof. exact: outer_measure0. Qed.

Lemma caratheodory_measure_ge0 (A : set U) : 0 <= mu A.
Proof. exact: outer_measure_ge0. Qed.

Lemma caratheodory_measure_sigma_additive : semi_sigma_additive (mu : set U -> _).
Proof.
move=> A mA tA mbigcupA; set B := \bigcup_k A k.
suff : forall X, mu X = \sum_(k <oo) mu (X `&` A k) + mu (X `&` ~` B).
  move/(_ B); rewrite setICr outer_measure0 adde0.
  rewrite (_ : (fun n => _) = fun n => \sum_(k < n) mu (A k)); last first.
    rewrite funeqE => n; rewrite big_mkord; apply eq_bigr => i _; congr (mu _).
    by rewrite setIC; apply/setIidPl => t Ait; exists i.
  move=> ->; have := fun n (_ : xpredT n) => outer_measure_ge0 mu (A n).
  move/(@is_cvg_ereal_nneg_series _ _) => /cvg_ex[l] hl.
  under [in X in _ --> X]eq_fun do rewrite -(big_mkord xpredT (mu \o A)).
  by move/(@cvg_lim _ (@ereal_hausdorff R)) : (hl) => ->.
move=> X.
have mB : mu.-measurable B := caratheodory_measurable_bigcup mA.
apply/eqP; rewrite eq_le (caratheodory_lim_lee mA tA X) andbT.
have /(lee_add2r (mu (X `&` ~` B))) := outer_measure_bigcup_lim mu A X.
by rewrite -le_caratheodory_measurable // => ?; rewrite -mB.
Qed.

Definition caratheodory_measure_mixin := Measure.Axioms caratheodory_measure0
  caratheodory_measure_ge0 caratheodory_measure_sigma_additive.
Definition caratheodory_measure : {measure set U -> \bar R} :=
  Measure caratheodory_measure_mixin.

Lemma measure_is_complete_caratheodory : measure_is_complete caratheodory_measure.
Proof.
move=> B [A [mA muA0 BA]]; apply le_caratheodory_measurable => X.
suff -> : mu (X `&` B) = 0.
  by rewrite add0e le_outer_measure //; apply subIset; left.
have muB0 : mu B = 0.
  apply/eqP; rewrite eq_le outer_measure_ge0 andbT.
  by apply: (le_trans (le_outer_measure mu BA)); rewrite -muA0.
apply/eqP; rewrite eq_le outer_measure_ge0 andbT.
have : X `&` B `<=` B by apply subIset; right.
by move/(le_outer_measure mu); rewrite muB0 => ->.
Qed.

End caratheodory_measure.

Lemma epsilon_trick (R : realType) (A : (\bar R)^nat) e
    (P : pred nat) : (forall n, 0 <= A n) -> (0 <= e)%R ->
  \sum_(i <oo | P i) (A i + (e / (2 ^ i.+1)%:R)%:E) <=
  \sum_(i <oo | P i) A i + e%:E.
Proof.
move=> A0 /nonnegP[{}e].
rewrite (@le_trans _ _ (lim (fun n => (\sum_(0 <= i < n | P i) A i) +
    \sum_(0 <= i < n) (e%:num / (2 ^ i.+1)%:R)%:E))) //.
  rewrite ereal_pseriesD //.
  rewrite ereal_limD //.
  - rewrite lee_add2l //; apply: lee_lim => //.
    + by apply: is_cvg_ereal_nneg_series => n _; apply: divr_ge0.
    + by apply: is_cvg_ereal_nneg_series => n _; apply: divr_ge0.
    + by near=> n; apply: lee_sum_nneg_subset => // i _; apply divr_ge0.
  - exact: is_cvg_ereal_nneg_series.
  - by apply: is_cvg_ereal_nneg_series => n _; apply: divr_ge0.
  - by apply: adde_def_nneg_series => // n _; apply: divr_ge0.
suff cvggeo : (fun n => \sum_(0 <= i < n) (e%:num / (2 ^ i.+1)%:R)%:E) -->
    e%:num%:E.
  rewrite ereal_limD //.
  - by rewrite lee_add2l // (cvg_lim _ cvggeo).
  - exact: is_cvg_ereal_nneg_series.
  - by apply: is_cvg_ereal_nneg_series => ?; rewrite lee_fin divr_ge0.
  - by rewrite (cvg_lim _ cvggeo) //= fin_num_adde_def.
rewrite (_ : (fun n => _) = EFin \o
    (fun n => \sum_(0 <= i < n) (e%:num / (2 ^ (i + 1))%:R))%R); last first.
  rewrite funeqE => n /=; rewrite (@big_morph _ _ EFin 0 adde)//.
  by under [in RHS]eq_bigr do rewrite addn1.
apply: cvg_comp; last apply cvg_refl.
have := cvg_geometric_series_half e%:num O.
by rewrite expr0 divr1; apply: cvg_trans.
Unshelve. all: by end_near. Qed.

Section measurable_cover.
Variable T : semiRingOfSetsType.
Implicit Types (X : set T) (F : (set T)^nat).

Definition measurable_cover X := [set F : (set T)^nat |
  (forall i, measurable (F i)) /\ X `<=` \bigcup_k (F k)].

Lemma cover_measurable X F : measurable_cover X F -> forall k, measurable (F k).
Proof. by move=> + k; rewrite /measurable_cover => -[] /(_ k). Qed.

Lemma cover_subset X F : measurable_cover X F -> X `<=` \bigcup_k (F k).
Proof. by case. Qed.

End measurable_cover.

Section measure_extension.
Variables (R : realType) (T : semiRingOfSetsType).
Variable mu : set T -> \bar R.
Hypothesis measure0 : mu set0 = 0.
Hypothesis measure_ge0 : forall X, mu X >= 0.
Hint Resolve measure_ge0 measure0 : core.

Definition mu_ext (X : set T) : \bar R :=
  ereal_inf [set \sum_(i <oo) mu (A i) | A in measurable_cover X].

Lemma le_mu_ext : {homo mu_ext : A B / A `<=` B >-> A <= B}.
Proof.
move=> A B AB; apply/le_ereal_inf => x [B' [mB' BB']].
by move=> <-{x}; exists B' => //; split => //; apply: subset_trans AB BB'.
Qed.

Lemma mu_ext_ge0 A : 0 <= mu_ext A.
Proof.
apply: lb_ereal_inf => x [B [mB AB] <-{x}]; rewrite ereal_lim_ge //=.
  by apply: is_cvg_ereal_nneg_series => // n _; exact: measure_ge0.
by near=> n; rewrite sume_ge0 // => i _; apply: measure_ge0.
Unshelve. all: by end_near. Qed.

Lemma mu_ext0 : mu_ext set0 = 0.
Proof.
apply/eqP; rewrite eq_le; apply/andP; split; last exact/mu_ext_ge0.
rewrite /mu_ext; apply ereal_inf_lb; exists (fun _ => set0); first by split.
by apply: (@lim_near_cst _ _ _ _ _ 0) => //; near=> n => /=; rewrite big1.
Unshelve. all: by end_near. Qed.

Lemma measurable_uncurry (G : ((set T)^nat)^nat) (x : nat * nat) :
  measurable (G x.1 x.2) -> measurable (uncurry G x).
Proof. by case: x. Qed.

Lemma mu_ext_sigma_subadditive : sigma_subadditive mu_ext.
Proof.
move=> A; have [[i ioo]|] := pselect (exists i, mu_ext (A i) = +oo).
  rewrite (ereal_nneg_series_pinfty _ _ ioo)// ?lee_pinfty// => n _.
  exact: mu_ext_ge0.
rewrite -forallNE => Aoo; apply: lee_adde => e.
rewrite (le_trans _ (epsilon_trick _ _ _))//; last first.
  by move=> n; apply: mu_ext_ge0.
pose P n (B : (set T)^nat) := measurable_cover (A n) B /\
  \sum_(k <oo) mu (B k) <= mu_ext (A n) + (e%:num / (2 ^ n.+1)%:R)%:E.
have [G PG] : {G : ((set T)^nat)^nat & forall n, P n (G n)}.
  apply: (@choice _ _ P) => n; rewrite /P /mu_ext.
  set S := (X in ereal_inf X); move infS : (ereal_inf S) => iS.
  case: iS infS => [r Sr|Soo|Soo].
  - have en1 : (0 < e%:num / (2 ^ n.+1)%:R)%R.
      by rewrite divr_gt0 // ltr0n expn_gt0.
    have /(lb_ereal_inf_adherent (PosNum en1)) : ereal_inf S \is a fin_num.
      by rewrite Sr.
    move=> [x [[B [mB AnB muBx]] xS]].
    exists B; split => //; rewrite muBx -Sr; apply/ltW.
    by rewrite (lt_le_trans xS) // lee_add2l //= lee_fin ler_pmul.
  - by have := Aoo n; rewrite /mu_ext Soo.
  - suff : lbound S 0 by move/lb_ereal_inf; rewrite Soo.
    move=> /= _ [B [mB AnB] <-].
    by apply: ereal_nneg_series_lim_ge0 => ? _; exact: measure_ge0.
have muG_ge0 x : 0 <= (mu \o uncurry G) x by exact/measure_ge0.
apply (@le_trans _ _ (\esum_(i in setT) (mu \o uncurry G) i)).
  rewrite /mu_ext; apply: ereal_inf_lb => /=.
  have /card_esym/ppcard_eqP[f] := card_nat2.
  exists (uncurry G \o f).
    split => [i|]; first exact/measurable_uncurry/(PG (f i).1).1.1.
    apply: (@subset_trans _  (\bigcup_n \bigcup_k G n k)) => [t [i _]|].
      by move=> /(cover_subset (PG i).1) -[j _ ?]; exists i => //; exists j.
    move=> t [i _ [j _ Bijt]]; exists (f^-1%FUN (i, j)) => //=.
    by rewrite invK ?inE.
  rewrite -(esum_pred_image (mu \o uncurry G) _ xpredT) ?[fun=> _]set_true//.
  congr esum.
  by rewrite -[RHS](image_eq f)predeqE=> -[a b]/=; split=> -[n _ <-]; exists n.
rewrite (_ : esum _ _ = \sum_(i <oo) \sum_(j <oo ) mu (G i j)); last first.
  pose J : nat -> set (nat * nat) := fun i => [set (i, j) | j in setT].
  rewrite (_ : setT = \bigcup_k J k); last first.
    by rewrite predeqE => -[a b]; split => // _; exists a => //; exists b.
  rewrite esum_bigcupT /=; last 2 first.
    - apply/trivIsetP => i j _ _ ij.
      rewrite predeqE => -[n m] /=; split => //= -[] [_] _ [<-{n} _].
      by move=> [m' _] [] /esym/eqP; rewrite (negbTE ij).
    - by move=> /= [n m]; apply/measure_ge0; exact: (cover_measurable (PG n).1).
  rewrite (_ : setT = id @` xpredT); last first.
    by rewrite image_id funeqE => x; rewrite trueE.
  rewrite esum_pred_image //; last by move=> n _; exact: esum_ge0.
  apply: eq_ereal_pseries => /= j.
  rewrite -(esum_pred_image (mu \o uncurry G) (pair j) predT)//=; last first.
    by move=> ? ? _ _; exact: (@can_inj _ _ _ snd).
  by congr esum; rewrite predeqE => -[a b]; split; move=> [i _ <-]; exists i.
apply lee_lim.
- apply: is_cvg_ereal_nneg_series => n _.
  by apply: ereal_nneg_series_lim_ge0 => m _; exact: (muG_ge0 (n, m)).
- apply: is_cvg_ereal_nneg_series => n _; apply: adde_ge0 => //.
  exact: mu_ext_ge0.
- by near=> n; apply: lee_sum => i _; exact: (PG i).2.
Unshelve. all: by end_near. Qed.

End measure_extension.

Section measure_extension.
Variables (R : realType) (T : semiRingOfSetsType).
Variable mu : {additive_measure set T -> \bar R}.

Canonical outer_measure_of_measure (R : realType) (T : semiRingOfSetsType)
  (mu : {additive_measure set T -> \bar R}) : {outer_measure set T -> \bar R} :=
    OuterMeasure (OuterMeasure.Axioms
      (mu_ext0 (measure0 mu) (measure_ge0 mu))
      (mu_ext_ge0 (measure_ge0 mu))
      (le_mu_ext mu)
      (mu_ext_sigma_subadditive (measure_ge0 mu))).

End measure_extension.

Section g_salgebra_measure_unique_trace.
Variables (R : realType) (T : measurableType) (G : set (set T)).
Variables (D : set T) (mD : measurable D).
Let H := [set X | G X /\ X `<=` D] (* "trace" of G wrt D *).
Hypotheses (Hm : H `<=` measurable) (setIH : setI_closed H) (HD : H D).
Variables m1 m2 : {measure set T -> \bar R}.
Hypotheses (m1m2 : forall A, H A -> m1 A = m2 A) (m1oo : (m1 D < +oo)%E).

Lemma g_salgebra_measure_unique_trace :
  (forall X, (<<s D, H >>) X -> X `<=` D) -> forall X, <<s D, H >> X ->
  m1 X = m2 X.
Proof.
move=> sDHD; set E := [set A | [/\ measurable A, m1 A = m2 A & A `<=` D] ].
have HE : H `<=` E.
  by move=> X HX; rewrite /E /=; split; [exact: Hm|exact: m1m2|case: HX].
have setDE : setD_closed E.
  move=> A B BA [mA m1m2A AD] [mB m1m2B BD]; split; first exact: measurableD.
  - rewrite measureD//; last first.
      by rewrite (le_lt_trans _ m1oo)//; apply: le_measure => //=; rewrite inE.
    rewrite setIidr// m1m2A m1m2B measureD// ?setIidr//.
    by rewrite (le_lt_trans _ m1oo)// -m1m2A; apply: le_measure => //;
      rewrite inE.
  - by rewrite setDE; apply: subIset; left.
have ndE : ndseq_closed E.
  move=> A ndA EA; split; have mA n : measurable (A n) by have [] := EA n.
  - exact: bigcupT_measurable.
  - transitivity (lim (m1 \o A)).
      by apply/esym/cvg_lim=>//; exact/(cvg_mu_inc mA _ ndA)/bigcupT_measurable.
    transitivity (lim (m2 \o A)).
      by congr (lim _); rewrite funeqE => n; have [] := EA n.
    by apply/cvg_lim => //; exact/(cvg_mu_inc mA _ ndA)/bigcupT_measurable.
  - by apply: bigcup_sub => n; have [] := EA n.
have sDHE : <<s D, H >> `<=` E.
  by apply: monotone_class_subset => //; split => //; [move=> A []|exact/HE].
by move=> X /sDHE[mX ?] _.
Qed.

End g_salgebra_measure_unique_trace.

Section g_salgebra_measure_unique.
Variables (R : realType) (T : measurableType) (G : set (set T)).
Hypothesis Gm : G `<=` measurable.
Variable g : (set T)^nat.
Hypotheses (Gg : forall i, G (g i)).
Hypothesis g_cover : \bigcup_k (g k) = setT.
Variables m1 m2 : {measure set T -> \bar R}.

Lemma g_salgebra_measure_unique_cover :
  (forall n A, <<s G >> A -> m1 (g n `&` A) = m2 (g n `&` A)) ->
  forall A, <<s G >> A -> m1 A = m2 A.
Proof.
move=> sGm1m2; pose g' k := \bigcup_(i in `I_k) g i.
have sGm := smallest_sub (@sigma_algebra_measurable T) Gm.
have Gg' i : <<s G >> (g' i).
  apply: (@fin_bigcup_measurable [the ringOfSetsType of g_measurable G]) => //.
  by move=> n _; apply: sub_sigma_algebra.
have sG'm1m2 n A : <<s G >> A -> m1 (g' n `&` A) = m2 (g' n `&` A).
  move=> sGA; rewrite setI_bigcupl bigcup_mkord.
  elim: n => [|n IHn] in A sGA *; rewrite (big_ord0, big_ord_recr) ?measure0//=.
  have sGgA i : <<s G >> (g i `&` A).
    apply: (@measurableI [the semiRingOfSetsType of g_measurable G]) => //.
    exact: sub_sigma_algebra.
  apply: eq_measureU; rewrite ?sGm1m2 ?IHn//; last first.
  - by rewrite -big_distrl -setIA big_distrl/= IHn// setICA setIid//.
  - exact/sGm.
  - by apply: bigsetU_measurable => i _; apply/sGm.
have g'_cover : \bigcup_k (g' k) = setT.
  by rewrite -subTset -g_cover => x [k _ gx]; exists k.+1 => //; exists k => /=.
have nd_g' : nondecreasing_seq g'.
  move=> m n lemn; rewrite subsetEset => x [k km gx]; exists k => //=.
  by rewrite (leq_trans _ lemn).
move=> A gA.
have -> : A = \bigcup_n (g' n `&` A) by rewrite -setI_bigcupl g'_cover setTI.
transitivity (lim (fun n => m1 (g' n `&` A))).
  apply/esym/cvg_lim => //; apply: cvg_mu_inc => //.
  - by move=> n; apply: measurableI; apply/sGm.
  - by apply: bigcupT_measurable => k; apply: measurableI; exact/sGm.
  - by move=> ? ? ?; apply/subsetPset; apply: setSI; exact/subsetPset/nd_g'.
transitivity (lim (fun n => m2 (g' n `&` A))).
  by congr (lim _); rewrite funeqE => x; apply: sG'm1m2 => //; exact/sGm.
apply/cvg_lim => //; apply: cvg_mu_inc => //.
- by move=> k; apply: measurableI => //; exact/sGm.
- by apply: bigcupT_measurable => k; apply: measurableI; exact/sGm.
- by move=> a b ab; apply/subsetPset; apply: setSI; exact/subsetPset/nd_g'.
Qed.

Hypothesis setIG : setI_closed G.
Hypothesis m1m2 : forall A, G A -> m1 A = m2 A.
Hypothesis m1goo : forall k, (m1 (g k) < +oo)%E.

Lemma g_salgebra_measure_unique : forall E, <<s G >> E -> m1 E = m2 E.
Proof.
pose G_ n := [set X | G X /\ X `<=` g n]. (* "trace" *)
have G_E n : G_ n = [set g n `&` C | C in G].
  rewrite eqEsubset; split.
    by move=> X [GX Xgn] /=; exists X => //; rewrite setIidr.
  by rewrite /G_ => X [Y GY <-{X}]; split; [exact: setIG|apply: subIset; left].
have gIsGE n : [set g n `&` A | A in <<s G >>] =
               <<s g n, preimage_class (g n) id G >>.
  rewrite sigma_algebra_preimage_classE eqEsubset; split.
    by move=> _ /= [Y sGY <-]; exists Y => //; rewrite preimage_id setIC.
  by move=> _ [Y mY <-] /=; exists Y => //; rewrite preimage_id setIC.
have preimg_gGE n : preimage_class (g n) id G = G_ n.
  rewrite eqEsubset; split => [_ [Y GY <-]|].
    by rewrite preimage_id G_E /=; exists Y => //; rewrite setIC.
  by move=> X [GX Xgn]; exists X => //; rewrite preimage_id setIidr.
apply: g_salgebra_measure_unique_cover => //.
move=> n A sGA; apply: (@g_salgebra_measure_unique_trace _ _ G (g n)) => //.
- exact: Gm.
- by move=> ? [? _]; exact/Gm.
- by move=> ? ? [? ?] [? ?]; split; [exact: setIG|apply: subIset; tauto].
- by split.
- by move=> ? [? ?]; exact: m1m2.
- move=> X; rewrite -/(G_ n) -preimg_gGE -gIsGE.
  by case=> B sGB <-{X}; apply: subIset; left.
- by rewrite -/(G_ n) -preimg_gGE -gIsGE; exists A.
Qed.

End g_salgebra_measure_unique.

Section measure_unique.
Variables (R : realType) (T : measurableType) (G : set (set T)).
Variable g : (set T)^nat.
Hypotheses (mG : measurable = <<s G >>) (setIG : setI_closed G).
Hypothesis Gg : forall i, G (g i).
Hypothesis g_cover : \bigcup_k (g k) = setT.
Variables m1 m2 : {measure set T -> \bar R}.
Hypothesis m1m2 : forall A, G A -> m1 A = m2 A.
Hypothesis m1goo : forall k, (m1 (g k) < +oo)%E.

Lemma measure_unique A : measurable A -> m1 A = m2 A.
Proof.
move=> mA; apply: (@g_salgebra_measure_unique _ _ G); rewrite -?mG//.
by rewrite mG; exact: sub_sigma_algebra.
Qed.

End measure_unique.
Arguments measure_unique {R T} G g.

Lemma measurable_mu_extE (R : realType) (T : semiRingOfSetsType)
    (mu : {additive_measure set T -> \bar R}) X :
  sigma_sub_additive mu ->
  measurable X -> mu_ext mu X = mu X.
Proof.
move=> muS mX; apply/eqP; rewrite eq_le; apply/andP; split.
  apply ereal_inf_lb; exists (fun n => if n is 0%N then X else set0).
    by split=> [[]// _|t Xt]; exists 0%N.
  apply/cvg_lim => //; rewrite -cvg_shiftS.
  rewrite (_ : [sequence _]_n = cst (mu X)); first exact: cvg_cst.
  by rewrite funeqE => n /=; rewrite big_nat_recl//= big1 ?adde0.
apply/lb_ereal_inf => x [A [mA XA] <-{x}].
have XUA : X = \bigcup_n (X `&` A n).
  rewrite predeqE => t; split => [Xt|[i _ []//]].
  by have [i _ Ait] := XA _ Xt; exists i; split.
apply: (@le_trans _ _ (\sum_(i <oo) mu (X `&` A i))).
  by rewrite muS//= -?XUA => // i; apply: measurableI.
apply lee_lim.
- by apply: is_cvg_ereal_nneg_series => n _; exact/measure_ge0.
- by apply: is_cvg_ereal_nneg_series => n _; exact/measure_ge0.
- near=> n; apply: lee_sum => i  _. apply: le_measure => // /[!inE]//=.
  exact: measurableI.
Unshelve. all: by end_near. Qed.

Section Rmu_ext.
Import SetRing.

Lemma Rmu_ext (R : realType) (T : semiRingOfSetsType) (rT := @SetRing.ring T)
    (mu : {additive_measure set T -> \bar R}) :
  mu_ext (measure mu) = mu_ext mu.
Proof.
apply/funeqP => /= X; rewrite /mu_ext/=; apply/eqP; rewrite eq_le.
rewrite ?lb_ereal_inf// => _ [F [Fm XS] <-]; rewrite ereal_inf_lb//; last first.
  exists F; first by split=> // i; apply: sub_gen_smallest.
  by rewrite (eq_ereal_pseries (fun=> RmuE _ _)).
pose K := [set: nat] `*`` fun i => [set` decomp (F i)].
have /ppcard_eqP[f] : (K #= [set: nat])%card.
  apply: cardMR_eq_nat => // i.
  by split=> //; apply/set0P; rewrite set_fset_eq0 decompN0.
pose g i := (f^-1%FUN i).2.
exists g; first split.
- move=> k; have [/= _] : K (f^-1%FUN k) by apply: funS.
  by apply: decomp_measurable.
- move=> i /XS [k _]; rewrite -[F k]cover_decomp => -[D /= DFk Di].
  by exists (f (k, D)) => //; rewrite /g invK// inE.
rewrite !ereal_pseries_esum//= /measure ?set_true.
rewrite -[RHS](eq_esum (fun _ _ => esum_fset _))// esum_esum//=.
rewrite (reindex_esum K setT f) => //=; apply: eq_esum => i Ki.
by rewrite /g funK ?inE.
Qed.

End Rmu_ext.

Lemma measurable_Rmu_extE (R : realType) (T : semiRingOfSetsType)
    (rT := @SetRing.ring T) (mu : {additive_measure set T -> \bar R}) X :
  sigma_sub_additive mu ->
  measurable (X : set rT)  -> mu_ext mu X = SetRing.measure mu X.
Proof.
move=> mu_sub Xm/=; rewrite -Rmu_ext/= measurable_mu_extE//.
exact: ring_sigma_sub_additive.
Qed.

Section Hahn_extension.
Variables (R : realType) (T : semiRingOfSetsType).
Variable mu : {additive_measure set T -> \bar R}.
Let M : measurableType := [the measurableType of caratheodory_type (mu_ext mu)].

Hypothesis mu_sub : sigma_sub_additive mu.
Local Notation rT := (@SetRing.ring T).
Let Rmu := SetRing.measure mu.

Lemma sub_caratheodory : <<s @measurable T >> `<=` @measurable M.
Proof.
suff: <<s @measurable rT >> `<=` @measurable M.
   apply: subset_trans;  apply: sub_smallest2r => //=.
   by apply: sub_smallest.
apply: smallest_sub.
  split => //; [by move=> X mX; rewrite setTD; exact: measurableC |
                by move=> u_ mu_; exact: bigcupT_measurable].
move=> A mA; apply le_caratheodory_measurable => // X.
apply lb_ereal_inf => _ [B [mB XB] <-].
rewrite -(eq_ereal_pseries (fun=> SetRing.RmuE _ _))=> //.
have RmB i : measurable (B i : set rT) by exact: sub_gen_smallest.
set BA := eseries (fun n => Rmu (B n `&` A)).
set BNA := eseries (fun n => Rmu (B n `&` ~` A)).
apply (@le_trans _ _ (lim BA + lim BNA)); [apply: lee_add|].
  - rewrite (_ : BA = eseries (fun n => mu_ext mu (B n `&` A))); last first.
      rewrite funeqE => n; apply: eq_bigr => k _.
      by rewrite /= measurable_Rmu_extE //; exact: measurableI.
    apply (@le_trans _ _ (mu_ext mu (\bigcup_k (B k `&` A)))).
      by apply le_mu_ext; rewrite -setI_bigcupl; apply setISS.
    exact: outer_measure_sigma_subadditive.
  - rewrite (_ : BNA = eseries (fun n => mu_ext mu (B n `\` A))); last first.
      rewrite funeqE => n; apply eq_bigr => k _.
      rewrite /= measurable_Rmu_extE //; exact: measurableD.
    apply (@le_trans _ _ (mu_ext mu (\bigcup_k (B k `\` A)))).
      by apply le_mu_ext; rewrite -setI_bigcupl; apply setISS.
    exact: outer_measure_sigma_subadditive.
have ? : cvg BNA.
  apply/is_cvg_ereal_nneg_series => n _.
  by rewrite -setDE; apply: measure_ge0 => //; apply: measurableD.
have ? : cvg BA.
  apply/is_cvg_ereal_nneg_series => n _.
  by apply: measure_ge0 => //; apply: measurableI.
have ? : cvg (eseries (Rmu \o B)).
  by apply/is_cvg_ereal_nneg_series => n _; exact: measure_ge0.
have [def|] := boolP (adde_def (lim BA) (lim BNA)); last first.
  rewrite /adde_def negb_and !negbK=> /orP[/andP[BAoo BNAoo]|/andP[BAoo BNAoo]].
  - suff -> : lim (eseries (Rmu \o B)) = +oo by rewrite lee_pinfty.
    apply/eqP; rewrite -lee_pinfty_eq -(eqP BAoo); apply/lee_lim => //.
    near=> n; apply: lee_sum => m _; apply: le_measure; rewrite /mkset; by
      [rewrite inE; exact: measurableI | rewrite inE | apply: subIset; left].
  - suff -> : lim (eseries (Rmu \o B)) = +oo by rewrite lee_pinfty.
    apply/eqP; rewrite -lee_pinfty_eq -(eqP BNAoo); apply/lee_lim => //.
    by near=> n; apply: lee_sum => m _; rewrite -setDE; apply: le_measure;
       rewrite /mkset ?inE//; apply: measurableD.
rewrite -ereal_limD // (_ : (fun _ => _) =
    eseries (fun k => Rmu (B k `&` A) + Rmu (B k `&` ~` A))); last first.
  by rewrite funeqE => n; rewrite -big_split /=; apply eq_bigr.
apply/lee_lim => //.
  apply/is_cvg_ereal_nneg_series => // n _; apply/adde_ge0.
  by apply: measure_ge0 => //; apply: measurableI.
  by rewrite -setDE; apply: measure_ge0; apply: measurableD.
near=> n; apply: lee_sum => i _; rewrite -measure_semi_additive2.
- apply: le_measure; rewrite /mkset ?inE//; [|by rewrite -setIUr setUCr setIT].
  by apply: measurableU; [exact:measurableI|rewrite -setDE; exact:measurableD].
- exact: measurableI.
- by rewrite -setDE; exact: measurableD.
- by apply: measurableU; [exact:measurableI|rewrite -setDE; exact:measurableD].
- by rewrite setIACA setICr setI0.
Unshelve. all: by end_near. Qed.

Let I : measurableType := g_measurableType (@measurable T).

Definition Hahn_ext : set I -> \bar R := mu_ext mu.

Local Lemma Hahn_ext0 : Hahn_ext set0 = 0.
Proof. exact: mu_ext0. Qed.

Local Lemma Hahn_ext_ge0 (A : set I) : 0 <= Hahn_ext A.
Proof. exact: mu_ext_ge0. Qed.

Local Lemma Hahn_ext_sigma_additive : semi_sigma_additive Hahn_ext.
Proof.
move=> F mF tF mUF; rewrite /Hahn_ext.
apply: caratheodory_measure_sigma_additive => //; last first.
  exact: sub_caratheodory.
by move=> i; exact: (sub_caratheodory (mF i)).
Qed.

Canonical Hahn_ext_measure : {measure set I -> \bar R} :=
  @Measure.Pack _ _ _ Hahn_ext (Measure.Axioms
    Hahn_ext0 Hahn_ext_ge0 Hahn_ext_sigma_additive).

Lemma Hahn_ext_sigma_finite : @sigma_finite _ T setT mu ->
  @sigma_finite _ I setT Hahn_ext.
Proof.
move=> -[S setTS mS]; exists S => //; move=> i; split.
  by have := (mS i).1; exact: sub_sigma_algebra.
by rewrite /Hahn_ext /= measurable_mu_extE //;
  [exact: (mS i).2 | exact: (mS i).1].
Qed.

Lemma Hahn_ext_unique : @sigma_finite _ T setT mu ->
  forall mu' : {measure set I -> \bar R},
    (forall X, @measurable T X -> mu X = mu' X) ->
    forall X, @measurable I X -> Hahn_ext X = mu' X.
Proof.
move=> [F TF /all_and2[Fm muF]] mu' mu'mu X mX.
apply: (@measure_unique _ I (@measurable T) F) => //=.
- by move=> A B Am Bm; apply: measurableI.
- by move=> A Am; rewrite /Hahn_ext/= measurable_mu_extE// mu'mu.
- by move=> k; rewrite /Hahn_ext/= measurable_mu_extE.
Qed.
End Hahn_extension.

Lemma caratheodory_measurable_mu_ext (R : realType) (T : measurableType)
    (mu : {measure set T -> \bar R}) A :
  measurable A -> (mu_ext mu).-measurable A.
Proof.
by move=> mu' Am; apply: sub_caratheodory => //;
  [exact: measure_sigma_sub_additive|exact: sub_sigma_algebra].
Qed.

Definition preimage_classes (T1 T2 : measurableType) (T : Type)
    (f1 : T -> T1) (f2 : T -> T2)  :=
  <<s preimage_class setT f1 measurable `|`
      preimage_class setT f2 measurable >>.

Section product_lemma.
Variables (T1 T2 : measurableType) (T : pointedType).
Variables (f1 : T -> T1) (f2 : T -> T2) (T3 : Type) (g : T3 -> T).

Lemma preimage_classes_comp : preimage_classes (f1 \o g) (f2 \o g) =
                              preimage_class setT g (preimage_classes f1 f2).
Proof.
rewrite {1}/preimage_classes -sigma_algebra_preimage_classE; congr (<<s _ >>).
rewrite predeqE => C; split.
- move=> [[A mA <-{C}]|[A mA <-{C}]].
  + by exists (f1 @^-1` A) => //; left; exists A => //; rewrite setTI.
  + by exists (f2 @^-1` A) => //; right; exists A => //; rewrite setTI.
- move=> [A [[B mB <-{A} <-{C}]|[B mB <-{A} <-{C}]]].
  + by left; rewrite !setTI; exists B => //; rewrite setTI.
  + by right; rewrite !setTI; exists B => //; rewrite setTI.
Qed.

End product_lemma.

Definition prod_measurable (T1 T2 : measurableType) := (T1 * T2)%type.

Section product_salgebra_instance.
Variables (T1 T2 : measurableType).
Let f1 := @fst T1 T2.
Let f2 := @snd T1 T2.

Let prod_salgebra_set0 : preimage_classes f1 f2 (set0 : set (T1 * T2)).
Proof. exact: sigma_algebra0. Qed.

Let prod_salgebra_setC A : preimage_classes f1 f2 A ->
  preimage_classes f1 f2 (~` A).
Proof. exact: sigma_algebraC. Qed.

Let prod_salgebra_bigcup (F : _^nat) : (forall i, preimage_classes f1 f2 (F i))
  -> preimage_classes f1 f2 (\bigcup_i (F i)).
Proof. exact: sigma_algebra_bigcup. Qed.

HB.instance Definition prod_salgebra_mixin :=
  @isMeasurable.Build (T1 * T2)%type (Pointed.class _) (preimage_classes f1 f2)
    (prod_salgebra_set0) (prod_salgebra_setC) (prod_salgebra_bigcup).

Definition prod_measurableType := [the measurableType of prod_measurable T1 T2].

End product_salgebra_instance.

Lemma measurableM (T1 T2 : measurableType) (A : set T1) (B : set T2) :
  measurable A -> measurable B -> measurable (A `*` B).
Proof.
move=> mA mB.
have -> : A `*` B = (A `*` setT) `&` (setT `*` B) :> set (T1 * T2).
  by rewrite -{1}(setIT A) -{1}(setTI B) setMI.
rewrite setMT setTM; apply: measurableI.
- by apply: sub_sigma_algebra; left; exists A => //; rewrite setTI.
- by apply: sub_sigma_algebra; right; exists B => //; rewrite setTI.
Qed.

Section product_salgebra_measurableType.
Variables (T1 T2 : measurableType).
Let M1 := @measurable T1.
Let M2 := @measurable T2.
Let M1xM2 := [set A `*` B | A in M1 & B in M2].

Lemma measurable_prod_measurableType :
  @measurable (prod_measurableType T1 T2) = <<s M1xM2 >>.
Proof.
rewrite eqEsubset; split.
  apply: smallest_sub; first exact: smallest_sigma_algebra.
  rewrite subUset; split.
  - have /subset_trans : preimage_class setT fst M1 `<=` M1xM2.
      by move=> _ [X MX <-]; exists X=> //; exists setT; rewrite /M2 // setIC//.
    by apply; exact: sub_sigma_algebra.
  - have /subset_trans : preimage_class setT snd M2 `<=` M1xM2.
      by move=> _ [Y MY <-]; exists setT; rewrite /M1 //; exists Y.
    by apply; exact: sub_sigma_algebra.
apply: smallest_sub; first exact: smallest_sigma_algebra.
by move=> _ [A MA] [B MB] <-; apply: measurableM=> //; exact: sub_sigma_algebra.
Qed.

End product_salgebra_measurableType.

Section product_salgebra_g_measurableTypeR.
Variables (T1 : measurableType) (T2 : pointedType) (C2 : set (set T2)).
Hypothesis (setTC2 : setT `<=` C2).

(* NB: useful? *)
Lemma measurable_prod_g_measurableTypeR :
  @measurable (prod_measurableType T1 (g_measurableType C2))
  = <<s [set A `*` B | A in measurable & B in C2] >>.
Proof.
rewrite measurable_prod_measurableType //; congr (<<s _ >>).
rewrite predeqE => X; split=> [[A mA] [B mB] <-{X}|[A C1A] [B C2B] <-{X}].
  by exists A => //; exists B => //; exact: setTC2.
by exists A => //; exists B => //; exact: sub_sigma_algebra.
Qed.

End product_salgebra_g_measurableTypeR.

Section product_salgebra_g_measurableType.
Variables (T1 T2 : pointedType) (C1 : set (set T1)) (C2 : set (set T2)).
Hypotheses (setTC1 : setT `<=` C1) (setTC2 : setT `<=` C2).

Lemma measurable_prod_g_measurableType :
  @measurable (prod_measurableType (g_measurableType C1) (g_measurableType C2))
  = <<s [set A `*` B | A in C1 & B in C2] >>.
Proof.
rewrite measurable_prod_measurableType //; congr (<<s _ >>).
rewrite predeqE => X; split=> [[A mA] [B mB] <-{X}|[A C1A] [B C2B] <-{X}].
  by exists A; [exact: setTC1|exists B => //; exact: setTC2].
by exists A; [exact: sub_sigma_algebra|exists B => //; exact: sub_sigma_algebra].
Qed.

End product_salgebra_g_measurableType.

Section prod_measurable_fun.
Variables (T T1 T2 : measurableType) (f : T -> prod_measurableType T1 T2).

Lemma prod_measurable_funP : measurable_fun setT f <->
  measurable_fun setT (fst \o f) /\ measurable_fun setT (snd \o f).
Proof.
apply: (@iff_trans _ (preimage_classes (fst \o f) (snd \o f) `<=` measurable)).
- rewrite preimage_classes_comp; split=> [mf A [C HC <-]|f12]; first exact: mf.
  by move=> _ A mA; apply: f12; exists A.
- split => [h|[mf1 mf2]].
    split => _ A mA; apply: h; apply: sub_sigma_algebra;
    by [left; exists A|right; exists A].
  apply: smallest_sub; first exact: sigma_algebra_measurable.
  by rewrite subUset; split=> [|] A [C mC <-]; [exact: mf1|exact: mf2].
Qed.

End prod_measurable_fun.

Section partial_measurable_fun.
Variables (T T1 T2 : measurableType) (f : prod_measurableType T1 T2 -> T).

Lemma measurable_fun_prod1 x :
  measurable_fun setT f -> measurable_fun setT (fun y => f (x, y)).
Proof.
move=> mf; pose pairx := fun y : T2 => (x, y).
have m1pairx : measurable_fun setT (fst \o pairx) by exact/measurable_fun_cst.
have m2pairx : measurable_fun setT (snd \o pairx) by exact/measurable_fun_id.
have : measurable_fun setT pairx by exact/(proj2 (prod_measurable_funP _)).
exact: measurable_fun_comp.
Qed.

Lemma measurable_fun_prod2 y :
  measurable_fun setT f -> measurable_fun setT (fun x => f (x, y)).
Proof.
move=> mf; pose pairy := fun x : T1 => (x, y).
have m1pairy : measurable_fun setT (fst \o pairy) by exact/measurable_fun_id.
have m2pairy : measurable_fun setT (snd \o pairy) by exact/measurable_fun_cst.
have : measurable_fun setT pairy by exact/(proj2 (prod_measurable_funP _)).
exact: measurable_fun_comp.
Qed.

End partial_measurable_fun.
