/***************************************************************************
 *   Copyright (C) 2004-2005 by Andreas Ramm                               *
 *   psychobrain@gmx.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "mateedit.h"

#include <qlabel.h>

#include <kmainwindow.h>
#include <klocale.h>


#include <kapplication.h>
#include <ktextedit.h>
#include <qcolor.h>
#include <kstatusbar.h>
#include <kmenubar.h>
#include <kresolver.h>
#include <ksocketaddress.h>
#include <kpopupmenu.h>
#include <kfiledialog.h>
#include <kaction.h>
#include <qserversocket.h>
#include <qsocket.h>

#include <kmessagebox.h>
#include <qptrlist.h>
#include <qclipboard.h>
#include <qinputdialog.h>
#include <qtimer.h>
#include <kpushbutton.h>

#include <kiconview.h>
#include <kicondialog.h>
#include <klineedit.h>
#include <qlayout.h>
#include <qimage.h>

#include <kspell.h>
#include <ksconfig.h>
#include <kfinddialog.h>
#include <kreplacedialog.h>

#include <kstandarddirs.h>
#include <kmainwindowiface.h>

#include <qradiobutton.h>

#include <userdetails.h>
#include <qpainter.h>
#include <kwin.h>
#include <kuser.h>


#include "message.h"
#ifdef ZEROCONF_MATEEDIT
    #include <dnssd/publicservice.h>
    #include "startdialog.h"
#endif

QColor * colors;
QString * users;
QImage * user_images;
bool server_ready;

#ifdef ZEROCONF_MATEEDIT
MateEditServerSocket::MateEditServerSocket( const QString& sessionName, int backlog, QObject * parent, const char * name, int port)
    : KNetwork::KServerSocket( parent, name ) ,publisher(sessionName,"_mateedit._tcp")
#else
MateEditServerSocket::MateEditServerSocket( const QString& sessionName, int backlog, QObject * parent, const char * name, int port)
    : KNetwork::KServerSocket( parent, name )
#endif
{
    m_client_count = 0;    //No active connections
    server_counter = 0;    //Counter for messages that passed through
    m_available_client_numbers = new QValueList<int>;
    m_available_client_numbers->append(1);
    m_available_client_numbers->append(2);
    m_available_client_numbers->append(3);
    m_available_client_numbers->append(4);
    m_available_client_numbers->append(5);
    m_server_users = new QString[MATEEDIT_MAX_USERS];
    m_server_images = new QImage[MATEEDIT_MAX_USERS];
    setFamily(KNetwork::KResolver::InetFamily);
    setAddress(QString::number(port)); //"0");
    connect(this,SIGNAL(readyAccept()),this,SLOT(newConnection()));
    listen(backlog);
#ifdef ZEROCONF_MATEEDIT
    publisher.setPort(this->localAddress().asInet().port());
    publisher.publish();
#endif

}

MateEditServerSocket::~MateEditServerSocket()
{
    delete m_available_client_numbers;
    delete[] m_server_users;
    delete[] m_server_images;
}

void MateEditServerSocket::newConnection()
{
    ++m_client_count;
    int client_number = m_available_client_numbers->first();

//    qDebug( "client_number: %i; count %i", client_number, client_count );

    m_available_client_numbers->remove( m_available_client_numbers->begin() );
    KNetwork::KStreamSocket* socket= static_cast<KNetwork::KStreamSocket*>(accept());
    MateEditClientSocket * newclient = new MateEditClientSocket( client_number, socket, this );
    m_client_list.append( newclient );
    MateEdit::Message * message = new MateEdit::Message( MateEdit::Message::Init, client_number, server_counter, 0,  ((MateEditMainWindow *) KApplication::kApplication()->mainWidget() )->getText() );
    message->send( newclient->socket() );
    delete message;
    connect( newclient, SIGNAL( removeConnection( int, MateEditClientSocket * ) ), this, SLOT( removeConnection( int, MateEditClientSocket * ) ) );

    return;
}

void MateEditServerSocket::removeConnection( int number, MateEditClientSocket * client )
{
    int index = m_client_list.findRef(client);
    if ( index != -1 )
    {
        m_available_client_numbers->append( number );
        --m_client_count;
        m_server_users[number] = "";
        m_client_list.remove();

        MateEditClientSocket * current_client;
        current_client = ( m_client_list.first() );
        MateEdit::Message * message = new MateEdit::Message( MateEdit::Message::RemoveUser, number, server_counter, 0, "" );
        while ( current_client )
        {
            message->send( current_client->socket() );
            current_client = m_client_list.next();
        }
        delete message;
    }
}

MateEditServerThread::MateEditServerThread(const QString& name, int port)
    :QThread(),
     m_name(name),
     m_port(port)
{}


void MateEditServerThread::run()
{
    m_serverSocket = new MateEditServerSocket( m_name, MATEEDIT_MAX_USERS,  NULL, "server", m_port );
    server_ready = true;
    m_port = m_serverSocket->localAddress().asInet().port();
    while ( 1 )
    {
        sleep( 10 );
    }
}

const unsigned short MateEditServerThread::port() const
{
    return m_port;
}

MateEditClientSocket::MateEditClientSocket( int client_number, KNetwork::KStreamSocket *s, QObject *parent, const char *name ) : QObject( parent, name ), m_socket(s)
{
    m_number = client_number;
    connect( m_socket, SIGNAL( readyRead() ), SLOT( readClient() ) );
    connect( m_socket, SIGNAL( closed() ), SLOT( connectionClosed() ) );
//    setSocket( sock );
}

void MateEditClientSocket::readClient()
{
    static int size = 0;
    MateEditClientSocket * current_client;
    QDataStream stream ( m_socket );
    if (size ==0)
    {
        stream >> size;
    }
    while ( m_socket->bytesAvailable() >= size  && size > 0 )
    {
        char *data = new char[size];
        m_socket->readBlock(data, size);
        QByteArray bytearray;
        bytearray.assign(data,size);
        QBuffer buffer(bytearray);
        MateEdit::Message * message = new MateEdit::Message;
        message->read(&buffer);

        ((MateEditServerSocket*)this->parent())->server_counter++;
        message->setServerState(((MateEditServerSocket*)this->parent())->server_counter);

        //Forward message to all clients including originator

        current_client = ( (MateEditServerSocket*)parent() )->m_client_list.first();
        while ( current_client )
        {
            message->send( current_client->socket() );
            current_client = ( (MateEditServerSocket*)parent() )->m_client_list.next();
        }
        if ( message->type() == MateEdit::Message::InitUserData )
        {
            ((MateEditServerSocket*)parent() )->m_server_users[message->sender()] = message->text();
            ((MateEditServerSocket*)parent() )->m_server_images[message->sender()] = message->image();
            for (int i = 1; i < MATEEDIT_MAX_USERS; i++ )
            {
                MateEdit::Message * initmessage = new MateEdit::Message( MateEdit::Message::InitUserData, i , ((MateEditServerSocket*)parent() )->server_counter, message->clientState(), ((MateEditServerSocket*)parent() )->m_server_users[i] );
                current_client = this;
                initmessage->setImage( ((MateEditServerSocket*)parent() )->m_server_images[i] );
                initmessage->send( current_client->socket() );
                delete initmessage;
            }
        }
        size = 0;
        delete message;
        if ( m_socket->bytesAvailable() >= sizeof (int) )
        {
            stream >> size;
        }
    }
}

MateEditMainWindow::MateEditMainWindow()
    : KMainWindow( 0, i18n("MateEdit") ),
      m_serverThread (NULL)
{
    // set the shell's ui resource file
//    setXMLFile("kdeneteditui.rc");

    users = new QString[MATEEDIT_MAX_USERS];
    users[0] = "Server";
    users[1] = "";
    users[2] = "";
    users[3] = "";
    users[4] = "";
    users[5] = "";

    m_color_selection = 2;

    colors = new QColor[MATEEDIT_MAX_USERS];
    colors[1] = Qt::darkRed;
    colors[2].setNamedColor("DarkGreen");
    colors[3].setNamedColor("MidnightBlue");
    colors[4] = Qt::darkBlue;
    colors[5].setRgb(20,50,50);

    fileNew = KStdAction::openNew(this, SLOT(slotFileNew()), actionCollection());
    fileOpen = KStdAction::open(this, SLOT(slotFileOpen()), actionCollection());
    fileSave = KStdAction::save(this, SLOT(slotFileSave()), actionCollection());
    fileSaveAs = KStdAction::saveAs(this, SLOT(slotFileSaveAs()), actionCollection());
    fileQuit = KStdAction::quit(this, SLOT(slotFileQuit()), actionCollection());
    editCopy = KStdAction::copy(this, SLOT(slotEditCopy()), actionCollection());
    editPaste = KStdAction::paste(this, SLOT(slotEditPaste()), actionCollection());
    editCut = KStdAction::cut(this, SLOT(slotEditCut()), actionCollection());
    editUndo = KStdAction::undo(this, SLOT(slotEditUndo()), actionCollection());
//    editRedo = KStdAction::redo(this, SLOT(slotEditRedo()), actionCollection());
//    spelling = KStdAction::spelling(this, SLOT(slotSpelling()), actionCollection());
//    print = KStdAction::print(this, SLOT(slotPrint()), actionCollection());
//    find = KStdAction::find(this, SLOT(slotFind()), actionCollection());
//    replace = KStdAction::replace(this, SLOT(slotReplace()), actionCollection());


    createGUI();
    this->statusBar();

    QWidget *widget=new QWidget(this);

    QBoxLayout * l = new QVBoxLayout( widget );

    m_chat_list = new KListBox ( widget );

    m_chat_edit = new KLineEdit ( widget );
    connect( (QObject *)m_chat_edit, SIGNAL( returnPressed() ), (QObject *)this, SLOT( sendChatMessage() ) );

    m_splitter = new QSplitter( widget );
    this->setCentralWidget( widget );

    l->addWidget( m_splitter );
    l->addWidget( m_chat_list );
    l->addWidget( m_chat_edit );
    l->setStretchFactor( m_splitter, 80  );
    l->setStretchFactor( m_chat_list, 15  );
    l->setStretchFactor( m_chat_edit, 5  );

    m_textfield = new KTextEdit( m_splitter );
    connect( (QObject *)m_textfield, SIGNAL( modificationChanged(bool) ), (QObject *)this, SLOT( slotTextfieldModified(bool) ) );
    this->setCaption ( i18n("untitled") );

    m_user_view = new KIconView ( m_splitter );
    m_splitter->moveToFirst ( m_textfield );
    m_splitter->setHandleWidth( 1 );

    m_textfield->setMinimumSize( 200, 200 );
    m_splitter->setCollapsible( m_textfield, false );


    m_user_view->setMinimumWidth( 65 );
    m_user_view->setMaximumWidth( 65 );

    user_images = new QImage[MATEEDIT_MAX_USERS];

    this->resize( 640, 440 );



    m_mateedit = new MateEdit::MateEdit ( this, m_textfield, true, true, true, true, true, true, true );
    m_highLighter = m_mateedit->syntaxHighlighter();
//    m_textfield->installEventFilter( this );
    KApplication::kApplication()->installEventFilter( this );


    QTimer::singleShot( 0, this, SLOT(initObject()) );
}

MateEditMainWindow::~MateEditMainWindow()
{
    delete[] colors;
    delete[] users;
    delete[] user_images;
    delete m_serverThread;
}

void MateEditMainWindow::initObject()
{

    QString _template;

    QString serverip;
    unsigned short port=0;

    bool ok = FALSE;

#ifdef ZEROCONF_MATEEDIT
    DNSSD::PublicService test("dummy","_test._udp");
    bool OK= test.publish();
    test.stop();

    if(OK)
    {
        StartDialog* dlg = new StartDialog(this);
        dlg->exec();


        if ( dlg->m_main->hostButton->isChecked() )
        {
            server_ready = false;
            m_serverThread = new MateEditServerThread(dlg->m_main->nameEdit->text());
            m_serverThread->start();
            serverip = "localhost";
        }
        else if ( dlg->m_main->guestButton->isChecked() )
        {
            ServiceItem *itm = static_cast<ServiceItem*>(dlg->m_main->sessions->selectedItem());
            if (itm)
            {
                serverip = itm->service->hostName();
                port = itm->service->port();
            }
            server_ready = true;
        }
        else
        {
            QString text = dlg->m_main->ipEdit->text();
            serverip = text.section(':', 0 , 0);
            port = text.section(':', 1 , 1).toInt();
            server_ready = true;
        }
        dlg->deleteLater();
    }
    else
    {
#endif //#ifdef ZEROCONF_MATEEDIT

        int rtc = KMessageBox::questionYesNo( this, i18n("Are you hosting a session?"), i18n("Host") );

        port=9723;

        if ( rtc == KMessageBox::Yes )
        {
            server_ready = false;
            m_serverThread = new MateEditServerThread("testname", port);
            m_serverThread->start();
            serverip = "localhost";
        }
        else
        {
            QString text = QInputDialog::getText( i18n("MateEdit"), i18n("Enter Host computer name or ip address and port number"), QLineEdit::Normal, "localhost:9723", &ok, this );
            serverip = text.section(':', 0 , 0);
            port = text.section(':', 1 , 1).toInt();
            server_ready = true;
        }
#ifdef ZEROCONF_MATEEDIT
    }
#endif
//    connect( (QObject *)m_mateedit, SIGNAL( readyRead() ), (QObject *)this, SLOT( readServer() ) );
//    connect( (QObject *)m_mateedit, SIGNAL( connectionClosed() ), (QObject *)this, SLOT( connectionClosed() ) );
//    connect( (QObject *)m_mateedit, SIGNAL( connected() ), (QObject *)this, SLOT( connectionConnected() ) );
//    connect( (QObject *)m_mateedit, SIGNAL( error(int) ), (QObject *)this, SLOT( connectionError(int) ) );

    connect( (QObject *)&m_timer, SIGNAL( timeout() ), (QObject *)this, SLOT( connectionPending() ) );

    while ( ! server_ready )
    {
        sleep (1);
    }

    if (!port && m_serverThread)
    {
        port = m_serverThread->port();
//        qDebug(m_serverThread->toInvitation().prettyURL());
    }
    m_chat_list->insertItem( i18n("Welcome to MateEdit V0.2") );
    m_chat_list->insertItem( i18n("MateEdit lets you share your documents with friends all over the world") );
    m_chat_list->insertItem( i18n("If you have any comments, suggestions or problems check the website at http://sourceforge.net/projects/mateedit/") );
    if( m_serverThread )
    {
        m_chat_list->insertItem( i18n("You have opened a server on port number: ") + QString::number(port) );
    }
    m_chat_list->setVScrollBarMode( QScrollView::AlwaysOn );

    m_mateedit->connectToServer( serverip , port );


    m_textfield->setFocus();
}

void MateEditMainWindow::connectionError(int error)
{
    KStatusBar * statusbar = this->statusBar();

    switch( error )
    {
        case KNetwork::KSocketBase::ConnectionRefused:
        {
            statusbar->message( i18n("Connection refused by host") );
        }
        case KNetwork::KSocketBase::LookupFailure:
        {
            statusbar->message( i18n("Connection faild. Host not found") );
        }
/*        case KNetwork::KSocketBase::SocketRead:
        {
            statusbar->message( i18n("Connection failed. Error reading data.") );
        }
*/    }
}

void MateEditMainWindow::connectionConnected()
{
    KStatusBar * statusbar = this->statusBar();

    statusbar->message( i18n("Connected"), 5000 );
}

void MateEditMainWindow::connectionPending()
{
    KStatusBar * statusbar = this->statusBar();

    static QString text = i18n("Connecting to server ");
    text += '.';

    statusbar->message( text );
}

void MateEditMainWindow::connectionClosed()
{
    KMessageBox::information (this, i18n("The Host has closed the session"), i18n("Server shutdown") );
    m_user_view->clear();
    for( int i = 0; i < MATEEDIT_MAX_USERS; i++ )
    {
        users[i] = "";
        m_user_icons[i] = NULL;
    }
}

QString MateEditMainWindow::getText( )
{
    return m_textfield->text();
}

bool MateEditMainWindow::eventFilter( QObject * o, QEvent * e )
{
//    qDebug("event type %i", e->type());
    if( e->type() == QEvent::ContextMenu )
    {
//        qDebug("ContextMenu");
        ((QContextMenuEvent *)e)->accept();
        return TRUE; // eat event
    }
    if( e->type() == QEvent::Drop )
    {
//        qDebug("Drop");
        ((QContextMenuEvent *)e)->accept();
        return TRUE; // eat event
    }
    if ( e->type() == QEvent::MouseButtonPress || e->type() == QEvent::MouseButtonRelease )
    {
        QMouseEvent * m = (QMouseEvent *)e;
        if(m->button() == Qt::LeftButton)
        {
//            qDebug("Mouse left button");
            return false;
//            QWidget::eventFilter( o, e );
        }
        if(m->button() == Qt::MidButton && e->type() == QEvent::MouseButtonRelease)
        {
//            qDebug("Mouse mid button");
            m->accept();
            return TRUE; // eat event
            int rtc;
            int pos_para;
            int pos_index;
            QClipboard *cb = KApplication::kApplication()->clipboard();
                // Copy text from the clipboard (paste)
            QString text = cb->text(QClipboard::Clipboard);
            if ( !text.isNull() )
            {
                int paraFrom;
                int indexFrom;
                int paraTo;
                int indexTo;
                if ( m_textfield->hasSelectedText() )
                {
                    m_textfield->getSelection( &paraFrom, &indexFrom, &paraTo, &indexTo );

                    rtc = m_mateedit->deleteText( m_textfield->selectedText(), paraFrom, indexFrom, paraTo, indexTo );
                    undo_stack.push(rtc);

                    m_textfield->removeSelectedText();  //remove here so we get a correct cursor position

                    m_textfield->getCursorPosition( &pos_para, &pos_index );
                }

                text.replace(QChar ('\r') ,QChar('\n'));

                int start;
                int end;
                int i;
                start = text.find('\n');
                for ( i = 1, end = text.find('\n', start + 1 ); end != -1; i++, end = text.find('\n', start + 1) )
                {
                    start = end;
                }
                if ( start == -1 )
                {
                    paraTo = pos_para;
                    indexTo = pos_index + text.length();
                }
                else
                {
                    int newlines = text.contains('\r');
                    newlines += text.contains('\n');
                    paraTo = pos_para + newlines;
                    indexTo = text.length() - start - 1;
                }
                rtc = m_mateedit->insertText( text, pos_para, pos_index, paraTo, indexTo );
                undo_stack.push(rtc);

                m_textfield->insert(text);
                m->accept();
                return TRUE; // eat event
            }
        }
//        qDebug("Mouse other button");
        m->accept();
        return TRUE; // eat event
    }
    if ( e->type() == QEvent::MouseButtonRelease )
    {
        QMouseEvent * m = (QMouseEvent *)e;
        m->accept();
        return TRUE; // eat event
    }
    if ( e->type() == QEvent::KeyPress )
    {
        QKeyEvent * k = (QKeyEvent *)e;
        int key = k->key();
        if ( (QObject *)m_textfield == o )
        {
            int pos_para;
            int pos_index;
            int rtc;
            m_textfield->getCursorPosition( &pos_para, &pos_index );
            if ( key == Qt::Key_Delete )
            {
                int paraFrom;
                int indexFrom;
                int paraTo;
                int indexTo;
                if ( m_textfield->hasSelectedText() )
                {
                    m_textfield->getSelection( &paraFrom, &indexFrom, &paraTo, &indexTo );

                    rtc = m_mateedit->deleteText( m_textfield->selectedText(), paraFrom, indexFrom, paraTo, indexTo );
                    undo_stack.push(rtc);

                    m_textfield->removeSelectedText();  //remove here so we get a correct cursor position
                }
                else
                {
                    paraFrom = pos_para;
                    indexFrom = pos_index;
                    m_textfield->moveCursor( QTextEdit::MoveForward, false );
                    m_textfield->getCursorPosition( &pos_para, &pos_index );
                    paraTo = pos_para;
                    indexTo = pos_index;
                    m_textfield->moveCursor( QTextEdit::MoveBackward, false );
                    QChar ch = m_textfield->text(paraFrom)[indexFrom];
                    rtc = m_mateedit->deleteText( ch, paraFrom, indexFrom, paraTo, indexTo );
                    undo_stack.push(rtc);

                    m_textfield->doKeyboardAction( QTextEdit::ActionDelete );

                }
                k->accept();
                return TRUE; // eat event
            }
            else if ( key == Qt::Key_Backspace )
            {
                if ( ( pos_index > 0 || pos_para > 0 ) || ( m_textfield->hasSelectedText() ) )
                {
                    int paraFrom;
                    int indexFrom;
                    int paraTo;
                    int indexTo;
                    if ( m_textfield->hasSelectedText() )
                    {
                        m_textfield->getSelection( &paraFrom, &indexFrom, &paraTo, &indexTo );

                        rtc = m_mateedit->deleteText( m_textfield->selectedText(), paraFrom, indexFrom, paraTo, indexTo );
                        undo_stack.push(rtc);

                        m_textfield->removeSelectedText();  //remove here so we get a correct cursor position
                    }
                    else
                    {
                        paraTo = pos_para;
                        indexTo = pos_index;
                        m_textfield->moveCursor( QTextEdit::MoveBackward, false );
                        m_textfield->getCursorPosition( &pos_para, &pos_index );
                        paraFrom = pos_para;
                        indexFrom = pos_index;
                        m_textfield->moveCursor( QTextEdit::MoveForward, false );

                        QChar ch = m_textfield->text(paraFrom)[indexFrom];
                        rtc = m_mateedit->deleteText(  ch, paraFrom, indexFrom, paraTo, indexTo );
                        undo_stack.push(rtc);

                        m_textfield->doKeyboardAction( QTextEdit::ActionBackspace );
                    }
                }
                k->accept();
                return TRUE; // eat event
            }
            else if ( ( key == Qt::Key_Home ) || ( key == Qt::Key_End ) || ( key == Qt::Key_Left ) || ( key == Qt::Key_Up ) || ( key == Qt::Key_Right ) || ( key == Qt::Key_Down ) )
            {
                return FALSE;
            }
            else if ( ( key == Qt::Key_V ) && ( k->state() == Qt::ControlButton ) )
            {
                QClipboard *cb = KApplication::kApplication()->clipboard();
                // Copy text from the clipboard (paste)
                QString text = cb->text(QClipboard::Clipboard);
                if ( !text.isNull() )
                {
                    int paraFrom;
                    int indexFrom;
                    int paraTo;
                    int indexTo;
                    if ( m_textfield->hasSelectedText() )
                    {
                        m_textfield->getSelection( &paraFrom, &indexFrom, &paraTo, &indexTo );

                        rtc = m_mateedit->deleteText( m_textfield->selectedText(), paraFrom, indexFrom, paraTo, indexTo );
                        undo_stack.push(rtc);

                        m_textfield->removeSelectedText();  //remove here so we get a correct cursor position

                        m_textfield->getCursorPosition( &pos_para, &pos_index );
                    }

                    text.replace(QChar ('\r') ,QChar('\n'));

                    int start;
                    int end;
                    int i;
                    start = text.find('\n');
                    for ( i = 1, end = text.find('\n', start + 1 ); end != -1; i++, end = text.find('\n', start + 1) )
                    {
                        start = end;
                    }
                    if ( start == -1 )
                    {
                        paraTo = pos_para;
                        indexTo = pos_index + text.length();
                    }
                    else
                    {
                        int newlines = text.contains('\r');
                        newlines += text.contains('\n');
                        paraTo = pos_para + newlines;
                        indexTo = text.length() - start - 1;
                    }
                    rtc = m_mateedit->insertText( text, pos_para, pos_index, paraTo, indexTo );
                    undo_stack.push(rtc);

                    m_textfield->insert(text);
                    k->accept();
                    return TRUE; // eat event
                }

            }
            else if ( ( key == Qt::Key_X ) && ( k->state() == Qt::ControlButton ) )
            {
                if ( m_textfield->hasSelectedText() )
                {
                    QClipboard *cb = KApplication::kApplication()->clipboard();
                    QString text = m_textfield->selectedText();
                    cb->setText( text, QClipboard::Clipboard );

                    int paraFrom;
                    int indexFrom;
                    int paraTo;

                    int indexTo;
                    m_textfield->getSelection( &paraFrom, &indexFrom, &paraTo, &indexTo );

                    rtc = m_mateedit->deleteText( text, paraFrom, indexFrom, paraTo, indexTo );
                    undo_stack.push(rtc);

                    m_textfield->removeSelectedText();  //remove here so we get a correct cursor position

                    m_textfield->getCursorPosition( &pos_para, &pos_index );
                }
                return FALSE;

            }
            else if ( ! ( k->text().isEmpty() || (k->state() == Qt::ControlButton) || (k->state() == Qt::AltButton) ) )
            {
                int paraFrom;
                int indexFrom;
                int paraTo;
                int indexTo;
                if ( m_textfield->hasSelectedText() )
                {
                    m_textfield->getSelection( &paraFrom, &indexFrom, &paraTo, &indexTo );

                    rtc = m_mateedit->deleteText( m_textfield->selectedText(), paraFrom, indexFrom, paraTo, indexTo );
                    undo_stack.push(rtc);

                    m_textfield->removeSelectedText();  //remove here so we get a correct cursor position

                    m_textfield->getCursorPosition( &pos_para, &pos_index );
                }

                QString text = k->text();
                text.replace(QChar ('\r') ,QChar('\n'));

                int start;
                int end;
                int i;
                start = text.find('\n');
                for ( i = 1, end = text.find('\n', start + 1 ); end != -1; i++, end = text.find('\n', start + 1) )
                {
                    start = end;
                }
                if ( start == -1 )
                {
                    paraTo = pos_para;
                    indexTo = pos_index + text.length();
                }
                else
                {
                    int newlines = text.contains('\r');
                    newlines += text.contains('\n');
                    paraTo = pos_para + newlines;
                    indexTo = text.length() - start - 1;
                }
                rtc = m_mateedit->insertText( text, pos_para, pos_index, paraTo, indexTo );
                undo_stack.push(rtc);

                m_textfield->insert( text );

                k->accept();
                return TRUE; // eat event
            }
        }
        return QWidget::eventFilter( o, e );
    }
    else
    {
        // standard event processing
        return QWidget::eventFilter( o, e );
    }
}


/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////
void MateEditMainWindow::sendChatMessage()
{
    QString text = m_chat_edit->text();

    if ( ! text.isEmpty() )
    {
        m_mateedit->chat( text );

        m_chat_edit->setText( "" );

        text = users[m_local_pc] + ": " + text;
        m_chat_list->insertItem( text );
        QTimer::singleShot( 10, this, SLOT(slotSetLastChatVisible()) );
    }
}

void MateEditMainWindow::slotFileNew()
{

}

void MateEditMainWindow::slotFileOpen()
{
    m_filename = KFileDialog::getOpenFileName(QString::null,"*.txt|"+i18n("Text files")+"\n*|"+i18n("All files"),this);
    if ( m_filename )
    {
        QFile file( m_filename );
        if (file.open(IO_ReadOnly) )
        {
            QString text;
            QTextStream t( &file );
            text = t.read();
            text.replace(QChar ('\r') ,QChar('\n'));
            int newlines = text.contains('\n');
            int to;
            int pos = text.findRev('\n');
            if ( pos != -1 )
            {
                to = text.length() - pos - 1;
            }
            else
            {
                to = text.length();
            }

//            owners->insert( MessageCoords( 0, 0 ), MessageCoords( newlines, to ), 1, text );
            int rtc;
            rtc = m_mateedit->insertText( text, 0, 0 , newlines, to );
            undo_stack.push(rtc);
            m_textfield->setText( text );
            this->setCaption ( m_filename );
            fileSave->setEnabled( false );
            m_textfield->setModified( false );
            file.close();
        }
        else
        {
            KMessageBox::error(this,i18n("Cannot read from file."));
        }
    }
}

void MateEditMainWindow::slotFileSave()
{

    if ( ! m_filename.isEmpty() && m_textfield->isModified() )
    {
        QFile file( m_filename );
        if (file.open(IO_WriteOnly) )
        {
            QString text;
            text = m_textfield->text();
            QTextStream t( &file );
            t << text;
            statusBar()->message(i18n("Data stored to the file."), 5);
            file.close();
            m_textfield->setModified( false );
        }
        else
        {
            KMessageBox::error(this,i18n("Cannot write to the file."));
        }
    }
    else if ( m_filename.isEmpty () && m_textfield->isModified() )
    {
        this->slotFileSaveAs();
    }
}

void MateEditMainWindow::slotFileSaveAs()
{
    QString text;
    text = m_textfield->text();
    m_filename = KFileDialog::getSaveFileName(QString::null,"*.txt|"+i18n("Text files")+"\n*|"+i18n("All files"),this);
    if ( ! m_filename.isEmpty() )
    {
        if ( m_filename.find('.') < 0 )
        {
            m_filename += ".txt";
        }
        QFile file( m_filename );
        if ( file.exists() )
        {
            if ( KMessageBox::questionYesNo (this,i18n("Do you want to overwrite existing file?"),i18n("File exists"))!=KMessageBox::Yes )
            {
                return;
            }
        }
        if (file.open(IO_WriteOnly) )
        {
            QTextStream t( &file );
            t << text;
            statusBar()->message(i18n("Data stored to the file."), 5);
            file.close();
            m_textfield->setModified( false );
        }
        else
        {
            KMessageBox::error(this,i18n("Cannot write to the file."));
        }
    }
}

void MateEditMainWindow::slotFileQuit()
{
    if ( m_textfield->isModified() && (KMessageBox::questionYesNo (this,i18n("Save Changes?"),i18n("File exists"))==KMessageBox::Yes))
    {
        if ( m_filename.isEmpty() )
        {
            m_filename = KFileDialog::getSaveFileName(QString::null,"*.txt|"+i18n("Text files")+"\n*|"+i18n("All files"),this);
            if ( ! m_filename.isEmpty() )
            {
                if ( m_filename.find('.') < 0 )
                {
                    m_filename += ".txt";
                }
            }
        }
        if ( ! m_filename.isEmpty() )
        {
            QFile file( m_filename );
            if ( file.exists() )
            {
                if (KMessageBox::questionYesNo (this,i18n("Do you want to overwrite existing file?"),i18n("File exists"))!=KMessageBox::Yes)
                {
                    return;
                }
            }
            if (file.open(IO_WriteOnly) )
            {
                QString text;
                text = m_textfield->text();
                QTextStream t( &file );
                t << text;
                statusBar()->message(i18n("Data stored to the file."), 5);
                file.close();
            }
            else
            {
                KMessageBox::error(this,i18n("Cannot write to the file."));
            }
        }
    }

    kapp->quit();
}

void MateEditMainWindow::slotEditCopy()
{
    QClipboard *cb = KApplication::kApplication()->clipboard();
    if ( m_textfield->hasSelectedText() )
    {
        cb->setText( m_textfield->selectedText( ), QClipboard::Clipboard );
    }
}

void MateEditMainWindow::slotEditPaste()
{
    int pos_para;
    int pos_index;
    QClipboard *cb = KApplication::kApplication()->clipboard();
    // Copy text from the clipboard (paste)
    QString text = cb->text(QClipboard::Clipboard);
    if ( ! text.isNull() )
    {
        int paraFrom;
        int indexFrom;
        int paraTo;
        int indexTo;
        if ( m_textfield->hasSelectedText() )
        {
            m_textfield->getSelection( &paraFrom, &indexFrom, &paraTo, &indexTo );

            int rtc;
            rtc = m_mateedit->deleteText( m_textfield->selectedText(), paraFrom, indexFrom, paraTo, indexTo );
            undo_stack.push(rtc);

            m_textfield->removeSelectedText();  //remove here so we get a correct cursor position
        }

        m_textfield->getCursorPosition( &pos_para, &pos_index );

//        Message * message = new Message( Message::Insert, m_local_pc, 0, timestamp, text );
//        message->setFrom( pos_para, pos_index );
        text.replace(QChar ('\r') ,QChar('\n'));

        int start;
        int end;
        int i;
        start = text.find('\n');
        for ( i = 1, end = text.find('\n', start + 1 ); end != -1; i++, end = text.find('\n', start + 1) )
        {
            start = end;
        }
        if ( start == -1 )
        {
            paraTo = pos_para;
            indexTo = pos_index + text.length();
        }
        else
        {
            int newlines = text.contains('\r');
            newlines += text.contains('\n');
            paraTo = pos_para + newlines;
            indexTo = text.length() - start - 1;
        }
        int rtc;
        rtc = m_mateedit->insertText( text, pos_para, pos_index, paraTo, indexTo );
        undo_stack.push(rtc);

        m_textfield->insert( text );
    }
}

void MateEditMainWindow::slotEditCut()
{
    if ( m_textfield->hasSelectedText() )
    {
        QClipboard *cb = KApplication::kApplication()->clipboard();
        QString text = m_textfield->selectedText();
        cb->setText( text, QClipboard::Clipboard );

        int paraFrom;
        int indexFrom;
        int paraTo;

        int indexTo;
        m_textfield->getSelection( &paraFrom, &indexFrom, &paraTo, &indexTo );

        int rtc;
        rtc = m_mateedit->deleteText( m_textfield->selectedText(), paraFrom, indexFrom, paraTo, indexTo );
        undo_stack.push(rtc);

        m_textfield->removeSelectedText();  //remove here so we get a correct cursor position
    }
}

void MateEditMainWindow::slotEditUndo()
{
    int id;
    int rtc;
    if( undo_stack.isEmpty() )
    {
        return;
    }
    id = undo_stack.pop();
    rtc = m_mateedit->undoMessage(id);
    if ( rtc >= 0 )
    {
        redo_stack.push( rtc );
    }
}

void MateEditMainWindow::slotEditRedo()
{
    int id;
    int rtc;
    id = redo_stack.pop();
    rtc = m_mateedit->redoMessage(id);
    if ( rtc >= 0 )
    {
        undo_stack.push( rtc );
    }
}

void MateEditMainWindow::slotSpelling()
{
    KSpellConfig spellconfig;

    QString text = m_textfield->text();
    KSpell::modalCheck ( text, &spellconfig );

}

void MateEditMainWindow::slotPrint()
{

}

void MateEditMainWindow::slotFind()
{
    KFindDialog * findDialog = new KFindDialog ( true, this );
    findDialog->exec();
    QString pattern = findDialog->pattern();
}

void MateEditMainWindow::slotReplace()
{
    KReplaceDialog * replaceDialog = new KReplaceDialog ( this );
    replaceDialog->setHasSelection( false );
    replaceDialog->exec();
    QString pattern = replaceDialog->pattern();
    QString replacement = replaceDialog->replacement();
}

void MateEditMainWindow::slotSetLastChatVisible()
{
    m_chat_list->verticalScrollBar()->setValue ( m_chat_list->verticalScrollBar()->maxValue() );
}

void MateEditMainWindow::slotTextfieldModified(bool m)
{
    if ( m )
    {
        if ( m_filename.isEmpty() )
        {
            setCaption ( i18n("untitled") + I18N_NOOP(" [") + i18n("modified") + I18N_NOOP("]")  );
        }
        else
        {
            setCaption ( m_filename + I18N_NOOP(" [") + i18n("modified") + I18N_NOOP("]")  );
        }
        fileSave->setEnabled( true );
    }
    else
    {
        if ( m_filename.isEmpty() )
        {
            setCaption ( i18n("untitled")  );
        }
        else
        {
            setCaption ( m_filename );
        }
        fileSave->setEnabled( false );
    }
}
bool MateEditMainWindow::mateEditInit( int user_number, QString & username, QImage & image )
{
    QString testusernames[MATEEDIT_MAX_USERS];
    testusernames[0] = "Server";
    testusernames[1] = "Brain";
    testusernames[2] = "Ginger";
    testusernames[3] = "Robot";
    testusernames[4] = "Twist";
    testusernames[5] = "Q";

    m_local_pc = user_number;
    UserDetailsDialog userDetails;
    QString imagenames[MATEEDIT_MAX_USERS];
    imagenames[0] = locate("appdata", "penguin.png");
    imagenames[1] = locate("appdata", "penguin.png");
    imagenames[2] = locate("appdata", "smiley.png");
    imagenames[3] = locate("appdata", "robot.png");
    imagenames[4] = locate("appdata", "personal.png");
    imagenames[5] = locate("appdata", "gv.png");

    KUser user;

    username = (user.loginName());
    qDebug(username);
    if(username.isEmpty())
    {
        username = testusernames[user_number];
    }
    userDetails.setIconPath( imagenames[user_number] );
    userDetails.userNickName->setText( username );
    userDetails.userNickName->setText( user.fullName() );
    userDetails.exec();

    QImage tmp_image;

    tmp_image.load( userDetails.getIconPath() );
    user_images[user_number] = tmp_image.scale( 55, 55 );
    image = user_images[user_number];

    for (int i = 0; i < 55; i++ )
    {
        user_images[user_number].setPixel( i, 50, Qt::black.rgb() );
        user_images[user_number].setPixel( i, 51, Qt::black.rgb() );
        user_images[user_number].setPixel( i, 52, Qt::black.rgb() );
        user_images[user_number].setPixel( i, 53, Qt::black.rgb() );
        user_images[user_number].setPixel( i, 54, Qt::black.rgb() );
    }

    username = userDetails.userNickName->text();
    m_user_icons[user_number] = new QIconViewItem( m_user_view, username, user_images[user_number] );
    users[user_number] = username;

    return true;
}

bool MateEditMainWindow::mateEditInitUser( int user_number, const QString & name, const QImage & image )
{
    user_images[user_number] = image;
    for (int i = 0; i < 55; i++ )
    {
        user_images[user_number].setPixel( i, 50, colors[user_number].rgb() );
        user_images[user_number].setPixel( i, 51, colors[user_number].rgb() );
        user_images[user_number].setPixel( i, 52, colors[user_number].rgb() );
        user_images[user_number].setPixel( i, 53, colors[user_number].rgb() );
        user_images[user_number].setPixel( i, 54, colors[user_number].rgb() );
    }
    m_user_icons[user_number] = new QIconViewItem( m_user_view, name, user_images[user_number] );
    users[user_number] = name;

    return true;
}


bool MateEditMainWindow::mateEditRemoveUser( int user_number, const QString & name )
{
    QString text = users[user_number] + i18n(" has left");
    m_chat_list->insertItem( text );
    QTimer::singleShot( 10, this, SLOT(slotSetLastChatVisible()) );

    delete m_user_icons[user_number];
    m_user_icons[user_number] = NULL;

    return true;
}


bool MateEditMainWindow::mateEditInsertText( const QString & text, const MateEdit::MessageCoords & from, const MateEdit::MessageCoords & to )
{
    int para;
    int pos;

    m_textfield->getCursorPosition( &para, &pos );
    MateEdit::MessageCoords coords( para, pos );
    m_textfield->insertAt( text, from.para(), from.idx() );
    coords.shiftUp( from, to );
    m_textfield->setCursorPosition( coords.para(), coords.idx() );

    KApplication* a = KApplication::kApplication();
    a->updateUserTimestamp();
    KMainWindowInterface ifc( this );
    KWin::demandAttention( ifc.getWinID() );

    return true;
}


bool MateEditMainWindow::mateEditDeleteText( const QString & text, const MateEdit::MessageCoords & from, const MateEdit::MessageCoords & to )
{
    int para;
    int pos;

    qDebug("MateEditMainWindow::mateEditDeleteText");
    m_textfield->setSelection( from.para(), from.idx(), to.para(), to.idx(), 1 );
    m_textfield->getCursorPosition( &para, (int *)&pos );
    MateEdit::MessageCoords coords(para, pos);
    coords.shiftDown( from, to );

    m_textfield->removeSelectedText( 1 );
    m_textfield->setCursorPosition( coords.para(), coords.idx() );

    KApplication* a = KApplication::kApplication();
    a->updateUserTimestamp();
    KMainWindowInterface ifc( this );
    KWin::demandAttention( ifc.getWinID() );

    return true;
}


bool MateEditMainWindow::mateEditMessage( const QString & text )
{
    KApplication* a = KApplication::kApplication();
    a->updateUserTimestamp();
    KMainWindowInterface ifc( this );
    KWin::demandAttention( ifc.getWinID() );

    KMessageBox::information (this, text, i18n("Message") );

    return true;
}


bool MateEditMainWindow::mateEditChatMessage(const QString & text )
{
    m_chat_list->insertItem( text );
    QTimer::singleShot( 10, this, SLOT(slotSetLastChatVisible()) );

    KApplication* a = KApplication::kApplication();
    a->updateUserTimestamp();
    KMainWindowInterface ifc( this );
    KWin::demandAttention( ifc.getWinID() );

    return true;
}

bool MateEditMainWindow::mateEditSessionClosed( const QString & reason )
{
    KApplication* a = KApplication::kApplication();
    a->updateUserTimestamp();
    KMainWindowInterface ifc( this );
    KWin::demandAttention( ifc.getWinID() );

    KMessageBox::information (this, i18n("The Host has closed the session"), reason );

    return true;
}

#if 0
    static uint size = 0;
    int counter = 0;
    QDataStream stream ( m_client_socket );
    if ( size ==0 )
    {
        stream >> size;
    }
    //Read messages from server and apply locally
    while ( m_client_socket->bytesAvailable() >= size && size > 0 )
    {
        ++counter;
        int rtc;
        int para;
        int pos;

        Message * message = new Message;

        char *data = new char[size];
        m_client_socket->readBlock(data, size);
        QByteArray bytearray;
        bytearray.assign(data,size);
        QBuffer buffer(bytearray);
        message->read(&buffer);

        m_timestamp = message->serverState();

        switch ( message->type() )
        {
            case Message::Init:
            {
                m_local_pc = message->sender();
                m_messageQueue.setLocalPc( m_local_pc );
                m_syntaxHighlighter->setLocalPc( m_local_pc );
                QString text = message->text();
                m_timestamp = message->serverState();
                if ( ! text.isEmpty() )
                {
                    text.replace(QChar ('\r') ,QChar('\n'));
                    int newlines = text.contains('\n');
                    int to;
                    int pos = text.findRev('\n');
                    if ( pos != -1 )
                    {
                        to = text.length() - pos - 1;
                    }
                    else
                    {
                        to = text.length();
                    }

                    m_owners->insert( MessageCoords( 0, 0 ), MessageCoords( newlines, to ), m_local_pc, text );
                    remoteInsertText( text, MessageCoords( 0, 0 ), MessageCoords( newlines, to ) );
//                    m_textfield->setText( text );
                }
                QString testusernames[MATEEDIT_MAX_USERS];
                testusernames[0] = "Server";
                testusernames[1] = "Brain";
                testusernames[2] = "Ginger";
                testusernames[3] = "Robot";
                testusernames[4] = "Twist";
                testusernames[5] = "Q";

                int user_number = m_local_pc;
                UserDetailsDialog userDetails;
                QString imagenames[MATEEDIT_MAX_USERS];
                imagenames[0] = locate("appdata", "penguin.png");
                imagenames[1] = locate("appdata", "penguin.png");
                imagenames[2] = locate("appdata", "smiley.png");
                imagenames[3] = locate("appdata", "robot.png");
                imagenames[4] = locate("appdata", "personal.png");
                imagenames[5] = locate("appdata", "gv.png");

                userDetails.setIconPath( imagenames[user_number] );
                userDetails.userNickName->setText( testusernames[user_number] );
                userDetails.exec();

                QImage tmp_image;

                tmp_image.load( userDetails.getIconPath() );
                user_images[user_number] = tmp_image.scale( 55, 55 );

                for (int i = 0; i < 55; i++ )
                {
                    user_images[user_number].setPixel( i, 50, Qt::black.rgb() );
                    user_images[user_number].setPixel( i, 51, Qt::black.rgb() );
                    user_images[user_number].setPixel( i, 52, Qt::black.rgb() );
                    user_images[user_number].setPixel( i, 53, Qt::black.rgb() );
                    user_images[user_number].setPixel( i, 54, Qt::black.rgb() );
                }
                QString username;
                username = userDetails.userNickName->text();
                m_user_icons[user_number] = new QIconViewItem( m_user_view, username, user_images[user_number] );
                users[user_number] = username;

                Message * initmessage = new Message( Message::InitUserData, m_local_pc, 0, m_timestamp, username );

                initmessage->setImage ( user_images[user_number] );

                initmessage->send( m_client_socket );

                delete initmessage;
                break;
            }
            case Message::InitUserData:
            {
                if ( m_local_pc != message->sender() && users[message->sender()].isEmpty() && ! message->text().isEmpty() )
                {
//                    QString text = message->text() + i18n(" has joined");
//                    m_chat_list->insertItem( text );
//                    QTimer::singleShot( 10, this, SLOT(slotSetLastChatVisible()) );

                    int user_number = message->sender();

                    user_images[user_number] = message->image();

                    for (int i = 0; i < 55; i++ )
                    {
                        user_images[user_number].setPixel( i, 50, colors[user_number].rgb() );
                        user_images[user_number].setPixel( i, 51, colors[user_number].rgb() );
                        user_images[user_number].setPixel( i, 52, colors[user_number].rgb() );
                        user_images[user_number].setPixel( i, 53, colors[user_number].rgb() );
                        user_images[user_number].setPixel( i, 54, colors[user_number].rgb() );
                    }
//                    m_user_icons[user_number] = new QIconViewItem( m_user_view, message->text(), user_images[user_number] );
                    users[message->sender()] = message->text();
                    newUser( user_number, message->text(), user_images[user_number] );
                }
                break;
            }
            case Message::Message:
            {
                if ( m_local_pc != message->sender() && users[message->sender()].isEmpty() && ! message->text().isEmpty() )
                {
                    QString text = users[message->sender()] + ": " + message->text();
                    KMessageBox::information (this, text, i18n("Message") );

                }
                break;
            }
            case Message::ChatMessage:
            {
                if ( m_local_pc != message->sender()  && ! message->text().isEmpty() )
                {
                    QString text = users[message->sender()] + ": " + message->text();
                    m_chat_list->insertItem( text );
                    QTimer::singleShot( 10, this, SLOT(slotSetLastChatVisible()) );
                }
                break;
            }
            case Message::RemoveUser:
            {
                int user_number = message->sender();
                delete m_user_icons[user_number];
                m_user_icons[user_number] = NULL;

                if ( user_number == -1 )
                {
                    KMessageBox::information (this, i18n("The Host has closed the session"), i18n("Server shutdown") );
                }
                else
                {
                    QString text = users[user_number] + i18n(" has left");
                    m_chat_list->insertItem( text );
                    QTimer::singleShot( 10, this, SLOT(slotSetLastChatVisible()) );
                }
                users[user_number] = "";
                break;
            }
            case Message::Insert:
            {
                rtc = m_messageQueue.processMessage( message );
                if (rtc >= 0)
                {
                    if ( message->type() != Message::NoOp )
                    {
                        if ( coords >= message->from( m_local_pc ) )
                        {
                            m_owners->insert( message->from( m_local_pc ), message->to( m_local_pc ), message->sender(), message->text() );
                            remoteInsertText( message->text(), message->from( m_local_pc ), message->to( m_local_pc ) )
//                            m_textfield->getCursorPosition( &para, &pos );
//                            MessageCoords coords( para, pos );
//                            m_textfield->insertAt( message->text(), message->from( m_local_pc ).para(), message->from( m_local_pc ).idx() );
//                            coords.shiftUp( message->from( m_local_pc ), message->to( m_local_pc ) );
//                            m_textfield->setCursorPosition( coords.para(), coords.idx() );
                        }
                        else
                        {
                            m_owners->insert( message->from( m_local_pc ), message->to( m_local_pc ), message->sender(), message->text() );
                            remoteInsertText( message->text(), message->from( m_local_pc ), message->to( m_local_pc ) )
//                            m_textfield->getCursorPosition( &para, &pos );
//                            MessageCoords coords( para, pos );
//                            m_textfield->insertAt( message->text(), message->from( m_local_pc ).para(), message->from( m_local_pc ).idx() );
//                            m_textfield->setCursorPosition( coords.para(), coords.idx() );
                        }
                    }
                }
                break;
            } //end case Message::Insert
            case Message::Delete:
            {
                rtc = m_messageQueue.processMessage( message );
                if (rtc >= 0)
                {
                    if ( message->type() != Message::NoOp )
                    {
//                        m_textfield->setSelection( message->from( m_local_pc ).para(), message->from( m_local_pc ).idx(), message->to( m_local_pc ).para(), message->to( m_local_pc ).idx(), 1 );
//                        m_textfield->getCursorPosition( &para, (int *)&pos );
//                        MessageCoords coords(para, pos);
//                        coords.shiftDown( message->from( m_local_pc ), message->to( m_local_pc ) );

                        m_owners->remove( message->from( m_local_pc ), message->to( m_local_pc ) );
                        remoteDeleteText( message->text(), message->from( m_local_pc ), message->to( m_local_pc ) )
//
//                        m_textfield->removeSelectedText( 1 );
//                        m_textfield->setCursorPosition( coords.para(), coords.idx() );
                    }
                }
                break;
            } //end case Message::Delete:
        } //end switch message->type
        KApplication* a = KApplication::kApplication();
        a->updateUserTimestamp();
        KMainWindowInterface ifc( this );
        KWin::demandAttention( ifc.getWinID() );

        size = 0;
        if ( m_client_socket->bytesAvailable() >= sizeof (uint) )
        {
            stream >> size;
        }
    } //end while ( client_socket->bytesAvailable() >= size && size > 0 )

#endif


#include "mateedit.moc"
