/*
 *  The ManaPlus Client
 *  Copyright (C) 2007  Joshua Langley <joshlangley@optusnet.com.au>
 *  Copyright (C) 2009-2010  The Mana Developers
 *  Copyright (C) 2011-2015  The ManaPlus Developers
 *
 *  This file is part of The ManaPlus Client.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef INPUT_INPUTACTIONMAP_H
#define INPUT_INPUTACTIONMAP_H

#include "sdlshared.h"

#include "actions/actions.h"
#include "actions/chat.h"
#include "actions/commands.h"
#include "actions/move.h"
#include "actions/pets.h"
#include "actions/statusbar.h"
#include "actions/tabs.h"
#include "actions/target.h"
#include "actions/windows.h"

#include "input/inputactiondata.h"
#include "input/inputcondition.h"
#include "input/inputmanager.h"
#include "input/inputtype.h"

#include "localconsts.h"

#define defaultAction(name) \
        InputType::UNKNOWN, InputAction::NO_VALUE, \
        InputType::UNKNOWN, InputAction::NO_VALUE, \
        Input::GRP_DEFAULT, \
        name, \
        InputAction::NO_VALUE, 50

#define addKey(name) InputType::KEYBOARD, name
#define emptyKey InputType::UNKNOWN, InputAction::NO_VALUE
#define joystickButton(num) InputType::JOYSTICK, num

// keyData must be in same order as enum keyAction.
static const InputActionData inputActionData[InputAction::TOTAL] = {
    {"keyMoveUp",
        addKey(SDLK_UP),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveUp,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        false},
    {"keyMoveDown",
        addKey(SDLK_DOWN),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveDown,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        false},
    {"keyMoveLeft",
        addKey(SDLK_LEFT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveLeft,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        false},
    {"keyMoveRight",
        addKey(SDLK_RIGHT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveRight,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        false},
    {"keyAttack",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::attack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOFOLLOW
            | InputCondition::VALIDSPEED,
        "attack",
        false},
    {"keyTargetAttack",
        addKey(SDLK_LCTRL),
        joystickButton(0),
        Input::GRP_DEFAULT,
        &Actions::targetAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOFOLLOW
            | InputCondition::VALIDSPEED | InputCondition::ALIVE,
        "targetattack",
        true},
    {"keyMoveToTarget",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveToTarget,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "movetotarget",
        false},
    {"keyChangeMoveToTarget",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMoveToTarget,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keyMoveToHome",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveToHome,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "movetohome",
        false},
    {"keySetHome",
        addKey(SDLK_KP5),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::setHome,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "sethome",
        false},
    {"keyMoveToPoint",
        addKey(SDLK_RSHIFT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        false},
    {"keyTalk",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::talk,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME,
        "talk",
        false},
    {"keyTarget",
        addKey(SDLK_LSHIFT),
        joystickButton(4),
        Input::GRP_DEFAULT,
        &Actions::stopAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME,
        "stopattack",
        false},
    {"keyUnTarget",
        defaultAction(&Actions::untarget),
        InputCondition::GAME,
        "untarget",
        false},
    {"keyTargetMonster",
        addKey(SDLK_a),
        joystickButton(3),
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::targetMonster,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET
            | InputCondition::VALIDSPEED,
        "targetmonster",
        false},
    {"keyTargetNPC",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::targetNPC,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "targetnpc",
        false},
    {"keyTargetPlayer",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::targetPlayer,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "targetplayer",
        false},
    {"keyPickup",
        addKey(SDLK_z),
        joystickButton(1),
        Input::GRP_DEFAULT,
        &Actions::pickup,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "pickup",
        false},
    {"keyChangePickupType",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changePickupType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keyHideWindows",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::hideWindows,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "hidewindows",
        false},
    {"keyBeingSit",
        addKey(SDLK_s),
        joystickButton(2),
        Input::GRP_DEFAULT,
        &Actions::sit,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "sit",
        false},
    {"keyScreenshot",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::screenshot,
        InputAction::NO_VALUE, 50,
        InputCondition::NOTARGET | InputCondition::NOINPUT,
        "screenshot",
        false},
    {"keyTrade",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeTrade,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "changetrade",
        false},
    {"keyPathfind",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMapMode,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET
            | InputCondition::EMODS,
        "",
        false},
    {"keyOK",
        addKey(SDLK_SPACE),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::ok,
        InputAction::NO_VALUE, 50,
        InputCondition::NOMODAL | InputCondition::NOAWAY
            | InputCondition::NONPCINPUT,
        "",
        false},
    {"keyQuit",
        addKey(SDLK_ESCAPE),
#if defined(USE_SDL2) && defined(ANDROID)
        addKey(SDLK_AC_BACK),
#else
        emptyKey,
#endif
        Input::GRP_DEFAULT,
        &Actions::quit,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "quit",
        false},
    {"keyShortcutsKey",
        addKey(SDLK_MENU),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        false},
    {"keyShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut16",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut17",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut18",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut19",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyShortcut20",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        false},
    {"keyWindowHelp",
        addKey(SDLK_F1),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::helpWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "help|openhelp",
        false},
    {"keyWindowStatus",
        addKey(SDLK_F2),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::statusWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "status|openstatus",
        false},
    {"keyWindowInventory",
        addKey(SDLK_F3),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::inventoryWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "inventory|openinventory",
        false},
    {"keyWindowEquipment",
        addKey(SDLK_F4),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::equipmentWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "equipment|openequipment",
        false},
    {"keyWindowSkill",
        addKey(SDLK_F5),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::skillDialogShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "skill|skills|openskills",
        false},
    {"keyWindowMinimap",
        addKey(SDLK_F6),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::minimapWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "minimap|openminimap",
        false},
    {"keyWindowChat",
        addKey(SDLK_F7),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::chatWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "openchat",
        false},
    {"keyWindowShortcut",
        addKey(SDLK_F8),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::shortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "openshortcut|openshortcuts",
        false},
    {"keyWindowSetup",
        addKey(SDLK_F9),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::setupWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::NOTARGET | InputCondition::NOINPUT
            | InputCondition::ENABLED,
        "setup|opensetup",
        false},
    {"keyWindowDebug",
        addKey(SDLK_F10),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::debugWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "debug|opendebug",
        false},
    {"keyWindowSocial",
        addKey(SDLK_F11),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::socialWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "social|opensocial",
        false},
    {"keyWindowEmoteBar",
        addKey(SDLK_F12),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::emoteShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "emotes|openemotes",
        false},
    {"keyWindowOutfit",
        addKey(SDLK_BACKQUOTE),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::outfitWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "openoutfit|openoutfits",
        false},
    {"keyWindowShop",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::shopWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "shop|openshop",
        false},
    {"keyWindowDrop",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::dropShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "opendrop",
        false},
    {"keyWindowKills",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::killStatsWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "kills|openkills",
        false},
    {"keyWindowSpells",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::spellShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "spells|openspells",
        false},
    {"keyWindowBotChecker",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::botcheckerWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "botchecker|openbotchecker",
        false},
    {"keyWindowOnline",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::whoIsOnlineWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "online|openonline",
        false},
    {"keyWindowDidYouKnow",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::didYouKnowWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "didyouknow|opendidyouknow",
        false},
    {"keyWindowQuests",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::questsWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "quests|openquests",
        false},
    {"keySocialPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevSocialTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "prevsocialtab",
        false},
    {"keySocialNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextSocialTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "nextsocialtab",
        false},
    {"keySmilie",
        addKey(SDLK_LALT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        false},
    {"keyEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyWearOutfit",
        addKey(SDLK_RCTRL),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        false},
    {"keyCopyOutfit",
        addKey(SDLK_RALT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        false},
    {"keyCopyEquipedOutfit",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::copyEquippedToOutfit,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        false},
    {"keyOutfitShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyOutfitShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyMoveToPointShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyChat",
        addKey(SDLK_RETURN),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_CHAT,
        &Actions::toggleChat,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::NOAWAY
            | InputCondition::NOMODAL | InputCondition::INGAME,
        "chat",
        false},
    {"keyChatScrollUp",
        addKey(SDLK_PAGEUP),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::scrollChatUp,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "chatscrollup",
        false},
    {"keyChatScrollDown",
        addKey(SDLK_PAGEDOWN),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::scrollChatDown,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "chatscrolldown",
        false},
    {"keyChatPrevTab",
        addKey(SDLK_KP7),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "prevtab",
        false},
    {"keyChatNextTab",
        addKey(SDLK_KP9),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "nexttab",
        false},
    {"keyChatCloseTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::closeChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "close",
        false},
    {"keyChatPrevHistory",
        addKey(SDLK_KP7),
        emptyKey,
        Input::GRP_CHAT | Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        false},
    {"keyChatNextHistory",
        addKey(SDLK_KP9),
        emptyKey,
        Input::GRP_CHAT | Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        false},
    {"keyIgnoreInput1",
#ifdef USE_SDL2
        emptyKey,
#else
        addKey(SDLK_LSUPER),
#endif
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::ignoreInput,
        InputAction::NO_VALUE, 500,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyIgnoreInput2",
#ifdef USE_SDL2
        emptyKey,
#else
        addKey(SDLK_RSUPER),
#endif
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::ignoreInput,
        InputAction::NO_VALUE, 500,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyDirectUp",
        defaultAction(&Actions::directUp),
        InputCondition::GAME,
        "directup|turnup",
        false},
    {"keyDirectDown",
        defaultAction(&Actions::directDown),
        InputCondition::GAME,
        "directdown|turndown",
        false},
    {"keyDirectLeft",
        defaultAction(&Actions::directLeft),
        InputCondition::GAME,
        "directleft|turnleft",
        false},
    {"keyDirectRight",
        defaultAction(&Actions::directRight),
        InputCondition::GAME,
        "directright|turnright",
        false},
    {"keyCrazyMoves",
        addKey(SDLK_SLASH),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::crazyMoves,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        false},
    {"keyChangeCrazyMoveType",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeCrazyMove,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keyQuickDrop",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::dropItem0,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "drop",
        false},
    {"keyQuickDropN",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::dropItem,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "dropn",
        false},
    {"keySwitchQuickDrop",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::switchQuickDrop,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        false},
    {"keyMagicInma1",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::heal,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "heal",
        true},
    {"keyMagicItenplz",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::itenplz,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "itenplz",
        false},
    {"keyMagicAttack",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::magicAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "magicattack",
        false},
    {"keySwitchMagicAttack",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMagicAttackType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keySwitchPvpAttack",
        defaultAction(&Actions::changePvpMode),
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keyInvertDirection",
        defaultAction(&Actions::changeMoveType),
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keyChangeAttackWeaponType",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAttackWeaponType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keyChangeAttackType",
        addKey(SDLK_END),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAttackType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keyChangeFollowMode",
        addKey(SDLK_KP1),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeFollowMode,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keyChangeImitationMode",
        addKey(SDLK_KP4),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeImitationMode,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keyDisableGameModifiers",
        addKey(SDLK_KP8),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeGameModifier,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        false},
    {"keyChangeAudio",
        addKey(SDLK_KP3),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAudio,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "audio",
        false},
    {"keyAway",
        addKey(SDLK_KP2),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::away,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        false},
    {"keyRightClick",
        addKey(SDLK_TAB),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::mouseClick,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::NOAWAY
            | InputCondition::NOMODAL | InputCondition::INGAME,
        "rightclick",
        false},
    {"keyCameraMode",
        addKey(SDLK_KP_PLUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::camera,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        false},
    {"keyMod",
        emptyKey,
        emptyKey,
        0,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIUp",
        addKey(SDLK_UP),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIDown",
        addKey(SDLK_DOWN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUILeft",
        addKey(SDLK_LEFT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIRight",
        addKey(SDLK_RIGHT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUISelect",
        addKey(SDLK_SPACE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUISelect2",
        addKey(SDLK_RETURN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUICancel",
        addKey(SDLK_ESCAPE),
#if defined(USE_SDL2) && defined(ANDROID)
        addKey(SDLK_AC_BACK),
#else
        emptyKey,
#endif
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIHome",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIEnd",
        addKey(SDLK_END),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIDelete",
        addKey(SDLK_DELETE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIBackSpace",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUITab",
        addKey(SDLK_TAB),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIPageUp",
        addKey(SDLK_PAGEUP),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIPageDown",
        addKey(SDLK_PAGEDOWN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIInsert",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIMod",
        addKey(SDLK_LSHIFT),
        addKey(SDLK_RSHIFT),
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keySafeVideo",
        addKey(SDLK_KP_MINUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::safeVideoMode,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyStopSit",
        defaultAction(&Actions::stopSit),
        InputCondition::DEFAULT | InputCondition::INGAME,
        "sit2",
        false},
    {"keyShowKeyboard",
        defaultAction(&Actions::showKeyboard),
        InputCondition::DEFAULT,
        "showkeyboard",
        false},
    {"keyShowWindows",
        defaultAction(&Actions::showWindows),
        InputCondition::DEFAULT,
        "showwindows",
        false},
    {"keyChatMod",
        addKey(SDLK_LSHIFT),
        emptyKey,
        Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        false},
    {"keyMoveForward",
        defaultAction(&Actions::moveForward),
        InputCondition::ARROWKEYS,
        "",
        false},
    {"keyGUICtrl",
        addKey(SDLK_LCTRL),
        addKey(SDLK_RCTRL),
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIb",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIc",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUId",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIe",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIf",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIh",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIk",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIu",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIv",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIw",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyShortcutsPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevShortcutsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "shortcutprevtab",
        false},
    {"keyShortcutsNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextShortcutsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "shortcutnexttab",
        false},
    {"keyCommandsPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevCommandsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "commandsprevtab",
        false},
    {"keyCommandsNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextCommandsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "commandsnexttab",
        false},
    {"keyOpenTrade",
        defaultAction(&Actions::openTrade),
        InputCondition::GAME | InputCondition::NOTARGET,
        "trade",
        false},
    {"keyGUIF1",
        addKey(SDLK_F1),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF2",
        addKey(SDLK_F2),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF3",
        addKey(SDLK_F3),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF4",
        addKey(SDLK_F4),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF5",
        addKey(SDLK_F5),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF6",
        addKey(SDLK_F6),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF7",
        addKey(SDLK_F7),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF8",
        addKey(SDLK_F8),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF9",
        addKey(SDLK_F9),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF10",
        addKey(SDLK_F10),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF11",
        addKey(SDLK_F11),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyGUIF12",
        addKey(SDLK_F12),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        false},
    {"keyWindowAbout",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::aboutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "about",
        false},
    {"keyWindowUpdater",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::updaterWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME,
        "updater",
        false},
    {"keyTargetClosestMonster",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::targetClosestMonster,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET
            | InputCondition::VALIDSPEED,
        "targetclosestmonster",
        false},
    {"keyChatCloseAllTabs",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::closeAllChatTabs,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "closeall",
        false},
    {"keyChatIgnoreAllWhispers",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::ignoreAllWhispers,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "ignoreallwhispers",
        false},
    {"keyChatAnnounce",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::chatAnnounce,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "announce",
        true},
    {"keyIpcToggle",
        defaultAction(&Actions::ipcToggle),
        InputCondition::INGAME,
        "ipctoggle",
        false},
    {"keyWhere",
        defaultAction(&Actions::where),
        InputCondition::INGAME,
        "where",
        false},
    {"keyWho",
        defaultAction(&Actions::who),
        InputCondition::INGAME,
        "who",
        false},
    {"keyWhisper",
        defaultAction(&Actions::msg),
        InputCondition::INGAME,
        "w|whisper|msg",
        true},
    {"keyQuery",
        defaultAction(&Actions::query),
        InputCondition::INGAME,
        "q|query|j|join",
        true},
    {"keyClearChatTab",
        defaultAction(&Actions::clearChatTab),
        InputCondition::INGAME,
        "clear",
        false},
    {"keyIgnore",
        defaultAction(&Actions::chatIgnore),
        InputCondition::INGAME,
        "ignore",
        true},
    {"keyUnignore",
        defaultAction(&Actions::chatUnignore),
        InputCondition::INGAME,
        "unignore",
        true},
    {"keyFriend",
        defaultAction(&Actions::chatFriend),
        InputCondition::INGAME,
        "friend|befriend",
        true},
    {"keyDisregard",
        defaultAction(&Actions::chatDisregard),
        InputCondition::INGAME,
        "disregard",
        true},
    {"keyNeutral",
        defaultAction(&Actions::chatNeutral),
        InputCondition::INGAME,
        "neutral",
        true},
    {"keyBlacklist",
        defaultAction(&Actions::chatBlackList),
        InputCondition::INGAME,
        "blacklist",
        true},
    {"keyEnemy",
        defaultAction(&Actions::chatEnemy),
        InputCondition::INGAME,
        "enemy",
        true},
    {"keyErase",
        defaultAction(&Actions::chatErase),
        InputCondition::INGAME,
        "erase",
        true},
    {"keyCleanGraphics",
        defaultAction(&Actions::cleanGraphics),
        InputCondition::INGAME,
        "cleangraphics",
        false},
    {"keyCleanFonts",
        defaultAction(&Actions::cleanFonts),
        InputCondition::INGAME,
        "cleanfonts",
        false},
    {"keyCreateParty",
        defaultAction(&Actions::createParty),
        InputCondition::INGAME,
        "createparty",
        true},
    {"keyCreateGuild",
        defaultAction(&Actions::createGuild),
        InputCondition::INGAME,
        "createguild",
        true},
    {"keyParty",
        defaultAction(&Actions::party),
        InputCondition::INGAME,
        "party",
        true},
    {"keyMe",
        defaultAction(&Actions::me),
        InputCondition::INGAME,
        "me",
        true},
    {"keyToggle",
        defaultAction(&Actions::toggle),
        InputCondition::INGAME,
        "toggle",
        false},
    {"keyPresent",
        defaultAction(&Actions::present),
        InputCondition::INGAME,
        "present",
        false},
    {"keyPrintAll",
        defaultAction(&Actions::printAll),
        InputCondition::INGAME,
        "all",
        false},
    {"keyMove",
        defaultAction(&Actions::move),
        InputCondition::INGAME,
        "move",
        true},
    {"keySetTarget",
        defaultAction(&Actions::setTarget),
        InputCondition::INGAME,
        "target",
        true},
    {"keyAttackHuman",
        defaultAction(&Actions::attackHuman),
        InputCondition::INGAME,
        "atkhuman",
        false},
    {"keyOutfit",
        defaultAction(&Actions::commandOutfit),
        InputCondition::INGAME,
        "outfit",
        true},
    {"keyEmote",
        defaultAction(&Actions::commandEmote),
        InputCondition::INGAME,
        "emote",
        true},
    {"keyEmotePet",
        defaultAction(&Actions::commandEmotePet),
        InputCondition::INGAME,
        "emotepet",
        true},
    {"keyAwayMessage",
        defaultAction(&Actions::awayMessage),
        InputCondition::INGAME,
        "away",
        true},
    {"keyPseudoAway",
        defaultAction(&Actions::pseudoAway),
        InputCondition::INGAME,
        "pseudoaway",
        true},
    {"keyFollow",
        defaultAction(&Actions::follow),
        InputCondition::INGAME,
        "follow",
        true},
    {"keyNavigate",
        defaultAction(&Actions::navigate),
        InputCondition::INGAME,
        "navigate",
        true},
    {"keyImitation",
        defaultAction(&Actions::imitation),
        InputCondition::INGAME,
        "imitation",
        true},
    {"keySendMail",
        defaultAction(&Actions::sendMail),
        InputCondition::INGAME,
        "mail",
        true},
    {"keyTradeCommand",
        defaultAction(&Actions::trade),
        InputCondition::INGAME,
        "trade",
        true},
    {"keyPriceLoad",
        defaultAction(&Actions::priceLoad),
        InputCondition::INGAME,
        "priceload",
        false},
    {"keyPriceSave",
        defaultAction(&Actions::priceSave),
        InputCondition::INGAME,
        "pricesave",
        false},
    {"keyCacheInfo",
        defaultAction(&Actions::cacheInfo),
        InputCondition::INGAME,
        "cacheinfo",
        false},
    {"keyDisconnect",
        defaultAction(&Actions::disconnect),
        InputCondition::INGAME,
        "disconnect",
        false},
    {"keyUndress",
        defaultAction(&Actions::undress),
        InputCondition::INGAME,
        "undress",
        true},
    {"keyDirs",
        defaultAction(&Actions::dirs),
        InputCondition::INGAME,
        "dirs",
        false},
    {"keyInfo",
        defaultAction(&Actions::info),
        InputCondition::INGAME,
        "info",
        false},
    {"keyWait",
        defaultAction(&Actions::wait),
        InputCondition::INGAME,
        "wait",
        true},
    {"keyUptime",
        defaultAction(&Actions::uptime),
        InputCondition::INGAME,
        "uptime",
        false},
    {"keyAddPriorityAttack",
        defaultAction(&Actions::addPriorityAttack),
        InputCondition::INGAME,
        "addpriorityattack",
        true},
    {"keyAddAttack",
        defaultAction(&Actions::addAttack),
        InputCondition::INGAME,
        "addattack",
        true},
    {"keyRemoveAttack",
        defaultAction(&Actions::removeAttack),
        InputCondition::INGAME,
        "removeattack|removeignoreattack",
        true},
    {"keyAddIgnoreAttack",
        defaultAction(&Actions::addIgnoreAttack),
        InputCondition::INGAME,
        "addignoreattack",
        true},
    {"keyDump",
        defaultAction(&Actions::dump),
        InputCondition::INGAME,
        "dump",
        false},
    {"keyServerIgnoreAll",
        defaultAction(&Actions::serverIgnoreAll),
        InputCondition::INGAME,
        "serverignoreall",
        false},
    {"keyServerUnIgnoreAll",
        defaultAction(&Actions::serverUnIgnoreAll),
        InputCondition::INGAME,
        "serverunignoreall",
        false},
    {"keySetDrop",
        defaultAction(&Actions::setDrop),
        InputCondition::INGAME,
        "setdrop",
        true},
    {"keyError",
        defaultAction(&Actions::error),
        InputCondition::INGAME,
        "error",
        false},
    {"keyDumpg",
        defaultAction(&Actions::dumpGraphics),
        InputCondition::INGAME,
        "dumpg",
        false},
    {"keyDumpe",
        defaultAction(&Actions::dumpEnvironment),
        InputCondition::INGAME,
        "dumpe",
        false},
    {"keyDumpt",
        defaultAction(&Actions::dumpTests),
        InputCondition::INGAME,
        "dumpt",
        false},
    {"keyDumpOGL",
        defaultAction(&Actions::dumpOGL),
        InputCondition::INGAME,
        "dumpogl",
        false},
    {"keyDumpGL",
        defaultAction(&Actions::dumpGL),
        InputCondition::INGAME,
        "dumpgl",
        false},
    {"keyDumpMods",
        defaultAction(&Actions::dumpMods),
        InputCondition::INGAME,
        "dumpMods",
        false},
    {"keyUrl",
        defaultAction(&Actions::url),
        InputCondition::INGAME,
        "url",
        true},
    {"keyOpenUrl",
        defaultAction(&Actions::openUrl),
        InputCondition::INGAME,
        "open",
        true},
    {"keyExecute",
        defaultAction(&Actions::execute),
        InputCondition::INGAME,
        "execute",
        true},
    {"keyTestSdlFont",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
#if defined USE_OPENGL && defined DEBUG_SDLFONT
        &Actions::testSdlFont,
#else
        nullptr,
#endif
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
#if defined USE_OPENGL && defined DEBUG_SDLFONT
        "testsdlfont",
#else
        "",
#endif
        false},
    {"keyEnableHightlight",
        defaultAction(&Actions::enableHighlight),
        InputCondition::INGAME,
        "enablehighlight",
        false},
    {"keyDisableHightlight",
        defaultAction(&Actions::disableHighlight),
        InputCondition::INGAME,
        "disablehighlight",
        false},
    {"keyDontRemoveName",
        defaultAction(&Actions::dontRemoveName),
        InputCondition::INGAME,
        "dontremovename",
        false},
    {"keyRemoveName",
        defaultAction(&Actions::removeName),
        InputCondition::INGAME,
        "removename",
        false},
    {"keyDisableAway",
        defaultAction(&Actions::disableAway),
        InputCondition::INGAME,
        "disableaway",
        false},
    {"keyEnableAway",
        defaultAction(&Actions::enableAway),
        InputCondition::INGAME,
        "enableaway",
        false},
    {"keyTestParticle",
        defaultAction(&Actions::testParticle),
        InputCondition::INGAME,
        "testparticle",
        true},
    {"keyCreateItems",
        defaultAction(&Actions::createItems),
        InputCondition::INGAME,
        "createitems",
        false},
    {"keyTalkRaw",
        defaultAction(&Actions::talkRaw),
        InputCondition::INGAME,
        "talkraw",
        true},
    {"keyTalkPet",
        defaultAction(&Actions::talkPet),
        InputCondition::INGAME,
        "talkpet",
        true},
    {"keyUploadConfig",
        defaultAction(&Actions::uploadConfig),
        InputCondition::INGAME,
        "uploadconfig",
        false},
    {"keyUploadServerConfig",
        defaultAction(&Actions::uploadServerConfig),
        InputCondition::INGAME,
        "uploadserverconfig",
        false},
    {"keyUploadLog",
        defaultAction(&Actions::uploadLog),
        InputCondition::INGAME,
        "uploadlog",
        false},
    {"keyGm",
        defaultAction(&Actions::gm),
        InputCondition::INGAME,
        "gm",
        true},
    {"keyHack",
        defaultAction(&Actions::hack),
        InputCondition::INGAME,
        "hack",
        true},
    {"keyDebugSpawn",
        defaultAction(&Actions::debugSpawn),
        InputCondition::INGAME,
        "debugspawn",
        true},
    {"keyPetSmilie",
        defaultAction(nullptr),
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        false},
    {"keyPetEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyPetEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyCatchPet",
        defaultAction(&Actions::catchPet),
        InputCondition::INGAME,
        "catchpet|catch",
        true},
    {"keyChatServerIgnoreWhisper",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::serverIgnoreWhisper,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "serverignore",
        false},
    {"keyChatServerUnIgnoreWhisper",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::serverUnIgnoreWhisper,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "serverunignore",
        false},
    {"keyMercenaryFire",
        defaultAction(&Actions::mercenaryFire),
        InputCondition::INGAME,
        "mercenaryfire|firemercenary",
        false},
    {"keyTargetMercenary",
        defaultAction(&Actions::targetMercenary),
        InputCondition::INGAME | InputCondition::NOTARGET,
        "targetmercenary",
        false},
    {"keyTargetPet",
        defaultAction(&Actions::targetPet),
        InputCondition::INGAME | InputCondition::NOTARGET,
        "targetpet",
        false},
    {"keyPetSetName",
        defaultAction(&Actions::setPetName),
        InputCondition::INGAME,
        "setpetname|petsetname",
        false},
    {"keyHomunculusSetName",
        defaultAction(&Actions::setHomunculusName),
        InputCondition::INGAME,
        "sethomunname|homunsetname|sethomunculusname|homunculussetname",
        false},
    {"keyHomunculusFire",
        defaultAction(&Actions::fireHomunculus),
        InputCondition::INGAME,
        "firehomunculus|homunculusfire",
        false},
    {"keyWindowBank",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::bankWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "bank|openbank",
        false},
    {"keyInventoryPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevInvTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "previnvtab",
        false},
    {"keyInventoryNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextInvTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "nextinvtab",
        false},
    {"keyContextMenu",
        defaultAction(&Actions::contextMenu),
        InputCondition::NOINPUT | InputCondition::NOAWAY
            | InputCondition::NOMODAL | InputCondition::INGAME,
        "contextmenu",
        false},
    {"keyBuy",
        defaultAction(&Actions::buy),
        InputCondition::GAME,
        "buy",
        false},
    {"keySell",
        defaultAction(&Actions::sell),
        InputCondition::GAME,
        "sell",
        false},
    {"keyWhisper2",
        defaultAction(&Actions::msg2),
        InputCondition::INGAME,
        "w2|whisper2|msg2",
        true},
    {"keyGuild",
        defaultAction(&Actions::guild),
        InputCondition::INGAME,
        "guild",
        true},
    {"keyNuke",
        defaultAction(&Actions::chatNuke),
        InputCondition::INGAME,
        "nuke",
        true},
    {"keyUse",
        defaultAction(&Actions::useItem),
        InputCondition::INGAME,
        "use|useitem",
        true},
    {"keyChatAdd",
        defaultAction(&Actions::chatAdd),
        InputCondition::INGAME,
        "addchat|chatadd",
        true},
    {"keyWhisperText",
        defaultAction(&Actions::msgText),
        InputCondition::INGAME,
        "wt|whispertext|msgtext",
        true},
    {"keyNavigateTo",
        defaultAction(&Actions::navigateTo),
        InputCondition::INGAME,
        "navigateto",
        true},
    {"keyCameraMove",
        defaultAction(&Actions::moveCamera),
        InputCondition::INGAME,
        "movecamera|cameramove",
        true},
    {"keyDropItem",
        defaultAction(&Actions::dropItemId),
        InputCondition::INGAME,
        "dropitem",
        true},
    {"keyDropItemAll",
        defaultAction(&Actions::dropItemIdAll),
        InputCondition::INGAME,
        "dropitemall",
        true},
    {"keyDropInv",
        defaultAction(&Actions::dropItemInv),
        InputCondition::INGAME,
        "dropinv",
        true},
    {"keyDropInvAll",
        defaultAction(&Actions::dropItemInvAll),
        InputCondition::INGAME,
        "dropinvall",
        true},
    {"keyUseInv",
        defaultAction(&Actions::useItemInv),
        InputCondition::INGAME,
        "useinv",
        true},
    {"keyInvToStorage",
        defaultAction(&Actions::invToStorage),
        InputCondition::INGAME,
        "invtostorage|invstorage",
        true},
    {"keyTradeAdd",
        defaultAction(&Actions::tradeAdd),
        InputCondition::INGAME,
        "tradeadd|addtrade",
        true},
    {"keyStorageToInv",
        defaultAction(&Actions::storageToInv),
        InputCondition::INGAME,
        "storagetoinv|storageinv",
        true},
    {"keyProtectItem",
        defaultAction(&Actions::protectItem),
        InputCondition::INGAME,
        "protectitem|itemprotect",
        true},
    {"keyUnProtectItem",
        defaultAction(&Actions::unprotectItem),
        InputCondition::INGAME,
        "unprotectitem|itemunprotect",
        true},
    {"keyKickParty",
        defaultAction(&Actions::kickParty),
        InputCondition::INGAME,
        "kickparty|partykick",
        true},
    {"keyAddText",
        defaultAction(&Actions::addText),
        InputCondition::INGAME,
        "addtext|textadd",
        true},
    {"keyKick",
        defaultAction(&Actions::kick),
        InputCondition::INGAME,
        "kick",
        true},
    {"keyLeaveParty",
        defaultAction(&Actions::leaveParty),
        InputCondition::INGAME,
        "leaveparty|partyleave",
        true},
    {"keyWarp",
        defaultAction(&Actions::warp),
        InputCondition::INGAME,
        "warp",
        true},
    {"keyClearChat",
        defaultAction(&Actions::clearChat),
        InputCondition::INGAME,
        "clearchat|chatclear",
        false},
    {"keyPetMoveUp",
        defaultAction(&Actions::petMoveUp),
        InputCondition::INGAME,
        "petmoveup|moveuppet",
        false},
    {"keyPetMoveDown",
        defaultAction(&Actions::petMoveDown),
        InputCondition::INGAME,
        "petmovedown|movedownpet",
        false},
    {"keyPetMoveLeft",
        defaultAction(&Actions::petMoveLeft),
        InputCondition::INGAME,
        "petmoveleft|moveleftpet",
        false},
    {"keyPetMoveRight",
        defaultAction(&Actions::petMoveRight),
        InputCondition::INGAME,
        "petmoveright|moverightpet",
        false},
    {"keyPetDirectUp",
        defaultAction(&Actions::petDirectUp),
        InputCondition::INGAME,
        "petdirectup|directuppet",
        false},
    {"keyPetDirectDown",
        defaultAction(&Actions::petDirectDown),
        InputCondition::INGAME,
        "petdirectdown|directdownpet",
        false},
    {"keyPetDirectLeft",
        defaultAction(&Actions::petDirectLeft),
        InputCondition::INGAME,
        "petdirectleft|directleftpet",
        false},
    {"keyPetDirectRight",
        defaultAction(&Actions::petDirectRight),
        InputCondition::INGAME,
        "petdirectright|directrightpet",
        false},
    {"keyPetAiStart",
        defaultAction(&Actions::petAiStart),
        InputCondition::INGAME,
        "petaistart|startpetai|enablepetai|petaienable",
        false},
    {"keyPetAiStop",
        defaultAction(&Actions::petAiStop),
        InputCondition::INGAME,
        "petaistop|stoppetai|disablepetai|petaidisable",
        false},
    {"keyGeneralTab",
        defaultAction(&Actions::chatGeneralTab),
        InputCondition::INGAME,
        "generaltab|generalchattab|general",
        false},
    {"keyDebugTab",
        defaultAction(&Actions::chatDebugTab),
        InputCondition::INGAME,
        "debugtab|debugchattab",
        false},
    {"keyBattleTab",
        defaultAction(&Actions::chatBattleTab),
        InputCondition::INGAME,
        "battletab|battlechattab",
        false},
    {"keyTradeTab",
        defaultAction(&Actions::chatTradeTab),
        InputCondition::INGAME,
        "tradetab|tradechattab",
        false},
    {"keyLangTab",
        defaultAction(&Actions::chatLangTab),
        InputCondition::INGAME,
        "langtab|langchattab",
        false},
    {"keyGmTab",
        defaultAction(&Actions::chatGmTab),
        InputCondition::INGAME,
        "gmtab|gmchattab",
        false},
    {"keyPartyTab",
        defaultAction(&Actions::chatPartyTab),
        InputCondition::INGAME,
        "partytab|partychattab",
        false},
    {"keyGuildTab",
        defaultAction(&Actions::chatGuildTab),
        InputCondition::INGAME,
        "guildtab|guildchattab",
        false},
    {"keyDropClear",
        defaultAction(&Actions::clearDrop),
        InputCondition::INGAME,
        "cleardrops|cleardrop|cleardropwindow|dropclear|dropsclear",
        false},
    {"keyWindowCart",
        defaultAction(&Actions::cartWindowShow),
        InputCondition::SHORTCUT0,
        "cart|opencart",
        false},
    {"keyMagicHealMostDamaged",
        defaultAction(&Actions::healmd),
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "healmd|saveass",
        false},
    {"keyPetMove",
        defaultAction(&Actions::petMove),
        InputCondition::INGAME,
        "movepet|petmove",
        true},
    {"keyHomunTalk",
        defaultAction(&Actions::homunTalk),
        InputCondition::INGAME,
        "homuntalk|merctalk",
        true},
    {"keyHomunEmote",
        defaultAction(&Actions::commandHomunEmote),
        InputCondition::INGAME,
        "homunemote|mercemote",
        true},
    {"keyHomunSmilie",
        defaultAction(nullptr),
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        false},
    {"keyHomunEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyHomunEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        false},
    {"keyKickGuild",
        defaultAction(&Actions::kickGuild),
        InputCondition::INGAME,
        "kickguild|guildkick",
        true},
    {"keyHat",
        defaultAction(&Actions::hat),
        InputCondition::INGAME,
        "hat|hats|showhat",
        false},
    {"keyCreatePublicRoom",
        defaultAction(&Actions::createPublicChatRoom),
        InputCondition::INGAME,
        "createroom|createpublicroom",
        true},
    {"keyJoinRoom",
        defaultAction(&Actions::joinChatRoom),
        InputCondition::INGAME,
        "joinroom",
        true},
    {"keyLeaveRoom",
        defaultAction(&Actions::leaveChatRoom),
        InputCondition::INGAME,
        "leaveroom",
        false},
    {"keyWindowStatusBar",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::quickWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "yellowbar|quickoptions",
        false},
};

#undef defaultAction
#undef addKey
#undef joystickButton

#endif  // INPUT_INPUTACTIONMAP_H
