/*
 *  The ManaPlus Client
 *  Copyright (C) 2011-2015  The ManaPlus Developers
 *
 *  This file is part of The ManaPlus Client.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef TOUCHMANAGER_H
#define TOUCHMANAGER_H

#include "listeners/configlistener.h"

#include "enums/input/inputaction.h"

#include "gui/rect.h"

#include <vector>

#include "localconsts.h"

class Image;
class ImageCollection;
class ImageRect;
class MouseInput;

typedef void (*TouchFuncPtr) (const MouseInput &mouseInput);

const int actionsSize = static_cast<int>(InputAction::TOTAL);
const int buttonsCount = 12;

struct TouchItem final
{
    TouchItem(const std::string &text0,
              const Rect &rect0, const int type0,
              const std::string &eventPressed0,
              const std::string &eventReleased0,
              ImageRect *const images0, Image *const icon0,
              const int x0, const int y0, const int width0, const int height0,
              const TouchFuncPtr ptrAll, const TouchFuncPtr ptrPressed,
              const TouchFuncPtr ptrReleased, const TouchFuncPtr ptrOut) :
        text(text0),
        rect(rect0),
        type(type0),
        eventPressed(eventPressed0),
        eventReleased(eventReleased0),
        images(images0),
        icon(icon0),
        x(x0),
        y(y0),
        width(width0),
        height(height0),
        funcAll(ptrAll),
        funcPressed(ptrPressed),
        funcReleased(ptrReleased),
        funcOut(ptrOut)
    {
    }

    A_DELETE_COPY(TouchItem)

    std::string text;
    Rect rect;
    int type;
    std::string eventPressed;
    std::string eventReleased;
    ImageRect *images;
    Image *icon;
    int x;
    int y;
    int width;
    int height;
    TouchFuncPtr funcAll;
    TouchFuncPtr funcPressed;
    TouchFuncPtr funcReleased;
    TouchFuncPtr funcOut;
};

typedef std::vector<TouchItem*> TouchItemVector;
typedef TouchItemVector::const_iterator TouchItemVectorCIter;
typedef TouchItemVector::iterator TouchItemVectorIter;

class TouchManager final : public ConfigListener
{
    public:
        TouchManager();

        ~TouchManager();

        A_DELETE_COPY(TouchManager)

        enum Types
        {
            NORMAL = 0,
            LEFT = 1,
            RIGHT = 2
        };

        void init();

        void loadTouchItem(TouchItem **item,
                           const std::string &name,
                           const std::string &imageName,
                           const std::string &text,
                           int x, int y,
                           const int width, const int height,
                           const int type,
                           const std::string &eventPressed,
                           const std::string &eventReleased,
                           const TouchFuncPtr fAll = nullptr,
                           const TouchFuncPtr fPressed = nullptr,
                           const TouchFuncPtr fReleased = nullptr,
                           const TouchFuncPtr fOut = nullptr) A_NONNULL(2);

        void clear();

        void draw();

        void safeDraw();

        void drawText();

        bool processEvent(const MouseInput &mouseInput);

        bool isActionActive(const InputActionT index) const;

        void setActionActive(const InputActionT index,
                             const bool value)
        {
            if (static_cast<int>(index) >= 0 &&
                static_cast<int>(index) < actionsSize)
            {
                mActions[static_cast<size_t>(index)] = value;
            }
        }

        void resize(const int width, const int height);

        static void unload(TouchItem *const item);

        void unloadTouchItem(TouchItem **unloadItem);

        void optionChanged(const std::string &value) override final;

        void loadPad();

        void loadButtons();

        void loadKeyboard();

        int getPadSize() const
        { return (mJoystickSize + 2) * 50; }

        void setInGame(const bool b);

        void setTempHide(const bool b);

        void shutdown();

        static void executeAction(const std::string &event);

    private:
        TouchItem *mKeyboard;
        TouchItem *mPad;
        TouchItem *mButtons[buttonsCount] A_NONNULLPOINTER;
        TouchItemVector mObjects;
        ImageCollection *mVertexes A_NONNULLPOINTER;
        bool mActions[actionsSize];
        bool mRedraw;
        bool mShowJoystick;
        bool mShowButtons;
        bool mShowKeyboard;
        int mButtonsSize;
        int mJoystickSize;
        int mButtonsFormat;
        int mWidth;
        int mHeight;
        bool mShow;
        bool mInGame;
        bool mTempHideButtons;
};

extern TouchManager touchManager;

#endif  // TOUCHMANAGER_H
