/*
 *  The ManaPlus Client
 *  Copyright (C) 2007  Joshua Langley <joshlangley@optusnet.com.au>
 *  Copyright (C) 2009-2010  The Mana Developers
 *  Copyright (C) 2011-2015  The ManaPlus Developers
 *
 *  This file is part of The ManaPlus Client.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef INPUT_INPUTACTIONMAP_H
#define INPUT_INPUTACTIONMAP_H

#include "sdlshared.h"

#include "actions/actions.h"
#include "actions/chat.h"
#include "actions/commands.h"
#include "actions/move.h"
#include "actions/pets.h"
#include "actions/statusbar.h"
#include "actions/tabs.h"
#include "actions/target.h"
#include "actions/windows.h"

#include "input/inputactiondata.h"
#include "input/inputcondition.h"
#include "input/inputmanager.h"
#include "input/inputtype.h"

#include "localconsts.h"

#define defaultAction(name) \
        InputType::UNKNOWN, -1, \
        InputType::UNKNOWN, -1, \
        Input::GRP_DEFAULT, \
        name, \
        InputAction::NO_VALUE, 50

#define addKey(name) InputType::KEYBOARD, name
#define emptyKey InputType::UNKNOWN, -1
#define joystickButton(num) InputType::JOYSTICK, num

// keyData must be in same order as enum keyAction.
static const InputActionData inputActionData
    [static_cast<size_t>(InputAction::TOTAL)] = {
    {"keyMoveUp",
        addKey(SDLK_UP),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveUp,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyMoveDown",
        addKey(SDLK_DOWN),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveDown,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyMoveLeft",
        addKey(SDLK_LEFT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveLeft,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyMoveRight",
        addKey(SDLK_RIGHT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveRight,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyAttack",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::attack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOFOLLOW
            | InputCondition::VALIDSPEED,
        "attack",
        UseArgs_false,
        Protected_false},
    {"keyTargetAttack",
        addKey(SDLK_LCTRL),
        joystickButton(0),
        Input::GRP_DEFAULT,
        &Actions::targetAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOFOLLOW
            | InputCondition::VALIDSPEED | InputCondition::ALIVE,
        "targetattack",
        UseArgs_true,
        Protected_false},
    {"keyMoveToTarget",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveToTarget,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "movetotarget",
        UseArgs_false,
        Protected_false},
    {"keyChangeMoveToTarget",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMoveToTarget,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToHome",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveToHome,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "movetohome",
        UseArgs_false,
        Protected_false},
    {"keySetHome",
        addKey(SDLK_KP5),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::setHome,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "sethome",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPoint",
        addKey(SDLK_RSHIFT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyTalk",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::talk,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME,
        "talk",
        UseArgs_false,
        Protected_false},
    {"keyTarget",
        addKey(SDLK_LSHIFT),
        joystickButton(4),
        Input::GRP_DEFAULT,
        &Actions::stopAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME,
        "stopattack",
        UseArgs_false,
        Protected_false},
    {"keyUnTarget",
        defaultAction(&Actions::untarget),
        InputCondition::GAME,
        "untarget",
        UseArgs_false,
        Protected_false},
    {"keyTargetMonster",
        addKey(SDLK_a),
        joystickButton(3),
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::targetMonster,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET
            | InputCondition::VALIDSPEED,
        "targetmonster",
        UseArgs_false,
        Protected_false},
    {"keyTargetNPC",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::targetNPC,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "targetnpc",
        UseArgs_false,
        Protected_false},
    {"keyTargetPlayer",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::targetPlayer,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "targetplayer",
        UseArgs_false,
        Protected_false},
    {"keyPickup",
        addKey(SDLK_z),
        joystickButton(1),
        Input::GRP_DEFAULT,
        &Actions::pickup,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "pickup",
        UseArgs_false,
        Protected_false},
    {"keyChangePickupType",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changePickupType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyHideWindows",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::hideWindows,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "hidewindows",
        UseArgs_false,
        Protected_false},
    {"keyBeingSit",
        addKey(SDLK_s),
        joystickButton(2),
        Input::GRP_DEFAULT,
        &Actions::sit,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET |
            InputCondition::NOBLOCK,
        "sit",
        UseArgs_false,
        Protected_false},
    {"keyScreenshot",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::screenshot,
        InputAction::NO_VALUE, 50,
        InputCondition::NOTARGET | InputCondition::NOINPUT,
        "screenshot",
        UseArgs_false,
        Protected_true},
    {"keyTrade",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeTrade,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "changetrade",
        UseArgs_false,
        Protected_true},
    {"keyPathfind",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMapMode,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyOK",
        addKey(SDLK_SPACE),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::ok,
        InputAction::NO_VALUE, 50,
        InputCondition::NOMODAL | InputCondition::NOAWAY
            | InputCondition::NONPCINPUT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyQuit",
        addKey(SDLK_ESCAPE),
#if defined(USE_SDL2) && defined(ANDROID)
        addKey(SDLK_AC_BACK),
#else
        emptyKey,
#endif
        Input::GRP_DEFAULT,
        &Actions::quit,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "quit",
        UseArgs_false,
        Protected_true},
    {"keyShortcutsKey",
        addKey(SDLK_MENU),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut16",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut17",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut18",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut19",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyShortcut20",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyWindowHelp",
        addKey(SDLK_F1),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::helpWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "help|openhelp",
        UseArgs_false,
        Protected_false},
    {"keyWindowStatus",
        addKey(SDLK_F2),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::statusWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "status|openstatus",
        UseArgs_false,
        Protected_false},
    {"keyWindowInventory",
        addKey(SDLK_F3),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::inventoryWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "inventory|openinventory",
        UseArgs_false,
        Protected_false},
    {"keyWindowEquipment",
        addKey(SDLK_F4),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::equipmentWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "equipment|openequipment",
        UseArgs_false,
        Protected_false},
    {"keyWindowSkill",
        addKey(SDLK_F5),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::skillDialogShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "skill|skills|openskills",
        UseArgs_false,
        Protected_false},
    {"keyWindowMinimap",
        addKey(SDLK_F6),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::minimapWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "minimap|openminimap",
        UseArgs_false,
        Protected_false},
    {"keyWindowChat",
        addKey(SDLK_F7),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::chatWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "openchat",
        UseArgs_false,
        Protected_false},
    {"keyWindowShortcut",
        addKey(SDLK_F8),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::shortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "openshortcut|openshortcuts",
        UseArgs_false,
        Protected_false},
    {"keyWindowSetup",
        addKey(SDLK_F9),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::setupWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::NOTARGET | InputCondition::NOINPUT
            | InputCondition::ENABLED,
        "setup|opensetup",
        UseArgs_false,
        Protected_false},
    {"keyWindowDebug",
        addKey(SDLK_F10),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::debugWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "debug|opendebug",
        UseArgs_false,
        Protected_false},
    {"keyWindowSocial",
        addKey(SDLK_F11),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::socialWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "social|opensocial",
        UseArgs_false,
        Protected_false},
    {"keyWindowEmoteBar",
        addKey(SDLK_F12),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::emoteShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "emotes|openemotes",
        UseArgs_false,
        Protected_false},
    {"keyWindowOutfit",
        addKey(SDLK_BACKQUOTE),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::outfitWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "openoutfit|openoutfits",
        UseArgs_false,
        Protected_false},
    {"keyWindowShop",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::shopWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "shop|openshop",
        UseArgs_false,
        Protected_false},
    {"keyWindowDrop",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::dropShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "opendrop",
        UseArgs_false,
        Protected_false},
    {"keyWindowKills",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::killStatsWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "kills|openkills",
        UseArgs_false,
        Protected_false},
    {"keyWindowSpells",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::spellShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "spells|openspells",
        UseArgs_false,
        Protected_false},
    {"keyWindowBotChecker",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "",
        UseArgs_false,
        Protected_false},
    {"keyWindowOnline",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::whoIsOnlineWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "online|openonline",
        UseArgs_false,
        Protected_false},
    {"keyWindowDidYouKnow",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::didYouKnowWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "didyouknow|opendidyouknow",
        UseArgs_false,
        Protected_false},
    {"keyWindowQuests",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::questsWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "quests|openquests",
        UseArgs_false,
        Protected_false},
    {"keySocialPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevSocialTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "prevsocialtab",
        UseArgs_false,
        Protected_false},
    {"keySocialNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextSocialTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "nextsocialtab",
        UseArgs_false,
        Protected_false},
    {"keySmilie",
        addKey(SDLK_LALT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyWearOutfit",
        addKey(SDLK_RCTRL),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCopyOutfit",
        addKey(SDLK_RALT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyCopyEquipedOutfit",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::copyEquippedToOutfit,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyOutfitShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMoveToPointShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChat",
        addKey(SDLK_RETURN),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_CHAT,
        &Actions::toggleChat,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::NOAWAY
            | InputCondition::NOMODAL | InputCondition::INGAME,
        "chat",
        UseArgs_false,
        Protected_false},
    {"keyChatScrollUp",
        addKey(SDLK_PAGEUP),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::scrollChatUp,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "chatscrollup",
        UseArgs_false,
        Protected_false},
    {"keyChatScrollDown",
        addKey(SDLK_PAGEDOWN),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::scrollChatDown,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "chatscrolldown",
        UseArgs_false,
        Protected_false},
    {"keyChatPrevTab",
        addKey(SDLK_KP7),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "prevtab",
        UseArgs_false,
        Protected_false},
    {"keyChatNextTab",
        addKey(SDLK_KP9),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "nexttab",
        UseArgs_false,
        Protected_false},
    {"keyChatCloseTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::closeChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "close",
        UseArgs_false,
        Protected_false},
    {"keyChatPrevHistory",
        addKey(SDLK_KP7),
        emptyKey,
        Input::GRP_CHAT | Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyChatNextHistory",
        addKey(SDLK_KP9),
        emptyKey,
        Input::GRP_CHAT | Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyIgnoreInput1",
#ifdef USE_SDL2
        emptyKey,
#else
        addKey(SDLK_LSUPER),
#endif
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::ignoreInput,
        InputAction::NO_VALUE, 500,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyIgnoreInput2",
#ifdef USE_SDL2
        emptyKey,
#else
        addKey(SDLK_RSUPER),
#endif
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::ignoreInput,
        InputAction::NO_VALUE, 500,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_true},
    {"keyDirectUp",
        defaultAction(&Actions::directUp),
        InputCondition::GAME,
        "directup|turnup",
        UseArgs_false,
        Protected_false},
    {"keyDirectDown",
        defaultAction(&Actions::directDown),
        InputCondition::GAME,
        "directdown|turndown",
        UseArgs_false,
        Protected_false},
    {"keyDirectLeft",
        defaultAction(&Actions::directLeft),
        InputCondition::GAME,
        "directleft|turnleft",
        UseArgs_false,
        Protected_false},
    {"keyDirectRight",
        defaultAction(&Actions::directRight),
        InputCondition::GAME,
        "directright|turnright",
        UseArgs_false,
        Protected_false},
    {"keyCrazyMoves",
        addKey(SDLK_SLASH),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::crazyMoves,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChangeCrazyMoveType",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeCrazyMove,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyQuickDrop",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::dropItem0,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "drop",
        UseArgs_false,
        Protected_false},
    {"keyQuickDropN",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::dropItem,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "dropn",
        UseArgs_false,
        Protected_false},
    {"keySwitchQuickDrop",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::switchQuickDrop,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        UseArgs_false,
        Protected_true},
    {"keyMagicInma1",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::heal,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "heal",
        UseArgs_true,
        Protected_false},
    {"keyMagicItenplz",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::itenplz,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "itenplz",
        UseArgs_false,
        Protected_false},
    {"keyMagicAttack",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::magicAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOFOLLOW
            | InputCondition::VALIDSPEED,
        "magicattack",
        UseArgs_false,
        Protected_false},
    {"keySwitchMagicAttack",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMagicAttackType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keySwitchPvpAttack",
        defaultAction(&Actions::changePvpMode),
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyInvertDirection",
        defaultAction(&Actions::changeMoveType),
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyChangeAttackWeaponType",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAttackWeaponType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChangeAttackType",
        addKey(SDLK_END),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAttackType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChangeFollowMode",
        addKey(SDLK_KP1),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeFollowMode,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChangeImitationMode",
        addKey(SDLK_KP4),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeImitationMode,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_true},
    {"keyDisableGameModifiers",
        addKey(SDLK_KP8),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeGameModifier,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        UseArgs_false,
        Protected_true},
    {"keyChangeAudio",
        addKey(SDLK_KP3),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAudio,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "audio",
        UseArgs_false,
        Protected_true},
    {"keyAway",
        addKey(SDLK_KP2),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::away,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        UseArgs_false,
        Protected_false},
    {"keyRightClick",
        addKey(SDLK_TAB),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::mouseClick,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::NOAWAY
            | InputCondition::NOMODAL | InputCondition::INGAME,
        "rightclick",
        UseArgs_false,
        Protected_false},
    {"keyCameraMode",
        addKey(SDLK_KP_PLUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::camera,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyMod",
        emptyKey,
        emptyKey,
        0,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIUp",
        addKey(SDLK_UP),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIDown",
        addKey(SDLK_DOWN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUILeft",
        addKey(SDLK_LEFT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIRight",
        addKey(SDLK_RIGHT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUISelect",
        addKey(SDLK_SPACE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUISelect2",
        addKey(SDLK_RETURN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUICancel",
        addKey(SDLK_ESCAPE),
#if defined(USE_SDL2) && defined(ANDROID)
        addKey(SDLK_AC_BACK),
#else
        emptyKey,
#endif
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIHome",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIEnd",
        addKey(SDLK_END),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIDelete",
        addKey(SDLK_DELETE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIBackSpace",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUITab",
        addKey(SDLK_TAB),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIPageUp",
        addKey(SDLK_PAGEUP),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIPageDown",
        addKey(SDLK_PAGEDOWN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIInsert",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIMod",
        addKey(SDLK_LSHIFT),
        addKey(SDLK_RSHIFT),
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keySafeVideo",
        addKey(SDLK_KP_MINUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::safeVideoMode,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyStopSit",
        defaultAction(&Actions::stopSit),
        InputCondition::DEFAULT | InputCondition::INGAME,
        "sit2",
        UseArgs_false,
        Protected_false},
    {"keyShowKeyboard",
        defaultAction(&Actions::showKeyboard),
        InputCondition::DEFAULT,
        "showkeyboard",
        UseArgs_false,
        Protected_false},
    {"keyShowWindows",
        defaultAction(&Actions::showWindows),
        InputCondition::DEFAULT,
        "showwindows",
        UseArgs_false,
        Protected_false},
    {"keyChatMod",
        addKey(SDLK_LSHIFT),
        emptyKey,
        Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyMoveForward",
        defaultAction(&Actions::moveForward),
        InputCondition::ARROWKEYS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUICtrl",
        addKey(SDLK_LCTRL),
        addKey(SDLK_RCTRL),
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIb",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIc",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUId",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIe",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIf",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIh",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIk",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIu",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIv",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIw",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyShortcutsPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevShortcutsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "shortcutprevtab",
        UseArgs_false,
        Protected_false},
    {"keyShortcutsNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextShortcutsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "shortcutnexttab",
        UseArgs_false,
        Protected_false},
    {"keyCommandsPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevCommandsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "commandsprevtab",
        UseArgs_false,
        Protected_false},
    {"keyCommandsNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextCommandsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "commandsnexttab",
        UseArgs_false,
        Protected_false},
    {"keyOpenTrade",
        defaultAction(&Actions::openTrade),
        InputCondition::GAME | InputCondition::NOTARGET,
        "trade",
        UseArgs_false,
        Protected_false},
    {"keyGUIF1",
        addKey(SDLK_F1),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF2",
        addKey(SDLK_F2),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF3",
        addKey(SDLK_F3),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF4",
        addKey(SDLK_F4),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF5",
        addKey(SDLK_F5),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF6",
        addKey(SDLK_F6),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF7",
        addKey(SDLK_F7),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF8",
        addKey(SDLK_F8),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF9",
        addKey(SDLK_F9),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF10",
        addKey(SDLK_F10),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF11",
        addKey(SDLK_F11),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyGUIF12",
        addKey(SDLK_F12),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false,
        Protected_false},
    {"keyWindowAbout",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::aboutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "about",
        UseArgs_false,
        Protected_false},
    {"keyWindowUpdater",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::updaterWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME,
        "updater",
        UseArgs_false,
        Protected_false},
    {"keyTargetClosestMonster",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::targetClosestMonster,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET
            | InputCondition::VALIDSPEED,
        "targetclosestmonster",
        UseArgs_false,
        Protected_false},
    {"keyChatCloseAllTabs",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::closeAllChatTabs,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "closeall",
        UseArgs_false,
        Protected_false},
    {"keyChatIgnoreAllWhispers",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::ignoreAllWhispers,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "ignoreallwhispers",
        UseArgs_false,
        Protected_false},
    {"keyChatAnnounce",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::chatAnnounce,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "announce",
        UseArgs_true,
        Protected_false},
    {"keyIpcToggle",
        defaultAction(&Actions::ipcToggle),
        InputCondition::INGAME,
        "ipctoggle",
        UseArgs_false,
        Protected_false},
    {"keyWhere",
        defaultAction(&Actions::where),
        InputCondition::INGAME,
        "where",
        UseArgs_false,
        Protected_false},
    {"keyWho",
        defaultAction(&Actions::who),
        InputCondition::INGAME,
        "who",
        UseArgs_false,
        Protected_false},
    {"keyWhisper",
        defaultAction(&Actions::msg),
        InputCondition::INGAME,
        "w|whisper|msg",
        UseArgs_true,
        Protected_false},
    {"keyQuery",
        defaultAction(&Actions::query),
        InputCondition::INGAME,
        "q|query|j|join",
        UseArgs_true,
        Protected_false},
    {"keyClearChatTab",
        defaultAction(&Actions::clearChatTab),
        InputCondition::INGAME,
        "clear",
        UseArgs_false,
        Protected_false},
    {"keyIgnore",
        defaultAction(&Actions::chatIgnore),
        InputCondition::INGAME,
        "ignore",
        UseArgs_true,
        Protected_false},
    {"keyUnignore",
        defaultAction(&Actions::chatUnignore),
        InputCondition::INGAME,
        "unignore",
        UseArgs_true,
        Protected_false},
    {"keyFriend",
        defaultAction(&Actions::chatFriend),
        InputCondition::INGAME,
        "friend|befriend",
        UseArgs_true,
        Protected_false},
    {"keyDisregard",
        defaultAction(&Actions::chatDisregard),
        InputCondition::INGAME,
        "disregard",
        UseArgs_true,
        Protected_false},
    {"keyNeutral",
        defaultAction(&Actions::chatNeutral),
        InputCondition::INGAME,
        "neutral",
        UseArgs_true,
        Protected_false},
    {"keyBlacklist",
        defaultAction(&Actions::chatBlackList),
        InputCondition::INGAME,
        "blacklist",
        UseArgs_true,
        Protected_false},
    {"keyEnemy",
        defaultAction(&Actions::chatEnemy),
        InputCondition::INGAME,
        "enemy",
        UseArgs_true,
        Protected_false},
    {"keyErase",
        defaultAction(&Actions::chatErase),
        InputCondition::INGAME,
        "erase",
        UseArgs_true,
        Protected_false},
    {"keyCleanGraphics",
        defaultAction(&Actions::cleanGraphics),
        InputCondition::INGAME,
        "cleangraphics",
        UseArgs_false,
        Protected_false},
    {"keyCleanFonts",
        defaultAction(&Actions::cleanFonts),
        InputCondition::INGAME,
        "cleanfonts",
        UseArgs_false,
        Protected_false},
    {"keyCreateParty",
        defaultAction(&Actions::createParty),
        InputCondition::INGAME,
        "createparty",
        UseArgs_true,
        Protected_false},
    {"keyCreateGuild",
        defaultAction(&Actions::createGuild),
        InputCondition::INGAME,
        "createguild",
        UseArgs_true,
        Protected_false},
    {"keyParty",
        defaultAction(&Actions::party),
        InputCondition::INGAME,
        "party",
        UseArgs_true,
        Protected_false},
    {"keyMe",
        defaultAction(&Actions::me),
        InputCondition::INGAME,
        "me",
        UseArgs_true,
        Protected_false},
    {"keyToggle",
        defaultAction(&Actions::toggle),
        InputCondition::INGAME,
        "toggle",
        UseArgs_false,
        Protected_false},
    {"keyPresent",
        defaultAction(&Actions::present),
        InputCondition::INGAME,
        "present",
        UseArgs_false,
        Protected_false},
    {"keyPrintAll",
        defaultAction(&Actions::printAll),
        InputCondition::INGAME,
        "all",
        UseArgs_false,
        Protected_false},
    {"keyMove",
        defaultAction(&Actions::move),
        InputCondition::INGAME,
        "move",
        UseArgs_true,
        Protected_false},
    {"keySetTarget",
        defaultAction(&Actions::setTarget),
        InputCondition::INGAME,
        "target",
        UseArgs_true,
        Protected_false},
    {"keyAttackHuman",
        defaultAction(&Actions::attackHuman),
        InputCondition::GAME | InputCondition::NOFOLLOW
            | InputCondition::VALIDSPEED,
        "atkhuman",
        UseArgs_false,
        Protected_false},
    {"keyOutfit",
        defaultAction(&Actions::commandOutfit),
        InputCondition::INGAME,
        "outfit",
        UseArgs_true,
        Protected_false},
    {"keyEmote",
        defaultAction(&Actions::commandEmote),
        InputCondition::INGAME,
        "emote",
        UseArgs_true,
        Protected_false},
    {"keyEmotePet",
        defaultAction(&Actions::commandEmotePet),
        InputCondition::INGAME,
        "emotepet",
        UseArgs_true,
        Protected_false},
    {"keyAwayMessage",
        defaultAction(&Actions::awayMessage),
        InputCondition::INGAME,
        "away",
        UseArgs_true,
        Protected_false},
    {"keyPseudoAway",
        defaultAction(&Actions::pseudoAway),
        InputCondition::INGAME,
        "pseudoaway",
        UseArgs_true,
        Protected_false},
    {"keyFollow",
        defaultAction(&Actions::follow),
        InputCondition::INGAME,
        "follow",
        UseArgs_true,
        Protected_false},
    {"keyNavigate",
        defaultAction(&Actions::navigate),
        InputCondition::INGAME,
        "navigate",
        UseArgs_true,
        Protected_false},
    {"keyImitation",
        defaultAction(&Actions::imitation),
        InputCondition::INGAME,
        "imitation",
        UseArgs_true,
        Protected_false},
    {"keySendMail",
        defaultAction(&Actions::sendMail),
        InputCondition::INGAME,
        "sendmail|mailsend|mailto",
        UseArgs_true,
        Protected_false},
    {"keyTradeCommand",
        defaultAction(&Actions::trade),
        InputCondition::INGAME,
        "trade",
        UseArgs_true,
        Protected_false},
    {"keyPriceLoad",
        defaultAction(&Actions::priceLoad),
        InputCondition::INGAME,
        "priceload",
        UseArgs_false,
        Protected_false},
    {"keyPriceSave",
        defaultAction(&Actions::priceSave),
        InputCondition::INGAME,
        "pricesave",
        UseArgs_false,
        Protected_false},
    {"keyCacheInfo",
        defaultAction(&Actions::cacheInfo),
        InputCondition::INGAME,
        "cacheinfo",
        UseArgs_false,
        Protected_false},
    {"keyDisconnect",
        defaultAction(&Actions::disconnect),
        InputCondition::INGAME,
        "disconnect",
        UseArgs_false,
        Protected_false},
    {"keyUndress",
        defaultAction(&Actions::undress),
        InputCondition::INGAME,
        "undress",
        UseArgs_true,
        Protected_false},
    {"keyDirs",
        defaultAction(&Actions::dirs),
        InputCondition::INGAME,
        "dirs",
        UseArgs_false,
        Protected_false},
    {"keyInfo",
        defaultAction(&Actions::info),
        InputCondition::INGAME,
        "info",
        UseArgs_false,
        Protected_false},
    {"keyWait",
        defaultAction(&Actions::wait),
        InputCondition::INGAME,
        "wait",
        UseArgs_true,
        Protected_false},
    {"keyUptime",
        defaultAction(&Actions::uptime),
        InputCondition::INGAME,
        "uptime",
        UseArgs_false,
        Protected_false},
    {"keyAddPriorityAttack",
        defaultAction(&Actions::addPriorityAttack),
        InputCondition::INGAME,
        "addpriorityattack",
        UseArgs_true,
        Protected_false},
    {"keyAddAttack",
        defaultAction(&Actions::addAttack),
        InputCondition::INGAME,
        "addattack",
        UseArgs_true,
        Protected_false},
    {"keyRemoveAttack",
        defaultAction(&Actions::removeAttack),
        InputCondition::INGAME,
        "removeattack|removeignoreattack",
        UseArgs_true,
        Protected_false},
    {"keyAddIgnoreAttack",
        defaultAction(&Actions::addIgnoreAttack),
        InputCondition::INGAME,
        "addignoreattack",
        UseArgs_true,
        Protected_false},
    {"keyDump",
        defaultAction(&Actions::dump),
        InputCondition::INGAME,
        "dump",
        UseArgs_false,
        Protected_false},
    {"keyServerIgnoreAll",
        defaultAction(&Actions::serverIgnoreAll),
        InputCondition::INGAME,
        "serverignoreall",
        UseArgs_false,
        Protected_false},
    {"keyServerUnIgnoreAll",
        defaultAction(&Actions::serverUnIgnoreAll),
        InputCondition::INGAME,
        "serverunignoreall",
        UseArgs_false,
        Protected_false},
    {"keySetDrop",
        defaultAction(&Actions::setDrop),
        InputCondition::INGAME,
        "setdrop",
        UseArgs_true,
        Protected_false},
    {"keyError",
        defaultAction(&Actions::error),
        InputCondition::INGAME,
        "error",
        UseArgs_false,
        Protected_false},
    {"keyDumpg",
        defaultAction(&Actions::dumpGraphics),
        InputCondition::INGAME,
        "dumpg",
        UseArgs_false,
        Protected_false},
    {"keyDumpe",
        defaultAction(&Actions::dumpEnvironment),
        InputCondition::INGAME,
        "dumpe",
        UseArgs_false,
        Protected_false},
    {"keyDumpt",
        defaultAction(&Actions::dumpTests),
        InputCondition::INGAME,
        "dumpt",
        UseArgs_false,
        Protected_false},
    {"keyDumpOGL",
        defaultAction(&Actions::dumpOGL),
        InputCondition::INGAME,
        "dumpogl",
        UseArgs_false,
        Protected_false},
    {"keyDumpGL",
        defaultAction(&Actions::dumpGL),
        InputCondition::INGAME,
        "dumpgl",
        UseArgs_false,
        Protected_false},
    {"keyDumpMods",
        defaultAction(&Actions::dumpMods),
        InputCondition::INGAME,
        "dumpMods",
        UseArgs_false,
        Protected_false},
    {"keyUrl",
        defaultAction(&Actions::url),
        InputCondition::INGAME,
        "url",
        UseArgs_true,
        Protected_false},
    {"keyOpenUrl",
        defaultAction(&Actions::openUrl),
        InputCondition::INGAME,
        "open",
        UseArgs_true,
        Protected_false},
    {"keyExecute",
        defaultAction(&Actions::execute),
        InputCondition::INGAME,
        "execute",
        UseArgs_true,
        Protected_false},
    {"keyTestSdlFont",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
#if defined USE_OPENGL && defined DEBUG_SDLFONT
        &Actions::testSdlFont,
#else
        nullptr,
#endif
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
#if defined USE_OPENGL && defined DEBUG_SDLFONT
        "testsdlfont",
#else
        "",
#endif
        UseArgs_false,
        Protected_false},
    {"keyEnableHightlight",
        defaultAction(&Actions::enableHighlight),
        InputCondition::INGAME,
        "enablehighlight",
        UseArgs_false,
        Protected_false},
    {"keyDisableHightlight",
        defaultAction(&Actions::disableHighlight),
        InputCondition::INGAME,
        "disablehighlight",
        UseArgs_false,
        Protected_false},
    {"keyDontRemoveName",
        defaultAction(&Actions::dontRemoveName),
        InputCondition::INGAME,
        "dontremovename",
        UseArgs_false,
        Protected_false},
    {"keyRemoveName",
        defaultAction(&Actions::removeName),
        InputCondition::INGAME,
        "removename",
        UseArgs_false,
        Protected_false},
    {"keyDisableAway",
        defaultAction(&Actions::disableAway),
        InputCondition::INGAME,
        "disableaway",
        UseArgs_false,
        Protected_false},
    {"keyEnableAway",
        defaultAction(&Actions::enableAway),
        InputCondition::INGAME,
        "enableaway",
        UseArgs_false,
        Protected_false},
    {"keyTestParticle",
        defaultAction(&Actions::testParticle),
        InputCondition::INGAME,
        "testparticle",
        UseArgs_true,
        Protected_false},
    {"keyCreateItems",
        defaultAction(&Actions::createItems),
        InputCondition::INGAME,
        "createitems",
        UseArgs_false,
        Protected_false},
    {"keyTalkRaw",
        defaultAction(&Actions::talkRaw),
        InputCondition::INGAME,
        "talkraw",
        UseArgs_true,
        Protected_false},
    {"keyTalkPet",
        defaultAction(&Actions::talkPet),
        InputCondition::INGAME,
        "talkpet",
        UseArgs_true,
        Protected_false},
    {"keyUploadConfig",
        defaultAction(&Actions::uploadConfig),
        InputCondition::INGAME,
        "uploadconfig",
        UseArgs_false,
        Protected_false},
    {"keyUploadServerConfig",
        defaultAction(&Actions::uploadServerConfig),
        InputCondition::INGAME,
        "uploadserverconfig",
        UseArgs_false,
        Protected_false},
    {"keyUploadLog",
        defaultAction(&Actions::uploadLog),
        InputCondition::INGAME,
        "uploadlog",
        UseArgs_false,
        Protected_false},
    {"keyGm",
        defaultAction(&Actions::gm),
        InputCondition::INGAME,
        "gm",
        UseArgs_true,
        Protected_false},
    {"keyHack",
        defaultAction(&Actions::hack),
        InputCondition::INGAME,
        "hack",
        UseArgs_true,
        Protected_false},
    {"keyDebugSpawn",
        defaultAction(&Actions::debugSpawn),
        InputCondition::INGAME,
        "debugspawn",
        UseArgs_true,
        Protected_false},
    {"keyPetSmilie",
        defaultAction(nullptr),
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyPetEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyCatchPet",
        defaultAction(&Actions::catchPet),
        InputCondition::INGAME,
        "catchpet|catch",
        UseArgs_true,
        Protected_false},
    {"keyChatServerIgnoreWhisper",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::serverIgnoreWhisper,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "serverignore",
        UseArgs_false,
        Protected_false},
    {"keyChatServerUnIgnoreWhisper",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::serverUnIgnoreWhisper,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "serverunignore",
        UseArgs_false,
        Protected_false},
    {"keyMercenaryFire",
        defaultAction(&Actions::mercenaryFire),
        InputCondition::INGAME,
        "mercenaryfire|firemercenary",
        UseArgs_false,
        Protected_false},
    {"keyTargetMercenary",
        defaultAction(&Actions::targetMercenary),
        InputCondition::INGAME | InputCondition::NOTARGET,
        "targetmercenary",
        UseArgs_false,
        Protected_false},
    {"keyTargetPet",
        defaultAction(&Actions::targetPet),
        InputCondition::INGAME | InputCondition::NOTARGET,
        "targetpet",
        UseArgs_false,
        Protected_false},
    {"keyPetSetName",
        defaultAction(&Actions::setPetName),
        InputCondition::INGAME,
        "setpetname|petsetname",
        UseArgs_false,
        Protected_false},
    {"keyHomunculusSetName",
        defaultAction(&Actions::setHomunculusName),
        InputCondition::INGAME,
        "sethomunname|homunsetname|sethomunculusname|homunculussetname",
        UseArgs_false,
        Protected_false},
    {"keyHomunculusFire",
        defaultAction(&Actions::fireHomunculus),
        InputCondition::INGAME,
        "firehomunculus|homunculusfire",
        UseArgs_false,
        Protected_false},
    {"keyWindowBank",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::bankWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "bank|openbank",
        UseArgs_false,
        Protected_false},
    {"keyInventoryPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevInvTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "previnvtab",
        UseArgs_false,
        Protected_false},
    {"keyInventoryNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextInvTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "nextinvtab",
        UseArgs_false,
        Protected_false},
    {"keyContextMenu",
        defaultAction(&Actions::contextMenu),
        InputCondition::NOINPUT | InputCondition::NOAWAY
            | InputCondition::NOMODAL | InputCondition::INGAME,
        "contextmenu",
        UseArgs_false,
        Protected_false},
    {"keyBuy",
        defaultAction(&Actions::buy),
        InputCondition::GAME,
        "buy",
        UseArgs_false,
        Protected_false},
    {"keySell",
        defaultAction(&Actions::sell),
        InputCondition::GAME,
        "sell",
        UseArgs_false,
        Protected_false},
    {"keyWhisper2",
        defaultAction(&Actions::msg2),
        InputCondition::INGAME,
        "w2|whisper2|msg2",
        UseArgs_true,
        Protected_false},
    {"keyGuild",
        defaultAction(&Actions::guild),
        InputCondition::INGAME,
        "guild",
        UseArgs_true,
        Protected_false},
    {"keyNuke",
        defaultAction(&Actions::chatNuke),
        InputCondition::INGAME,
        "nuke",
        UseArgs_true,
        Protected_false},
    {"keyUse",
        defaultAction(&Actions::useItem),
        InputCondition::INGAME,
        "use|useitem",
        UseArgs_true,
        Protected_false},
    {"keyChatAdd",
        defaultAction(&Actions::chatAdd),
        InputCondition::INGAME,
        "addchat|chatadd",
        UseArgs_true,
        Protected_false},
    {"keyWhisperText",
        defaultAction(&Actions::msgText),
        InputCondition::INGAME,
        "wt|whispertext|msgtext",
        UseArgs_true,
        Protected_false},
    {"keyNavigateTo",
        defaultAction(&Actions::navigateTo),
        InputCondition::INGAME,
        "navigateto",
        UseArgs_true,
        Protected_false},
    {"keyCameraMove",
        defaultAction(&Actions::moveCamera),
        InputCondition::INGAME,
        "movecamera|cameramove",
        UseArgs_true,
        Protected_false},
    {"keyDropItem",
        defaultAction(&Actions::dropItemId),
        InputCondition::INGAME,
        "dropitem",
        UseArgs_true,
        Protected_false},
    {"keyDropItemAll",
        defaultAction(&Actions::dropItemIdAll),
        InputCondition::INGAME,
        "dropitemall",
        UseArgs_true,
        Protected_false},
    {"keyDropInv",
        defaultAction(&Actions::dropItemInv),
        InputCondition::INGAME,
        "dropinv",
        UseArgs_true,
        Protected_false},
    {"keyDropInvAll",
        defaultAction(&Actions::dropItemInvAll),
        InputCondition::INGAME,
        "dropinvall",
        UseArgs_true,
        Protected_false},
    {"keyUseInv",
        defaultAction(&Actions::useItemInv),
        InputCondition::INGAME,
        "useinv",
        UseArgs_true,
        Protected_false},
    {"keyInvToStorage",
        defaultAction(&Actions::invToStorage),
        InputCondition::INGAME,
        "invtostorage|invstorage",
        UseArgs_true,
        Protected_false},
    {"keyTradeAdd",
        defaultAction(&Actions::tradeAdd),
        InputCondition::INGAME,
        "tradeadd|addtrade",
        UseArgs_true,
        Protected_false},
    {"keyStorageToInv",
        defaultAction(&Actions::storageToInv),
        InputCondition::INGAME,
        "storagetoinv|storageinv",
        UseArgs_true,
        Protected_false},
    {"keyProtectItem",
        defaultAction(&Actions::protectItem),
        InputCondition::INGAME,
        "protectitem|itemprotect",
        UseArgs_true,
        Protected_false},
    {"keyUnProtectItem",
        defaultAction(&Actions::unprotectItem),
        InputCondition::INGAME,
        "unprotectitem|itemunprotect",
        UseArgs_true,
        Protected_false},
    {"keyKickParty",
        defaultAction(&Actions::kickParty),
        InputCondition::INGAME,
        "kickparty|partykick",
        UseArgs_true,
        Protected_false},
    {"keyAddText",
        defaultAction(&Actions::addText),
        InputCondition::INGAME,
        "addtext|textadd",
        UseArgs_true,
        Protected_false},
    {"keyKick",
        defaultAction(&Actions::kick),
        InputCondition::INGAME,
        "kick",
        UseArgs_true,
        Protected_false},
    {"keyLeaveParty",
        defaultAction(&Actions::leaveParty),
        InputCondition::INGAME,
        "leaveparty|partyleave",
        UseArgs_true,
        Protected_false},
    {"keyWarp",
        defaultAction(&Actions::warp),
        InputCondition::INGAME,
        "warp",
        UseArgs_true,
        Protected_false},
    {"keyClearChat",
        defaultAction(&Actions::clearChat),
        InputCondition::INGAME,
        "clearchat|chatclear",
        UseArgs_false,
        Protected_false},
    {"keyPetMoveUp",
        defaultAction(&Actions::petMoveUp),
        InputCondition::INGAME,
        "petmoveup|moveuppet",
        UseArgs_false,
        Protected_false},
    {"keyPetMoveDown",
        defaultAction(&Actions::petMoveDown),
        InputCondition::INGAME,
        "petmovedown|movedownpet",
        UseArgs_false,
        Protected_false},
    {"keyPetMoveLeft",
        defaultAction(&Actions::petMoveLeft),
        InputCondition::INGAME,
        "petmoveleft|moveleftpet",
        UseArgs_false,
        Protected_false},
    {"keyPetMoveRight",
        defaultAction(&Actions::petMoveRight),
        InputCondition::INGAME,
        "petmoveright|moverightpet",
        UseArgs_false,
        Protected_false},
    {"keyPetDirectUp",
        defaultAction(&Actions::petDirectUp),
        InputCondition::INGAME,
        "petdirectup|directuppet",
        UseArgs_false,
        Protected_false},
    {"keyPetDirectDown",
        defaultAction(&Actions::petDirectDown),
        InputCondition::INGAME,
        "petdirectdown|directdownpet",
        UseArgs_false,
        Protected_false},
    {"keyPetDirectLeft",
        defaultAction(&Actions::petDirectLeft),
        InputCondition::INGAME,
        "petdirectleft|directleftpet",
        UseArgs_false,
        Protected_false},
    {"keyPetDirectRight",
        defaultAction(&Actions::petDirectRight),
        InputCondition::INGAME,
        "petdirectright|directrightpet",
        UseArgs_false,
        Protected_false},
    {"keyPetAiStart",
        defaultAction(&Actions::petAiStart),
        InputCondition::INGAME,
        "petaistart|startpetai|enablepetai|petaienable",
        UseArgs_false,
        Protected_false},
    {"keyPetAiStop",
        defaultAction(&Actions::petAiStop),
        InputCondition::INGAME,
        "petaistop|stoppetai|disablepetai|petaidisable",
        UseArgs_false,
        Protected_false},
    {"keyGeneralTab",
        defaultAction(&Actions::chatGeneralTab),
        InputCondition::INGAME,
        "generaltab|generalchattab|general",
        UseArgs_false,
        Protected_false},
    {"keyDebugTab",
        defaultAction(&Actions::chatDebugTab),
        InputCondition::INGAME,
        "debugtab|debugchattab",
        UseArgs_false,
        Protected_false},
    {"keyBattleTab",
        defaultAction(&Actions::chatBattleTab),
        InputCondition::INGAME,
        "battletab|battlechattab",
        UseArgs_false,
        Protected_false},
    {"keyTradeTab",
        defaultAction(&Actions::chatTradeTab),
        InputCondition::INGAME,
        "tradetab|tradechattab",
        UseArgs_false,
        Protected_false},
    {"keyLangTab",
        defaultAction(&Actions::chatLangTab),
        InputCondition::INGAME,
        "langtab|langchattab",
        UseArgs_false,
        Protected_false},
    {"keyGmTab",
        defaultAction(&Actions::chatGmTab),
        InputCondition::INGAME,
        "gmtab|gmchattab",
        UseArgs_false,
        Protected_false},
    {"keyPartyTab",
        defaultAction(&Actions::chatPartyTab),
        InputCondition::INGAME,
        "partytab|partychattab",
        UseArgs_false,
        Protected_false},
    {"keyGuildTab",
        defaultAction(&Actions::chatGuildTab),
        InputCondition::INGAME,
        "guildtab|guildchattab",
        UseArgs_false,
        Protected_false},
    {"keyDropClear",
        defaultAction(&Actions::clearDrop),
        InputCondition::INGAME,
        "cleardrops|cleardrop|cleardropwindow|dropclear|dropsclear",
        UseArgs_false,
        Protected_false},
    {"keyWindowCart",
        defaultAction(&Actions::cartWindowShow),
        InputCondition::SHORTCUT0,
        "cart|opencart",
        UseArgs_false,
        Protected_false},
    {"keyMagicHealMostDamaged",
        defaultAction(&Actions::healmd),
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "healmd|saveass",
        UseArgs_false,
        Protected_false},
    {"keyPetMove",
        defaultAction(&Actions::petMove),
        InputCondition::INGAME,
        "movepet|petmove",
        UseArgs_true,
        Protected_false},
    {"keyHomunTalk",
        defaultAction(&Actions::homunTalk),
        InputCondition::INGAME,
        "homuntalk|merctalk",
        UseArgs_true,
        Protected_false},
    {"keyHomunEmote",
        defaultAction(&Actions::commandHomunEmote),
        InputCondition::INGAME,
        "homunemote|mercemote",
        UseArgs_true,
        Protected_false},
    {"keyHomunSmilie",
        defaultAction(nullptr),
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyHomunEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false,
        Protected_false},
    {"keyKickGuild",
        defaultAction(&Actions::kickGuild),
        InputCondition::INGAME,
        "kickguild|guildkick",
        UseArgs_true,
        Protected_false},
    {"keyHat",
        defaultAction(&Actions::hat),
        InputCondition::INGAME,
        "hat|hats|showhat",
        UseArgs_false,
        Protected_false},
    {"keyCreatePublicRoom",
        defaultAction(&Actions::createPublicChatRoom),
        InputCondition::INGAME,
        "createroom|createpublicroom",
        UseArgs_true,
        Protected_false},
    {"keyJoinRoom",
        defaultAction(&Actions::joinChatRoom),
        InputCondition::INGAME,
        "joinroom",
        UseArgs_true,
        Protected_false},
    {"keyLeaveRoom",
        defaultAction(&Actions::leaveChatRoom),
        InputCondition::INGAME,
        "leaveroom",
        UseArgs_false,
        Protected_false},
    {"keyWindowStatusBar",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::quickWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "yellowbar|quickoptions",
        UseArgs_false,
        Protected_false},
    {"keyWindowMail",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::mailWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "mailwindow|mail",
        UseArgs_false,
        Protected_false},
    {"keyConfSet",
        defaultAction(&Actions::confSet),
        InputCondition::INGAME,
        "confset|setconf",
        UseArgs_true,
        Protected_false},
    {"keyServerConfSet",
        defaultAction(&Actions::serverConfSet),
        InputCondition::INGAME,
        "servconfset|setservconf",
        UseArgs_true,
        Protected_false},
    {"keyConfGet",
        defaultAction(&Actions::confGet),
        InputCondition::INGAME,
        "confget|getconf",
        UseArgs_true,
        Protected_false},
    {"keyServerConfGet",
        defaultAction(&Actions::serverConfGet),
        InputCondition::INGAME,
        "servconfget|getservconf",
        UseArgs_true,
        Protected_false},
    {"keyChangeTargetingType",
        defaultAction(&Actions::changeTargetingType),
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false,
        Protected_false},
    {"keyTestInfo",
        defaultAction(&Actions::testInfo),
        InputCondition::INGAME,
        "testinfo",
        UseArgs_false,
        Protected_false},
};

#undef defaultAction
#undef addKey
#undef joystickButton

#endif  // INPUT_INPUTACTIONMAP_H
