/*
 *  The ManaPlus Client
 *  Copyright (C) 2007  Joshua Langley <joshlangley@optusnet.com.au>
 *  Copyright (C) 2009-2010  The Mana Developers
 *  Copyright (C) 2011-2015  The ManaPlus Developers
 *
 *  This file is part of The ManaPlus Client.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef INPUT_INPUTACTIONMAP_H
#define INPUT_INPUTACTIONMAP_H

#include "sdlshared.h"

#include "actions/actions.h"
#include "actions/chat.h"
#include "actions/commands.h"
#include "actions/move.h"
#include "actions/pets.h"
#include "actions/statusbar.h"
#include "actions/tabs.h"
#include "actions/target.h"
#include "actions/windows.h"

#include "input/inputactiondata.h"
#include "input/inputcondition.h"
#include "input/inputmanager.h"
#include "input/inputtype.h"

#include "localconsts.h"

#define defaultAction(name) \
        InputType::UNKNOWN, -1, \
        InputType::UNKNOWN, -1, \
        Input::GRP_DEFAULT, \
        name, \
        InputAction::NO_VALUE, 50

#define addKey(name) InputType::KEYBOARD, name
#define emptyKey InputType::UNKNOWN, -1
#define joystickButton(num) InputType::JOYSTICK, num

// keyData must be in same order as enum keyAction.
static const InputActionData inputActionData
    [static_cast<size_t>(InputAction::TOTAL)] = {
    {"keyMoveUp",
        addKey(SDLK_UP),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveUp,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        UseArgs_false},
    {"keyMoveDown",
        addKey(SDLK_DOWN),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveDown,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        UseArgs_false},
    {"keyMoveLeft",
        addKey(SDLK_LEFT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveLeft,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        UseArgs_false},
    {"keyMoveRight",
        addKey(SDLK_RIGHT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveRight,
        InputAction::NO_VALUE, 50,
        InputCondition::ARROWKEYS,
        "",
        UseArgs_false},
    {"keyAttack",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::attack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOFOLLOW
            | InputCondition::VALIDSPEED,
        "attack",
        UseArgs_false},
    {"keyTargetAttack",
        addKey(SDLK_LCTRL),
        joystickButton(0),
        Input::GRP_DEFAULT,
        &Actions::targetAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOFOLLOW
            | InputCondition::VALIDSPEED | InputCondition::ALIVE,
        "targetattack",
        UseArgs_true},
    {"keyMoveToTarget",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveToTarget,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "movetotarget",
        UseArgs_false},
    {"keyChangeMoveToTarget",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMoveToTarget,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyMoveToHome",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::moveToHome,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "movetohome",
        UseArgs_false},
    {"keySetHome",
        addKey(SDLK_KP5),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::setHome,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "sethome",
        UseArgs_false},
    {"keyMoveToPoint",
        addKey(SDLK_RSHIFT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false},
    {"keyTalk",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::talk,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME,
        "talk",
        UseArgs_false},
    {"keyTarget",
        addKey(SDLK_LSHIFT),
        joystickButton(4),
        Input::GRP_DEFAULT,
        &Actions::stopAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME,
        "stopattack",
        UseArgs_false},
    {"keyUnTarget",
        defaultAction(&Actions::untarget),
        InputCondition::GAME,
        "untarget",
        UseArgs_false},
    {"keyTargetMonster",
        addKey(SDLK_a),
        joystickButton(3),
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::targetMonster,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET
            | InputCondition::VALIDSPEED,
        "targetmonster",
        UseArgs_false},
    {"keyTargetNPC",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::targetNPC,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "targetnpc",
        UseArgs_false},
    {"keyTargetPlayer",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::targetPlayer,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "targetplayer",
        UseArgs_false},
    {"keyPickup",
        addKey(SDLK_z),
        joystickButton(1),
        Input::GRP_DEFAULT,
        &Actions::pickup,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "pickup",
        UseArgs_false},
    {"keyChangePickupType",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changePickupType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyHideWindows",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::hideWindows,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "hidewindows",
        UseArgs_false},
    {"keyBeingSit",
        addKey(SDLK_s),
        joystickButton(2),
        Input::GRP_DEFAULT,
        &Actions::sit,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "sit",
        UseArgs_false},
    {"keyScreenshot",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::screenshot,
        InputAction::NO_VALUE, 50,
        InputCondition::NOTARGET | InputCondition::NOINPUT,
        "screenshot",
        UseArgs_false},
    {"keyTrade",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeTrade,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "changetrade",
        UseArgs_false},
    {"keyPathfind",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMapMode,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyOK",
        addKey(SDLK_SPACE),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::ok,
        InputAction::NO_VALUE, 50,
        InputCondition::NOMODAL | InputCondition::NOAWAY
            | InputCondition::NONPCINPUT,
        "",
        UseArgs_false},
    {"keyQuit",
        addKey(SDLK_ESCAPE),
#if defined(USE_SDL2) && defined(ANDROID)
        addKey(SDLK_AC_BACK),
#else
        emptyKey,
#endif
        Input::GRP_DEFAULT,
        &Actions::quit,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "quit",
        UseArgs_false},
    {"keyShortcutsKey",
        addKey(SDLK_MENU),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false},
    {"keyShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::shortcut,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut16",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut17",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut18",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut19",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyShortcut20",
        defaultAction(&Actions::shortcut),
        InputCondition::SHORTCUT,
        "",
        UseArgs_false},
    {"keyWindowHelp",
        addKey(SDLK_F1),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::helpWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "help|openhelp",
        UseArgs_false},
    {"keyWindowStatus",
        addKey(SDLK_F2),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::statusWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "status|openstatus",
        UseArgs_false},
    {"keyWindowInventory",
        addKey(SDLK_F3),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::inventoryWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "inventory|openinventory",
        UseArgs_false},
    {"keyWindowEquipment",
        addKey(SDLK_F4),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::equipmentWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "equipment|openequipment",
        UseArgs_false},
    {"keyWindowSkill",
        addKey(SDLK_F5),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::skillDialogShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "skill|skills|openskills",
        UseArgs_false},
    {"keyWindowMinimap",
        addKey(SDLK_F6),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::minimapWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "minimap|openminimap",
        UseArgs_false},
    {"keyWindowChat",
        addKey(SDLK_F7),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::chatWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "openchat",
        UseArgs_false},
    {"keyWindowShortcut",
        addKey(SDLK_F8),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::shortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "openshortcut|openshortcuts",
        UseArgs_false},
    {"keyWindowSetup",
        addKey(SDLK_F9),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::setupWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::NOTARGET | InputCondition::NOINPUT
            | InputCondition::ENABLED,
        "setup|opensetup",
        UseArgs_false},
    {"keyWindowDebug",
        addKey(SDLK_F10),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::debugWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "debug|opendebug",
        UseArgs_false},
    {"keyWindowSocial",
        addKey(SDLK_F11),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::socialWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "social|opensocial",
        UseArgs_false},
    {"keyWindowEmoteBar",
        addKey(SDLK_F12),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::emoteShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "emotes|openemotes",
        UseArgs_false},
    {"keyWindowOutfit",
        addKey(SDLK_BACKQUOTE),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::outfitWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "openoutfit|openoutfits",
        UseArgs_false},
    {"keyWindowShop",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::shopWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "shop|openshop",
        UseArgs_false},
    {"keyWindowDrop",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::dropShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "opendrop",
        UseArgs_false},
    {"keyWindowKills",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::killStatsWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "kills|openkills",
        UseArgs_false},
    {"keyWindowSpells",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::spellShortcutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "spells|openspells",
        UseArgs_false},
    {"keyWindowBotChecker",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "",
        UseArgs_false},
    {"keyWindowOnline",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::whoIsOnlineWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "online|openonline",
        UseArgs_false},
    {"keyWindowDidYouKnow",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::didYouKnowWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "didyouknow|opendidyouknow",
        UseArgs_false},
    {"keyWindowQuests",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::questsWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "quests|openquests",
        UseArgs_false},
    {"keySocialPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevSocialTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "prevsocialtab",
        UseArgs_false},
    {"keySocialNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextSocialTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "nextsocialtab",
        UseArgs_false},
    {"keySmilie",
        addKey(SDLK_LALT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_EMOTION,
        &Actions::emote,
        InputAction::EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyWearOutfit",
        addKey(SDLK_RCTRL),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false},
    {"keyCopyOutfit",
        addKey(SDLK_RALT),
        emptyKey,
        Input::GRP_DEFAULT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false},
    {"keyCopyEquipedOutfit",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::copyEquippedToOutfit,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        UseArgs_false},
    {"keyOutfitShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyOutfitShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_OUTFIT,
        &Actions::outfit,
        InputAction::NO_VALUE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyMoveToPointShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_MOVETOPOINT,
        &Actions::moveToPoint,
        InputAction::MOVE_TO_POINT, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyChat",
        addKey(SDLK_RETURN),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_CHAT,
        &Actions::toggleChat,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::NOAWAY
            | InputCondition::NOMODAL | InputCondition::INGAME,
        "chat",
        UseArgs_false},
    {"keyChatScrollUp",
        addKey(SDLK_PAGEUP),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::scrollChatUp,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "chatscrollup",
        UseArgs_false},
    {"keyChatScrollDown",
        addKey(SDLK_PAGEDOWN),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::scrollChatDown,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "chatscrolldown",
        UseArgs_false},
    {"keyChatPrevTab",
        addKey(SDLK_KP7),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "prevtab",
        UseArgs_false},
    {"keyChatNextTab",
        addKey(SDLK_KP9),
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "nexttab",
        UseArgs_false},
    {"keyChatCloseTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::closeChatTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "close",
        UseArgs_false},
    {"keyChatPrevHistory",
        addKey(SDLK_KP7),
        emptyKey,
        Input::GRP_CHAT | Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false},
    {"keyChatNextHistory",
        addKey(SDLK_KP9),
        emptyKey,
        Input::GRP_CHAT | Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false},
    {"keyIgnoreInput1",
#ifdef USE_SDL2
        emptyKey,
#else
        addKey(SDLK_LSUPER),
#endif
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::ignoreInput,
        InputAction::NO_VALUE, 500,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyIgnoreInput2",
#ifdef USE_SDL2
        emptyKey,
#else
        addKey(SDLK_RSUPER),
#endif
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::ignoreInput,
        InputAction::NO_VALUE, 500,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyDirectUp",
        defaultAction(&Actions::directUp),
        InputCondition::GAME,
        "directup|turnup",
        UseArgs_false},
    {"keyDirectDown",
        defaultAction(&Actions::directDown),
        InputCondition::GAME,
        "directdown|turndown",
        UseArgs_false},
    {"keyDirectLeft",
        defaultAction(&Actions::directLeft),
        InputCondition::GAME,
        "directleft|turnleft",
        UseArgs_false},
    {"keyDirectRight",
        defaultAction(&Actions::directRight),
        InputCondition::GAME,
        "directright|turnright",
        UseArgs_false},
    {"keyCrazyMoves",
        addKey(SDLK_SLASH),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::crazyMoves,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        UseArgs_false},
    {"keyChangeCrazyMoveType",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeCrazyMove,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyQuickDrop",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::dropItem0,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "drop",
        UseArgs_false},
    {"keyQuickDropN",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::dropItem,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "dropn",
        UseArgs_false},
    {"keySwitchQuickDrop",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::switchQuickDrop,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        UseArgs_false},
    {"keyMagicInma1",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::heal,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "heal",
        UseArgs_true},
    {"keyMagicItenplz",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::itenplz,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "itenplz",
        UseArgs_false},
    {"keyMagicAttack",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::magicAttack,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOFOLLOW
            | InputCondition::VALIDSPEED,
        "magicattack",
        UseArgs_false},
    {"keySwitchMagicAttack",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeMagicAttackType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keySwitchPvpAttack",
        defaultAction(&Actions::changePvpMode),
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyInvertDirection",
        defaultAction(&Actions::changeMoveType),
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyChangeAttackWeaponType",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAttackWeaponType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyChangeAttackType",
        addKey(SDLK_END),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAttackType,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyChangeFollowMode",
        addKey(SDLK_KP1),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeFollowMode,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyChangeImitationMode",
        addKey(SDLK_KP4),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeImitationMode,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyDisableGameModifiers",
        addKey(SDLK_KP8),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeGameModifier,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        UseArgs_false},
    {"keyChangeAudio",
        addKey(SDLK_KP3),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::changeAudio,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "audio",
        UseArgs_false},
    {"keyAway",
        addKey(SDLK_KP2),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::away,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "",
        UseArgs_false},
    {"keyRightClick",
        addKey(SDLK_TAB),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::mouseClick,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::NOAWAY
            | InputCondition::NOMODAL | InputCondition::INGAME,
        "rightclick",
        UseArgs_false},
    {"keyCameraMode",
        addKey(SDLK_KP_PLUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::camera,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyMod",
        emptyKey,
        emptyKey,
        0,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIUp",
        addKey(SDLK_UP),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIDown",
        addKey(SDLK_DOWN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUILeft",
        addKey(SDLK_LEFT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIRight",
        addKey(SDLK_RIGHT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUISelect",
        addKey(SDLK_SPACE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUISelect2",
        addKey(SDLK_RETURN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUICancel",
        addKey(SDLK_ESCAPE),
#if defined(USE_SDL2) && defined(ANDROID)
        addKey(SDLK_AC_BACK),
#else
        emptyKey,
#endif
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIHome",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIEnd",
        addKey(SDLK_END),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIDelete",
        addKey(SDLK_DELETE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIBackSpace",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUITab",
        addKey(SDLK_TAB),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIPageUp",
        addKey(SDLK_PAGEUP),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIPageDown",
        addKey(SDLK_PAGEDOWN),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIInsert",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIMod",
        addKey(SDLK_LSHIFT),
        addKey(SDLK_RSHIFT),
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keySafeVideo",
        addKey(SDLK_KP_MINUS),
        emptyKey,
        Input::GRP_DEFAULT,
        &Actions::safeVideoMode,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyStopSit",
        defaultAction(&Actions::stopSit),
        InputCondition::DEFAULT | InputCondition::INGAME,
        "sit2",
        UseArgs_false},
    {"keyShowKeyboard",
        defaultAction(&Actions::showKeyboard),
        InputCondition::DEFAULT,
        "showkeyboard",
        UseArgs_false},
    {"keyShowWindows",
        defaultAction(&Actions::showWindows),
        InputCondition::DEFAULT,
        "showwindows",
        UseArgs_false},
    {"keyChatMod",
        addKey(SDLK_LSHIFT),
        emptyKey,
        Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false},
    {"keyMoveForward",
        defaultAction(&Actions::moveForward),
        InputCondition::ARROWKEYS,
        "",
        UseArgs_false},
    {"keyGUICtrl",
        addKey(SDLK_LCTRL),
        addKey(SDLK_RCTRL),
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIb",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIc",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUId",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIe",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIf",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIh",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIk",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIu",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIv",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIw",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_GUICHAN,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyShortcutsPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevShortcutsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "shortcutprevtab",
        UseArgs_false},
    {"keyShortcutsNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextShortcutsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "shortcutnexttab",
        UseArgs_false},
    {"keyCommandsPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevCommandsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "commandsprevtab",
        UseArgs_false},
    {"keyCommandsNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextCommandsTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "commandsnexttab",
        UseArgs_false},
    {"keyOpenTrade",
        defaultAction(&Actions::openTrade),
        InputCondition::GAME | InputCondition::NOTARGET,
        "trade",
        UseArgs_false},
    {"keyGUIF1",
        addKey(SDLK_F1),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF2",
        addKey(SDLK_F2),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF3",
        addKey(SDLK_F3),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF4",
        addKey(SDLK_F4),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF5",
        addKey(SDLK_F5),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF6",
        addKey(SDLK_F6),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF7",
        addKey(SDLK_F7),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF8",
        addKey(SDLK_F8),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF9",
        addKey(SDLK_F9),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF10",
        addKey(SDLK_F10),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF11",
        addKey(SDLK_F11),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyGUIF12",
        addKey(SDLK_F12),
        emptyKey,
        Input::GRP_GUICHAN | Input::GRP_CHAT,
        nullptr,
        InputAction::NO_VALUE, 50,
        InputCondition::DEFAULT,
        "",
        UseArgs_false},
    {"keyWindowAbout",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::aboutWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "about",
        UseArgs_false},
    {"keyWindowUpdater",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::updaterWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME,
        "updater",
        UseArgs_false},
    {"keyTargetClosestMonster",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_REPEAT,
        &Actions::targetClosestMonster,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET
            | InputCondition::VALIDSPEED,
        "targetclosestmonster",
        UseArgs_false},
    {"keyChatCloseAllTabs",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::closeAllChatTabs,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "closeall",
        UseArgs_false},
    {"keyChatIgnoreAllWhispers",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::ignoreAllWhispers,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "ignoreallwhispers",
        UseArgs_false},
    {"keyChatAnnounce",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::chatAnnounce,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "announce",
        UseArgs_true},
    {"keyIpcToggle",
        defaultAction(&Actions::ipcToggle),
        InputCondition::INGAME,
        "ipctoggle",
        UseArgs_false},
    {"keyWhere",
        defaultAction(&Actions::where),
        InputCondition::INGAME,
        "where",
        UseArgs_false},
    {"keyWho",
        defaultAction(&Actions::who),
        InputCondition::INGAME,
        "who",
        UseArgs_false},
    {"keyWhisper",
        defaultAction(&Actions::msg),
        InputCondition::INGAME,
        "w|whisper|msg",
        UseArgs_true},
    {"keyQuery",
        defaultAction(&Actions::query),
        InputCondition::INGAME,
        "q|query|j|join",
        UseArgs_true},
    {"keyClearChatTab",
        defaultAction(&Actions::clearChatTab),
        InputCondition::INGAME,
        "clear",
        UseArgs_false},
    {"keyIgnore",
        defaultAction(&Actions::chatIgnore),
        InputCondition::INGAME,
        "ignore",
        UseArgs_true},
    {"keyUnignore",
        defaultAction(&Actions::chatUnignore),
        InputCondition::INGAME,
        "unignore",
        UseArgs_true},
    {"keyFriend",
        defaultAction(&Actions::chatFriend),
        InputCondition::INGAME,
        "friend|befriend",
        UseArgs_true},
    {"keyDisregard",
        defaultAction(&Actions::chatDisregard),
        InputCondition::INGAME,
        "disregard",
        UseArgs_true},
    {"keyNeutral",
        defaultAction(&Actions::chatNeutral),
        InputCondition::INGAME,
        "neutral",
        UseArgs_true},
    {"keyBlacklist",
        defaultAction(&Actions::chatBlackList),
        InputCondition::INGAME,
        "blacklist",
        UseArgs_true},
    {"keyEnemy",
        defaultAction(&Actions::chatEnemy),
        InputCondition::INGAME,
        "enemy",
        UseArgs_true},
    {"keyErase",
        defaultAction(&Actions::chatErase),
        InputCondition::INGAME,
        "erase",
        UseArgs_true},
    {"keyCleanGraphics",
        defaultAction(&Actions::cleanGraphics),
        InputCondition::INGAME,
        "cleangraphics",
        UseArgs_false},
    {"keyCleanFonts",
        defaultAction(&Actions::cleanFonts),
        InputCondition::INGAME,
        "cleanfonts",
        UseArgs_false},
    {"keyCreateParty",
        defaultAction(&Actions::createParty),
        InputCondition::INGAME,
        "createparty",
        UseArgs_true},
    {"keyCreateGuild",
        defaultAction(&Actions::createGuild),
        InputCondition::INGAME,
        "createguild",
        UseArgs_true},
    {"keyParty",
        defaultAction(&Actions::party),
        InputCondition::INGAME,
        "party",
        UseArgs_true},
    {"keyMe",
        defaultAction(&Actions::me),
        InputCondition::INGAME,
        "me",
        UseArgs_true},
    {"keyToggle",
        defaultAction(&Actions::toggle),
        InputCondition::INGAME,
        "toggle",
        UseArgs_false},
    {"keyPresent",
        defaultAction(&Actions::present),
        InputCondition::INGAME,
        "present",
        UseArgs_false},
    {"keyPrintAll",
        defaultAction(&Actions::printAll),
        InputCondition::INGAME,
        "all",
        UseArgs_false},
    {"keyMove",
        defaultAction(&Actions::move),
        InputCondition::INGAME,
        "move",
        UseArgs_true},
    {"keySetTarget",
        defaultAction(&Actions::setTarget),
        InputCondition::INGAME,
        "target",
        UseArgs_true},
    {"keyAttackHuman",
        defaultAction(&Actions::attackHuman),
        InputCondition::GAME | InputCondition::NOFOLLOW
            | InputCondition::VALIDSPEED,
        "atkhuman",
        UseArgs_false},
    {"keyOutfit",
        defaultAction(&Actions::commandOutfit),
        InputCondition::INGAME,
        "outfit",
        UseArgs_true},
    {"keyEmote",
        defaultAction(&Actions::commandEmote),
        InputCondition::INGAME,
        "emote",
        UseArgs_true},
    {"keyEmotePet",
        defaultAction(&Actions::commandEmotePet),
        InputCondition::INGAME,
        "emotepet",
        UseArgs_true},
    {"keyAwayMessage",
        defaultAction(&Actions::awayMessage),
        InputCondition::INGAME,
        "away",
        UseArgs_true},
    {"keyPseudoAway",
        defaultAction(&Actions::pseudoAway),
        InputCondition::INGAME,
        "pseudoaway",
        UseArgs_true},
    {"keyFollow",
        defaultAction(&Actions::follow),
        InputCondition::INGAME,
        "follow",
        UseArgs_true},
    {"keyNavigate",
        defaultAction(&Actions::navigate),
        InputCondition::INGAME,
        "navigate",
        UseArgs_true},
    {"keyImitation",
        defaultAction(&Actions::imitation),
        InputCondition::INGAME,
        "imitation",
        UseArgs_true},
    {"keySendMail",
        defaultAction(&Actions::sendMail),
        InputCondition::INGAME,
        "sendmail|mailsend|mailto",
        UseArgs_true},
    {"keyTradeCommand",
        defaultAction(&Actions::trade),
        InputCondition::INGAME,
        "trade",
        UseArgs_true},
    {"keyPriceLoad",
        defaultAction(&Actions::priceLoad),
        InputCondition::INGAME,
        "priceload",
        UseArgs_false},
    {"keyPriceSave",
        defaultAction(&Actions::priceSave),
        InputCondition::INGAME,
        "pricesave",
        UseArgs_false},
    {"keyCacheInfo",
        defaultAction(&Actions::cacheInfo),
        InputCondition::INGAME,
        "cacheinfo",
        UseArgs_false},
    {"keyDisconnect",
        defaultAction(&Actions::disconnect),
        InputCondition::INGAME,
        "disconnect",
        UseArgs_false},
    {"keyUndress",
        defaultAction(&Actions::undress),
        InputCondition::INGAME,
        "undress",
        UseArgs_true},
    {"keyDirs",
        defaultAction(&Actions::dirs),
        InputCondition::INGAME,
        "dirs",
        UseArgs_false},
    {"keyInfo",
        defaultAction(&Actions::info),
        InputCondition::INGAME,
        "info",
        UseArgs_false},
    {"keyWait",
        defaultAction(&Actions::wait),
        InputCondition::INGAME,
        "wait",
        UseArgs_true},
    {"keyUptime",
        defaultAction(&Actions::uptime),
        InputCondition::INGAME,
        "uptime",
        UseArgs_false},
    {"keyAddPriorityAttack",
        defaultAction(&Actions::addPriorityAttack),
        InputCondition::INGAME,
        "addpriorityattack",
        UseArgs_true},
    {"keyAddAttack",
        defaultAction(&Actions::addAttack),
        InputCondition::INGAME,
        "addattack",
        UseArgs_true},
    {"keyRemoveAttack",
        defaultAction(&Actions::removeAttack),
        InputCondition::INGAME,
        "removeattack|removeignoreattack",
        UseArgs_true},
    {"keyAddIgnoreAttack",
        defaultAction(&Actions::addIgnoreAttack),
        InputCondition::INGAME,
        "addignoreattack",
        UseArgs_true},
    {"keyDump",
        defaultAction(&Actions::dump),
        InputCondition::INGAME,
        "dump",
        UseArgs_false},
    {"keyServerIgnoreAll",
        defaultAction(&Actions::serverIgnoreAll),
        InputCondition::INGAME,
        "serverignoreall",
        UseArgs_false},
    {"keyServerUnIgnoreAll",
        defaultAction(&Actions::serverUnIgnoreAll),
        InputCondition::INGAME,
        "serverunignoreall",
        UseArgs_false},
    {"keySetDrop",
        defaultAction(&Actions::setDrop),
        InputCondition::INGAME,
        "setdrop",
        UseArgs_true},
    {"keyError",
        defaultAction(&Actions::error),
        InputCondition::INGAME,
        "error",
        UseArgs_false},
    {"keyDumpg",
        defaultAction(&Actions::dumpGraphics),
        InputCondition::INGAME,
        "dumpg",
        UseArgs_false},
    {"keyDumpe",
        defaultAction(&Actions::dumpEnvironment),
        InputCondition::INGAME,
        "dumpe",
        UseArgs_false},
    {"keyDumpt",
        defaultAction(&Actions::dumpTests),
        InputCondition::INGAME,
        "dumpt",
        UseArgs_false},
    {"keyDumpOGL",
        defaultAction(&Actions::dumpOGL),
        InputCondition::INGAME,
        "dumpogl",
        UseArgs_false},
    {"keyDumpGL",
        defaultAction(&Actions::dumpGL),
        InputCondition::INGAME,
        "dumpgl",
        UseArgs_false},
    {"keyDumpMods",
        defaultAction(&Actions::dumpMods),
        InputCondition::INGAME,
        "dumpMods",
        UseArgs_false},
    {"keyUrl",
        defaultAction(&Actions::url),
        InputCondition::INGAME,
        "url",
        UseArgs_true},
    {"keyOpenUrl",
        defaultAction(&Actions::openUrl),
        InputCondition::INGAME,
        "open",
        UseArgs_true},
    {"keyExecute",
        defaultAction(&Actions::execute),
        InputCondition::INGAME,
        "execute",
        UseArgs_true},
    {"keyTestSdlFont",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT,
#if defined USE_OPENGL && defined DEBUG_SDLFONT
        &Actions::testSdlFont,
#else
        nullptr,
#endif
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
#if defined USE_OPENGL && defined DEBUG_SDLFONT
        "testsdlfont",
#else
        "",
#endif
        UseArgs_false},
    {"keyEnableHightlight",
        defaultAction(&Actions::enableHighlight),
        InputCondition::INGAME,
        "enablehighlight",
        UseArgs_false},
    {"keyDisableHightlight",
        defaultAction(&Actions::disableHighlight),
        InputCondition::INGAME,
        "disablehighlight",
        UseArgs_false},
    {"keyDontRemoveName",
        defaultAction(&Actions::dontRemoveName),
        InputCondition::INGAME,
        "dontremovename",
        UseArgs_false},
    {"keyRemoveName",
        defaultAction(&Actions::removeName),
        InputCondition::INGAME,
        "removename",
        UseArgs_false},
    {"keyDisableAway",
        defaultAction(&Actions::disableAway),
        InputCondition::INGAME,
        "disableaway",
        UseArgs_false},
    {"keyEnableAway",
        defaultAction(&Actions::enableAway),
        InputCondition::INGAME,
        "enableaway",
        UseArgs_false},
    {"keyTestParticle",
        defaultAction(&Actions::testParticle),
        InputCondition::INGAME,
        "testparticle",
        UseArgs_true},
    {"keyCreateItems",
        defaultAction(&Actions::createItems),
        InputCondition::INGAME,
        "createitems",
        UseArgs_false},
    {"keyTalkRaw",
        defaultAction(&Actions::talkRaw),
        InputCondition::INGAME,
        "talkraw",
        UseArgs_true},
    {"keyTalkPet",
        defaultAction(&Actions::talkPet),
        InputCondition::INGAME,
        "talkpet",
        UseArgs_true},
    {"keyUploadConfig",
        defaultAction(&Actions::uploadConfig),
        InputCondition::INGAME,
        "uploadconfig",
        UseArgs_false},
    {"keyUploadServerConfig",
        defaultAction(&Actions::uploadServerConfig),
        InputCondition::INGAME,
        "uploadserverconfig",
        UseArgs_false},
    {"keyUploadLog",
        defaultAction(&Actions::uploadLog),
        InputCondition::INGAME,
        "uploadlog",
        UseArgs_false},
    {"keyGm",
        defaultAction(&Actions::gm),
        InputCondition::INGAME,
        "gm",
        UseArgs_true},
    {"keyHack",
        defaultAction(&Actions::hack),
        InputCondition::INGAME,
        "hack",
        UseArgs_true},
    {"keyDebugSpawn",
        defaultAction(&Actions::debugSpawn),
        InputCondition::INGAME,
        "debugspawn",
        UseArgs_true},
    {"keyPetSmilie",
        defaultAction(nullptr),
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyPetEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_PET_EMOTION,
        &Actions::petEmote,
        InputAction::PET_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyCatchPet",
        defaultAction(&Actions::catchPet),
        InputCondition::INGAME,
        "catchpet|catch",
        UseArgs_true},
    {"keyChatServerIgnoreWhisper",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::serverIgnoreWhisper,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "serverignore",
        UseArgs_false},
    {"keyChatServerUnIgnoreWhisper",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::serverUnIgnoreWhisper,
        InputAction::NO_VALUE, 50,
        InputCondition::INGAME,
        "serverunignore",
        UseArgs_false},
    {"keyMercenaryFire",
        defaultAction(&Actions::mercenaryFire),
        InputCondition::INGAME,
        "mercenaryfire|firemercenary",
        UseArgs_false},
    {"keyTargetMercenary",
        defaultAction(&Actions::targetMercenary),
        InputCondition::INGAME | InputCondition::NOTARGET,
        "targetmercenary",
        UseArgs_false},
    {"keyTargetPet",
        defaultAction(&Actions::targetPet),
        InputCondition::INGAME | InputCondition::NOTARGET,
        "targetpet",
        UseArgs_false},
    {"keyPetSetName",
        defaultAction(&Actions::setPetName),
        InputCondition::INGAME,
        "setpetname|petsetname",
        UseArgs_false},
    {"keyHomunculusSetName",
        defaultAction(&Actions::setHomunculusName),
        InputCondition::INGAME,
        "sethomunname|homunsetname|sethomunculusname|homunculussetname",
        UseArgs_false},
    {"keyHomunculusFire",
        defaultAction(&Actions::fireHomunculus),
        InputCondition::INGAME,
        "firehomunculus|homunculusfire",
        UseArgs_false},
    {"keyWindowBank",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::bankWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::SHORTCUT0,
        "bank|openbank",
        UseArgs_false},
    {"keyInventoryPrevTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::prevInvTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "previnvtab",
        UseArgs_false},
    {"keyInventoryNextTab",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::nextInvTab,
        InputAction::NO_VALUE, 50,
        InputCondition::NOINPUT | InputCondition::INGAME,
        "nextinvtab",
        UseArgs_false},
    {"keyContextMenu",
        defaultAction(&Actions::contextMenu),
        InputCondition::NOINPUT | InputCondition::NOAWAY
            | InputCondition::NOMODAL | InputCondition::INGAME,
        "contextmenu",
        UseArgs_false},
    {"keyBuy",
        defaultAction(&Actions::buy),
        InputCondition::GAME,
        "buy",
        UseArgs_false},
    {"keySell",
        defaultAction(&Actions::sell),
        InputCondition::GAME,
        "sell",
        UseArgs_false},
    {"keyWhisper2",
        defaultAction(&Actions::msg2),
        InputCondition::INGAME,
        "w2|whisper2|msg2",
        UseArgs_true},
    {"keyGuild",
        defaultAction(&Actions::guild),
        InputCondition::INGAME,
        "guild",
        UseArgs_true},
    {"keyNuke",
        defaultAction(&Actions::chatNuke),
        InputCondition::INGAME,
        "nuke",
        UseArgs_true},
    {"keyUse",
        defaultAction(&Actions::useItem),
        InputCondition::INGAME,
        "use|useitem",
        UseArgs_true},
    {"keyChatAdd",
        defaultAction(&Actions::chatAdd),
        InputCondition::INGAME,
        "addchat|chatadd",
        UseArgs_true},
    {"keyWhisperText",
        defaultAction(&Actions::msgText),
        InputCondition::INGAME,
        "wt|whispertext|msgtext",
        UseArgs_true},
    {"keyNavigateTo",
        defaultAction(&Actions::navigateTo),
        InputCondition::INGAME,
        "navigateto",
        UseArgs_true},
    {"keyCameraMove",
        defaultAction(&Actions::moveCamera),
        InputCondition::INGAME,
        "movecamera|cameramove",
        UseArgs_true},
    {"keyDropItem",
        defaultAction(&Actions::dropItemId),
        InputCondition::INGAME,
        "dropitem",
        UseArgs_true},
    {"keyDropItemAll",
        defaultAction(&Actions::dropItemIdAll),
        InputCondition::INGAME,
        "dropitemall",
        UseArgs_true},
    {"keyDropInv",
        defaultAction(&Actions::dropItemInv),
        InputCondition::INGAME,
        "dropinv",
        UseArgs_true},
    {"keyDropInvAll",
        defaultAction(&Actions::dropItemInvAll),
        InputCondition::INGAME,
        "dropinvall",
        UseArgs_true},
    {"keyUseInv",
        defaultAction(&Actions::useItemInv),
        InputCondition::INGAME,
        "useinv",
        UseArgs_true},
    {"keyInvToStorage",
        defaultAction(&Actions::invToStorage),
        InputCondition::INGAME,
        "invtostorage|invstorage",
        UseArgs_true},
    {"keyTradeAdd",
        defaultAction(&Actions::tradeAdd),
        InputCondition::INGAME,
        "tradeadd|addtrade",
        UseArgs_true},
    {"keyStorageToInv",
        defaultAction(&Actions::storageToInv),
        InputCondition::INGAME,
        "storagetoinv|storageinv",
        UseArgs_true},
    {"keyProtectItem",
        defaultAction(&Actions::protectItem),
        InputCondition::INGAME,
        "protectitem|itemprotect",
        UseArgs_true},
    {"keyUnProtectItem",
        defaultAction(&Actions::unprotectItem),
        InputCondition::INGAME,
        "unprotectitem|itemunprotect",
        UseArgs_true},
    {"keyKickParty",
        defaultAction(&Actions::kickParty),
        InputCondition::INGAME,
        "kickparty|partykick",
        UseArgs_true},
    {"keyAddText",
        defaultAction(&Actions::addText),
        InputCondition::INGAME,
        "addtext|textadd",
        UseArgs_true},
    {"keyKick",
        defaultAction(&Actions::kick),
        InputCondition::INGAME,
        "kick",
        UseArgs_true},
    {"keyLeaveParty",
        defaultAction(&Actions::leaveParty),
        InputCondition::INGAME,
        "leaveparty|partyleave",
        UseArgs_true},
    {"keyWarp",
        defaultAction(&Actions::warp),
        InputCondition::INGAME,
        "warp",
        UseArgs_true},
    {"keyClearChat",
        defaultAction(&Actions::clearChat),
        InputCondition::INGAME,
        "clearchat|chatclear",
        UseArgs_false},
    {"keyPetMoveUp",
        defaultAction(&Actions::petMoveUp),
        InputCondition::INGAME,
        "petmoveup|moveuppet",
        UseArgs_false},
    {"keyPetMoveDown",
        defaultAction(&Actions::petMoveDown),
        InputCondition::INGAME,
        "petmovedown|movedownpet",
        UseArgs_false},
    {"keyPetMoveLeft",
        defaultAction(&Actions::petMoveLeft),
        InputCondition::INGAME,
        "petmoveleft|moveleftpet",
        UseArgs_false},
    {"keyPetMoveRight",
        defaultAction(&Actions::petMoveRight),
        InputCondition::INGAME,
        "petmoveright|moverightpet",
        UseArgs_false},
    {"keyPetDirectUp",
        defaultAction(&Actions::petDirectUp),
        InputCondition::INGAME,
        "petdirectup|directuppet",
        UseArgs_false},
    {"keyPetDirectDown",
        defaultAction(&Actions::petDirectDown),
        InputCondition::INGAME,
        "petdirectdown|directdownpet",
        UseArgs_false},
    {"keyPetDirectLeft",
        defaultAction(&Actions::petDirectLeft),
        InputCondition::INGAME,
        "petdirectleft|directleftpet",
        UseArgs_false},
    {"keyPetDirectRight",
        defaultAction(&Actions::petDirectRight),
        InputCondition::INGAME,
        "petdirectright|directrightpet",
        UseArgs_false},
    {"keyPetAiStart",
        defaultAction(&Actions::petAiStart),
        InputCondition::INGAME,
        "petaistart|startpetai|enablepetai|petaienable",
        UseArgs_false},
    {"keyPetAiStop",
        defaultAction(&Actions::petAiStop),
        InputCondition::INGAME,
        "petaistop|stoppetai|disablepetai|petaidisable",
        UseArgs_false},
    {"keyGeneralTab",
        defaultAction(&Actions::chatGeneralTab),
        InputCondition::INGAME,
        "generaltab|generalchattab|general",
        UseArgs_false},
    {"keyDebugTab",
        defaultAction(&Actions::chatDebugTab),
        InputCondition::INGAME,
        "debugtab|debugchattab",
        UseArgs_false},
    {"keyBattleTab",
        defaultAction(&Actions::chatBattleTab),
        InputCondition::INGAME,
        "battletab|battlechattab",
        UseArgs_false},
    {"keyTradeTab",
        defaultAction(&Actions::chatTradeTab),
        InputCondition::INGAME,
        "tradetab|tradechattab",
        UseArgs_false},
    {"keyLangTab",
        defaultAction(&Actions::chatLangTab),
        InputCondition::INGAME,
        "langtab|langchattab",
        UseArgs_false},
    {"keyGmTab",
        defaultAction(&Actions::chatGmTab),
        InputCondition::INGAME,
        "gmtab|gmchattab",
        UseArgs_false},
    {"keyPartyTab",
        defaultAction(&Actions::chatPartyTab),
        InputCondition::INGAME,
        "partytab|partychattab",
        UseArgs_false},
    {"keyGuildTab",
        defaultAction(&Actions::chatGuildTab),
        InputCondition::INGAME,
        "guildtab|guildchattab",
        UseArgs_false},
    {"keyDropClear",
        defaultAction(&Actions::clearDrop),
        InputCondition::INGAME,
        "cleardrops|cleardrop|cleardropwindow|dropclear|dropsclear",
        UseArgs_false},
    {"keyWindowCart",
        defaultAction(&Actions::cartWindowShow),
        InputCondition::SHORTCUT0,
        "cart|opencart",
        UseArgs_false},
    {"keyMagicHealMostDamaged",
        defaultAction(&Actions::healmd),
        InputCondition::GAME | InputCondition::VALIDSPEED,
        "healmd|saveass",
        UseArgs_false},
    {"keyPetMove",
        defaultAction(&Actions::petMove),
        InputCondition::INGAME,
        "movepet|petmove",
        UseArgs_true},
    {"keyHomunTalk",
        defaultAction(&Actions::homunTalk),
        InputCondition::INGAME,
        "homuntalk|merctalk",
        UseArgs_true},
    {"keyHomunEmote",
        defaultAction(&Actions::commandHomunEmote),
        InputCondition::INGAME,
        "homunemote|mercemote",
        UseArgs_true},
    {"keyHomunSmilie",
        defaultAction(nullptr),
        InputCondition::DEFAULT | InputCondition::INGAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut1",
        addKey(SDLK_1),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut2",
        addKey(SDLK_2),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut3",
        addKey(SDLK_3),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut4",
        addKey(SDLK_4),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut5",
        addKey(SDLK_5),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut6",
        addKey(SDLK_6),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut7",
        addKey(SDLK_7),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut8",
        addKey(SDLK_8),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut9",
        addKey(SDLK_9),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut10",
        addKey(SDLK_0),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut11",
        addKey(SDLK_MINUS),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut12",
        addKey(SDLK_EQUALS),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut13",
        addKey(SDLK_BACKSPACE),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut14",
        addKey(SDLK_INSERT),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut15",
        addKey(SDLK_HOME),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut16",
        addKey(SDLK_q),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut17",
        addKey(SDLK_w),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut18",
        addKey(SDLK_e),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut19",
        addKey(SDLK_r),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut20",
        addKey(SDLK_t),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut21",
        addKey(SDLK_y),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut22",
        addKey(SDLK_u),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut23",
        addKey(SDLK_i),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut24",
        addKey(SDLK_o),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut25",
        addKey(SDLK_p),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut26",
        addKey(SDLK_LEFTBRACKET),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut27",
        addKey(SDLK_RIGHTBRACKET),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut28",
        addKey(SDLK_BACKSLASH),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut29",
        addKey(SDLK_a),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut30",
        addKey(SDLK_s),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut31",
        addKey(SDLK_d),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut32",
        addKey(SDLK_f),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut33",
        addKey(SDLK_g),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut34",
        addKey(SDLK_h),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut35",
        addKey(SDLK_j),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut36",
        addKey(SDLK_k),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut37",
        addKey(SDLK_l),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut38",
        addKey(SDLK_SEMICOLON),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut39",
        addKey(SDLK_QUOTE),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut40",
        addKey(SDLK_z),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut41",
        addKey(SDLK_x),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut42",
        addKey(SDLK_c),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut43",
        addKey(SDLK_v),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut44",
        addKey(SDLK_b),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut45",
        addKey(SDLK_n),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut46",
        addKey(SDLK_m),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut47",
        addKey(SDLK_COMMA),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyHomunEmoteShortcut48",
        addKey(SDLK_PERIOD),
        emptyKey,
        Input::GRP_HOMUN_EMOTION,
        &Actions::homunEmote,
        InputAction::HOMUN_EMOTE, 100,
        InputCondition::GAME,
        "",
        UseArgs_false},
    {"keyKickGuild",
        defaultAction(&Actions::kickGuild),
        InputCondition::INGAME,
        "kickguild|guildkick",
        UseArgs_true},
    {"keyHat",
        defaultAction(&Actions::hat),
        InputCondition::INGAME,
        "hat|hats|showhat",
        UseArgs_false},
    {"keyCreatePublicRoom",
        defaultAction(&Actions::createPublicChatRoom),
        InputCondition::INGAME,
        "createroom|createpublicroom",
        UseArgs_true},
    {"keyJoinRoom",
        defaultAction(&Actions::joinChatRoom),
        InputCondition::INGAME,
        "joinroom",
        UseArgs_true},
    {"keyLeaveRoom",
        defaultAction(&Actions::leaveChatRoom),
        InputCondition::INGAME,
        "leaveroom",
        UseArgs_false},
    {"keyWindowStatusBar",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::quickWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "yellowbar|quickoptions",
        UseArgs_false},
    {"keyWindowMail",
        emptyKey,
        emptyKey,
        Input::GRP_DEFAULT | Input::GRP_GUI,
        &Actions::mailWindowShow,
        InputAction::NO_VALUE, 50,
        InputCondition::GAME | InputCondition::NOTARGET,
        "mailwindow|mail",
        UseArgs_false},
    {"keyConfSet",
        defaultAction(&Actions::confSet),
        InputCondition::INGAME,
        "confset|setconf",
        UseArgs_true},
    {"keyServerConfSet",
        defaultAction(&Actions::serverConfSet),
        InputCondition::INGAME,
        "servconfset|setservconf",
        UseArgs_true},
    {"keyConfGet",
        defaultAction(&Actions::confGet),
        InputCondition::INGAME,
        "confget|getconf",
        UseArgs_true},
    {"keyServerConfGet",
        defaultAction(&Actions::serverConfGet),
        InputCondition::INGAME,
        "servconfget|getservconf",
        UseArgs_true},
    {"keyChangeTargetingType",
        defaultAction(&Actions::changeTargetingType),
        InputCondition::GAME | InputCondition::VALIDSPEED
            | InputCondition::EMODS,
        "",
        UseArgs_false},
    {"keyTestInfo",
        defaultAction(&Actions::testInfo),
        InputCondition::INGAME,
        "testinfo",
        UseArgs_false},
};

#undef defaultAction
#undef addKey
#undef joystickButton

#endif  // INPUT_INPUTACTIONMAP_H
