// license:BSD-3-Clause
// copyright-holders:David Haywood

// TV Word Processor, with printer

#include "emu.h"
#include "cpu/nec/v25.h"
#include "softlist_dev.h"

namespace {

class tvdear_state : public driver_device
{
public:
	tvdear_state(const machine_config &mconfig, device_type type, const char *tag)
		: driver_device(mconfig, type, tag)
		, m_maincpu(*this, "systemcpu")
		, m_keyboard(*this, "KEY%u", 0U)
	{
	}

	void tvdear(machine_config &config);

protected:
	virtual void machine_start() override ATTR_COLD;

private:
	void p0_w(u8 data);
	u8 pt_r();

	void mem_map(address_map &map);
	void io_map(address_map &map);

	required_device<v25_device> m_maincpu;
	required_ioport_array<10> m_keyboard;

	u8 m_p0 = 0;
};

void tvdear_state::machine_start()
{
	save_item(NAME(m_p0));
}

void tvdear_state::p0_w(u8 data)
{
	m_p0 = data;
}

u8 tvdear_state::pt_r()
{
	if (m_p0 < 0xa0)
		return m_keyboard[m_p0 >> 4]->read();
	else
		return 0xff;
}

void tvdear_state::mem_map(address_map &map)
{
	map(0x00000, 0x07fff).ram();
	map(0x10000, 0x17fff).ram();
	map(0x80000, 0xfffff).rom().region("maincpu", 0x080000);
}

void tvdear_state::io_map(address_map &map)
{
	map(0x0500, 0x0500).nopw(); // watchdog?
}

static INPUT_PORTS_START(tvdear)
	PORT_START("KEY0")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('O') PORT_CHAR(0x3089) PORT_CODE(KEYCODE_O) // ら
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('9') PORT_CHAR(0x3088) PORT_CODE(KEYCODE_9) // よ
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('.') PORT_CHAR(0x308b) PORT_CODE(KEYCODE_STOP) // る
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('L') PORT_CHAR(0x308a) PORT_CODE(KEYCODE_L) // り
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 001D")
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 001F")
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 0016")

	PORT_START("KEY1")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('P') PORT_CHAR(0x305b) PORT_CODE(KEYCODE_P) // せ
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('0') PORT_CHAR(0x308f) PORT_CODE(KEYCODE_0) // わ
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('%') PORT_CHAR(0x307b) PORT_CODE(KEYCODE_MINUS) // ほ
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(';') PORT_CHAR(0x308c) PORT_CODE(KEYCODE_COLON) // れ
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('&') PORT_CHAR(0x308d) PORT_CODE(KEYCODE_RSHIFT) // ろ
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('(') PORT_CHAR(0x309b) PORT_CODE(KEYCODE_OPENBRACE) // ■゛
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(':') PORT_CHAR(0x3051) PORT_CODE(KEYCODE_QUOTE) // け
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('/') PORT_CHAR(0x3081) PORT_CODE(KEYCODE_SLASH) // め

	PORT_START("KEY2")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('U') PORT_CHAR(0x306a) PORT_CODE(KEYCODE_U) // な
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('7') PORT_CHAR(0x3084) PORT_CODE(KEYCODE_7) // や
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('M') PORT_CHAR(0x3082) PORT_CODE(KEYCODE_M) // も
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('J') PORT_CHAR(0x307e) PORT_CODE(KEYCODE_J) // ま
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('6') PORT_CHAR(0x304a) PORT_CODE(KEYCODE_6) // お
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('Y') PORT_CHAR(0x3093) PORT_CODE(KEYCODE_Y) // ん
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('H') PORT_CHAR(0x304f) PORT_CODE(KEYCODE_H) // く
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('N') PORT_CHAR(0x307f) PORT_CODE(KEYCODE_N) // み

	PORT_START("KEY3")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('I') PORT_CHAR(0x306b) PORT_CODE(KEYCODE_I) // に
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('8') PORT_CHAR(0x3086) PORT_CODE(KEYCODE_8) // ゆ
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(',') PORT_CHAR(0x306d) PORT_CODE(KEYCODE_COMMA) // ね
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('K') PORT_CHAR(0x306e) PORT_CODE(KEYCODE_K) // の
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(0x00a5) PORT_CHAR(0x3078) PORT_CODE(KEYCODE_EQUALS) // ¥ へ
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(')') PORT_CHAR(0x309c) PORT_CODE(KEYCODE_CLOSEBRACE) // ■゜
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 0015")
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_UNUSED)

	PORT_START("KEY4")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('W') PORT_CHAR(0x3066) PORT_CODE(KEYCODE_W) // て
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('2') PORT_CHAR(0x3075) PORT_CODE(KEYCODE_2) // ふ
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('X') PORT_CHAR(0x3055) PORT_CODE(KEYCODE_X) // さ
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('S') PORT_CHAR(0x3068) PORT_CODE(KEYCODE_S) // と
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 0011")
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Special Key L")
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 0006")
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 0009")

	PORT_START("KEY5")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('R') PORT_CHAR(0x3059) PORT_CODE(KEYCODE_R) // す
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('4') PORT_CHAR(0x3046) PORT_CODE(KEYCODE_4) // う
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('V') PORT_CHAR(0x3072) PORT_CODE(KEYCODE_V) // ひ
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('F') PORT_CHAR(0x306f) PORT_CODE(KEYCODE_F) // は
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('5') PORT_CHAR(0x3048) PORT_CODE(KEYCODE_5) // え
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('T') PORT_CHAR(0x304b) PORT_CODE(KEYCODE_T) // か
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('G') PORT_CHAR(0x304d) PORT_CODE(KEYCODE_G) // き
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('B') PORT_CHAR(0x3053) PORT_CODE(KEYCODE_B) // こ

	PORT_START("KEY6")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR(0x2014) PORT_CHAR(0x30fc) PORT_CODE(KEYCODE_BACKSLASH) // — ー
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('\'') PORT_CHAR(0x3080) PORT_CODE(KEYCODE_ENTER) // む
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 001E")
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 001C")
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 0014")
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME(u8"\u30B9\u30DA\u30FC\u30B9 (Space)") PORT_CHAR(0x20) PORT_CODE(KEYCODE_SPACE) // スペース

	PORT_START("KEY7")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('E') PORT_CHAR(0x3044) PORT_CODE(KEYCODE_E) // い
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('3') PORT_CHAR(0x3042) PORT_CODE(KEYCODE_3) // あ
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('C') PORT_CHAR(0x305d) PORT_CODE(KEYCODE_C) // そ
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('D') PORT_CHAR(0x3057) PORT_CODE(KEYCODE_D) // し
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Special Key H")
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 0012")
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 0013")
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_UNUSED)

	PORT_START("KEY8")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('Q') PORT_CHAR(0x305f) PORT_CODE(KEYCODE_Q) // た
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('1') PORT_CHAR(0x306c) PORT_CODE(KEYCODE_1) // ぬ
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('Z') PORT_CHAR(0x3064) PORT_CODE(KEYCODE_Z) // つ
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_CHAR('A') PORT_CHAR(0x3061) PORT_CODE(KEYCODE_A) // ち
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 0003")
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 000F")
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_UNUSED)

	PORT_START("KEY9")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 000B")
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 007F")
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 000D")
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_KEYBOARD) PORT_NAME("Key 0017")
INPUT_PORTS_END

void tvdear_state::tvdear(machine_config &config)
{
	V25(config, m_maincpu, 16000000); // NEC D70320DGJ-8; XTAL marked 16AKSS5HT
	m_maincpu->set_addrmap(AS_PROGRAM, &tvdear_state::mem_map);
	m_maincpu->set_addrmap(AS_IO, &tvdear_state::io_map);
	m_maincpu->p0_out_cb().set(FUNC(tvdear_state::p0_w));
	m_maincpu->pt_in_cb().set(FUNC(tvdear_state::pt_r));

	//MB90076(config, "tvvc", 14318181); // XTAL marked 14AKSS5JT

	SOFTWARE_LIST(config, "cart_list").set_original("tvdear");
}

ROM_START(tvdear)
	ROM_REGION(0x200000, "maincpu", 0)
	ROM_LOAD("d23c160000.u5", 0x00000, 0x200000, CRC(41ec9890) SHA1(20cfdfec7eeb39a9ce971f23fdc97b42a5d68301) )
ROM_END

} // anonymous namespace

CONS( 1995, tvdear,  0,          0,  tvdear,  tvdear, tvdear_state, empty_init, "Takara", "TV Dear Multi Word Processor", MACHINE_IS_SKELETON | MACHINE_NODEVICE_PRINTER ) // テレビディア マルチワープロ
