// license:BSD-3-Clause
// copyright-holders:Bryan McPhail
#include "emu.h"
#include "includes/deco32.h"

/******************************************************************************/

WRITE32_MEMBER( deco32_state::pri_w )
{
	m_pri = data;
}

WRITE32_MEMBER( dragngun_state::sprite_control_w )
{
	m_sprite_ctrl = data;
}

WRITE32_MEMBER( dragngun_state::spriteram_dma_w )
{
	/* DMA spriteram to private sprite chip area, and clear cpu ram */
	m_spriteram->copy();
	memset(m_spriteram->live(),0,0x2000);
}

/******************************************************************************/

/* Later games have double buffered paletteram - the real palette ram is
only updated on a DMA call */

WRITE32_MEMBER( deco32_state::buffered_palette_w )
{
	COMBINE_DATA(&m_generic_paletteram_32[offset]);
	m_dirty_palette[offset]=1;
}

WRITE32_MEMBER( deco32_state::palette_dma_w )
{
	for (int i = 0; i < m_palette->entries(); i++)
	{
		if (m_dirty_palette[i])
		{
			m_dirty_palette[i] = 0;

			uint8_t b = (m_generic_paletteram_32[i] >>16) & 0xff;
			uint8_t g = (m_generic_paletteram_32[i] >> 8) & 0xff;
			uint8_t r = (m_generic_paletteram_32[i] >> 0) & 0xff;

			m_palette->set_pen_color(i,rgb_t(r,g,b));
		}
	}
}

/******************************************************************************/

void deco32_state::video_start()
{
	save_item(NAME(m_pri));
}

void deco32_state::allocate_spriteram(int chip)
{
	m_spriteram16[chip] = std::make_unique<uint16_t[]>(0x2000/4);
	m_spriteram16_buffered[chip] = std::make_unique<uint16_t[]>(0x2000/4);
	save_pointer(NAME(m_spriteram16[chip].get()), 0x2000/4, chip);
	save_pointer(NAME(m_spriteram16_buffered[chip].get()), 0x2000/4, chip);
}

void deco32_state::allocate_buffered_palette()
{
	m_dirty_palette = make_unique_clear<uint8_t[]>(2048);
	save_pointer(NAME(m_dirty_palette.get()), 2048);
}

void deco32_state::allocate_rowscroll(int size1, int size2, int size3, int size4)
{
	m_pf_rowscroll[0] = make_unique_clear<uint16_t[]>(size1);
	m_pf_rowscroll[1] = make_unique_clear<uint16_t[]>(size2);
	m_pf_rowscroll[2] = make_unique_clear<uint16_t[]>(size3);
	m_pf_rowscroll[3] = make_unique_clear<uint16_t[]>(size4);
	save_pointer(NAME(m_pf_rowscroll[0].get()), size1);
	save_pointer(NAME(m_pf_rowscroll[1].get()), size2);
	save_pointer(NAME(m_pf_rowscroll[2].get()), size3);
	save_pointer(NAME(m_pf_rowscroll[3].get()), size4);
}

VIDEO_START_MEMBER( captaven_state, captaven )
{
	deco32_state::allocate_spriteram(0);
	deco32_state::allocate_rowscroll(0x4000/4, 0x2000/4, 0x4000/4, 0x2000/4);
	deco32_state::video_start();
}

VIDEO_START_MEMBER( fghthist_state, fghthist )
{
	m_sprgen[0]->alloc_sprite_bitmap();
	deco32_state::allocate_spriteram(0);
	deco32_state::allocate_rowscroll(0x2000/4, 0x2000/4, 0x2000/4, 0x2000/4);
	deco32_state::allocate_buffered_palette();
	deco32_state::video_start();
}

VIDEO_START_MEMBER( nslasher_state, nslasher )
{
	int width, height;
	width = m_screen->width();
	height = m_screen->height();
	m_tilemap_alpha_bitmap=std::make_unique<bitmap_ind16>(width, height );
	for (int chip = 0; chip < 2; chip++)
	{
		m_sprgen[chip]->alloc_sprite_bitmap();
		deco32_state::allocate_spriteram(chip);
	}
	deco32_state::allocate_rowscroll(0x2000/4, 0x2000/4, 0x2000/4, 0x2000/4);
	deco32_state::video_start();
}

VIDEO_START_MEMBER( dragngun_state, dragngun )
{
	m_screen->register_screen_bitmap(m_temp_render_bitmap);
	deco32_state::allocate_rowscroll(0x4000/4, 0x2000/4, 0x4000/4, 0x2000/4);
	deco32_state::allocate_buffered_palette();
	save_item(NAME(m_sprite_ctrl));
}
/******************************************************************************/


/******************************************************************************/

uint32_t captaven_state::screen_update_captaven(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	address_space &space = machine().dummy_space();
	uint16_t flip = m_deco_tilegen[0]->pf_control_r(space, 0, 0xffff);
	flip_screen_set(BIT(flip, 7));
	m_sprgen[0]->set_flip_screen(BIT(flip, 7));

	screen.priority().fill(0, cliprect);
	bitmap.fill(m_palette->pen(0x000), cliprect); // Palette index not confirmed

	m_deco_tilegen[1]->set_pf1_8bpp_mode(1);

	m_deco_tilegen[0]->pf_update(m_pf_rowscroll[0].get(), m_pf_rowscroll[1].get());
	m_deco_tilegen[1]->pf_update(m_pf_rowscroll[2].get(), m_pf_rowscroll[3].get());

	// pf4 not used (because pf3 is in 8bpp mode)

	if ((m_pri&1)==0)
	{
		m_deco_tilegen[1]->tilemap_1_draw(screen, bitmap, cliprect, 0, 1);
		m_deco_tilegen[0]->tilemap_2_draw(screen, bitmap, cliprect, 0, 2);
	}
	else
	{
		m_deco_tilegen[0]->tilemap_2_draw(screen, bitmap, cliprect, 0, 1);
		m_deco_tilegen[1]->tilemap_1_draw(screen, bitmap, cliprect, 0, 2);
	}

	m_deco_tilegen[0]->tilemap_1_draw(screen, bitmap, cliprect, 0, 4);

	m_sprgen[0]->set_alt_format(true);
	m_sprgen[0]->draw_sprites(bitmap, cliprect, m_spriteram16_buffered[0].get(), 0x400); // only low half of sprite ram is used?

	return 0;
}

uint32_t dragngun_state::screen_update_dragngun(screen_device &screen, bitmap_rgb32 &bitmap, const rectangle &cliprect)
{
	screen.priority().fill(0, cliprect);
	bitmap.fill(m_palette->black_pen(), cliprect);

	m_deco_tilegen[0]->pf_update(m_pf_rowscroll[0].get(), m_pf_rowscroll[1].get());
	m_deco_tilegen[1]->pf_update(m_pf_rowscroll[2].get(), m_pf_rowscroll[3].get());

	//m_deco_tilegen[0]->set_pf3_8bpp_mode(1); // despite being 8bpp this doesn't require the same shifting as captaven, why not?

	m_deco_tilegen[1]->tilemap_2_draw(screen, bitmap, cliprect, 0, 1); // it uses pf3 in 8bpp mode instead, like captaven
	m_deco_tilegen[1]->tilemap_1_draw(screen, bitmap, cliprect, 0, 2);
	m_deco_tilegen[0]->tilemap_2_draw(screen, bitmap, cliprect, 0, 4);
	m_deco_tilegen[0]->tilemap_1_draw(screen, bitmap, cliprect, 0, 8);

	// zooming sprite draw is very slow, and sprites are buffered.. however, one of the levels attempts to use
	// partial updates for every line, which causes things to be very slow... the sprites appear to support
	// multiple layers of alpha, so rendering to a buffer for layer isn't easy (maybe there are multiple sprite
	// chips at work?)
	//
	// really, it needs optimizing ..
	// so for now we only draw these 2 layers on the last update call
	if (cliprect.max_y == 247)
	{
		rectangle clip(cliprect.min_x, cliprect.max_x, 8, 247);

		m_sprgenzoom->dragngun_draw_sprites(bitmap,clip,m_spriteram->buffer(), m_sprite_layout_ram[0], m_sprite_layout_ram[1], m_sprite_lookup_ram[0], m_sprite_lookup_ram[1], m_sprite_ctrl, screen.priority(), m_temp_render_bitmap );

	}

	return 0;
}


uint32_t fghthist_state::screen_update_fghthist(screen_device &screen, bitmap_rgb32 &bitmap, const rectangle &cliprect)
{
	screen.priority().fill(0, cliprect);
	bitmap.fill(m_palette->pen(0x300), cliprect); // Palette index not confirmed

	m_deco_tilegen[0]->pf_update(m_pf_rowscroll[0].get(), m_pf_rowscroll[1].get());
	m_deco_tilegen[1]->pf_update(m_pf_rowscroll[2].get(), m_pf_rowscroll[3].get());

	// sprites are flipped relative to tilemaps
	m_sprgen[0]->set_flip_screen(true);
	m_sprgen[0]->draw_sprites(bitmap, cliprect, m_spriteram16_buffered[0].get(), 0x800);

	/* Draw screen */
	m_deco_tilegen[1]->tilemap_2_draw(screen, bitmap, cliprect, 0, 1);

	if(m_pri&1)
	{
		m_deco_tilegen[0]->tilemap_2_draw(screen, bitmap, cliprect, 0, 2);
		m_sprgen[0]->inefficient_copy_sprite_bitmap(bitmap, cliprect, 0x0800, 0x0800, 1024, 0x1ff);
		m_deco_tilegen[1]->tilemap_1_draw(screen, bitmap, cliprect, 0, 4);
	}
	else
	{
		m_deco_tilegen[1]->tilemap_1_draw(screen, bitmap, cliprect, 0, 2);
		m_sprgen[0]->inefficient_copy_sprite_bitmap(bitmap, cliprect, 0x0800, 0x0800, 1024, 0x1ff);
		m_deco_tilegen[0]->tilemap_2_draw(screen, bitmap, cliprect, 0, 4);
	}

	m_sprgen[0]->inefficient_copy_sprite_bitmap(bitmap, cliprect, 0x0000, 0x0800, 1024, 0x1ff);

	m_deco_tilegen[0]->tilemap_1_draw(screen, bitmap, cliprect, 0, 0);
	return 0;
}

/*
    This function mimics the priority PROM/circuit on the pcb.  It takes
    the tilemaps & sprite bitmaps as inputs, and outputs a final pixel
    based on alpha & priority values.  Rendering sprites to temporary
    bitmaps is the only reasonable way to implement proper priority &
    blending support - it can't be done in-place on the final framebuffer
    without a lot of support bitmaps.
*/
void nslasher_state::mixDualAlphaSprites(screen_device &screen, bitmap_rgb32 &bitmap, const rectangle &cliprect, gfx_element *gfx0, gfx_element *gfx1, int mixAlphaTilemap)
{
	const pen_t *pens = m_deco_ace->pens();
	const pen_t *pal0 = &pens[gfx0->colorbase()];
	const pen_t *pal1 = &pens[gfx1->colorbase()];
	const pen_t *pal2 = &pens[m_gfxdecode->gfx((m_pri&1) ? 1 : 2)->colorbase()];
	int x,y;
	bitmap_ind16& sprite0_mix_bitmap = m_sprgen[0]->get_sprite_temp_bitmap();
	bitmap_ind16& sprite1_mix_bitmap = m_sprgen[1]->get_sprite_temp_bitmap();


	/* Mix sprites into main bitmap, based on priority & alpha */
	for (y=8; y<248; y++) {
		uint8_t* tilemapPri=&screen.priority().pix8(y);
		uint16_t* sprite0=&sprite0_mix_bitmap.pix16(y);
		uint16_t* sprite1=&sprite1_mix_bitmap.pix16(y);
		uint32_t* destLine=&bitmap.pix32(y);
		uint16_t* alphaTilemap=&m_tilemap_alpha_bitmap->pix16(y);

		for (x=0; x<320; x++) {
			uint16_t priColAlphaPal0=sprite0[x];
			uint16_t priColAlphaPal1=sprite1[x];
			uint16_t pri0=(priColAlphaPal0&0x6000)>>13;
			uint16_t pri1=(priColAlphaPal1&0x6000)>>13;
			uint16_t col0=((priColAlphaPal0&0x1f00)>>8) % gfx0->colors();
			uint16_t col1=((priColAlphaPal1&0x0f00)>>8) % gfx1->colors();
			uint16_t alpha1=priColAlphaPal1&0x8000;

			// Apply sprite bitmap 0 according to priority rules
			if ((priColAlphaPal0&0xff)!=0)
			{
				/*
				    Sprite 0 priority rules:

				    0 = Sprite above all layers
				    1 = Sprite under top playfield
				    2 = Sprite under top two playfields
				    3 = Sprite under all playfields
				*/
				if ((pri0&0x3)==0 || (pri0&0x3)==1 || ((pri0&0x3)==2 && mixAlphaTilemap)) // Spri0 on top of everything, or under alpha playfield
				{
					destLine[x]=pal0[(priColAlphaPal0&0xff) + (gfx0->granularity() * col0)];
				}
				else if ((pri0&0x3)==2) // Spri0 under top playfield
				{
					if (tilemapPri[x]<4)
						destLine[x]=pal0[(priColAlphaPal0&0xff) + (gfx0->granularity() * col0)];
				}
				else // Spri0 under top & middle playfields
				{
					if (tilemapPri[x]<2)
						destLine[x]=pal0[(priColAlphaPal0&0xff) + (gfx0->granularity() * col0)];
				}
			}

			// Apply sprite bitmap 1 according to priority rules
			if ((priColAlphaPal1&0xff)!=0)
			{
				// Apply alpha for this pixel based on Ace setting
				if (alpha1)
				{
					/*
					    Alpha rules:

					    Pri 0 - Over all tilemaps, but under sprite 0 pri 0, pri 1, pri 2
					    Pri 1 -
					    Pri 2 -
					    Pri 3 -
					*/

					/* Alpha values are tied to ACE ram... */
					//int alpha=m_deco_ace->get_alpha(((priColAlphaPal1&0xf0)>>4)/2);
					//if (alpha<0)
					//  alpha=0;

					/* I don't really understand how object ACE ram is really hooked up,
					    the only obvious place in Night Slashers is the stagecoach in level 2 */

					if (pri1==0 && (((priColAlphaPal0&0xff)==0 || ((pri0&0x3)!=0 && (pri0&0x3)!=1 && (pri0&0x3)!=2))))
					{
						if ((m_pri&1)==0 || ((m_pri&1)==1 && tilemapPri[x]<4) || ((m_pri&1)==1 && mixAlphaTilemap))
							destLine[x]=alpha_blend_r32(destLine[x], pal1[(priColAlphaPal1&0xff) + (gfx1->granularity() * col1)], 0x80);
					}
					else if (pri1==1 && ((priColAlphaPal0&0xff)==0 || ((pri0&0x3)!=0 && (pri0&0x3)!=1 && (pri0&0x3)!=2)))
						destLine[x]=alpha_blend_r32(destLine[x], pal1[(priColAlphaPal1&0xff) + (gfx1->granularity() * col1)], 0x80);
					else if (pri1==2)// TOdo
						destLine[x]=alpha_blend_r32(destLine[x], pal1[(priColAlphaPal1&0xff) + (gfx1->granularity() * col1)], 0x80);
					else if (pri1==3)// TOdo
						destLine[x]=alpha_blend_r32(destLine[x], pal1[(priColAlphaPal1&0xff) + (gfx1->granularity() * col1)], 0x80);
				}
				else
				{
					/*
					    Non alpha rules:

					    Pri 0 - Under sprite 0 pri 0, over all tilemaps
					*/
					if (pri1==0 && ((priColAlphaPal0&0xff)==0 || ((pri0&0x3)!=0)))
						destLine[x]=pal1[(priColAlphaPal1&0xff) + (gfx1->granularity() * col1)];
					else if (pri1==1) // todo
						destLine[x]=pal1[(priColAlphaPal1&0xff) + (gfx1->granularity() * col1)];
					else if (pri1==2) // todo
						destLine[x]=pal1[(priColAlphaPal1&0xff) + (gfx1->granularity() * col1)];
					else if (pri1==3) // todo
						destLine[x]=pal1[(priColAlphaPal1&0xff) + (gfx1->granularity() * col1)];
				}
			}

			/* Optionally mix in alpha tilemap */
			if (mixAlphaTilemap)
			{
				uint16_t p=alphaTilemap[x];
				if (p&0xf)
				{
					/* Alpha tilemap under top two sprite 0 priorities */
					if (((priColAlphaPal0&0xff)==0 || (pri0&0x3)==2 || (pri0&0x3)==3)
						&& ((priColAlphaPal1&0xff)==0 || (pri1&0x3)==2 || (pri1&0x3)==3 || alpha1))
					{
						/* Alpha values are tied to ACE ram */
						int alpha=m_deco_ace->get_alpha(0x17 + (((p&0xf0)>>4)/2));
						if (alpha<0)
							alpha=0;

						destLine[x]=alpha_blend_r32(destLine[x], pal2[p], alpha);
					}
				}
			}
		}
	}
}

uint32_t nslasher_state::screen_update_nslasher(screen_device &screen, bitmap_rgb32 &bitmap, const rectangle &cliprect)
{
	int alphaTilemap=0;
	m_deco_tilegen[0]->pf_update(m_pf_rowscroll[0].get(), m_pf_rowscroll[1].get());
	m_deco_tilegen[1]->pf_update(m_pf_rowscroll[2].get(), m_pf_rowscroll[3].get());

	/* This is not a conclusive test for deciding if tilemap needs alpha blending */
	if (m_deco_ace->get_aceram(0x17)!=0x0 && m_pri)
		alphaTilemap=1;

	screen.priority().fill(0, cliprect);

	bitmap.fill(m_deco_ace->pen(0x200), cliprect);

	/* Draw sprites to temporary bitmaps, saving alpha & priority info for later mixing */
	m_sprgen[0]->set_pix_raw_shift(8);
	m_sprgen[1]->set_pix_raw_shift(8);

	// sprites are flipped relative to tilemaps
	m_sprgen[0]->set_flip_screen(true);
	m_sprgen[1]->set_flip_screen(true);
	m_sprgen[0]->draw_sprites(bitmap, cliprect, m_spriteram16_buffered[0].get(), 0x800);
	m_sprgen[1]->draw_sprites(bitmap, cliprect, m_spriteram16_buffered[1].get(), 0x800);


	/* Render alpha-blended tilemap to separate buffer for proper mixing */
	m_tilemap_alpha_bitmap->fill(0, cliprect);

	/* Draw playfields & sprites */
	if (m_pri&2)
	{
		m_deco_tilegen[1]->tilemap_12_combine_draw(screen, bitmap, cliprect, 0, 1, 1);
		m_deco_tilegen[0]->tilemap_2_draw(screen, bitmap, cliprect, 0, 4);
	}
	else
	{
		m_deco_tilegen[1]->tilemap_2_draw(screen, bitmap, cliprect, 0, 1);
		if (m_pri&1)
		{
			m_deco_tilegen[0]->tilemap_2_draw(screen, bitmap, cliprect, 0, 2);
			if (alphaTilemap)
				m_deco_tilegen[1]->tilemap_1_draw(screen, *m_tilemap_alpha_bitmap, cliprect, 0, 4);
			else
				m_deco_tilegen[1]->tilemap_1_draw(screen, bitmap, cliprect, 0, 4);
		}
		else
		{
			m_deco_tilegen[1]->tilemap_1_draw(screen, bitmap, cliprect, 0, 2);
			if (alphaTilemap)
				m_deco_tilegen[0]->tilemap_2_draw(screen, *m_tilemap_alpha_bitmap, cliprect, 0, 4);
			else
				m_deco_tilegen[0]->tilemap_2_draw(screen, bitmap, cliprect, 0, 4);
		}
	}

	mixDualAlphaSprites(screen, bitmap, cliprect, m_gfxdecode->gfx(3), m_gfxdecode->gfx(4), alphaTilemap);

	m_deco_tilegen[0]->tilemap_1_draw(screen, bitmap, cliprect, 0, 0);
	return 0;
}
