// license:BSD-3-Clause
// copyright-holders:Pierpaolo Prazzoli
/********************************************************************
 Eolith 32 bits hardware: Vega system

 driver by Pierpaolo Prazzoli

 Games dumped
 - Crazy War

 TODO:
 - where are mapped the unused dip switches?

 *********************************************************************/

#include "emu.h"
#include "includes/eolith.h"

#include "cpu/e132xs/e132xs.h"
#include "machine/at28c16.h"
#include "machine/gen_latch.h"
#include "sound/qs1000.h"
#include "speaker.h"


class vegaeo_state : public eolith_state
{
public:
	vegaeo_state(const machine_config &mconfig, device_type type, const char *tag)
		: eolith_state(mconfig, type, tag)
		, m_soundlatch(*this, "soundlatch")
		, m_system_io(*this, "SYSTEM")
	{
	}

	required_device<generic_latch_8_device> m_soundlatch;
	required_ioport m_system_io;

	std::unique_ptr<uint8_t[]> m_vram;
	int m_vbuffer;

	DECLARE_WRITE8_MEMBER(vram_w);
	DECLARE_READ8_MEMBER(vram_r);
	DECLARE_WRITE32_MEMBER(vega_misc_w);
	DECLARE_READ32_MEMBER(vegaeo_custom_read);
	DECLARE_WRITE8_MEMBER(qs1000_p1_w);
	DECLARE_WRITE8_MEMBER(qs1000_p2_w);
	DECLARE_WRITE8_MEMBER(qs1000_p3_w);

	void init_vegaeo();
	DECLARE_VIDEO_START(vega);

	uint32_t screen_update_vega(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect);
	void vega(machine_config &config);
	void vega_map(address_map &map);
};

WRITE8_MEMBER( vegaeo_state::qs1000_p1_w )
{
}

WRITE8_MEMBER( vegaeo_state::qs1000_p2_w )
{
}

WRITE8_MEMBER( vegaeo_state::qs1000_p3_w )
{
	// .... .xxx - Data ROM bank (64kB)
	// ...x .... - ?
	// ..x. .... - /IRQ clear

	membank("qs1000:bank")->set_entry(data & 0x07);

	if (!BIT(data, 5))
		m_soundlatch->acknowledge_w(space, 0, !BIT(data, 5));
}

WRITE8_MEMBER(vegaeo_state::vram_w)
{
	// don't write transparent pen
	if (data != 0xff)
		COMBINE_DATA(&m_vram[offset + m_vbuffer * 0x14000]);
}

READ8_MEMBER(vegaeo_state::vram_r)
{
	return m_vram[offset + 0x14000 * m_vbuffer];
}

WRITE32_MEMBER(vegaeo_state::vega_misc_w)
{
	// other bits ???

	m_vbuffer = data & 1;
}


READ32_MEMBER(vegaeo_state::vegaeo_custom_read)
{
	speedup_read();
	return m_system_io->read();
}


void vegaeo_state::vega_map(address_map &map)
{
	map(0x00000000, 0x001fffff).ram();
	map(0x80000000, 0x80013fff).rw(FUNC(vegaeo_state::vram_r), FUNC(vegaeo_state::vram_w));
	map(0xfc000000, 0xfc0000ff).rw("at28c16", FUNC(at28c16_device::read), FUNC(at28c16_device::write)).umask32(0x000000ff);
	map(0xfc200000, 0xfc2003ff).rw("palette", FUNC(palette_device::read16), FUNC(palette_device::write16)).umask32(0x0000ffff).share("palette");
	map(0xfc400000, 0xfc40005b).nopw(); // crt registers ?
	map(0xfc600000, 0xfc600003).w(m_soundlatch, FUNC(generic_latch_8_device::write)).umask32(0x000000ff).cswidth(32);
	map(0xfca00000, 0xfca00003).w(FUNC(vegaeo_state::vega_misc_w));
	map(0xfcc00000, 0xfcc00003).r(FUNC(vegaeo_state::vegaeo_custom_read));
	map(0xfce00000, 0xfce00003).portr("P1_P2");
	map(0xfd000000, 0xfeffffff).rom().region("maindata", 0);
	map(0xfff80000, 0xffffffff).rom().region("maincpu", 0);
}

static INPUT_PORTS_START( crazywar )
	PORT_START("SYSTEM")
	PORT_BIT( 0x00000001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x00000002, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x00000004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x00000008, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x00000010, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_SERVICE_NO_TOGGLE( 0x00000020, IP_ACTIVE_LOW )
	PORT_BIT( 0x00000040, IP_ACTIVE_LOW, IPT_CUSTOM ) PORT_CUSTOM_MEMBER(DEVICE_SELF, vegaeo_state, eolith_speedup_getvblank, nullptr)
	PORT_BIT( 0x00000080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xffffff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START("P1_P2")
	PORT_BIT( 0x00000001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x00000002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x00000004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x00000008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x00000010, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x00000020, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x00000040, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x00000080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x00000100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x00000200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x00000400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x00000800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x00001000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x00002000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x00004000, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x00008000, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xffff0000, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


VIDEO_START_MEMBER(vegaeo_state,vega)
{
	m_vram = std::make_unique<uint8_t[]>(0x14000*2);
	save_pointer(NAME(m_vram.get()), 0x14000*2);
	save_item(NAME(m_vbuffer));
}

uint32_t vegaeo_state::screen_update_vega(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	for (int y = 0; y < 240; y++)
	{
		for (int x = 0; x < 320; x++)
		{
			bitmap.pix16(y, x) = m_vram[0x14000 * (m_vbuffer ^ 1) + (y * 320) + x] & 0xff;
		}
	}
	return 0;
}


MACHINE_CONFIG_START(vegaeo_state::vega)
	MCFG_DEVICE_ADD("maincpu", GMS30C2132, XTAL(55'000'000))
	MCFG_DEVICE_PROGRAM_MAP(vega_map)
	MCFG_TIMER_DRIVER_ADD_SCANLINE("scantimer", vegaeo_state, eolith_speedup, "screen", 0, 1)

	MCFG_DEVICE_ADD("at28c16", AT28C16, 0)

	/* video hardware */
	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_REFRESH_RATE(60)
	MCFG_SCREEN_VBLANK_TIME(ATTOSECONDS_IN_USEC(0))
	MCFG_SCREEN_SIZE(512, 262)
	MCFG_SCREEN_VISIBLE_AREA(0, 319, 0, 239)
	MCFG_SCREEN_UPDATE_DRIVER(vegaeo_state, screen_update_vega)
	MCFG_SCREEN_PALETTE("palette")

	MCFG_PALETTE_ADD("palette", 256)
	MCFG_PALETTE_FORMAT(xRRRRRGGGGGBBBBB)
	MCFG_PALETTE_MEMBITS(16)

	MCFG_VIDEO_START_OVERRIDE(vegaeo_state,vega)

	/* sound hardware */
	SPEAKER(config, "lspeaker").front_left();
	SPEAKER(config, "rspeaker").front_right();

	MCFG_GENERIC_LATCH_8_ADD("soundlatch")
	MCFG_GENERIC_LATCH_DATA_PENDING_CB(WRITELINE("qs1000", qs1000_device, set_irq))
	MCFG_GENERIC_LATCH_SEPARATE_ACKNOWLEDGE(true)

	MCFG_DEVICE_ADD("qs1000", QS1000, XTAL(24'000'000))
	MCFG_QS1000_EXTERNAL_ROM(true)
	MCFG_QS1000_IN_P1_CB(READ8("soundlatch", generic_latch_8_device, read))
	MCFG_QS1000_OUT_P1_CB(WRITE8(*this, vegaeo_state, qs1000_p1_w))
	MCFG_QS1000_OUT_P2_CB(WRITE8(*this, vegaeo_state, qs1000_p2_w))
	MCFG_QS1000_OUT_P3_CB(WRITE8(*this, vegaeo_state, qs1000_p3_w))
	MCFG_SOUND_ROUTE(0, "lspeaker", 1.0)
	MCFG_SOUND_ROUTE(1, "rspeaker", 1.0)
MACHINE_CONFIG_END

/*
Crazy Wars
Eolith

This game runs on Eolith Vega II V1.2 hardware

PCB Layout
----------

VEGA II V1.2
|-------------------------------------------------------|
|   VOL_L    QDSP      BGM.U84                          |
|   VOL_R    QS1000              61C256         61C256  |
|                                                       |
|   DA1311                       61C256         61C256  |
|                    24MHz                              |
|           EFFECT.U85           61C256         61C256  |
|                  14.31818MHz                          |
|J   QS-1001A.U86                61C256         61C256  |
|A                    EOLITH                            |
|M                    EV0514-001                        |
|M                                                  PAL |
|A   SERVICE_SW                                RESET_SW |
|                                GMS30C2132             |
|    TEST_SW    KT76C28K-10              41C16256       |
|               KT76C28K-10              41C16256  55MHz|
|                   |----------------------------|      |
|          28C16.U29|  06 04 02 00 14 12 10 08   |      |
|                   |                            |      |
|DSW2(4)            |                            |      |
|DSW1(4)            |  07 05 03 01 15 13 11 09   |      |
|                   |                            |   U7 |
|-------------------|----------------------------|------|
Notes:
      GMS30C2132 - Hyperstone CPU @ 55.0MHz
      61C256     - 32kx8 SRAM
      41C16256   - ISSI 256kx16 DRAM
      K76C28K-10 - SRAM, probably 2kx8 or 8kx8
      00 - 15    - Macronix MX29F1610M 16MBit SOP44 FlashROMs
      U7         - 27C040 EPROM
      U85        - 27C801 EPROM
      U84        - 27C4001 EPROM
      U29        - 2kx8 EEPROM
*/


ROM_START( crazywar )
	ROM_REGION( 0x80000, "maincpu", 0 ) /* Hyperstone CPU Code */
	ROM_LOAD( "u7",         0x00000, 0x80000, CRC(697c2505) SHA1(c787007f05d2ddf1706e15e9d9ef9b2479708f12) )

	ROM_REGION32_BE( 0x2000000, "maindata", ROMREGION_ERASE00 ) /* Game Data - banked ROM, swapping necessary */
	ROM_LOAD32_WORD_SWAP( "00", 0x0000000, 0x200000, CRC(fbb917ae) SHA1(1fd975cda06b3cb748503b7c8009e6184b46af3f) )
	ROM_LOAD32_WORD_SWAP( "01", 0x0000002, 0x200000, CRC(59308556) SHA1(bc8c28531fca009be5b7b3b1a4a9b3ebcc9d3c3a) )
	ROM_LOAD32_WORD_SWAP( "02", 0x0400000, 0x200000, CRC(34813167) SHA1(d04c71164b36af78425dcd637e60aee45c39a1ba) )
	ROM_LOAD32_WORD_SWAP( "03", 0x0400002, 0x200000, CRC(7fcb0a53) SHA1(f74e0512b5d4854d0c4b04bf8c917f8dccb4dc0f) )
	ROM_LOAD32_WORD_SWAP( "04", 0x0800000, 0x200000, CRC(f8eb8ce5) SHA1(a631f6979a9df2fda622483256ea569c6b4d1586) )
	ROM_LOAD32_WORD_SWAP( "05", 0x0800002, 0x200000, CRC(14d854df) SHA1(5527fb1a12193e27a3fad5ca7f4e3027f462ee50) )
	ROM_LOAD32_WORD_SWAP( "06", 0x0c00000, 0x200000, CRC(31c67f0a) SHA1(7a587bb86bc6450c66016c82efe047f2d350d586) )
	ROM_LOAD32_WORD_SWAP( "07", 0x0c00002, 0x200000, CRC(dddf93d2) SHA1(c982f18c4bd242885a6150252c9c2fa4a07ebf4b) )
	ROM_LOAD32_WORD_SWAP( "08", 0x1000000, 0x200000, CRC(dc37bcb9) SHA1(144050056905e3dce08795d1a4ac17a45f2a1fec) )
	ROM_LOAD32_WORD_SWAP( "09", 0x1000002, 0x200000, CRC(86ba59cc) SHA1(566cc6527188e24a6eae4a64131deca7e2140ada) )
	ROM_LOAD32_WORD_SWAP( "10", 0x1400000, 0x200000, CRC(524bf126) SHA1(85a27a74ba4caaf3ab1e1f0e8e8b516bb0182ae7) )
	ROM_LOAD32_WORD_SWAP( "11", 0x1400002, 0x200000, CRC(613b2764) SHA1(7a7c85c8cf1cba74e2e98a3b77d5ea44bb76a563) )
	ROM_LOAD32_WORD_SWAP( "12", 0x1800000, 0x200000, CRC(3c81d117) SHA1(76d6728f8c55e68c84d68ff2f242684bde30f4dd) )
	ROM_LOAD32_WORD_SWAP( "13", 0x1800002, 0x200000, CRC(b86545a0) SHA1(4aaa23c37d776647f3288ba541cefa79ddbd962d) )
	ROM_LOAD32_WORD_SWAP( "14", 0x1c00000, 0x200000, CRC(38ede322) SHA1(9496685a1280885a61a568047c4a8c2cd70d1b83) )
	ROM_LOAD32_WORD_SWAP( "15", 0x1c00002, 0x200000, CRC(d35e630a) SHA1(8c220f1baddd39cc978e3e5a874cc58e78b74c62) )

	ROM_REGION( 0x080000, "qs1000:cpu", 0 )  /* QDSP (8052) Code */
	ROM_LOAD( "bgm.u84",      0x000000, 0x080000, CRC(13aa7778) SHA1(131f74e1b73dd7a7038864593dc7ca24af0ffc30) )

	ROM_REGION( 0x1000000, "qs1000", 0 )
	ROM_LOAD( "effect.u85",   0x000000, 0x100000, CRC(9159fcc6) SHA1(2be9a197a51303a0da9484dced12a3f6d3b0d867) )
	ROM_LOAD( "qs1001a.u86",  0x200000, 0x080000, CRC(d13c6407) SHA1(57b14f97c7d4f9b5d9745d3571a0b7115fbe3176) )
ROM_END

void vegaeo_state::init_vegaeo()
{
	// Set up the QS1000 program ROM banking, taking care not to overlap the internal RAM
	machine().device("qs1000:cpu")->memory().space(AS_IO).install_read_bank(0x0100, 0xffff, "bank");
	membank("qs1000:bank")->configure_entries(0, 8, memregion("qs1000:cpu")->base()+0x100, 0x10000);

	init_speedup();
}

GAME( 2002, crazywar, 0, vega, crazywar, vegaeo_state, init_vegaeo, ROT0, "Eolith", "Crazy War", MACHINE_IMPERFECT_SOUND | MACHINE_SUPPORTS_SAVE )
