/*
   Copyright (C) 2006-2007 KovoKs <info@kovoks.nl>
   Copyright (C) 2007 Frode M. Døving <frode@lnix.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <qpainter.h>
#include <qcursor.h>
#include <qdragobject.h>
#include <qtimer.h>
#include <qheader.h>
#include <klistview.h>
#include <klocale.h>
#include <kdebug.h>
#include <kiconloader.h>

#include "datalistview.h"
#include "mailboxlistview.h"
#include "tooltip.h"

namespace Mailody {

MailBoxListView::MailBoxListView( QWidget* parent)
    : DataListView( parent, 0, 2 )
{
    setSize(25);    // size of the second column.
    setColumnText(0, i18n("Folder"));
    setColumnText(1, i18n("Information", "I"));
    setColumnAlignment(1, Qt::AlignCenter);

    setFullWidth(true);
    setRootIsDecorated(true);
    setDragEnabled(false);
    setAutoOpen(true);
    setShowSortIndicator(true);
    setColumnWidthMode(0,QListView::Maximum);
    setAcceptDrops(true);
    setDropVisualizer(false);
    setDropHighlighter(true);
    m_drop = new QTimer(this);
    connect(m_drop, SIGNAL(timeout()), this, SLOT(slotOpen()));
    connect(this, SIGNAL(doubleClicked( QListViewItem*, const QPoint&, int )),
            this, SLOT(slotToggleOpen( QListViewItem*, const QPoint&,int )));

    connect(this,
            SIGNAL( infoLeftMouse( MailBoxListViewItem*, int )),
            this, SLOT( slotShowMBDetails( MailBoxListViewItem*, int ) ));
    connect(this, SIGNAL(leftWidget()), SLOT(slotHideMBDetails()));
    connect(this, SIGNAL(onViewport()), SLOT(slotHideMBDetails()));
    m_mbtip = new ToolTip(this);
}

bool MailBoxListView::acceptDrag(QDropEvent *event) const
{
    return QTextDrag::canDecode( event );
}

void MailBoxListView::contentsDragEnterEvent(QDragEnterEvent *e)
{
    m_drop->start(1400);
    KListView::contentsDragEnterEvent(e);
}

void MailBoxListView::contentsDragLeaveEvent(QDragLeaveEvent *e)
{
    m_drop->stop();
    KListView::contentsDragLeaveEvent(e);
}

void MailBoxListView::contentsDropEvent(QDropEvent *e)
{
    m_drop->stop();
    KListView::contentsDropEvent(e);
}

void MailBoxListView::slotOpen()
{
    QPoint pos = viewport()->mapFromGlobal( QCursor::pos() );

    QListViewItem* lv = itemAt(pos);
    if (lv)
         lv->setOpen(true);
}

void MailBoxListView::slotToggleOpen(QListViewItem* lv, const QPoint&, int c)
{
    if (c == 1)
        return;

    lv->setOpen(!lv->isOpen());
}

void MailBoxListView::slotShowMBDetails(MailBoxListViewItem* item, int c)
{
    if (!c == 1)
        return;

    QString text = "<table style=\"margin-right: 20px;\" ";
    text.append("cellpadding=0 cellspacing=0 width=\"100%\">");
    text.append("<tr><td colspan=2><b><nobr>" + item->text(0)
            + "</nobr></b></td></tr>");
    if (item->getMessages() > 0)
        text.append("<tr><td><nobr>" + i18n("Messages")
                + "</nobr></td><td><nobr>"
                + QString::number(item->getMessages())
                + "</nobr></td><tr>");
    if (item->getTotalMessages() > 0)
        text.append("<tr><td><nobr>" + i18n("Messages Including Sub-Folders")
                + "</nobr></td><td><nobr>"
                + QString::number(item->getTotalMessages())
                + "</nobr></td></tr>");
    if (item->getMessages() == 0
        && item->getTotalMessages() == 0)
        text.append("<tr><td><nobr>" + i18n("Empty or Unknown")
                + "</nobr></td></tr>");
    text.append("</table>");

    m_mbtip->setText(QCursor::pos(), text);
}

void MailBoxListView::slotHideMBDetails()
{
    m_mbtip->hide();
}

void MailBoxListView::contentsMousePressEvent( QMouseEvent *e )
{
    QPoint point = e->pos();
    MailBoxListViewItem* item = static_cast<MailBoxListViewItem*>(itemAt(point));
    int c = header()->sectionAt(e->x());

//    kdDebug() << "column: " << c << endl;

    if (item && e->button() == Qt::LeftButton && c == 1)
    {
        e->accept();
        emit infoLeftMouse(item, c);
        return;
    }

 KListView::contentsMousePressEvent(e);
}


MailBoxListViewItem* MailBoxListView::nextWithUnread()
{
    // iterate from the selected item if its there or from the started.
    QListViewItem* start = selectedItems( false ).first();

    if (!start)
    {
        QListViewItemIterator it( this );
        while ( it.current() )
        {
            MailBoxListViewItem* item =
                    static_cast<MailBoxListViewItem*>(it.current());
            if (item->getUnseen() > 0)
                return item;

            ++it;
        }
    }
    else
    {
        // start from the selected one.
        QListViewItemIterator it( start );
        while ( it.current() )
        {
            MailBoxListViewItem* item =
                    static_cast<MailBoxListViewItem*>(it.current());
            if (item->getUnseen() > 0)
                return item;

            ++it;
        }

        // not found. start at the beginning...
        it = this;
        while ( it.current() )
        {
            MailBoxListViewItem* item =
                    static_cast<MailBoxListViewItem*>(it.current());
            if (item == start)
                break;

            if (item->getUnseen() > 0)
                return item;

            ++it;
        }
    }
    return 0;
}


//------------------------------------------------------------------
MailBoxListViewItem::MailBoxListViewItem( MailBoxListViewItem* parent,
                                          const QString& visualName,
                                          const QString& fullName,
                                          const bool mailCheck)
    : MailodyBaseListViewItem( parent ), m_unseen(0), m_messages(0),
            m_totalMessages(0)
{
    setDropEnabled( true );
    setText(0, visualName);
    setText(1, ".");
    m_visualName = visualName;
    m_full = fullName;
    m_check = mailCheck;
}

MailBoxListViewItem::MailBoxListViewItem( KListView* parent, const QString& name,
                        const QString& fullName, const bool mailCheck)
    : MailodyBaseListViewItem( parent ), m_unseen(0), m_messages(0),
            m_totalMessages(0)
{
    setDropEnabled( true );
    if (name == "INBOX")
    {
        setLevel(0);
        m_visualName = i18n("Inbox");
    }
    else
    {
        setLevel(1);
        m_visualName = name;
    }

    setText(0, m_visualName);
    setText(1, ".");
    m_full = fullName;
    m_check = mailCheck;
}

MailBoxListViewItem::~MailBoxListViewItem()
{
}

void MailBoxListViewItem::updateUnseen( int i )
{
    m_unseen = i;
    if ( i > 0)
        setText(0, m_visualName + " (" + QString::number(i) + ')');
    else
        setText(0, m_visualName);
}

void MailBoxListViewItem::paintCell( QPainter * p, const QColorGroup & cg,
                            int column, int width, int align )
{
    QColorGroup _cg( cg );
    QColor c = _cg.text();


    if (m_unseen > 0)
        _cg.setColor( QColorGroup::Text, Qt::darkBlue );

    KListViewItem::paintCell( p, _cg, column, width, align );

    /*
    if (column != 0)
    {
        p->save();
        p->setPen( QPen( cg.dark(), 1, Qt::DotLine  ) );
        p->drawLine( width - 1, 0, width - 1, height() );
        p->restore();
    }
    */

}

}

#include "mailboxlistview.moc"
