/* This file is part of the KDE project
   Copyright (C) 2008 Omat Holding B.V. <info@omat.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "attachmentwidget.h"

#include <KDebug>
#include <KIcon>
#include <KLocale>
#include <KMimeType>
#include <KRun>

#include <QContextMenuEvent>
#include <QHBoxLayout>
#include <QListView>
#include <QMenu>
#include <QStandardItemModel>

using namespace Mailody;

AttachmentWidget::AttachmentWidget( QWidget* parent )
        : QWidget( parent )
{
    QHBoxLayout* box = new QHBoxLayout( this );

    setAcceptDrops( true );
    m_model = new QStandardItemModel( this );
    QListView *list = new AttachmentView( this );
    list->setModel( m_model );
    connect( list, SIGNAL( openItems( const QModelIndexList& ) ),
             SLOT( slotOpen( const QModelIndexList& ) ) );
    connect( list, SIGNAL( removeItems( const QModelIndexList& ) ),
             SLOT( slotRemove( const QModelIndexList& ) ) );

    box->addWidget( list );
}

QHash<KUrl, QString> AttachmentWidget::attachments()
{
    QHash<KUrl, QString> results;

    QHash<QStandardItem*, KUrl>::Iterator ita;
    for ( ita = allItems.begin(); ita != allItems.end(); ++ita )
        results[ ita.value()] = ita.key()->text();

    return results;
}

void AttachmentWidget::dropEvent( QDropEvent *e )
{
    KUrl::List uriList = KUrl::List::fromMimeData( e->mimeData() );
    if ( !uriList.isEmpty() ) {
        for ( KUrl::List::Iterator it = uriList.begin();
                it != uriList.end(); ++it )
            addAttachment( *it );
    }
}

void AttachmentWidget::slotOpen( const QModelIndexList &list )
{
    foreach( const QModelIndex &index, list ) {
        QStandardItem *item = m_model->item( index.row() );
        new KRun( allItems.value( item ), this );
    }
}

void AttachmentWidget::slotRemove( const QModelIndexList &list )
{
    // when a row is deleted the other items shift up one item, so keep track of it...
    int i = 0;
    foreach( const QModelIndex &index, list ) {
        QStandardItem *item = m_model->item( index.row() );
        allItems.remove( item );
        m_model->removeRow( index.row()-i, index.parent() );
        i++;
    }

    if ( allItems.count() == 0 )
        emit noAttachmentsLeft();
}

void AttachmentWidget::addAttachment( const KUrl &url, const QString& name )
{
    const QString icon = KMimeType::iconNameForUrl( url );
    QString fileName = name;
    if ( fileName.isEmpty() )
        fileName = url.fileName();
    QStandardItem *item = new QStandardItem( KIcon( icon ), fileName );
    m_model->setItem( allItems.count(), 0, item );
    allItems.insert( item, url );
}


/* ############################################################### */
/*                  reimplementation of QListView                  */
/* ############################################################### */

AttachmentView::AttachmentView( QWidget* parent )
        : QListView( parent )
{
    setViewMode( QListView::ListMode );
    setFlow( QListView::LeftToRight );
    setWrapping( true );
    setMovement( QListView::Static );
    setResizeMode( QListView::Adjust );
    setSelectionMode( QAbstractItemView::ExtendedSelection );
}

void AttachmentView::contextMenuEvent( QContextMenuEvent *e )
{
    QMenu* p = new QMenu( this );
    QAction* open = new QAction( i18n( "Open" ), this );
    QAction* remove = new QAction( i18n( "Remove" ), this );

    p->addAction( open );
    p->addSeparator();
    p->addAction( remove );
    QAction* choice = p->exec( e->globalPos() );

    if ( choice == remove )
        emit removeItems( selectionModel()->selectedIndexes() );
    else if ( choice == open )
        emit openItems( selectionModel()->selectedRows() );
}

void AttachmentView::mouseDoubleClickEvent( QMouseEvent * )
{
    emit openItems( selectionModel()->selectedRows() );
}

#include "attachmentwidget.moc"
