/******************************** LICENSE ********************************


 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)
 
 Licensed under the Apache License, Version 2.0 (the "License"); 
 you may not use this file except in compliance with the License. 
 You may obtain a copy of the License at 
 
 	http://www.apache.org/licenses/LICENSE-2.0
 
 Unless required by applicable law or agreed to in writing, software 
 distributed under the License is distributed on an "AS IS" BASIS, 
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 See the License for the specific language governing permissions and 
 limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file OpenGLGui.cc
    \brief Implementation of the OpenGLGui class.
    \author Graphics Section, ECMWF

    Started: July 2008
*/
#include <OpenGLGui.h>
#include <OpenGLDriver.h>
#include <OpenGLTextureItem.h>
#include <OpenGLFboItem.h>
#include <OpenGLPushButtonWidget.h>
#include <OpenGLLabelWidget.h>
#include <OpenGLPainter.h>
#include <OpenGLTooltip.h>
#include <OpenGLCore.h>

#define GL_GLEXT_PROTOTYPES

#include <GL/gl.h>
#include <GL/glu.h>
   
#include <MtInputEvent.h>

OpenGLGui::OpenGLGui(OpenGLDriver *driver, string name, 
		     int x, int y, int width, int height) : 
	             OpenGLManagerWidget(driver,name) 
{
	setColour("border",Colour(0.,0.,0.));
	setColour("background",Colour(211./255.,215./255.,223./255.));
	
	//setColour("background",Colour(0./255.,0./255.,0./255.));

	borderWidth_=10;
	borderHeight_=6;

	alphaFocus_=1.0;
	alphaUnfocus_=0.5;
	alphaCurrent_=alphaUnfocus_;

	tex_ = new OpenGLTextureItem;
	//tex_->generateFromImage(0,width_,height_,GL_RGBA);
	tex_->setBlending(GL_CONSTANT_ALPHA,GL_ONE_MINUS_CONSTANT_ALPHA,alphaCurrent_);
	
	texBg_ = new OpenGLTextureItem;
	texBg_->transparent(false);

	fbo_=new OpenGLFboItem;
	/*fbo_->bind();
	fbo_->attachTexture(tex_->id());
	fbo_->checkStatus();
	fbo_->unBind();*/
	
	texDropShadow_= new OpenGLTextureItem[2];

	built_=false;
	enabled_=false;
	visible_=false;
	changed_=false;

	minX_=0;
	maxX_=driver_->dimensionX();
	minY_=0;
	maxY_=driver_->dimensionY();
	borderColour_=Colour(0.2,0.2,0.2);
	borderColour_=Colour(0.8,0.8,0.8);

	cornerSize_=10;
	dropShadowOutSize_=8;
	dropShadowInSize_=3;
	
	closeButtonSize_=0;

	createDropShadowTexture();

	dragType_=OpenGLWidget::NoDrag;
	dragArea_=OpenGLWidget::NoDragArea;

	titleBarHeight_=0;
}

OpenGLGui::~OpenGLGui()
{
	delete tex_;
	delete [] texDropShadow_;
	delete texBg_;
}

bool OpenGLGui::checkRectInWindow(int &x,int &y,const int w, const int h)
{
	bool retv=true;

	if(x < minX_)
	{
		x=minX_;
		retv=false;
	}

	if(x+w > maxX_)
	{
		x=maxX_-w;
		retv=false;
	}

	if(y < minY_)
	{
		y=minY_;
		retv=false;
	}

	if(y+h > maxY_)
	{
		y=maxY_-h;
		retv=false;
	}

	return retv;
}

void OpenGLGui::clearTexture()
{
	tex_->clear();
	texBg_->clear();
	rendered_=false;
}

void OpenGLGui::render()
{
	if(!enabled_) return;
	
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		tex_->generateFromImage(0,width_,height_,GL_RGBA);
		
		fbo_->clear();
		fbo_->bind();
		fbo_->attachTexture(tex_->id());
		glClear(GL_COLOR_BUFFER_BIT);
		fbo_->checkStatus();
		fbo_->unBind();
	}

	if(OpenGLCore::instance()->renderMode() == OpenGLCore::BackBufferMode)
	{
		texBg_->generateFromFb(x_-2,y_-2,width_+14,height_+14);
	}	

	outlineX_.clear();
	outlineY_.clear();	

	painter_->buildRoundedRectPolygon(0,0,width_,height_,cornerSize_,outlineX_,outlineY_);
	
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		fbo_->bind();
		glPushMatrix();
		glLoadIdentity();
		glTranslatef(-x_,-y_,0);
	}
                      	
	Colour col(235./255.,238./255.,240./255.);
	painter_->renderFilledRoundedRect(x_,y_,x_+width_,y_+height_,cornerSize_,col);

	//Render the gui
	OpenGLManagerWidget::render(); 

	rendered_=true;


	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		glPopMatrix();
		fbo_->unBind();
	}
	//Save it to a texture (gui texture)
	else if(OpenGLCore::instance()->renderMode() == OpenGLCore::BackBufferMode)
	{
		if(tex_->isEmpty())
		{
			tex_->generateFromFb(x_,y_,width_,height_);
		}
		else
		{
			tex_->modifyFromFb(x_,y_,width_,height_,0.,0.);
		}

		texBg_->mapTexture(x_-2,y_-2,x_+width_+12,y_+height_+12);
	}
}	


void OpenGLGui::renderChild(OpenGLBaseWidget *w)
{
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		fbo_->bind();
		glPushMatrix();
		glLoadIdentity();
		glTranslatef(-x_,-y_,0);
	}

	w->render();
	
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		glPopMatrix();
		fbo_->unBind();
	}
	else if(OpenGLCore::instance()->renderMode() == OpenGLCore::BackBufferMode)
	{
		tex_->modifyFromFb(w->getX()-1,w->getY()-1,
                           	   w->getWidth()+2,w->getHeight()+2,
			           w->getX()-1-x_,w->getY()-1-y_);
	}
}

void OpenGLGui::renderChildren(list<OpenGLBaseWidget*> &w)
{
	/*tex_->setBlendingAlpha(1.);
	tex_->mapTexture(x_,y_,x_+width_,y_+height_);

	for(list<OpenGLBaseWidget*>::iterator it=w.begin(); it != w.end(); it++)
	{
		(*it)->render();
	}

	tex_->modifyFromFb(x_,y_,width_,height_,0.,0.);

	//Map the gui texture with blending	
	map(true,true);*/
}



void OpenGLGui::renderBorder()
{		
	painter_->renderRoundedRect(x_,y_,x_+width_,y_+height_,cornerSize_,borderColour_,1,false);
}            

void OpenGLGui::createDropShadowTexture()
{
	int size=cornerSize_;  //dropShadowInSize_+dropShadowOutSize_;
	float rad=(size-1);
	float d;
	int num=size*size*4;
	GLubyte img[num]; 
	
	for(int i=0; i< num; i+=4)
	{
		img[i]=(GLubyte)60;
		img[i+1]=(GLubyte)60;
		img[i+2]=(GLubyte)60;
		img[i+3]=(GLubyte)0;
	}

	for(int i=0; i< size; i++)
	{	
		for(int j=0; j < size; j++)
		{
			d=sqrtf(i*i+(size-1-j)*(size-1-j));
			if(d < rad)
			{
				img[j*size*4+i*4+3] =(GLubyte) (255.-255.*d/rad);
			}
		}
	}			

	texDropShadow_[0].generateFromImage(img,size,size,GL_RGBA);

	for(int i=0; i< size; i++)
	{	
		for(int j=0; j < size; j++)
		{			
						
			img[j*size*4+i*4+3] =(GLubyte) (static_cast<float>(j)/static_cast<float>(size-1)*255.);		
		}
	}	

	texDropShadow_[1].generateFromImage(img,size,size,GL_RGBA);
}

void OpenGLGui::renderDropShadow()
{
	int size=dropShadowInSize_+dropShadowOutSize_;

	glPushMatrix();
	glTranslatef(x_+dropShadowOutSize_,y_-dropShadowOutSize_,0);

	//Bottom
	texDropShadow_[1].mapTexture(size,0,width_-size,size);

	//Left
	glPushMatrix();
	glTranslatef(width_-size,size,0);
	glRotatef(90.,0.,0.,1.);
	texDropShadow_[1].mapTexture(0,-size,height_-2*size,0);
	glPopMatrix();

	//Bottom left corner
	glPushMatrix();
	glTranslatef(size,size,0);
	glRotatef(-90.,0.,0.,1.);
	texDropShadow_[0].mapTexture(0,-size,size,0);
	glPopMatrix();

	//Bottom right corner
	glPushMatrix();
	glTranslatef(width_-size,0,0);
	texDropShadow_[0].mapTexture(0,0,size,size);
	glPopMatrix();

	//Top right corner
	glPushMatrix();
	glTranslatef(width_,height_-size,0);
	glRotatef(90.,0.,0.,1.);
	texDropShadow_[0].mapTexture(0,0,size,size);
	glPopMatrix();

	glPopMatrix();
}


//Map the background lying under the gui
void OpenGLGui::mapBg()
{	
	driver_->mapBgImageToFb(1,x_-12,y_-12,x_+width_+12,y_+height_+12);
}

void OpenGLGui::map()
{
	//Map the gui texture with blending
	if(visible_)
	{
		map(true,false);
	}
	else
	{
		mapBg();
		driver_->swapFb();
	}
}


void OpenGLGui::show()
{
	if(!enabled_ || !visible_) return;

	if(!rendered_)
	{
		render();
	}
   
	//Bind fbo
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		OpenGLCore::instance()->fboFb()->bind();	
	}
	
	if(focused_)
	{
		renderDropShadow();
	}

	//Map the gui texture with blending	
	tex_->setBlendingAlpha(alphaCurrent_);	
	//tex_->mapTexture(x_-200,y_-250,x_+width_,y_+height_);

	vector<PaperPoint> sp, tp;
	float xp,yp;
	float w=width_;
	float h=height_;

	for(int i=0; i< outlineX_.size(); i++)
	{
		float xp=1.-(w-outlineX_[i])/w;
		float yp=1.-(h-outlineY_[i])/h;
		sp.push_back(PaperPoint(xp,yp));

		xp=x_+outlineX_[i];
		yp=y_+outlineY_[i];
		tp.push_back(PaperPoint(xp,yp));
	}
	tex_->mapTexture(sp,tp);

	renderBorder();
	
	//Tooltip comes here!!
	/*if(!OpenGLTooltip::instance()->empty())
	{
		OpenGLTooltip::instance()->display();
	}*/

	//Unbind fbo
	if(OpenGLCore::instance()->renderMode() == OpenGLCore::FboMode)
	{
		OpenGLCore::instance()->fboFb()->unBind();	
	}
}


void OpenGLGui::map(bool mapbg,bool swap)
{
	if(!visible_) return;

	if(! rendered_)
	{
		render();
		mapBg();
	}

	//Map the background lying under the gui
	if(mapbg)
	{
		mapBg();
	}	

	if(focused_)
	{
		renderDropShadow();
	}

	//Map the gui texture with blending	
	tex_->setBlendingAlpha(alphaCurrent_);	
	//tex_->mapTexture(x_,y_,x_+width_,y_+height_);

	vector<PaperPoint> sp, tp;
	float xp,yp;
	float w=width_;
	float h=height_;

	for(int i=0; i< outlineX_.size(); i++)
	{
		float xp=1.-(w-outlineX_[i])/w;
		float yp=1.-(h-outlineY_[i])/h;
		sp.push_back(PaperPoint(xp,yp));

		xp=x_+outlineX_[i];
		yp=y_+outlineY_[i];
		tp.push_back(PaperPoint(xp,yp));
	}
	tex_->mapTexture(sp,tp);

	renderBorder();
	

	//Tooltip comes here!!
	if(!OpenGLTooltip::instance()->empty())
	{
		OpenGLTooltip::instance()->display();
	}

	//Swap buffers
	if(swap)
	{	
		driver_->swapFb();
	}
}
		
bool OpenGLGui::checkFocus(MtInputEvent *event)
{
	if(event->type() == Mt::MouseMoveEvent)
	{	
		MtMouseEvent *mev = (MtMouseEvent *) event;
		int x = mev->x();
 		int y = mev->y();
				
		bool orival=focused_;	
		
                //A gui is selected if the pointer is in the gui area
                focused_=checkPointInWidget(x,y);
	       // Log::debug() << "---> focus check: " <<  orival << " " << focused_<< " xy:"  << x  << "  " << y << " gui: " << x_ << " " << y_ << endl;
		
                //If selection has changed
		if(focused_ !=  orival)
		{
                        //Log::debug() << "focus changed: " << orival << "  -->  " << focused_ << endl; 
 
			if(focused_) 
			{
				//manager_->show(); //mapAll(true,this,false);
				alphaCurrent_=alphaFocus_;
			}
			else
			{
				alphaCurrent_=alphaUnfocus_;
			}
			//manager_->show();

			return true;

			//map(true,true);
		}	
	}
	return false;					
}

void OpenGLGui::event(MtInputEvent *event)
{		
	if(!enabled_ || !visible_) return;

	if(dragType_ == OpenGLWidget::NoDrag)
	{
		if(checkFocus(event) == true)
		{
			if(focused_)
			{
				active_=true;
			}
			return;
		}

		if(focused_ == true || active_ == true)
		{	
			active_=false;
			for(mglWidgetList::iterator it=children_.begin(); it !=children_.end(); it++)
			{
				//Log::dev() << "OpenGLGui::event ---> child: " << (*it)->name() << endl;

				(*it)->event(event);			
                        
                        	//Children are active only in case of drag action
                       	 	if((*it)->active())
				{
					//Log::dev() << " OpenGLGui::event ---> Child is active!" << endl;
					renderChild(*it); 
					//Log::dev() << "OpenGLGui::event --->  Gui active" << endl;
					active_=true;
					break;
				}
			}
		}

		if(active_ || !focused_) return;
	}

	//Check drag action
	int x, y; 	
        MtMouseEvent *mev;
      	
 	switch(event->type())
 	{
	
	case Mt::MousePressEvent:
 		mev = (MtMouseEvent *) event;
		x = mev->x();
 		y = mev->y();				
		
		if(mev->button() & Mt::LeftButton)
		{
			if(dragType_ == OpenGLWidget::NoDrag) 
 			{			
 				drag_start(x,y);
			}
			
		}				
 		break;	
		
 	case Mt::MouseMoveEvent: 		
		mev = (MtMouseEvent *) event;
		x = mev->x();
 		y = mev->y();
							
		if(mev->button() & Mt::LeftButton)
		{					
			if(dragType_ == OpenGLWidget::ReplaceDrag)
 			{	
				replace(x,y);
			}
			else if(dragType_ == OpenGLWidget::ResizeDrag)
 			{	
				resize(x,y);
			}			
		}		
		
		break;	
		
	case Mt::MouseReleaseEvent:
		mev = (MtMouseEvent *) event;
		x = mev->x();
 		y = mev->y();

		if(mev->button() & Mt::LeftButton)
		{
			if(dragType_ == OpenGLWidget::ReplaceDrag)
 			{	
				replace_end();
			}
			else if(dragType_ == OpenGLWidget::ResizeDrag)
 			{	
				resize_end();
			}
		}
		break;			
	}
}

/*void OpenGLGui::replace_start(int x,int y)
{			
	if(y < y_+height_ && 
	   y > y_+height_-borderHeight_-titleBarHeight_ && 
	   x < x_+width_-borderWidth_-closeButtonSize_ && x> x_ )
	{
		replace_=true;
		prevX_=x;
		prevY_=y;
	}
}*/

void OpenGLGui::drag_start(int x,int y)
{
	if(!checkPointInWidget(x,y)) return;

	int dragDistance=5;

	//Right border
	/*if(x > x_+width_-dragDistance && x <= x_+width_)
	{
		dragType_   =  OpenGLWidget::ResizeDrag;
		dragArea_   =  OpenGLWidget::RightBorder; 
	
		prevX_=x;
		prevY_=y;
	}

	else if(y < y_+height_ && 
	   	y > y_+height_-borderHeight_-titleBarHeight_ && 
	   	x < x_+width_-borderWidth_-closeButtonSize_ && x> x_ )
	{
		dragType_   =  OpenGLWidget::ReplaceDrag;
	
		prevX_=x;
		prevY_=y;
	}*/

	dragType_   =  OpenGLWidget::ReplaceDrag;	
	prevX_=x;
	prevY_=y;


}

void OpenGLGui::resize(int x,int y)
{
	if(dragType_ !=  OpenGLWidget::ResizeDrag) return;

	int xp,yp, w,h;

	switch(dragArea_)
	{
	case OpenGLWidget::RightBorder:
		w=width_+x-prevX_;
		//checkRectInWindow(x_,y_,w,height_);
		
		width_=w;
		renderBorder();		
		driver_->swapFb();

		prevX_=x;
		prevY_=y;

		break;
	default:
		break;
	}
}

void OpenGLGui::resize_end()
{
	dragType_ =  OpenGLWidget::NoDrag;	

	setPos(x_,y_);
	setWidth(width_);
	setHeight(height_);

	rendered_=false;
	map(true,true);
}



void OpenGLGui::replace(int x,int y)
{			
	int xp=x_+x-prevX_;
	int yp=y_+y-prevY_;

	checkRectInWindow(xp,yp,width_,height_);

	if(1)
	{
		mapBg();

		//setX(xp);
		//setY(yp);
		setPos(xp,yp);
		resetGeometry();
		
		prevX_=x;
		prevY_=y;		
		
		//map(true,true);

		//renderBorder();
	}
}
	
void OpenGLGui::replace_end()
{				
	dragType_ =  OpenGLWidget::NoDrag;	
}


bool OpenGLGui::active()
{
	//return (focused_ == true || active_ == true);
	return (active_== true || dragType_ !=  OpenGLWidget::NoDrag || changed_ == true);
}

void OpenGLGui::setVisible(bool v)
{	
	if(visible_ != v)
	{
		changeVisibility();
	}
}	

void OpenGLGui::changeVisibility()
{
	if(visible_== true)
	{
		visible_=false;
		active_=false;
		focused_=false;
		alphaCurrent_=alphaUnfocus_;
		dragType_=OpenGLWidget::NoDrag;
	}
	else
	{
		visible_=true;
	}
}


//-------------------------------------------------
// Callbacks - static wrappers
//-------------------------------------------------

void OpenGLGui::close_cb(void *object,void *data)
{
	OpenGLGui *instance = (OpenGLGui*) object;
	instance->close();
}	

//-------------------------------------------------
// Callbacks - concrete implemntations
//-------------------------------------------------

void OpenGLGui::close()
{
	if(visible_ == true)
	{	
		visible_= false;
		active_=false;
		focused_=false;
		alphaCurrent_=alphaUnfocus_;
		mapBg();
		driver_->swapFb();
	}

	notifyObserverOfVisibilityChange(false);

}
