/******************************** LICENSE ********************************


  Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.


 ******************************** LICENSE ********************************/

/*!
    \file CairoDriver.h
    \brief Definition of CairoDriver.
    \author Graphics Section, ECMWF

    Started: Mon Oct 15 20:49:32 2007
*/

#ifndef _CairoDriver_H
#define _CairoDriver_H

#include <cairo.h>
#include <BaseDriver.h>
#include <CairoDriverAttributes.h>
#include <XmlNode.h>

namespace magics
{

/*! \class CairoDriver
    \brief This driver produces output for Cairo
    \ingroup drivers

    This driver ...
*/
class CairoDriver: public BaseDriver, public CairoDriverAttributes
{

public:
	CairoDriver();
	~CairoDriver();
	void open();
	void close();

	/*!
	  \brief sets a new XML node
	*/
	void set(const XmlNode& node)
	{
		if ( magCompare(node.name(), "png") ||
		     magCompare(node.name(), "pdf") ||
		     magCompare(node.name(), "cps") ||
		     magCompare(node.name(), "csvg") ||
		     magCompare(node.name(), "x") ||
		     magCompare(node.name(), "ceps") )
		{
			XmlNode basic = node;
			basic.name("driver");
			BaseDriver::set(basic);
			basic.name("cairo");
			CairoDriverAttributes::set(basic);
		}
	}

	/*!
	  \brief sets a new map
	*/
	void set(const map<string, string>& map)
	{
		BaseDriver::set(map);
		CairoDriverAttributes::set(map);
	}

	void setPNG() const {backend_ = "PNG";}
	void setPDF() const {backend_ = "PDF";}
	void setPS () const {backend_ = "PS";}
	void setEPS() const {backend_ = "EPS";}
	void setSVG() const {backend_ = "SVG";}
	void setX()   const {backend_ = "X";}
	void setCairo() const {backend_ = "cairo";}

private:
	MAGICS_NO_EXPORT void startPage() const;
	MAGICS_NO_EXPORT void endPage() const;
	MAGICS_NO_EXPORT void project(const Layout& lay) const;
	MAGICS_NO_EXPORT void unproject() const;

        MAGICS_NO_EXPORT void setNewLineWidth(const float) const;
	MAGICS_NO_EXPORT void setNewColour(const Colour &col) const;
	MAGICS_NO_EXPORT int  setLineParameters(const LineStyle style, const float w) const;

	MAGICS_NO_EXPORT void renderPolyline(const int, float *, float *) const;
	MAGICS_NO_EXPORT void renderPolyline2(const int n, float *x, float *y) const;
	MAGICS_NO_EXPORT void renderSimplePolygon(const int, float *, float *) const;
	MAGICS_NO_EXPORT void renderText(const Text& text) const;
	MAGICS_NO_EXPORT void circle(const float x, const float y, const float r, const int) const;
	MAGICS_NO_EXPORT bool renderPixmap(float,float,float,float,int,int,unsigned char*,int,bool) const;
	MAGICS_NO_EXPORT bool renderCellArray(const Image& obj) const;
	MAGICS_NO_EXPORT void renderImage(const ImportObject& obj) const;
	MAGICS_NO_EXPORT void renderSymbols(const Symbol& symbol) const;
	MAGICS_NO_EXPORT bool convertToPixmap(const string &fname, const GraphicsFormat format, const int reso,
		             const float wx0, const float wy0,const float wx1,const float wy1) const;
	//! Method to print string about this class on to a stream of type ostream (virtual).
	void print(ostream&) const;
	MAGICS_NO_EXPORT void debugOutput(const string &s) const;

	mutable string	filename_;

	mutable cairo_t*	cr_;
	mutable cairo_surface_t* surface_;

	MAGICS_NO_EXPORT float projectX(const float x) const {return offsetX_+(x*coordRatioX_);}
	MAGICS_NO_EXPORT float projectY(const float y) const {return offsetY_+(y*coordRatioY_);}
	MAGICS_NO_EXPORT float setAngleY(const float y) const {return -y;}
	MAGICS_NO_EXPORT float setSymbolY(const float y) const {return -y;}
	MAGICS_NO_EXPORT float setY(const float y) const {return y;}

	mutable float offsetX_;
	mutable float offsetY_;
	mutable stack<float> offsetsX_;
	mutable stack<float> offsetsY_;

	mutable string backend_;
	mutable int dimensionXglobal_;
	mutable int dimensionYglobal_;

	//! Copy constructor - No copy allowed
	CairoDriver(const CairoDriver&);
	//! Overloaded << operator to copy - No copy allowed
	CairoDriver& operator=(const CairoDriver&);

	// -- Friends
	//! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const CairoDriver& p)
		{ p.print(s); return s; }
};

} // namespace magics
#endif
