/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \defgroup drivers Output drivers

   \section generalDriver Overview

   Magics++ supports various vector and raster output formats. It
   produces a generic descriptions of the output plot which
   than gets converted by a chosen driver to the requested format(s).

   \sa DriverManager::DriverManager(), OutputHandler(), OutputFactory()

   \section usingDrivers Using drivers

   In MagML the \<drivers\> Tag is used to define all drivers being used. Each output
   format

   Example test/MagML/drivers.magml test/C/drivers.c

   \section addDriver How to add a new output format / driver

   It is assumed you try to add a driver called <i>NewDriver</i>.

   - go into the <i>tools</i> directory and run <i>perl newdriver New</i>
   - edit your <i>NewDriver.xml</i> and copy it to <i>src/xml</i>
   - add this file in <i>src/xml/Makefile.am</i>
   - edit <i>NewDriver.cc/.h</i> and copy them to <i>src/drivers/</i>
   - add the new files in <i>src/drivers/Makefile.am</i>
   - look at <i>NewDriver.todo</i> and add factories in:
		-# <i>src/common/OutputHandler.cc</i>
		-# <i>src/common/OutputFactory.cc/.h</i>


\example drivers.c Example how mulitiples drivers can be used in C.
\example drivers.magml Example how drivers can be used in MagML.
*/

/*! \file BaseDriver.h
    \brief Definition of driver base class.
    \author Graphics Section, ECMWF

    Started: Jan 2004

*/
#ifndef MPP_BaseDriver_H
#define MPP_BaseDriver_H

#include <magics.h>
#include <Translator.h>
#include <Colour.h>
#include <BaseDriverAttributes.h>

class MtInputEvent;
#include "MagicsObserver.h"

namespace magics{

class Layer;
class Layout;
class RootLayout;
class PreviewLayout;
class MagnifierLayout;
class StepRenderer;

class Polyline;
class Text;
class Symbol;
class TextSymbol;

class ComplexSymbol;
class TextItem;
class FlagItem;
class SymbolItem;

class Image;
class Arrow;
class Flag;
class ImportObject;
class SelectionMode;
class PaperPoint;

struct ShadingProperties;
struct FillShadingProperties;
struct DotShadingProperties;
struct HatchShadingProperties;


/*!
 \brief helper compare function for string maps

 The function follows the descriptionn of Josuttis's STL book at page 213.
*/
class RuntimeStringCompare
{
	static bool nocase_compare(char c1, char c2) {return toupper(c1)<toupper(c2);}
public:
	RuntimeStringCompare(){}
	bool operator() (const string& s1, const string& s2) const 
	{
		return lexicographical_compare(s1.begin(),s1.end(),s2.begin(),s2.end(),nocase_compare);
	}
};

/*!
 \brief struct describing an XML element

 Helper construct for symbol plotting.
*/
struct xmlElement
{
	string name;
	std::map<string, string> attributes;
	xmlElement() : name("") {}
};

/*!
 \brief struct describing an SVG element

 Helper construct for symbol plotting.
*/
struct svgSymbol
{
	string id;
	vector<xmlElement> elements;
	svgSymbol() : id("") {}
};

typedef vector<svgSymbol> vSymbols;

/*!
 \brief struct describing a entry in the font table

 All fonts read in by readFonts() are stored in this way.

 \sa readFonts()
*/
struct magFont
{
	int	id;
	string	magics_name;
	string	ps_name;
	string	ps_filename;
	string	ttf_filename;
	string	css_name;
};


/*! \class BaseDriver
    \brief Base class for all drivers of Magics++.
    \ingroup drivers

    This abstract class provides an interface for Magics
    to call various device drivers.

    Every new driver has to inherit this class and has
    to provide own implementation of abstract members.
*/
class BaseDriver : public BaseDriverAttributes, public MagicsObserver
{
public:
	BaseDriver();
	virtual ~BaseDriver();

	virtual void set(const XmlNode& node)
	{
		BaseDriverAttributes::set(node);
	}

	virtual void set(const std::map<string, string>& map)
	{
		BaseDriverAttributes::set(map);
	}

	virtual void open() {}
	virtual void close() {}
	void printOutputName(const std::string & str) const;

#ifdef MAGICS_OPENGL
	//! Selection of Interactive mode: tooltip : user callback called when mouse moves.
	virtual void tooltip(const SelectionMode&) {}
	//! Selection of Interactive mode: pointSelection : user callback called when a point is selected.
	virtual void pointSelection(const SelectionMode&) {}
	//! Selection of Interactive mode: areaSelection : user callback called when an area is selected.
	virtual void areaSelection(const SelectionMode&) {}
	//! Selection of Interactive mode: polygoneSelection : user callback called when a polygone is selected.
	virtual void polygonSelection(const SelectionMode&) {}
	//! Selection of Interactive mode: lineSelection : user callback called when a line is selected.
	virtual void lineSelection(const SelectionMode&) {}
	//! Selection of Interactive mode: polylineSelection : user callback called when a line is selected.
	virtual void polylineSelection(const SelectionMode&) {}
	//! Selection of Interactive mode: polylineSelection : user callback called when a line is selected.
	virtual void pickSelection(const SelectionMode&) {}
	//! Selection of Interactive mode: magnification
	virtual void magnification(const SelectionMode&) {}
	//! Selection of Interactive mode: magnification
	virtual void noModes() {}
	


	//! Restore the framebuffer contents from the background texture (OpenGL only)
	virtual void restoreFb() {}
	//! Input event handling
	virtual void inputEvent(MtInputEvent*) {}
#endif

	virtual void project(const Layout& ) const {}
	virtual void unproject() const {}

	virtual void newLayer() const { debugOutput("newLayer");}
	virtual void closeLayer() const { debugOutput("closeLayer");}
	virtual void newLayer(const Layer&) const {debugOutput("newLayer");}
	virtual void closeLayer(const Layer&) const {debugOutput("closeLayer");}

	MAGICS_NO_EXPORT void redisplay(const Layout&) const;
	MAGICS_NO_EXPORT void redisplay(const RootLayout&) const;
	virtual MAGICS_NO_EXPORT void redisplay(const Layer&) const;
	MAGICS_NO_EXPORT void redisplay(const Polyline&) const;

	virtual MAGICS_NO_EXPORT void redisplay(const PreviewLayout&) const {};
	virtual MAGICS_NO_EXPORT void redisplay(const MagnifierLayout&) const {};
	virtual MAGICS_NO_EXPORT void redisplay(const StepRenderer&) const;
	 
	//! Method to redisplay a Text using PaperPoint coordinates.
	MAGICS_NO_EXPORT void redisplay(const Text&) const;
	//! Method to redisplay a Symbol using PaperPoint coordinates.
	MAGICS_NO_EXPORT void redisplay(const Symbol&) const;
	//! Method to redisplay a TextSymbol using PaperPoint coordinates.
	MAGICS_NO_EXPORT void redisplay(const TextSymbol&) const;
	//! Method to redisplay a ComlpexSymbol using PaperPoint coordinates.
	MAGICS_NO_EXPORT void redisplay(const ComplexSymbol&) const;
	MAGICS_NO_EXPORT void redisplay(const TextItem&, const ComplexSymbol&) const;
	MAGICS_NO_EXPORT void redisplay(const FlagItem&, const ComplexSymbol&) const;
	MAGICS_NO_EXPORT void redisplay(const SymbolItem&, const ComplexSymbol&) const;
	
	//! Method to redisplay an Arrow using PaperPoint coordinates .
	MAGICS_NO_EXPORT void redisplay(const Arrow&) const;
	//! Method to redisplay an Flag using PaperPoint coordinates.
	MAGICS_NO_EXPORT void redisplay(const Flag&) const;
	//! Method to redisplay a external file using PaperPoint coordinates.
	MAGICS_NO_EXPORT void redisplay(const ImportObject&) const;
	//! Method to redisplay a external file using PaperPoint coordinates.
	MAGICS_NO_EXPORT void redisplay(const Image&) const;
	
	void shade(const FillShadingProperties&) const;
	void shade(const DotShadingProperties&) const;
	void shade(const HatchShadingProperties&) const;

	bool disable() const { return disabled_; }
	void disable(bool disabled) { disabled_ = disabled; }


	/*!
	 \param SUPER_PAGE_X_LENGTH Default: 29.7
	*/
	MAGICS_NO_EXPORT void setXDeviceLength(float xdevicelength)
		{ xDeviceLength_ =  xdevicelength; }
	MAGICS_NO_EXPORT float getXDeviceLength() const
		{ return xDeviceLength_; }

	/*!
	 \param SUPER_PAGE_Y_LENGTH Default: 21.0
	*/
	MAGICS_NO_EXPORT void setYDeviceLength(float ydevicelength)
		{ yDeviceLength_ =  ydevicelength; }
	MAGICS_NO_EXPORT float getYDeviceLength() const
		{ return yDeviceLength_; }

	typedef void (BaseDriver::*ModeFunction)(const SelectionMode&);
	typedef void (BaseDriver::*ControlFunction)(bool);
	typedef void (BaseDriver::*InputEventFunction)(MtInputEvent*);

protected:
	float convertCM(const float cm) const {return cm*cmScale_;}
	void setCMscale(const float scale) const {cmScale_ = scale;}

	/*!
	  \brief set Y values positive or negative

	  Some drivers hava a Y axis which is positive towards the bottom of the plot
	  and these need to set this method negative.
	*/
	virtual MAGICS_NO_EXPORT float setY(const float y) const {return y;}

	/*!
	  \brief set Y values positive or negative dependent how symbols are interpreted

	*/
	virtual MAGICS_NO_EXPORT float setSymbolY(const float y) const {return y;}

	/*!
	  \brief set Y values positive or negative dependent how angles are interpreted

	  Some drivers hava a angle orientation different to PS
	  and these need to set this method negative.
	*/
	virtual MAGICS_NO_EXPORT float setAngleY(const float y) const {return y;}

	string getFileName(const string &extension, const unsigned int no = 0) const;

	//! Method to print string about this class on to a stream of type ostream (virtual).
	virtual void print(ostream&) const;
	virtual void startPage() const{};
	virtual void endPage() const{};
	virtual void setNewColour(const Colour &) const {};
	virtual void printLine(const Polyline &line) const;

	virtual void renderText(const Text &) const {}
	virtual void debugOutput(const string &s) const 
	{
		if(getDebug()) Log::debug() <<" DRIVERS: "<<s<< "\n";
	}

	virtual float projectX(const float x) const {return coordRatioX_*x;}
	virtual float projectY(const float y) const {return coordRatioY_*y;}
		
	string getTmpName() const;

	double LSF(float *x,float *y, int i0) const;

	mutable int	currentPage_;
	mutable string	fileName_;
	mutable string	currentLayer_;  // from Layer (open & close)
	mutable string	currentGroup_;  // from property object layer

	mutable LineStyle currentLineType_;
	mutable float	currentLineWidth_;
	mutable int	currentLineStyle_;
	mutable Colour	currentColour_;

	mutable float	coordRatioX_;
	mutable float	coordRatioY_;
	mutable float	dimensionX_;
	mutable float	dimensionY_;

	mutable std::stack<float>  dimensionStack_;
	mutable std::stack<float>  scalesX_;
	mutable std::stack<float>  scalesY_;

	mutable float	lastAreaHeightPercentage_;
	mutable float	lastAreaWidthPercentage_;
	mutable bool	newPage_;
	mutable	bool	newLayout_;
	bool    disabled_;
	bool    alphaEnabled_;
	mutable bool    polylineAntialiasing_;

	mutable vSymbols  sym_;

	mutable stack<const Layout *>	staLayouts_;

	virtual MAGICS_NO_EXPORT void setNewLineWidth(const float w) const {currentLineWidth_ = w;}
	MAGICS_NO_EXPORT float getNewLineWidth() const {return currentLineWidth_;}

	// Load svg symbols from file
	MAGICS_NO_EXPORT void loadSymbols() const;

	// Symbol calculating methods
	virtual MAGICS_NO_EXPORT void renderTextSymbols(const TextSymbol& symbol) const;
	
	virtual MAGICS_NO_EXPORT void renderComplexSymbols(const ComplexSymbol& symbol) const;
	virtual MAGICS_NO_EXPORT void renderTextItem(const TextItem&, const ComplexSymbol&) const;
	virtual MAGICS_NO_EXPORT void renderFlagItem(const FlagItem&, const ComplexSymbol& symbol) const;
	virtual MAGICS_NO_EXPORT void renderSymbolItem(const SymbolItem&, const ComplexSymbol& symbol) const;


	
	virtual MAGICS_NO_EXPORT void renderSymbols(const Symbol& symbol) const;
	virtual MAGICS_NO_EXPORT void renderPolyline(const int, float *, float *) const {};
	virtual MAGICS_NO_EXPORT void renderPolyline2(const int, float *, float *) const {}
	MAGICS_NO_EXPORT void renderPolyline(vector<PaperPoint> &vP) const;
	MAGICS_NO_EXPORT void renderPolyline2(vector<PaperPoint> &vP) const;
	virtual MAGICS_NO_EXPORT void renderSimplePolygon(const int, float *, float *) const = 0;
	MAGICS_NO_EXPORT void renderSimplePolygon(vector<PaperPoint> &vP) const;
	virtual MAGICS_NO_EXPORT void circle(const float, const float, const float, const int) const {}
	MAGICS_NO_EXPORT void snowflake(const float, const float, const float) const;
	MAGICS_NO_EXPORT void drizzle(const float, const float, const float) const;
	MAGICS_NO_EXPORT void triangle(const float, const float, const float, const int, const int) const;
	MAGICS_NO_EXPORT void lightning(const float x, const float y, const float size) const;

	virtual MAGICS_NO_EXPORT int setLineParameters(const LineStyle st, const float w) const {currentLineType_=st; setNewLineWidth(w);return 0;}

	// Filling calculating methods
	// PolylineSets rendering methods
	//MAGICS_NO_EXPORT void renderPolylineSets(const PolylineSet<PaperPoint>&) const;
	mutable unsigned int indexHatch_;
	mutable Shading   currentShading_;
	mutable const ShadingProperties *currentShadingProperties_;

	// wind calculating methods
	MAGICS_NO_EXPORT void renderWindArrow(const Arrow &arrow) const;
	MAGICS_NO_EXPORT void renderWindFlag(const Flag &flag) const;

	// images + bitmap methods
	virtual MAGICS_NO_EXPORT void renderImage(const ImportObject& object) const;
	MAGICS_NO_EXPORT void renderImage(const Image& obj) const {renderCellArray(obj);}
        virtual MAGICS_NO_EXPORT bool convertToPixmap(const string &fname, const GraphicsFormat format, const int reso,
	             const float wx0, const float wy0,const float wx1,const float wy1) const;
	virtual MAGICS_NO_EXPORT bool renderPixmap(float,float,float,float,int,int,unsigned char*,int,bool hasAlpha=false) const;
	virtual MAGICS_NO_EXPORT bool renderCellArray(const Image& ) const;

	mutable std::map<string, magFont, RuntimeStringCompare> FontMap_;
	typedef std::map<string, magFont, RuntimeStringCompare>::const_iterator fontMapIter;

private:
	void readFonts() const;

	mutable float cmScale_;
	mutable float xDeviceLength_;
	mutable float yDeviceLength_;
	static int numFiles_;

	//! Copy constructor - No copy allowed
	BaseDriver(const BaseDriver&);
	//! Overloaded << operator to copy - No copy allowed
	BaseDriver& operator=(const BaseDriver&);

// -- Friends
	//! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const BaseDriver& p) { p.print(s); return s; }
};

/*!
 \brief translates a string to a driver
*/
template<>
class Translator<string, BaseDriver>
{
public:
	MAGICS_NO_EXPORT BaseDriver* operator()(const string& val )
	{
		 return SimpleObjectMaker<BaseDriver>::create(val);
	}
	MAGICS_NO_EXPORT BaseDriver* magics(const string& param)
	{
		BaseDriver* object;
		ParameterManager::update(param, object);
		return object;
	}
};

} // namespace magics
#endif
