#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <pwd.h>
#include <sys/types.h>
#ifdef __xlc__
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#endif

#include <stdarg.h>


typedef struct _odb_stm {
	struct _odb_stm *next;
	struct _odb     *odb;
	int              id;
	int              count;
	int              size;
	double*          data;
	int              soc;

	char**           names;
	int*             types;

	char*            host;
	int              port;

	int              params;

} odb_stm;

typedef struct _odb {

	char*            host;

	
	int soc;

	int  err;
	char *msg;

	FILE* in;
	FILE* out;

	char buffer[1024];
	char *line;

	odb_stm* stms;


} odb;


int create_socket(const char *host, int port,int local_port,const char* interface)
{
	struct sockaddr_in s_in;
	struct sockaddr_in sin;
	struct hostent *him;
	int s;
	int status;
	int tries = 0;
	int flg;
#ifdef __alpha
	unsigned int  addr;
	unsigned int  none = (unsigned int)~0;
#elif defined(fujitsu)
	u_int addr;
	u_int none = (u_int)~0;
#elif defined(__64BIT__)
	unsigned long addr;
	uint32_t none = (uint32_t)-1;
#else
	unsigned long addr;
	unsigned long none = (unsigned long)-1;
#endif

	bzero(&s_in,sizeof(s_in));

	s_in.sin_port = htons(port);
	s_in.sin_family = AF_INET;

	addr = inet_addr(host);
	s_in.sin_addr.s_addr = addr;

	if(addr == none) {
		if ((him=gethostbyname(host))==NULL)
		{
			perror(host);
			return -1;
		}

		s_in.sin_family = him->h_addrtype;

		bcopy(him->h_addr_list[0],&s_in.sin_addr,him->h_length);
	}

	s = socket(AF_INET, SOCK_STREAM, 0);
	if (s < 0)
	{
		perror("socket");
		return(-1);
	}

	memset(&sin, 0, sizeof(struct sockaddr_in));
	sin.sin_port        = htons(local_port);
	sin.sin_family      = AF_INET;

	if(interface)
		sin.sin_addr.s_addr = inet_addr(interface);

#if 0
	if(local_port == 0)
	{
		if(bindresvport(s,&sin) < 0)
		{
			perror("bindresvport");
			close(s);
			return -1;
		}
		/* fprintf(stderr,"bind : %d\n",ntohs(sin.sin_port)); */
	}
	else
#endif
	{
		if(bind(s,(struct sockaddr*)&sin,sizeof(sin)) == -1)
		{
			perror("bind");
			close(s);
			return -1;
		}
	}


	status = connect(s,(struct sockaddr*)&s_in,sizeof(s_in));
	if (status < 0)
	{
		perror("connect");
		close(s);
		return -1;
	}

	/*
	flg = 1;
	if(setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, (msockopt)&flg, sizeof(flg))<0)
		marslog(LOG_WARN|LOG_PERR,"setsockopt SO_KEEPALIVE");

*/

	signal(SIGPIPE,SIG_IGN);

	return s;   /* OK */
}


static void odb_set_error(odb* o,const char* msg)
{
	fprintf(stderr,"odb error '%s'\n",msg);
	o->err = 1;
	if(o->msg) free(o->msg);
	o->msg = strdup(o->line);
}

void odb_clear_error(odb* o)
{
	o->err = 0;
}

int odb_get_error(odb* o)
{
	return o->err;
}

const char* odb_get_error_message(odb* o)
{
	return o->msg ? o->msg : "no error";
}

static int odb_next_code(odb* o)
{
	int code = 999;
	int len;

	fgets(o->buffer,sizeof(o->buffer)-1,o->in);


	len = strlen(o->buffer);
	if(len && o->buffer[len-1] == '\n')
		o->buffer[len-1] = 0;

	sscanf(o->buffer,"%d",&code);
	o->line = o->buffer+4;

	 o->err = 0;
	if(code == 999)
		odb_set_error(o,o->line);

	return code;
}

static void odb_answer(odb* o)
{
	int code = odb_next_code(o);
	while(code != 999 && code != 0)
		code = odb_next_code(o);
}

static void odb_free_statement(odb_stm* s)
{
	if(s->host) free(s->host);
	if(s->data) free(s->data);
	free(s);
}

void odb_destroy(odb* o)
{
	odb_stm *s = o->stms;
	while(s)
	{
		odb_free_statement(s);
		s = s->next;
	}

	if(o->msg) 
		free(o->msg);

	close(o->soc);
	fclose(o->in);
	fclose(o->out);
	free(o->host);
	free(o);
}

static void odb_send(odb* o,const char* what)
{
	int len = strlen(what);
	if(write(o->soc,what,len) != len)
		odb_set_error(o,"odb_send: short write");
		
}

void odb_do(odb* o,const char* what)
{
	char *semi = ";"; const char* p = what; while(*p)  { if(*p++ == ';') semi = ""; }

	
	fprintf(o->out,"DO %s%s\n",what,semi); fflush(o->out);
	odb_answer(o);
}

void odb_destroy_statement(odb_stm* stm)
{
	/* nothing for now */
}

odb_stm* odb_prepare_statement(odb* o,const char* what)
{
	char line[1024];
	int i = 0;
	int t = 0;
	int code;
	int done = 0;

	odb_stm* s = (odb_stm*)calloc(1,sizeof(odb_stm));
	const char* p = what; 
	char semi = ';';

	if(!s) return 0;

	fprintf(o->out,"PREPARE ");
	while(*p)  
	{ 	
		fputc(*p,o->out);
		if(*p == '?') fprintf(o->out,"%d",s->params++);
		if(*p == ';') semi = ' '; 
		p++;
	}

	fputc(semi,o->out); fputc('\n',o->out); fflush(o->out);

	s->soc  = -1;
	s->host = strdup(o->host);
	
	while(!done)
	{
		switch(code = odb_next_code(o))
		{
			case 0:
				done = 1;
				break;

			case 999:
				done = 1;
				break;

			case 100:
				s->id = atol(o->line);
				printf("id %d\n",s->id);
				break;

			case 101:
				s->host = strdup(o->line);
				printf("host %s\n",s->host);
				break;

			case 102:
				s->port = atol(o->line);
				printf("port %d\n",s->port);
				break;

			case 103:
				s->count = atol(o->line);
				s->names = (char**)calloc(s->count,sizeof(char*));
				s->types = (int*)calloc(s->count,sizeof(int));
				printf("size %d\n",s->count);
				i = 0;
				t = 0;
				break;

			case 104:
				printf("%d = %s\n",i,o->line);
				s->names[i++] = strdup(o->line);
				break;

			case 105:
				printf("%d = %s\n",i,o->line);
				s->types[t++] = atol(o->line);
				break;
				
			default:
				fprintf(stderr,"odb ignore %d %s\n",code,o->line);
				break;
				
		}
	}

	if(o->err)
	{
		odb_free_statement(s);
		return 0;
	}

	s->size  = sizeof(double) * s->count;
	s->data  = (double*)malloc(s->size);
	if(!s->data)
	{
		odb_free_statement(s);
		return 0;
	}

	s->odb   = o;
	s->next  = o->stms;
	o->stms  = s;

	return s;
}

int odb_no_columns(odb_stm* s)
{
	return s->count;
}

const char* odb_column_name(odb_stm* s,int i)
{
	return s->names[i];
}

int odb_column_type(odb_stm* s,int i)
{
	return s->types[i];
}

void odb_execute_with_params(odb_stm* s,int count,const double* params)
{
	if(s->params != count)
	{
		odb_set_error(s->odb,"wrong number of parameters");
		return;
	}

	if(s->params)
	{
		int i;
		fprintf(s->odb->out,"PARAM %d\n",s->params); 

		for(i = 0; i < s->params; i++)
			fprintf(s->odb->out,"%g\n",params[i]);
		
		fflush(s->odb->out);
		odb_answer(s->odb);
	}


	fprintf(s->odb->out,"EXECUTE %d\n",s->id); fflush(s->odb->out);
}

void odb_execute(odb_stm* s)
{
	odb_execute_with_params(s,0,0);
}

void odb_execute_with_vparams(odb_stm* s,...)
{
	double params[1024];
	int i;
	va_list list;

	va_start(list,s);

	if(s->params >= sizeof(params)/sizeof(params[0]))
	{
		odb_set_error(s->odb,"two many parameters");
		return;
	}

	for(i = 0; i < s->params; i++)
		params[i] = va_arg(list,double); 

	va_end(list);

	odb_execute_with_params(s,s->params,params);
}

const double* odb_fetch_row(odb_stm* s)
{
	int n,m;
	char *p;

	if(s->soc < 0)
	{
		printf("call %s %d\n",s->host,s->port);
		s->soc = create_socket(s->host,s->port,0,0);
		if(s->soc < 0)
		{
			odb_set_error(s->odb,"connect error");
			return 0;
		}
	}

	p = (char*)s->data;
	m = s->size;

	while(m)
	{
		n = read(s->soc,p,m);
		if(n < 0)
		{
			odb_set_error(s->odb,"read error");
			return 0;
		}

		if(n == 0)
		{
			/* End of file */
			close(s->soc);
			s->soc = -1;
			odb_answer(s->odb);
			return 0;
		}

		m -= n;
		p += n;
			
	}

	return s->data;

	/* error checking here */


}

static void odb_login(odb* o)
{
	/* inform of endian */
	char*            byte = "BYTE";
	unsigned long *endian = (unsigned long*)byte;

	
	fprintf(o->out,"LOGIN %u\n",*endian); fflush(o->out);
	odb_answer(o);
}

odb* odb_connect(const char* host,int port)
{
	int soc =  create_socket(host,port,0,0);
	if(soc != -1)
	{
		odb* o = (odb*)calloc(1,sizeof(odb));
		if(o)
		{
			o->soc  = soc;
			o->in   = fdopen(soc,"r");
			o->out  = fdopen(soc,"w");
			o->host = strdup(host);

			odb_login(o);

			return o;
		}
	}
	return NULL;
}
