/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file Wind.cc
    \brief Implementation of the Template class Wind.
    
    Magics Team - ECMWF 2005
    
    Started: Thu 17-Mar-2005
    
    Changes:
    
*/

#include "Wind.h"
#include "MatrixHandler.h"
#include "Layout.h"
#include "CustomisedPoint.h"

using namespace magics;

template <class P>
Wind<P>::Wind() 
{
}

template <class P>
Wind<P>::~Wind() 
{
}

/*!
 Class information are given to the output-stream.
*/	
template <class P>	
void Wind<P>::print(ostream& out)  const
{
	out << "Wind<P>";
}

template <class P>	
void Wind<P>::operator()(Data<P>& data, BasicGraphicsObjectContainer& parent)
{
	
	const Transformation& transformation = parent.transformation();
	try {
		MatrixHandler<P>& inx = data.xComponent();
		MatrixHandler<P>& iny = data.yComponent();
		
		vector<GeoPoint> thinx;
		vector<GeoPoint> thiny;
		
     int xfactor = this->thinning_factor_;
		if ( magCompare(this->thinning_method_, "automatic" ) ) {
			// We need to recalcalate the thinning ...			
			double width = parent.transformation().getMaxX() - parent.transformation().getMinX();
			this->thinning_factor_ = (width/this->thinning_factor_) / inx .XResolution();
			xfactor = (int) ceil((float)  this->thinning_factor_);
			 
			Log::dev() << " Thinning factor->" << this->thinning_factor_ << endl;
			
		}
		this->thinning_factor_ = xfactor;
		(*this->type_).prepare(parent, xfactor*inx .XResolution());

		transformation.thin(inx, this->thinning_factor_, this->thinning_factor_, thinx);
		transformation.thin(iny, this->thinning_factor_, this->thinning_factor_, thiny);

		typename vector<P>::const_iterator x = thinx.begin();
		typename vector<P>::const_iterator y = thiny.begin();

		while ( x != thinx.end() && y != thiny.end() )
		{
			
			if ( x->value() != inx.missing() &&  y->value() != iny.missing() ) {
			    pair<double, double> component = make_pair( x->value(), y->value());
			    transformation.reprojectComponents(*x, component);
				(*this->type_)(transformation(*x), component.first,  component.second, 0);
			}
			// Here we could check that the lat/lon of the x and y coordinates are the same...
			x++;
			y++;
		} 
	}
	catch ( exception& e) 
	{
		// We try to get some customised points!
		
		CustomisedPointsList points;
		std::set<string> request;
		request.insert("x_component");
		request.insert("y_component");
		request.insert("colour_component");
		data.customisedPoints(transformation, request, points);
		
	
		
		(*this->type_).adjust(points, transformation);
		
		for (vector<CustomisedPoint*>::const_iterator point =points.begin(); point != points.end(); ++point) {
			 P position((*point)->longitude(), (*point)->latitude());	 
			 if ( transformation.in(position) )  {
				 pair<double, double> component = make_pair(  (**point)["x_component"], (**point)["y_component"]);		 
				 transformation.reprojectComponents(position, component);
				 (*this->type_)(transformation(position), component.first,  component.second, 0);
			 }	
		}
	}
	(*this->type_).finish(parent);
}

template <class P>	
void Wind<P>::visit(LegendVisitor& legend)
{
	(*this->type_).visit(legend);
}
