# Copyright 2012 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Test maasserver middleware classes."""

from __future__ import (
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = []

import httplib
import json

from django.core.exceptions import ValidationError
from django.test.client import RequestFactory
from maasserver.exceptions import (
    MaasAPIException,
    MaasAPINotFound,
    )
from maasserver.middleware import (
    APIErrorsMiddleware,
    ExceptionMiddleware,
    )
from maasserver.testing import (
    factory,
    TestCase,
    )


def fake_request(base_path):
    """Create a fake request.

    :param base_path: The base path to make the request to.
    """
    rf = RequestFactory()
    return rf.get('%s/hello/' % base_path)


class ExceptionMiddlewareTest(TestCase):

    def make_base_path(self):
        """Return a path to handle exceptions for."""
        return "/%s" % factory.getRandomString()

    def make_middleware(self, base_path):
        """Create an ExceptionMiddleware for base_path."""
        class TestingExceptionMiddleware(ExceptionMiddleware):
            path_regex = base_path

        return TestingExceptionMiddleware()

    def process_exception(self, exception):
        """Run a given exception through a fake ExceptionMiddleware.

        :param exception: The exception to simulate.
        :type exception: Exception
        :return: The response as returned by the ExceptionMiddleware.
        :rtype: HttpResponse or None.
        """
        base_path = self.make_base_path()
        middleware = self.make_middleware(base_path)
        request = fake_request(base_path)
        return middleware.process_exception(request, exception)

    def test_ignores_paths_outside_path_regex(self):
        middleware = self.make_middleware(self.make_base_path())
        request = fake_request(self.make_base_path())
        exception = MaasAPINotFound("Huh?")
        self.assertIsNone(middleware.process_exception(request, exception))

    def test_ignores_unknown_exception(self):
        # An unknown exception is not processed by the middleware
        # (returns None).
        self.assertIsNone(
            self.process_exception(ValueError("Error occurred!")))

    def test_reports_MaasAPIException_with_appropriate_api_error(self):
        class MyException(MaasAPIException):
            api_error = httplib.UNAUTHORIZED

        exception = MyException("Error occurred!")
        response = self.process_exception(exception)
        self.assertEqual(
            (httplib.UNAUTHORIZED, "Error occurred!"),
            (response.status_code, response.content))

    def test_renders_MaasAPIException_as_unicode(self):
        class MyException(MaasAPIException):
            api_error = httplib.UNAUTHORIZED

        error_message = "Error %s" % unichr(233)
        response = self.process_exception(MyException(error_message))
        self.assertEqual(
            (httplib.UNAUTHORIZED, error_message),
            (response.status_code, response.content.decode('utf-8')))

    def test_reports_ValidationError_as_Bad_Request(self):
        response = self.process_exception(ValidationError("Validation Error"))
        self.assertEqual(
            (httplib.BAD_REQUEST, "Validation Error"),
            (response.status_code, response.content))

    def test_returns_ValidationError_message_dict_as_json(self):
        exception = ValidationError("Error")
        exception_dict = {'hostname': 'invalid'}
        setattr(exception, 'message_dict', exception_dict)
        response = self.process_exception(exception)
        self.assertEqual(exception_dict, json.loads(response.content))
        self.assertIn('application/json', response['Content-Type'])


class APIErrorsMiddlewareTest(TestCase):

    def test_handles_error_on_API(self):
        middleware = APIErrorsMiddleware()
        non_api_request = fake_request("/api/hello")
        exception = MaasAPINotFound("Have you looked under the couch?")
        response = middleware.process_exception(non_api_request, exception)
        self.assertEqual(
            (httplib.NOT_FOUND, "Have you looked under the couch?"),
            (response.status_code, response.content))

    def test_ignores_error_outside_API(self):
        middleware = APIErrorsMiddleware()
        non_api_request = fake_request("/middleware/api/hello")
        exception = MaasAPINotFound("Have you looked under the couch?")
        self.assertIsNone(
            middleware.process_exception(non_api_request, exception))
