/**
 * \file insetexternal.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Asger Alstrup Nielsen
 *
 * Full author contact details are available in file CREDITS
 */

#include <config.h>

#include "insetexternal.h"
#include "ExternalTemplate.h"
#include "BufferView.h"
#include "buffer.h"
#include "frontends/LyXView.h"
#include "LaTeXFeatures.h"
#include "gettext.h"
#include "debug.h"
#include "lyxlex.h"

#include "frontends/Dialogs.h"

#include "support/FileInfo.h"
#include "support/filetools.h"
#include "support/forkedcall.h"
#include "support/lstrings.h"
#include "support/lyxalgo.h"
#include "support/os.h"
#include "support/package.h"
#include "support/path.h"

#include <cstdio>
#include <utility>


using lyx::FileInfo;

using std::ostream;
using std::endl;


InsetExternal::InsetExternal()
	: view_(0)
{
	tempname_ = lyx::tempName(string(), "lyxext");
	//ExternalTemplateManager::Templates::const_iterator i1;
	params_.templ = ExternalTemplateManager::get().getTemplates().begin()->second;
}


InsetExternal::~InsetExternal()
{
	lyx::unlink(tempname_);
	hideDialog();
}


InsetExternal::Params InsetExternal::params() const
{
	return params_;
}


void InsetExternal::setFromParams(Params const & p)
{
	params_.filename = p.filename;
	params_.parameters = p.parameters;
	params_.templ = p.templ;
}


string const InsetExternal::editMessage() const
{
	return doSubstitution(0, params_.templ.guiName, false);
}


void InsetExternal::edit(BufferView * bv,
			 int /*x*/, int /*y*/, mouse_button::state)
{
	view_ = bv;
	view_->owner()->getDialogs().showExternal(this);
}


void InsetExternal::edit(BufferView * bv, bool)
{
	edit(bv, 0, 0, mouse_button::none);
}


void InsetExternal::write(Buffer const *, ostream & os) const
{
	os << "External " << params_.templ.lyxName << ",\""
	   << params_.filename << "\",\"" << params_.parameters << "\"\n";
}


void InsetExternal::read(Buffer const *, LyXLex & lex)
{
	string format;
	string token;

	// Read inset data from lex and store in format
	if (lex.eatLine()) {
		format = lex.getString();
	} else {
		lex.printError("InsetExternal: Parse error: `$$Token'");
	}

	while (lex.isOK()) {
		lex.nextToken();
		token = lex.getString();
		if (token == "\\end_inset")
			break;
	}
	if (token != "\\end_inset") {
		lex.printError("Missing \\end_inset at this point. "
			       "Read: `$$Token'");
	}

	// Parse string format...
	string::size_type const pos1 = format.find(',');
	params_.templ = ExternalTemplateManager::get().getTemplateByName(format.substr(0, pos1));
	string::size_type const pos2 = format.find("\",\"", pos1);
	params_.filename = format.substr(pos1 + 2, pos2 - (pos1 + 2));
	params_.parameters = format.substr(pos2 + 3, format.length() - (pos2 + 4));

	lyxerr[Debug::INFO] << "InsetExternal::Read: " << params_.templ.lyxName
			    << ' ' << params_.filename
			    << ' ' << params_.parameters << endl;
}


int InsetExternal::write(string const & format,
			 Buffer const * buf, ostream & os) const
{
	ExternalTemplate const & et = params_.templ;
	ExternalTemplate::Formats::const_iterator cit =
		et.formats.find(format);
	if (cit == et.formats.end()) {
		lyxerr << "External template format '" << format
		       << "' not specified in template "
		       << params_.templ.lyxName << endl;
		return 0;
	}

	updateExternal(format, buf);
	bool const use_latex_path = format == "LaTeX";
	string const outstring = doSubstitution(buf, cit->second.product,
						use_latex_path);
	os << outstring;
	return lyx::count(outstring.begin(), outstring.end(), '\n');
}


int InsetExternal::latex(Buffer const * buf,
			 ostream & os, bool, bool) const
{
	return write("LaTeX", buf, os);
}


int InsetExternal::ascii(Buffer const * buf, ostream & os, int) const
{
	return write("Ascii", buf, os);
}


int InsetExternal::linuxdoc(Buffer const * buf, ostream & os) const
{
	return write("LinuxDoc", buf, os);
}


int InsetExternal::docbook(Buffer const * buf, ostream & os, bool) const
{
	return write("DocBook", buf, os);
}


void InsetExternal::validate(LaTeXFeatures & features) const
{
	ExternalTemplate const & et = params_.templ;
	ExternalTemplate::Formats::const_iterator cit =
		et.formats.find("LaTeX");

	if (cit == et.formats.end())
		return;

	if (!cit->second.requirement.empty()) {
		features.require(cit->second.requirement);
	}
	if (!cit->second.preamble.empty()) {
		features.addExternalPreamble(cit->second.preamble + "\n");
	}
}


Inset * InsetExternal::clone(Buffer const &, bool same_id) const
{
	InsetExternal * inset = new InsetExternal;
	inset->params_ = params_;
	inset->view_ = view_;
	if (same_id)
		inset->id_ = id_;
	return inset;
}


string const InsetExternal::getScreenLabel(Buffer const *) const
{
	ExternalTemplate const & et = params_.templ;
	if (et.guiName.empty())
		return _("External");
	else
		return doSubstitution(0, et.guiName, false);
}


void InsetExternal::executeCommand(string const & s,
				   Buffer const * buffer) const
{
	Path p(buffer->filePath());
	Forkedcall one;
	if (lyxerr.debugging()) {
		lyxerr << "Executing '" << s << "' in '"
		       << buffer->filePath() << '\'' << endl;
	}
	one.startscript(Forkedcall::Wait, s);
}


namespace {

string const subst_path(string const & input,
			string const & placeholder,
			string const & path,
			bool use_latex_path)
{
	if (input.find(placeholder) == string::npos)
		return input;
	string const path2 = use_latex_path ?
		latex_path(path) : os::external_path(path);
	return subst(input, placeholder, path2);
}

} // namespace anon


string const InsetExternal::doSubstitution(Buffer const * buffer,
					   string const & s,
					   bool use_latex_path) const
{
	string result;
	string const basename = ChangeExtension(params_.filename, string());
	string filepath;
	if (buffer && !buffer->tmppath.empty() && !buffer->niceFile) {
		filepath = buffer->filePath();
	}
	result = subst_path(s, "$$FName", params_.filename, use_latex_path);
	result = subst_path(result, "$$Basename", basename, use_latex_path);
	result = subst(result, "$$Parameters", params_.parameters);
	result = subst_path(result, "$$FPath", filepath, use_latex_path);
	result = subst_path(result, "$$Tempname", tempname_, use_latex_path);
	result = subst_path(result, "$$Sysdir",
			    lyx::package().system_support(), use_latex_path);

	// Handle the $$Contents(filename) syntax
	if (contains(result, "$$Contents(\"")) {

		string::size_type const pos = result.find("$$Contents(\"");
		string::size_type const end = result.find("\")", pos);
		string const file = result.substr(pos + 12, end - (pos + 12));
		string contents;
		if (buffer) {
			// Make sure we are in the directory of the buffer
			Path p(buffer->filePath());
			contents = GetFileContents(file);
		} else {
			contents = GetFileContents(file);
		}
		result = subst(result,
			       ("$$Contents(\"" + file + "\")").c_str(),
			       contents);
	}

	return result;
}


void InsetExternal::updateExternal() const
{
	updateExternal("LaTeX", view_->buffer());
}

void InsetExternal::updateExternal(string const & format,
				   Buffer const * buf) const
{
	ExternalTemplate const & et = params_.templ;
	ExternalTemplate::Formats::const_iterator cit =
		et.formats.find(format);

	if (cit == et.formats.end() ||
	    cit->second.updateCommand.empty() ||
	    !et.automaticProduction)
		return;

	if (!cit->second.updateResult.empty()) {
		string const resultfile =
			doSubstitution(buf, cit->second.updateResult, false);
		FileInfo fi(params_.filename);
		FileInfo fi2(resultfile);
		if (fi2.exist() && fi.exist() &&
		    difftime(fi2.getModificationTime(),
			     fi.getModificationTime()) >= 0) {
			lyxerr[Debug::FILES] << resultfile
					     << " is up to date" << endl;
			return;
		}
	}

	executeCommand(doSubstitution(buf, cit->second.updateCommand, false),
		       buf);
}


void InsetExternal::viewExternal() const
{
	ExternalTemplate const & et = params_.templ;
	if (et.viewCommand.empty())
		return;

	updateExternal();
	executeCommand(doSubstitution(view_->buffer(),
				      et.viewCommand, false),
		       view_->buffer());
}


void InsetExternal::editExternal() const
{
	ExternalTemplate const & et = params_.templ;
	if (et.editCommand.empty())
		return;

	updateExternal();
	executeCommand(doSubstitution(view_->buffer(),
				      et.editCommand, false),
		       view_->buffer());
}


bool operator==(InsetExternal::Params const & left,
		InsetExternal::Params const & right)
{
	return ((left.filename   == right.filename) &&
		(left.parameters == right.parameters) &&
		(left.templ.lyxName == right.templ.lyxName));
}


bool operator!=(InsetExternal::Params const & left,
		InsetExternal::Params const & right)
{
	return	!(left == right);
}
