/*
 *  client/cache.c
 *		Interface to the LysKOM services.  Also does some
 *		caching of information from the server, thus the
 *		name.
 *
 *
 *  Copyright (C) 1990	Lysator Computer Club,
 *			Linkoping University,  Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author:	Thomas Bellman
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 *  email:	Bellman@Lysator.LiU.SE
 *
 *
 *  Any opinions expressed in this code are the author's PERSONAL opinions,
 *  and does NOT, repeat NOT, represent any official standpoint of Lysator,
 *  even if so stated.
 */

#include <config.h>
#if STDC_HEADERS || HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <signal.h>
#if MALLOC_H
#include <malloc.h>
#endif

#ifdef HAVE_ASSERT_H
#include <assert.h>
#endif

#include <kom-types.h> /* Before zmalloc.h on systems having size_t in
			  sys/types.h */
#ifdef __vax__
#define __SIZE_T included from string.h
#endif
#include <zmalloc.h>
#include <s-string.h>

#include <misc-types.h>
#include <services.h>
#include <kom-errno.h>

#include "copy.h"

#define CLIENT_CACHE

#include "error.h"

#include "xoutput.h"
#include "misc-parser.h"
#include "cache.h"
#include "cmds.h"
#include "main.h"
#include "offline.h"

#define Export



/*  ==================================================================  */
/*		    	    Global variables				*/


Export  Pers_no		  cpn	= 0;	/* Current Person Number */
Export  Conf_no		  ccn	= 0;	/* Current Conference Number */



/*  ==================================================================	*/
/*			    Local variables				*/


/*  Variables containing the cached data  */

static  int		  ccnpos		= -1;
static  Person		* this_person_stat	= NULL;
static	Conf_no		  saved_member_list_no	= 0;
static	Member_list	* saved_member_list	= NULL;
static	Pers_no		  saved_pers_stat_no	= 0;
static	Person		* saved_pers_stat	= NULL;
static	Text_no		  saved_text_no		= 0;
static	String		  saved_text		= EMPTY_STRING_i;
static	Conf_no		  saved_text_map_conf	= 0;
static	Text_list	  saved_text_map	= EMPTY_TEXT_LIST_i;
static	Membership_list	  saved_membership_list	= EMPTY_MEMBERSHIP_LIST_i;
static	Pers_no		  saved_membership_pers	= 0;
static	Bool		  saved_membership_hvit = FALSE;
static	int		  saved_membership_next = 0;
static	int		  saved_conf_names_count = -1;
static  String		**saved_conf_names	= NULL;
static	int		* saved_conf_last	= NULL;


/*  ==================================================================  */
/*		Macros and declarations of local functions.		*/


/*
 *  Lock and unlock the cache, so it won't get disturbed by
 *  signals and other strange things.  There must be a variable
 *  of type 'Signal_state' named 'save_signal' in the function
 *  calling LOCK and UNLOCK.  (And you must do the LOCK and
 *  UNLOCK in the same function.)
 */
#define LOCK	oldsigmask = sigblock (sigmask (SIGINT))
#define UNLOCK	sigsetmask (oldsigmask)



/*  Return position of conf CONF in LIST, or -1 if not present  */
inline static  int	conf_pos (Conf_no conf, Membership_list *list);
void within_cache_mark_text_as_read(Local_text_no, Membership*);




/*  ==================================================================  */
/*			    Support functions				*/


/*
 *  Find the position of the conference CONF in the membership list LIST.
 *  Returns -1 if not found.
 */
inline static  int
conf_pos (Conf_no		  conf,
	  Membership_list	* list)
{
    int		  i;

    i = 0;
    while (i < list->no_of_confs)
    {
	if (list->confs[i].conf_no == conf)
	    return  i;
	i++;
    }
    return  -1;
}   /* END: conf_pos */

/*  ==================================================================  */
/*	Functions to get various data from the server, possibly		*/
/*	through the cache.						*/



/*
 *  Function to handle a newly created text.
 *  Push all the info on a small queue.
 */
struct atext {
    Text_no		  text_no;
    Text_stat		  text_s;
    struct atext	* next;
};
static	struct atext 	* first = NULL;	/* The empty list is represented by */
static	struct atext	* last  = NULL;	/* first being NULL. */


/* Catches a text and puts it in the list. */
static void
catch_a_new_text(Text_no text_no, Text_stat text_stat_val)
{
    struct atext	* element;

    element = (struct atext *) zmalloc(sizeof(struct atext));

    element->text_no = text_no;
    element->text_s = text_stat_val;/* Copy the whole struct */
    (void) mark_text_stat (&text_stat_val);
    element->next = NULL;

    if (first == NULL)		/* We have an empty queue. */
    {
	first = element;
	last = element;
    } else {
	last->next = element;
	last = element;
    }
}

/* Saves a messages in a list. */
struct amessage {
    Conf_no		  to, from;
    String		  message;
    struct amessage 	* next;
};
static	struct amessage * first_message = NULL;
static	struct amessage * last_message = NULL;

static void
catch_a_directed_message(Conf_no to, Conf_no from, String message)
{
    struct amessage	* element;

    element = (struct amessage *) zmalloc(sizeof(struct amessage));

    element->to = to;
    element->from = from;
    element->message = message;
    (void) s_mark(&(element->message));
    element->next = NULL;

    if (first_message == NULL)	/* We have an empty queue. */
    {
	first_message = element;
	last_message = element;
    } else {
	last_message->next = element;
	last_message = element;
    }
    (void) s_release(&(element->message)); /* A *really* smart compiler could 
					      remove this one and the s_mark
					      above.
					      Perhaps I will do it manually 
					      in a later version.
					    */
}


/* Handle all asyncs. Fills in caches... Done when we can safely do so. */
Export void
handle_all_asyncs(void)
{
    /* Handle async messages */
    while (first_message != NULL)
    {
	struct amessage	* element;

	show_directed_message(first_message->to, 
			      first_message->from, 
			      first_message->message);

	element = first_message;
	first_message = first_message->next;
	(void)s_release(&element->message);
	zfree(element);
    }


    /* Handle texts */
    while (first != NULL)
    {
	Misc_info_group		  misc_data;
	const Misc_info		* misc_list;
	struct atext		* root_text;

	/* Update the last text in conf accordingly (if saved) */
	misc_list = first->text_s.misc_items;
	while ((misc_data = parse_next_misc (&misc_list,
					     first->text_s.misc_items + 
					     first->text_s.no_of_misc)).type
	       != m_end_of_list)
	{
	    Conf_no	recp;

#ifdef HAVE_ASSERT_H
	    assert(misc_data.type != m_error);
#endif

	    switch (misc_data.type)
	    {
	    case  m_recpt:
	    case  m_cc_recpt:
		recp = misc_data.recipient;
		break;
	    default:
		recp = 0;
	    }

	    if (recp)		/* No recpt at this misc-position. */
	    {
		if ((int)recp < saved_conf_names_count /* theoretically it 
							   can be -1 */
		    /* We have saved this one */
		    && misc_data.local_no > saved_conf_last[recp])
				/* This async is current. */
		{
		    saved_conf_last[recp] = misc_data.local_no;
		}
	    }
	}

	root_text = first;
	first = first->next;
	release_text_stat(&(root_text->text_s));
	zfree(root_text);
    }
}


/*
 *  Initialize the cache.  Must be called at the start of the
 *  program.
 */
Export  void
init_cache (void)

{
    register_new_text(&catch_a_new_text);
    register_directed_message(&catch_a_directed_message);
}





/*
 *  Changes what the current user is.  The new user must be logged in
 *  on the server.
 */
Export  void
cache_change_person (Pers_no	pers)

{
#define FUNCTION "cache_change_person()"

    Kom_err		  error;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": %lu\n", pers);
#endif

    /* First, forget all in cache */
    zfree (release_person (this_person_stat));
    this_person_stat = NULL;
    cpn				= 0;
    zfree (release_member_list (saved_member_list));
    saved_member_list		= NULL;
    saved_member_list_no	= 0;
    zfree (release_person (saved_pers_stat));
    saved_pers_stat		= NULL;
    saved_pers_stat_no		= 0;
    s_clear (&saved_text);
    saved_text_no		= 0;
    release_text_list (&saved_text_map);
    saved_text_map		= EMPTY_TEXT_LIST;
    saved_text_map_conf		= 0;
    release_membership_list (&saved_membership_list);
    saved_membership_list	= EMPTY_MEMBERSHIP_LIST;
    saved_membership_pers	= 0;

    if (!offline)
    {
	/* Then, get some initial cache data */
	this_person_stat = pers_stat (pers, &error);
	/* If we can't get the status for ourself, then there's not much we
	 * do.  Let's just go and die then. 
	 */
	if (this_person_stat == NULL)
	  {
	    fatal1 (CLIENT_UNEXPECTED_SERVER_ERROR, 
		    "Can't get our own status");
	  }
    }


    /* And remember who we are */
    cpn = pers;

    /* And forget where we were */
    ccn = 0;
#undef FUNCTION
}



/*
 *  Indicate for the cache that the user has switched the current conference.
 *  Returns zero in case of error, otherwise NEW_CONF.
 */
Export  Conf_no
cache_change_conf(Conf_no	  new_conf,
		  Kom_err	* error)

{
#define FUNCTION "cache_change_conf()"

    Success		  retval;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": %lu\n", new_conf);
#endif

    if (offline)
    {
    }
    else
    {
	retval = kom_change_conference (new_conf);
	if (retval != FAILURE)
	    *error = KOM_NO_ERROR;
	else
	{
	    *error = kom_errno;
	    switch (kom_errno)
	    {
	    case  KOM_UNDEF_CONF:
	    case  KOM_NOT_MEMBER:
		return (Conf_no)0;
		/*NOTREACHED*/
		break;

	    case  KOM_NO_CONNECT:
	    case  KOM_OUT_OF_MEMORY:
		fatal2 (CLIENT_SERVER_ERROR, "change_conf(%d)", new_conf);
		break;

	    default:
		fatal2 (CLIENT_UNEXPECTED_SERVER_ERROR, "pepsi(%d)", new_conf);
		break;
	    }
	}   /* if (retval == FAILURE) */
    }
    ccn = new_conf;

    return new_conf;
#undef FUNCTION
}   /* END: cache_change_conf */



/*
 *  Get the name of a conference.  Returns EMPTY_STRING if
 *  it can't find the name.
 */
Export  String
conf_name (Conf_no	  conf,
	   Kom_err	* error)

{
#define FUNCTION "conf_name()"
    Conference		* stat;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": %lu\n", conf);
#endif

    *error = KOM_NO_ERROR;
    if (conf == 0)
    {
	*error = KOM_CONF_ZERO;
	return  EMPTY_STRING;
    }
    if ((int) conf <= saved_conf_names_count
	&& saved_conf_names[conf]) 
    {
        /* We found the entry in the cache. */
	return *s_mark(saved_conf_names[conf]);
    }
    stat = conf_stat(conf, error);

    switch (*error)
    {
    case KOM_UNDEF_CONF:
	/* There was no such conf. */
        return EMPTY_STRING;
    default:
        break;
    }

    /* Now the conf is in the cache. */

    release_conf(stat);
    zfree(stat);

    if (saved_conf_names[conf])
    {
	return  *s_mark (saved_conf_names[conf]);
    }
    else
    {
	/* Failed to find the name */
	return EMPTY_STRING;
    }
#undef FUNCTION
}   /* END: conf_name */



/*
 *  Get the last local_no of a conference.  Returns -1 if
 *  it can't find the conf.
 */
Export  int
conf_last (Conf_no	  conf,
	   Kom_err	* error)

{
#define FUNCTION "conf_last()"

    Conference		* stat;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": %lu\n", conf);
#endif

    *error = KOM_NO_ERROR;
    if (conf == 0)
    {
	*error = KOM_CONF_ZERO;
	return -1;
    }
    if ((int) conf <= saved_conf_names_count
	&& (saved_conf_last[conf] != -1)) {
	return saved_conf_last[conf];
    }
    stat = conf_stat(conf, error);
    release_conf(stat);
    zfree(stat);

    if (*error == KOM_UNDEF_CONF)
    {
	return -1;
    }

    return saved_conf_last[conf];

#undef FUNCTION
}   /* END: conf_last */



/*
 *  Return a pointer to a Conference struct for the requested
 *  conference.
 *  A NULL pointer is returned if no data about the conference
 *  can be obtained.
 */
Export  Conference *
conf_stat (Conf_no	  conf,
	   Kom_err	* error)

{
#define FUNCTION "conf_stat()"
    Conference	 	* stat;
    
    Success		  retval;

#ifdef TRACE
    if (TRACE(3))
	xprintf("DEBUG: " FUNCTION ": %lu\n", conf);
#endif

    *error = KOM_NO_ERROR;
    if (conf == 0)
    {
	*error = KOM_CONF_ZERO;
	return  NULL;
    }

    stat = (Conference *) zmalloc( sizeof (Conference) );
    if (stat == NULL) {
	fatal1 (CLIENT_OUT_OF_MEMORY, "zmalloc()");
    }
    *stat = EMPTY_CONFERENCE;

    if (offline)
    {
        if (fetch_Conference(conf, stat) == FAILURE)
	{
	    *error = KOM_UNDEF_CONF;
	    return NULL;
	}
    }
    else
    {
	retval = kom_get_conf_stat (conf, stat);
	if (retval != FAILURE)
	    *error = KOM_NO_ERROR;
	else
	{
	    *error = kom_errno;
	    switch (*error)
	    {
	    case  KOM_UNDEF_CONF:
		zfree (release_conf(stat));
		return  NULL;

	    case  KOM_NO_CONNECT:
		fatal2 (CLIENT_SERVER_ERROR, "conf_stat(%d)", conf);
		/*NOTREACHED*/

	    default:
		fatal2 (CLIENT_UNEXPECTED_SERVER_ERROR, "conf_stat(%d)", conf);
		/*NOTREACHED*/
	    }
	}
    }

    if ((int) conf > saved_conf_names_count) {
	int r;

	if (!saved_conf_names) {
	    saved_conf_names = (String **) zmalloc((conf + 1) * sizeof(String *));
	    if (saved_conf_names == NULL) {
		fatal1 (CLIENT_OUT_OF_MEMORY, "zmalloc()");
		/*NOTREACHED*/
	    }
	    saved_conf_last = (int *) zmalloc((conf + 1) * sizeof(int));
	    if (saved_conf_last == NULL) {
		fatal1 (CLIENT_OUT_OF_MEMORY, "zmalloc()");
		/*NOTREACHED*/
	    }
	}
	saved_conf_names = (String **) zrealloc((char *) saved_conf_names,
						(conf + 1) * sizeof(String *));
	if (saved_conf_names == NULL) {
	    fatal1 (CLIENT_OUT_OF_MEMORY, "zrealloc()");
	}
	saved_conf_last = (int *) zrealloc((char *) saved_conf_last,
					   (conf + 1) * sizeof(int));
	if (saved_conf_last == NULL) {
	    fatal1 (CLIENT_OUT_OF_MEMORY, "zrealloc()");
	}
	for (r = saved_conf_names_count + 1; r <= conf; r++) {
	    saved_conf_names[r] = NULL;
	    saved_conf_last[r] = -1;
	}
	saved_conf_names_count = (int) conf;
    }	
    if (saved_conf_names[conf])
	(void) s_release(saved_conf_names[conf]);

    (void) s_mark(&stat->name);
    saved_conf_names[conf] = &stat->name;
    saved_conf_last[conf] = stat->first_local_no + stat->no_of_texts - 1;

    return (Conference *) zuse (mark_conf(stat));
#undef FUNCTION
}   /* END: conf_stat */



/*
 *  Get the list of members for a conference.  Returns NULL if none
 *  can be found.
 */
Export  Member_list *
member_list (Conf_no	  conf,
	     Kom_err	* error)

{
#define FUNCTION "member_list()"

    Success		  retval;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": %lu\n", conf);
#endif

    *error = KOM_NO_ERROR;
    if (conf == 0)
    {
	*error = KOM_CONF_ZERO;
	return  NULL;
    }
    else if (conf == saved_member_list_no)
	return  zuse (mark_member_list (saved_member_list));
    else
    {
	zfree (release_member_list (saved_member_list));
	saved_member_list = zmalloc (sizeof (Member_list));
	if (saved_member_list == NULL)
	{
	    fatal1 (CLIENT_OUT_OF_MEMORY, "zmalloc()");
	}
	*saved_member_list = EMPTY_MEMBER_LIST;
	saved_member_list_no = 0;

	retval = kom_get_members (conf, 0, MAX_CONF_NO,
				  saved_member_list);
	if (retval != FAILURE)
	    *error = KOM_NO_ERROR;
	else
	{
	    *error = kom_errno;
	    switch (*error)
	    {
	    case KOM_UNDEF_CONF: /* This conference did not exist */
		return NULL;

	    case KOM_INDEX_OUT_OF_RANGE: /* This conference did not have any
					    members. That is an acceptable
					    fact. */	      
	        saved_member_list_no = conf;
		return zuse(mark_member_list(saved_member_list));
	        break;


	    case KOM_NO_CONNECT:
		fatal2(CLIENT_SERVER_ERROR, "conf_stat(%d)", conf);
		break;

	    default:
		fatal4(CLIENT_UNEXPECTED_SERVER_ERROR,
		       "conf_stat(%d) error %d %s",
		       conf, kom_errno, kom_errno_string());
		break;
	    }
	    return  NULL;
	}

	saved_member_list_no = conf;
	return  zuse (mark_member_list (saved_member_list));
    }
#undef FUNCTION
}   /* END: member_list() */
    



/*
 *  Return a pointer to a Person struct for the requested
 *  person.
 *  A NULL pointer is returned if no data about the person
 *  can be obtained.
 */
Export  Person *
pers_stat (Pers_no	  pers,
	   Kom_err	* error)

{
#define FUNCTION "pers_stat()"

    Success		  retval;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": %lu\n", pers);
#endif

    *error = KOM_NO_ERROR;
    if (pers == 0)
    {
	*error = KOM_CONF_ZERO;	/* //////////////////// KOM_PERS_ZERO */
	return  NULL;
    }
    else if (pers == cpn)
	return  zuse (mark_person (this_person_stat));
    else if (pers == saved_pers_stat_no)
	return  zuse (mark_person (saved_pers_stat));
    else
    {
	zfree (release_person (saved_pers_stat));
	saved_pers_stat = zmalloc (sizeof (Person));
	if (saved_pers_stat == NULL)
	{
	    fatal1 (CLIENT_OUT_OF_MEMORY, "zrealloc()");
	}
	*saved_pers_stat = EMPTY_PERSON;
	saved_pers_stat_no = 0;

	retval = kom_get_person_stat (pers, saved_pers_stat);
	if (retval != FAILURE)
	    *error = KOM_NO_ERROR;
	else
	{
	    *error = kom_errno;
	    switch (*error)
	    {
	    case  KOM_UNDEF_PERS:
		return  NULL;

	    case  KOM_NO_CONNECT:
		fatal2 (CLIENT_SERVER_ERROR, "pers_stat(%d)", pers);
		break;

	    default:
		fatal2 (CLIENT_UNEXPECTED_SERVER_ERROR, "pers_stat(%d)", pers);
		break;
	    }
	    return  NULL;
	}

	saved_pers_stat_no = pers;
	return  zuse (mark_person (saved_pers_stat));
    }
#undef FUNCTION
}   /* END: pers_stat */



/*
 *  Copies a Text_stat and the misc-list in it from SOURCE to DEST.
 *  Both SOURCE and DEST are pointers.  The value of DEST is set.
 */
#define COPY_TEXT_STAT(source, dest)					\
{									\
    if (((dest) = malloc (sizeof (Text_stat))) == NULL)			\
    {									\
	fatal_client_error (CLIENT_OUT_OF_MEMORY,			\
			    this_function, "malloc() 1 failed",		\
			    (fatal_info) text);				\
	}								\
	*(dest) = *(source);						\
	(dest)->misc_items = malloc ( (dest)->no_of_misc		\
				     * sizeof (Misc_info));		\
	if ((dest)->misc_items == NULL)					\
	{								\
	    fatal_client_error (CLIENT_OUT_OF_MEMORY,			\
				this_function, "malloc() 2 failed",	\
				(fatal_info) text);			\
	}								\
	memcpy ((dest)->misc_items, (source)->misc_items,		\
		(dest)->no_of_misc * sizeof (Misc_info));		\
}



/*
 *  Return pointer to Text_stat struct for the requested text.
 *  A NULL pointer is returned if no data about the text can be
 *  obtained.
 *  The returned Text_stat is in zmalloced area and the caller
 *  is responsible for its zfreeing.
 *  Note: no real caching is done, since the text stats tend to
 *  change rather often.
 */
Export  Text_stat *
text_stat (Text_no	  text)
{
#define FUNCTION "text_stat()"

    Success		  retval;
    Text_stat		  temp_stat	= EMPTY_TEXT_STAT;
    Text_stat		* result	= NULL;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": %lu\n", text);
#endif

    if (text == 0)
	return  NULL;
    else
    {
        if (offline)
	{
	    if (fetch_Text_stat(text, &temp_stat) == FAILURE)
	        return NULL;
	}
	else
	{
	    retval = kom_get_text_stat(text, &temp_stat);
	    if (retval == FAILURE)
	    {
		switch (kom_errno)
		{
		case KOM_NO_SUCH_TEXT:
		    return NULL;

		case KOM_NO_CONNECT:
		    fatal4(CLIENT_SERVER_ERROR, 
			   "get_text_stat(%ld) kom_errno: %d %s",
			   text, kom_errno, kom_errno_string());
		    break;

		case KOM_OUT_OF_MEMORY:
		    fatal2(CLIENT_OUT_OF_MEMORY, "get_pers_stat(%ld)", text);
		    break;
		default:
		    fatal4(CLIENT_UNEXPECTED_SERVER_ERROR, 
			   "get_text_stat(%ld) kom_errno: %d %s",
			   text, kom_errno, kom_errno_string());
		    break;
		}
		return  NULL;
	    }
	}

	if ((result = zmalloc (sizeof (Text_stat))) == NULL)
	{
	    fatal1 (CLIENT_OUT_OF_MEMORY, "zmalloc() copying struct");
	}
	*result = temp_stat;
	result->misc_items = zmalloc ( result->no_of_misc
				      * sizeof (Misc_info));
	if (result->misc_items == NULL)
	{
	    fatal1 (CLIENT_OUT_OF_MEMORY, "zmalloc() copying misc");
	}
	memcpy (result->misc_items, temp_stat.misc_items,
		result->no_of_misc * sizeof (Misc_info));
	zfree (temp_stat.misc_items);

	return  result;
    }   /* text != 0 */
#undef FUNCTION
}   /* END: text_stat */



/*
 *  Return a 'String' object containing the text of the requested
 *  text, or EMPTY_STRING if the text can't be obtained.
 */
Export  String
get_text (Text_no	  text,
	  String_size	  start_char,
	  String_size	  end_char,
	  Kom_err	* error	     )

{
#define FUNCTION "get_text()"

    Success		  retval;
    String		  result	= EMPTY_STRING;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": %lu, %ld, %ld\n", text, start_char, end_char);
#endif

    *error = KOM_NO_ERROR;
    if (text == 0)
	return  EMPTY_STRING;
    else if (text != saved_text_no)
    {
	(void)s_release (&saved_text);
	saved_text = EMPTY_STRING;
	if (offline)
	{
	    if (fetch_text(text, &saved_text) == FAILURE)
	    {
		*error = KOM_NO_CONNECT;
	        return EMPTY_STRING;
	    }
	}
	else
	{
	    retval = kom_get_text (text, start_char, end_char, &saved_text);
	    if (retval == FAILURE)
	    {
		*error = kom_errno;
		switch (kom_errno)
		{
		case  KOM_NO_SUCH_TEXT:
		    return  EMPTY_STRING;

		case  KOM_NO_CONNECT:
		    fatal2 (CLIENT_SERVER_ERROR, "get_text(%ld)", text);
		    break;

		default:
		    fatal2 (CLIENT_UNEXPECTED_SERVER_ERROR,
			    "get_text(%ld)", text);
		    break;
		}
		return  EMPTY_STRING;
	    }
	}

	saved_text_no = text;
    }
    if (start_char >= s_strlen (saved_text))	/* If outside the text */
	return  EMPTY_STRING;

    if (end_char >= s_strlen (saved_text))	/* If larger than the text */
	end_char = END_OF_STRING;
    if (s_substr (&result, saved_text,
		  start_char, end_char) == FAILURE)
    {
	fatal2 (CLIENT_OUT_OF_MEMORY, "s_substr(%ld)", text);
    }
    return  result;
#undef FUNCTION
}   /* END: get_text */



/*
 *  Check if a text with the local number LOCAL_NO currently is present
 *  in the Text_list *LIST.  LIST is a pointer to a Text_list.
 */
#define LOCAL_NO_PRESENT(local_no, list)		\
    (  (local_no) >= (list)->first_local_no		\
     && (local_no) < ( (list)->first_local_no		\
		      + (list)->no_of_texts))


/*
 *  Get the global Text_no for the text with local number LOCAL_NO
 *  in the Text_list pointed to by LIST.  The number *must* be
 *  present in the list, i e the local number must not be out of
 *  range for *LIST.
 */
#define GLOBAL_NO(local_no, list)				\
    ((list)->texts[(local_no) - (list)->first_local_no])


/*
 *  In the conference with the number CONF, find the global
 *  Text_no of a text with the local number LOCAL_NO, or 0 if it
 *  can't find it, in which case it sets 'kom_errno' to the
 *  error, or KOM_NO_ERROR if the cause for the 0 was that the
 *  text had been GC:ed, deleted or removed from the conference.
 */
Export  Text_no
map_local_to_global (Conf_no		  conf,
		     Local_text_no	  local_no,
		     Kom_err		* error	   )

{
#define FUNCTION "map_local_to_global()"

    Text_no tno;
    Success retval;

#define LOOK_BACK					20
#define CHUNK_SIZE					128

    if (conf == 0 || local_no == 0)
    {
#ifdef TRACE
	if (TRACE(3))
	    xprintf("DEBUG: " FUNCTION ": %lu, %lu => error\n",
		    conf, local_no);
#endif

	*error = KOM_TEXT_ZERO;
	return  0;
    }

    if (conf == saved_text_map_conf
	&& LOCAL_NO_PRESENT (local_no, &(saved_text_map)))
    {
	tno = GLOBAL_NO (local_no, &(saved_text_map));

#ifdef TRACE
	if (TRACE(3))
	    xprintf("DEBUG: " FUNCTION ": %lu, %lu => %lu (cached)\n",
		    conf, local_no, tno);
#endif

	return tno;
    }
    else
    {
        if (offline)
	{
	    Time ts;
	    if (fetch_map(conf, &saved_text_map, &ts) == FAILURE)
	    {
#ifdef TRACE
		if (TRACE(3))
		    xprintf("DEBUG: " FUNCTION ": %lu, %lu => error (offl)\n",
		    conf, local_no);
#endif

		*error = KOM_NO_CONNECT;
	        return 0;
	    }
	}
	else
	{
	    retval = kom_get_map (conf, (  (local_no >= LOOK_BACK)
					   ? local_no - LOOK_BACK : 1),
				  CHUNK_SIZE, &saved_text_map);
	    if (retval == FAILURE)
	    {
		saved_text_map_conf = 0;
		*error = kom_errno;
		switch (*error)
		{
		case  KOM_UNDEF_CONF:
		case  KOM_ACCESS:
		case  KOM_NO_SUCH_LOCAL_TEXT:
#ifdef TRACE
		    if (TRACE(3))
			xprintf("DEBUG: " FUNCTION ": %lu, %lu => error getmap\n",
				conf, local_no);
#endif

		    return  0;
		    /*NOTREACHED*/
		    break;

		case  KOM_NO_CONNECT:
		    fatal4 (CLIENT_SERVER_ERROR, 
			    "get_map(conf=%d, local_no=%ld) failed. "
			    "LysKOM-errno: %d", 
			    conf, local_no, *error);
		    break;

		default:
		    fatal4 (CLIENT_UNEXPECTED_SERVER_ERROR,
			    "get_map(conf=%d, local_no=%ld) failed. "
			    "LysKOM-errno: %d", 
			    conf, local_no, *error);
		    break;
		}
	    }
	}

	saved_text_map_conf = conf;

	error = KOM_NO_ERROR;
	if (LOCAL_NO_PRESENT (local_no, &saved_text_map))
	    tno = GLOBAL_NO (local_no, &saved_text_map);
	else
	    tno = 0;

#ifdef TRACE
	if (TRACE(3))
	    xprintf("DEBUG: " FUNCTION ": %lu, %lu => %lu\n",
		    conf, local_no, tno);
#endif

	return tno;
    }
#undef LOOK_BACK
#undef CHUNK_SIZE
#undef FUNCTION
}   /* END: map_local_to_global */


#define CHUNK_SIZE				4

static void
get_next_chunk_membership(void)
{
#define FUNCTION "get_next_chunk_membership()"
    Success			  retval;
    Membership_list		  memb_list = EMPTY_MEMBERSHIP_LIST_i;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION "\n");
#endif

    if (offline)
    {
        if (saved_membership_hvit)
	    return;

        if (fetch_Membership_list(&memb_list) == FAILURE)
	{
	    fatal1(CLIENT_SERVER_ERROR,
		   "Cannot read cache files.");
	}
	saved_membership_hvit = TRUE;

	saved_membership_next = memb_list.no_of_confs;
    }
    else
    {
	retval = kom_get_membership (saved_membership_pers,
				     saved_membership_next, CHUNK_SIZE,
				     TRUE, &memb_list);
	if (retval == FAILURE)
	{
	    switch (kom_errno)
	    {
	    case  KOM_INDEX_OUT_OF_RANGE:
		saved_membership_hvit = TRUE;
		return;
		/*NOTREACHED*/
		break;

	    case  KOM_UNDEF_PERS:
		return;/* BUG not handled */
		/*NOTREACHED*/
		break;

	    case  KOM_NO_CONNECT:
		fatal4 (CLIENT_SERVER_ERROR, 
			"get_membership(pers=%d, next_to_fetch=%d,chunk=%d)",
			saved_membership_pers, saved_membership_next, 
			CHUNK_SIZE);
		break;

	    default:
		fatal4 (CLIENT_UNEXPECTED_SERVER_ERROR,
			"get_membership(pers=%d, next_to_fetch=%d,chunk=%d)",
			saved_membership_pers, saved_membership_next, 
			CHUNK_SIZE);
		break;
	    }
	}
	if (memb_list.no_of_confs < CHUNK_SIZE)
	{			/* We have it all. */
	    saved_membership_hvit = TRUE;
	}
	saved_membership_next += CHUNK_SIZE;
    }
    saved_membership_list.confs = (Membership *)
        zrealloc((char *) saved_membership_list.confs,
		 saved_membership_next * sizeof(Membership));

    {
	int r;

	for (r = 0; r < memb_list.no_of_confs; r++) {
	    int noc = saved_membership_list.no_of_confs;

	    saved_membership_list.confs[noc] = memb_list.confs[r]; /* copy */
	    zuse(saved_membership_list.confs[noc].read_texts);
	    saved_membership_list.no_of_confs++;
	}
    }
    release_membership_list(&memb_list); /* This frees the list of read also */
#undef FUNCTION
}


/*
 * Finds the membership for the conf first in the membership_list.
 * Returns -1 if not found. (this is not an error).
 * To invalidate the membership cache call this function with Pers_no 0.
 *
 * If MEMBERSHIP is not-null, then the membership found is copied in there. It
 * is the callers responsibility to zfree what it points to.
 */
Export	int
first_membership (Pers_no	  pers,
		  Membership	* result)
{
#define FUNCTION "first_membership()"

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": %lu\n", pers);
#endif

    if (pers != saved_membership_pers) /* Invalidate everything. */
    {
        if (saved_membership_hvit && offline)
	{
	    store_Membership_list(&saved_membership_list);
	}

	saved_membership_hvit = FALSE;
	saved_membership_pers = pers;
	release_membership_list(&saved_membership_list);
	saved_membership_list.no_of_confs = 0;
	saved_membership_list.confs = NULL;
	saved_membership_next = 0;
	ccnpos = -1;
    }

    if (pers == 0)		/* Just invalidation */
    {
	return -1;
    }

    if (saved_membership_list.no_of_confs == 0)
    {
	get_next_chunk_membership();
    }

    if (saved_membership_list.no_of_confs == 0)
    {
	return -1;
    }
    ccnpos = 0;
    if (result != NULL)
    {
        mark_membership(&saved_membership_list.confs[0]);
	*result = saved_membership_list.confs[0];
    }
    return 0;

#undef FUNCTION
}

/* Finds the membership for the conf located after the last call to
 * locate_membership, first_membership or next_membership.
 * Returns -1 if last.
 */
Export	int
next_membership (Membership	* result)
{
#define FUNCTION "next_membership()"
    int 			position;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION "\n");
#endif

    if (ccnpos >= saved_membership_list.no_of_confs - 1)
        get_next_chunk_membership(); /* This might modify 
					saved_membership_list */

    if (ccnpos < saved_membership_list.no_of_confs - 1) {
	ccnpos++;
        position = ccnpos;
    }
    else {
	return -1;
    } 
    if (result != NULL && position >= 0)
    {
	*result = saved_membership_list.confs[position];
    }
    return position;

#undef FUNCTION
}
    

/*
 *  Find the data about person PERS:s membership in CONF and put the
 *  data in RESULT.  Returns the position in the (zero-originated)
 *  list, or -1 if not found.
 *
 *  If the third argument MEMBERSHIP is non-null it is a pointer to a 
 *  location where the membership is copied. If null this function does
 *  not search the conference for non-existing texts.
 */
/* BUG? Say why we didn't find anything? */
Export  int
locate_membership (Conf_no	  conf,
		   Pers_no	  pers,
		   Membership	* result)

{
#define FUNCTION "locate_membership()"

    int				  position;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": conf=%lu, pers=%lu\n", conf, pers);
#endif

    position = -1;		/* Not found yet */
    if (conf == 0 || pers == 0)
	return  -1;		/* Doesn't alter the cache. */

    /*  This might need some explanation...
     *  The loop fetches another chunk of the membership list from
     *  the server.  This should be done if the requested conference
     *  can't be found in the part we have in memory, and we haven't
     *  walked through the entire list.  We know we have reached the
     *  end of the list if we we didn't get as long part as we asked
     *  for.
     *  Everything gotten from the server is saved in the cache.
     *  It is only thrown out when locate_membership is called with
     *  a new user (status, start_anew).
     */
    first_membership(pers,NULL); /* Maybe invalidate */
    
    while ((position = conf_pos(conf, &saved_membership_list)) == -1)
    {
	if (saved_membership_hvit)
	{
	    return -1;
	}

	get_next_chunk_membership();
    }

    if (result != NULL && position >= 0)
    {
	Kom_err 	error;

	/* Lets search the membership and move the first_unread marker
	   forward. */
	while (saved_membership_list.confs[position].last_text_read 
	       		< conf_last(conf, &error)
	       && map_local_to_global(conf,
		   saved_membership_list.confs[position].last_text_read + 1,
		   &error) == 0
	       && error == KOM_NO_ERROR)
	{
	    within_cache_mark_text_as_read(saved_membership_list.confs[position].last_text_read + 1,
					   &saved_membership_list.confs[position]); /* This changes the cache. */
	}

	if (error == KOM_NO_CONNECT)
	{
	    return -1;
	}

	/* Here we do copy this. */
	*result = saved_membership_list.confs[position];
    }
    return  position;

#undef CHUNK_SIZE
#undef FUNCTION
}   /* END: locate_membership */


/*
 * mark the text LOCALNO as read in the cache. SHIP is a membership 
 * that is modified.
 */
void
within_cache_mark_text_as_read(Local_text_no	lno,
			       Membership     * ship)
{
#define FUNCTION "within_cache_mark_text_as_read()"

#ifdef TRACE
  if (TRACE(3))
      xprintf("DEBUG: " FUNCTION ": %lu\n", lno);
#endif

    if (lno == ship->last_text_read + 1) /* We just need to adjust. */
    {
	++(ship->last_text_read);
    }
    else			/* We need to add to the vector. */
    {
	int i;

	if (ship->read_texts)	/* There already where an allocated area. */
	{
	    ship->read_texts = (Local_text_no *) 
	        zrealloc ((char *) ship->read_texts, 
			  (ship->no_of_read + 1) * sizeof (Local_text_no));
	}
	else			/* Allocate a new area. */
	{
	    ship->read_texts = (Local_text_no *) 
	        zmalloc ((ship->no_of_read + 1) * sizeof (Local_text_no));
	}

	for (i = ship->no_of_read - 1 ; i >= 0 ; i--) {
	    /* We will allocate one entry for every time we try to mark 
	       a text as read in a certain conf. This won't be detected
	       until the last_text_read field reaches this spot.
	     */

	    if (lno > ship->read_texts[i])
	    {
		ship->read_texts[i+1] = lno;
		break;
	    }

	    ship->read_texts[i+1] = ship->read_texts[i];
	}
	if ((ship->no_of_read > 0 && ship->read_texts[0] > lno)
	    || ship->no_of_read == 0)
	{
	    ship->read_texts[0] = lno;
	}

	++(ship->no_of_read);
    }

    while (ship->no_of_read
	   && (ship->last_text_read + 1 == ship->read_texts[0]
	       || ship->last_text_read >= ship->read_texts[0]))
    {
	int i;

	if (ship->last_text_read + 1 == ship->read_texts[0])
	{
	    (ship->last_text_read)++;
	}

	for (i = 1; i < ship->no_of_read; i++)
	    ship->read_texts[i-1] = ship->read_texts[i];
	/* We could free one int's use of memory here! */
	(ship->no_of_read)--;
    }

#ifdef TRACE
    if (TRACE(3))
    {
	if (ship->no_of_read)
	{
	    int i;

	    xprintf("\nDEBUG: Shiplist for conf %d now 1-%d", ship->conf_no, 
		    ship->last_text_read);
	    for (i = 0; i < ship->no_of_read; i++)
	    {	
		if (i > 0 
		    && ship->read_texts[i-1] == ship->read_texts[i] - 1
		    && i+1 < ship->no_of_read
		    && ship->read_texts[i+1] == ship->read_texts[i] + 1)
				    ;
		else
		    if (i && ship->read_texts[i-1] == ship->read_texts[i]-1)
			xprintf("-%d", ship->read_texts[i]);
		    else
			xprintf(",%d", ship->read_texts[i]);
	    }
	    xprintf("\n");
	}
    }
#endif

#undef FUNCTION
}

/* Mark a global TEXT_NO as read in all confs but BUT_CONF */
Export void
mark_global_as_read(Text_no		global_text_no,
		    Conf_no		but_conf)
{
#define FUNCTION "mark_global_as_read()"
    Text_stat      		* text;
    Misc_info_group		  misc_data;
    const Misc_info		* misc_list;
    Membership	      		* ship;
    int				  position;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION ": %lu\n", global_text_no);
#endif

    text = text_stat(global_text_no);

    if (text == NULL) {	/* No such text */
	return;
    }
    misc_list = text->misc_items;

    while ((misc_data = parse_next_misc (&misc_list,
					 text->misc_items +	
					 text->no_of_misc)).type
	   != m_end_of_list)
    {
	Conf_no	  recp;

	switch (misc_data.type)
	{
	case	m_recpt:
	case	m_cc_recpt:
	    recp = misc_data.recipient;
	    break;
	default:
	    recp = 0;
	}

	if (recp)		/* No recpt at this misc-position */
	{
	    /* Mark as read in the conf if it is another one only */
	    if (but_conf != recp)
	    {
		position = locate_membership (recp, cpn, NULL);
		if (position != -1) /* We are members */
		{
		    if (offline)
		    {
			store_mark_in_conf(recp, misc_data.local_no);
		    }
		    else
		    {
			kom_mark_as_read (recp, 1, &(misc_data.local_no));
		    }
		    
		    ship = &(saved_membership_list.confs[position]);

		    within_cache_mark_text_as_read(misc_data.local_no, 
						   ship);
		}
	    }	    
	}
    }
    release_text_stat(text);
    zfree(text);
#undef FUNCTION
}


/*
 * mark_as_read_locally
 *
 * Mark a list of texts as read in the cache but does not send anything to
 * the server or store it anywere.
 */
Export Success
mark_as_read_locally(Conf_no			conf_no,
		     int			len,
		     const Local_text_no      * text_arr)
{
#define FUNCTION "mark_as_read_locally()"
    int			r;
    int			position;
    Membership	      * ship;

#ifdef TRACE
    if (TRACE(3))
    {
	int i;

	xprintf("DEBUG: " FUNCTION ": %lu\n", conf_no);
	xprintf("DEBUG: %d inlgg:", len);
	for (i = 0; i < len; i++)
	{
	    xprintf(" %d", text_arr[i]);
	}
	xprintf("\n");
    }
#endif

    position = locate_membership (conf_no, cpn, NULL);

    if (position == -1)
    {
	fatal3 (CLIENT_SHOULDNT_HAPPEN,
		"We cant find the conf %d in the membership. (current conf: %d)",
		conf_no, ccn);
    }
    ship = &(saved_membership_list.confs[position]);

    for ( r = 0; r < len; r++ ) {
	if (text_arr[r] <= ship->last_text_read)
	    continue;

	within_cache_mark_text_as_read(text_arr[r], ship);
    }
    return OK;
#undef FUNCTION
}


/*
 * mark_as_read
 *
 * Mark a list of texts as read in the cache and sends the message on to the
 * server.
 * Arguments as the kom_mark_as_read.
 */
Export Success
mark_as_read (Conf_no			conf_no,
	      int			len,
	      const Local_text_no     * text_arr    )
{
#define FUNCTION "mark_as_read()"
    int			r;
    int			position;
    Membership	      * ship;
    Text_no		global_text_no;
    Kom_err	        error;

#ifdef TRACE
    if (TRACE(3))
    {
	int i;

	xprintf("DEBUG: " FUNCTION ": %lu\n", conf_no);
	xprintf("DEBUG: %d inlgg:", len);
	for (i = 0; i < len; i++)
	{
	    xprintf(" %d", text_arr[i]);
	}
	xprintf("\n");
    }
#endif

    if (offline)
    {
        for (r = 0; r < len; r++)
	    store_mark_in_conf(conf_no, text_arr[r]);
    }
    else
    {
	kom_mark_as_read (conf_no, len, text_arr);
    }
    position = locate_membership (conf_no, cpn, NULL);

    if (position == -1)
    {
	fatal3 (CLIENT_SHOULDNT_HAPPEN,
		"We cant find the conf %d in the membership. (current conf: %d)",
		conf_no, ccn);
    }
    ship = &(saved_membership_list.confs[position]);

    for ( r = 0; r < len; r++ ) {
	if (text_arr[r] <= ship->last_text_read)
	    continue;


	within_cache_mark_text_as_read(text_arr[r], ship);

	/* Mark texts as read in other confs. */
	global_text_no = map_local_to_global(ccn, text_arr[r], &error);
	/* BUG! Musnt go wrong */

	mark_global_as_read(global_text_no, conf_no);
	/* Could change membership_cach */

	/* Make sure we really have the membership again. */
	position = locate_membership (conf_no, cpn, NULL);
	if (position == -1)
	{
	    fatal3(CLIENT_SHOULDNT_HAPPEN,
		   "We have lost the conf %d in the membership. (current conf: %d)",
		   conf_no, ccn);
	}
	ship = &(saved_membership_list.confs[position]);
    }
    return OK;
#undef FUNCTION
}
    

/*
 * get_unread_confs_count
 *
 * Returns the number of unread confs.
 */
int
get_unread_confs_count()
{
#define FUNCTION "get_unread_confs_count()"

    Conf_no_list	  list = EMPTY_CONF_NO_LIST;
    int			  count_of_uconfs;

#ifdef TRACE
    if (TRACE(3))
        xprintf("DEBUG: " FUNCTION "\n");
#endif

    if (offline)
    {
	return offline_get_unread_confs_count();
    }

    if (kom_get_unread_confs (cpn, &list) == FAILURE) {
	switch (kom_errno) {
	case  KOM_OUT_OF_MEMORY:
	    fatal1 (CLIENT_OUT_OF_MEMORY, "get_conf_no_list() failed");
	    break;

	case  KOM_NO_CONNECT:
	    fatal1 (CLIENT_SERVER_ERROR, "get_conf_no_list() failed");
	    break;

	default:
	    fatal1 (CLIENT_UNEXPECTED_SERVER_ERROR, 
		    "get_conf_no_list() failed");
	    break;
	}
    }

    count_of_uconfs = list.no_of_confs;
    release_conf_no_list(&list); 
    return count_of_uconfs;
#undef FUNCTION
}
