/**!# Light Unix I/O for Lua
 * Copyright 2012 Rob Kendrick <rjek+luxio@rjek.com>
 *
 * Copyright 2012 Daniel Silverstone <dsilvers@digital-scurf.org>
 *
 * Distributed under the same terms as Lua itself (MIT).
 */

#define LUXIO_RELEASE 1 /* clients use to check for features/bug fixes */
#define LUXIO_ABI 0 /* clients use to check the ABI of calls is same */
#define LUXIO_COPYRIGHT "Copyright 2012 Rob Kendrick <rjek+luxio@rjek.com>\n" \
	"Copyright 2012 Daniel Silverstone <dsilvers@digital-scurf.org>"

#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <lua.h>
#include <lauxlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/wait.h>
#include <sys/utsname.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#include <net/if.h>
#include <fcntl.h>
#include <poll.h>
#include <unistd.h>
#include <dirent.h>
#include <time.h>
#include <limits.h>
#include <signal.h>

#ifdef HAVE_SENDFILE
# include <sys/sendfile.h>
#endif

#if (LUA_VERSION_NUM == 501)
# define lua_rawlen(L, idx) lua_objlen((L), (idx))
#endif

#define INVALID_MODE ((mode_t) -1)

/* External interface to Lua *************************************************/

int luaopen_luxio(lua_State *L);

/**# Introduction ************************************************************/

/**>
 * Luxio provides a very light-weight binding to many of the standard POSIX
 * and common Unix library calls.  Where possible, calls are very raw.  In
 * cases such as the `dirent` family, BSD sockets, the `getaddrinfo` family,
 * and some other cases, they interfaces are somewhat "cooked" either to make
 * them more efficient or even possible.
 *
 * For the simple raw uncooked functions, all we present here is an example
 * of the C prototype, and possible styles for use in Lua.  You'll have to
 * go looking in man pages for actual information on their use.
 *
 * Not all systems will provide all the functions described here.
 */

/**# Process creation and execution ******************************************/

/**% fork
 *  retval = fork();
 *  retval, errno = fork()
 */
static int
luxio_fork(lua_State *L) /* 3.1.1 */
{
	lua_pushinteger(L, fork());
	lua_pushinteger(L, errno);

	return 2;
}

static int
luxio__exec(lua_State *L, bool usep)
{
	const char *path = luaL_checkstring(L, 1);
	int params = lua_gettop(L) - 1;
	char **args;
	int c, ret;

	/* we probably at least need them to fill in arg[0] ... */
	luaL_checkstring(L, 2);

	args = calloc(params + 1, sizeof(*args));

	for (c = 0; c < params; c++) {
		/* gah!  constness */
		args[c] = (char *)luaL_checkstring(L, c + 2);
	}

	args[c] = NULL;

	if (usep) {
		ret = execvp(path, args);
	} else {
		ret = execv(path, args);
	}

	/* if we got here, there's an error. */
	free(args);
	lua_pushinteger(L, ret);
	lua_pushinteger(L, errno);

	return 2;
}

/**% exec
 *  retval = execv(path, argv[]);
 *  retval, errno = exec(path, ...)
 */
static int
luxio_exec(lua_State *L) /* 3.1.2 */
{
	return luxio__exec(L, false);
}

/**% execp
 *  retval = execvp(file, argv[]);
 *  retval, errno = execp(file, ...)
 */
static int
luxio_execp(lua_State *L) /* 3.1.2 */
{
	return luxio__exec(L, true);
}

/* TODO: pthread_atfork() 3.1.3 */

/**# Process termination */

/**% waitpid
 *  retval = waitpid(pid, *status, options);
 *  retval, status = waitpid(pid, options)
 *  retval, errno = waitpid(pid, options)
 */
static int
luxio_waitpid(lua_State *L) /* 3.2.1 */
{
	pid_t pid = luaL_checkinteger(L, 1);
	int options = luaL_checkinteger(L, 2);
	int status;
	pid_t proc;

	proc = waitpid(pid, &status, options);
	lua_pushinteger(L, proc);
	if (proc == -1) {
		lua_pushinteger(L, errno);
	} else {
		lua_pushinteger(L, status);
	}

	return 2;
}

#define WAITPID_STATUS(x) static int luxio_##x(lua_State *L) { \
		int status = luaL_checkinteger(L, 1);	      \
		lua_pushinteger(L, x(status));		      \
		return 1;				      \
	}

/**% WIFEXITED
 *  retval = WIFEXITED(status);
 *  retval = WIFEXITED(status)
 */
WAITPID_STATUS(WIFEXITED)

/**% WEXITSTATUS
 *  retval = WEXITSTATUS(status);
 *  retval = WIFEXITED(status)
 */
WAITPID_STATUS(WEXITSTATUS)

/**% WIFSIGNALED
 *  retval = WIFSIGNALED(status);
 *  retval = WIFSIGNALED(status)
 */
WAITPID_STATUS(WIFSIGNALED)

/**% WTERMSIG
 *  retval = WTERMSIG(status);
 *  retval = WTERMSIG(status);
 */
WAITPID_STATUS(WTERMSIG)

#ifdef WCOREDUMP
/**% WCOREDUMP
 *  retval = WCOREDUMP(status);
 *  retval = WCOREDUMP(status)
 */
WAITPID_STATUS(WCOREDUMP)
#endif

/**% WIFSTOPPED
 *  retval = WIFSTOPPED(status);
 *  retval = WIFSTOPPED(status)
 */
WAITPID_STATUS(WIFSTOPPED)

/**% WSTOPSIG
 *  retval = WSTOPSIG(status);
 *  retval = WSTOPSIG(status)
 */
WAITPID_STATUS(WSTOPSIG)

#ifdef WIFCONTINUED
/**% WIFCONTINUED
 *  retval = WIFCONTINUED(status);
 *  retval = WIFCONTINUED(status)
 */
WAITPID_STATUS(WIFCONTINUED)
#endif

#undef WAITPID_STATUS

/**% _exit
 *  _exit(status);
 *  _exit(status)
 */
static int
luxio__exit(lua_State *L) /* 3.2.2 */
{
	int ret = luaL_optinteger(L, 1, 0);

	_exit(ret);

	return 0;
}

/**# Signals *****************************************************************/

/**% kill
 *  retval = kill(pid, sig);
 *  retval, errno = kill(pid, sig)
 */
static int
luxio_kill(lua_State *L) /* 3.3.2 */
{
	pid_t pid = luaL_checkinteger(L, 1);
	int sig = luaL_checkinteger(L, 2);

	lua_pushinteger(L, kill(pid, sig));
	lua_pushinteger(L, errno);

	return 2;
}

/* Signals are going to be almost impossible to do nicely and safely. */

/* TODO: Manipulate Signal Sets 3.3.3 */
/* TODO: sigaction() 3.3.4 */
/* TODO: pthread_sigmask(), sigprocmask() 3.3.5 */
/* TODO: sigpending() 3.3.6 */
/* TODO: sigsuspend() 3.3.7 */
/* TODO: sigwait(), sigwaitinfo(), sigtimedwait() 3.3.8 */
/* TODO: sigqueue() 3.3.9 */
/* TODO: pthread_kill() 3.3.10 */

/**# Timer operations ********************************************************/

/**% alarm
 *  retval = alarm(seconds);
 *  retval = alarm(seconds)
 */
static int
luxio_alarm(lua_State *L) /* 3.4.1 */
{
	unsigned int seconds = luaL_checkinteger(L, 1);
	lua_pushinteger(L, alarm(seconds));

	return 1;
}

/**% pause
 *  retval = pause();
 *  retval, errno = pause()
 */
static int
luxio_pause(lua_State *L) /* 3.4.2 */
{
	lua_pushinteger(L, pause());
	lua_pushinteger(L, errno);

	return 2;
}

/**% sleep
 *  retval = sleep(seconds);
 *  retval = sleep(seconds)
 */
static int
luxio_sleep(lua_State *L) /* 3.4.3 */
{
	unsigned int seconds = luaL_checkinteger(L, 1);

	lua_pushinteger(L, sleep(seconds));

	return 1;
}

/**# Process identification **************************************************/

/**% getpid
 *  retval = getpid();
 *  retval = getpid()
 */
static int
luxio_getpid(lua_State *L) /* 4.1.1 */
{
	lua_pushinteger(L, getpid());

	return 1;
}

/**% getppid
 *  retval = getppid();
 *  retval = getppid()
 */
static int
luxio_getppid(lua_State *L) /* 4.1.1 */
{
	lua_pushinteger(L, getppid());

	return 1;
}

/**# User identification *****************************************************/

/**% getuid
 *  retval = getuid();
 *  retval = getuid()
 */
static int
luxio_getuid(lua_State *L) /* 4.2.1 */
{
	lua_pushinteger(L, getuid());

	return 1;
}

/**% geteuid
 *  retval = geteuid();
 *  retval = geteuid()
 */
static int
luxio_geteuid(lua_State *L) /* 4.2.1 */
{
	lua_pushinteger(L, geteuid());

	return 1;
}

/**% getgid
 *  retval = getgid();
 *  retval = getgid()
 */
static int
luxio_getgid(lua_State *L) /* 4.2.1 */
{
	lua_pushinteger(L, getgid());

	return 1;
}

/**% getegid
 *  retval = getegid();
 *  retval = getegid()
 */
static int
luxio_getegid(lua_State *L) /* 4.2.1 */
{
	lua_pushinteger(L, getegid());

	return 1;
}

/**% setuid
 *  retval = setuid(uid);
 *  retval, errno = setuid(uid)
 */
static int
luxio_setuid(lua_State *L) /* 4.2.2 */
{
	uid_t uid = luaL_checkinteger(L, 1);

	lua_pushinteger(L, setuid(uid));
	lua_pushinteger(L, errno);

	return 2;
}

/**% setgid
 *  retval = setgid(gid);
 *  retval, errno = setgid(gid)
 */
static int
luxio_setgid(lua_State *L) /* 4.2.2 */
{
	gid_t gid = luaL_checkinteger(L, 1);

	lua_pushinteger(L, setgid(gid));
	lua_pushinteger(L, errno);

	return 2;
}

/* TODO: getgroups() 4.2.3 */

/**% getlogin
 *  retval = getlogin_r(buf, bufsize);
 *  retval, login = getlogin()
 *  retval, errno = getlogin()
 */
static int
luxio_getlogin(lua_State *L) /* 4.2.4 */
{
	char buf[LOGIN_NAME_MAX];
	int r = getlogin_r(buf, sizeof(buf));

	if (r != 0) {
		lua_pushinteger(L, r);
		lua_pushinteger(L, errno);

		return 2;
	}

	lua_pushinteger(L, r);
	lua_pushstring(L, buf);

	return 2;
}

/* 4.3 Process groups ********************************************************/

/* TODO: getpgrp() 4.3.1 */
/* TODO: setsid() 4.3.2 */
/* TODO: setpgid() 4.3.3 */

/**# System identification ***************************************************/

/**% uname
 *  retval = uname(buf);
 *  retval, utsname = uname() -- A table containing struct fields
 *  retval, errno = uname()
 */
static int
luxio_uname(lua_State *L) /* 4.4.1 */
{
	struct utsname buf;
	int r = uname(&buf);

	lua_pushinteger(L, r);

	if (r < 0) {
		lua_pushinteger(L, errno);

		return 2;
	}

	lua_createtable(L, 0, 6);

#define UNAME_FIELD(n) do { lua_pushstring(L, #n); \
	lua_pushstring(L, buf.n); \
	lua_settable(L, 2); } while (0)

	UNAME_FIELD(sysname);
	UNAME_FIELD(nodename);
	UNAME_FIELD(release);
	UNAME_FIELD(version);
	UNAME_FIELD(machine);
#ifdef _GNU_SOURCE
	UNAME_FIELD(domainname);
#endif

#undef UNAME_FIELD

	return 2;
}

/**# Time ********************************************************************/

/**% time
 *  retval = time(NULL);
 *  retval, errno = time()
 */
static int
luxio_time(lua_State *L) /* 4.5.1 */
{
	lua_pushinteger(L, time(NULL));
	lua_pushinteger(L, errno);

	return 2;
}

/**% times
 *  retval = times(buf);
 *  retval, tms = times() -- A table containing utime, stime, cutime, cstime
 *  retval, errno = times()
 */
static int
luxio_times(lua_State *L) /* 4.5.2 */
{
	struct tms buf;
	clock_t r = times(&buf);

	if (r == (clock_t)-1) {
		lua_pushinteger(L, r);
		lua_pushinteger(L, errno);
		return 2;
	}

	lua_pushinteger(L, r);
	lua_createtable(L, 0, 4);

#define TIMES_FIELD(n) do { lua_pushstring(L, #n); \
	lua_pushinteger(L, buf.tms_##n); \
	lua_settable(L, 2); } while (0)

	TIMES_FIELD(utime);
	TIMES_FIELD(stime);
	TIMES_FIELD(cutime);
	TIMES_FIELD(cstime);

#undef TIMES_FIELD
	return 2;
}

/**# Environment variables ****************************************************/

/**% getenv
 *  retval = getenv(name);
 *  retval = getenv(name)
 */
static int
luxio_getenv(lua_State *L) /* 4.6.1 */
{
	const char *envvar = luaL_checkstring(L, 1);

	char *envval = getenv(envvar);

	if (envval == NULL)
		return 0;

	lua_pushstring(L, envval);

	return 1;
}

/**% setenv
 *  retval = setenv(name, value, overwrite);
 *  retval, errno = setenv(name, value[, overwrite/1])
 */
static int
luxio_setenv(lua_State *L) /* POSIX.1-2001 */
{
	const char *envvar = luaL_checkstring(L, 1);
	const char *envval = luaL_checkstring(L, 2);
	int overwrite = luaL_optint(L, 3, 1);

	lua_pushinteger(L, setenv(envvar, envval, overwrite));
	lua_pushinteger(L, errno);

	return 2;
}

/**% unsetenv
 *  retval = unsetenv(name);
 *  retval, errno = unsetenv(name)
 */
static int
luxio_unsetenv(lua_State *L) /* POSIX.1-2001 */
{
	const char *envvar = luaL_checkstring(L, 1);

	lua_pushinteger(L, unsetenv(envvar));
	lua_pushinteger(L, errno);

	return 2;
}

/* 4.7 Terminal identification ***********************************************/

/* TODO: ctermid() 4.7.1 */
/* TODO: ttyname(), ttyname_r(), isatty() 4.7.2 */

/* 4.8 Configurable system variables *****************************************/

/* TODO: sysconf() 4.8.1 */

/**# Directories *************************************************************/

/**>
 * `readdir()` is a hideous API.  As such, we need to be reasonably high-level
 * here, otherwise everything is just too painful.
 */

#define LUXIO_READDIR_METATABLE "luxio.readdir"

typedef struct {
	DIR *dirp;
	struct dirent *buf, *ent;
} luxio_readdir_state;

static int
luxio_readdir_gc(lua_State *L)
{
	luxio_readdir_state *s = luaL_checkudata(L, 1, LUXIO_READDIR_METATABLE);

	closedir(s->dirp);
	free(s->buf);

	return 0;
}

static int
luxio_readdir_tostring(lua_State *L)
{
	luxio_readdir_state *s = luaL_checkudata(L, 1, LUXIO_READDIR_METATABLE);
	char buf[sizeof("dirent: 0xffffffffffffffff")];

	/* we can't use lua_pushfstring here, because our pointer might
	 * be 64 bits.
	 */
	snprintf(buf, sizeof(buf), "dirent: %p", s);
	lua_pushstring(L, buf);

	return 1;
}

static void
luxio__bless_readdir(lua_State *L)
{
	int create = luaL_newmetatable(L, LUXIO_READDIR_METATABLE);

	if (create != 0) {
		lua_pushcfunction(L, luxio_readdir_gc);
		lua_setfield(L, -2, "__gc");
		lua_pushcfunction(L, luxio_readdir_tostring);
		lua_setfield(L, -2, "__tostring");
	}

	lua_setmetatable(L, -2);
}
/**% opendir
 * handle = opendir(path);
 * nil, errno = opendir(path)
 * handle, errno = opendir(path)
 */
static int
luxio_opendir(lua_State *L) /* 5.1.2 */
{
	const char *path = luaL_checkstring(L, 1);
	DIR *d = opendir(path);
	size_t bufz;
	luxio_readdir_state *s;

	if (d == NULL) {
		lua_pushnil(L);
		lua_pushinteger(L, errno);
		return 2;
	}

	s = lua_newuserdata(L, sizeof(*s));

	s->dirp = d;
	/* + 256 because it'd always be +1 if it weren't for the horrors
	 * of Solaris.  If we were using autoconf, we could use Ben
	 * Hutchings' function mentioned in his article "readdir_r considered
	 * harmful".
	 */
	bufz = sizeof(struct dirent) + pathconf(path, _PC_NAME_MAX) + 256;
	s->buf = malloc(bufz);

	luxio__bless_readdir(L);

	return 1;
}

/**% fdopendir
 * retval = fdopendir(fd);
 * nil, errno = fdopendir(fd)
 * handle, errno = fdopendir(fd)
 */
static int
luxio_fdopendir(lua_State *L) /* POSIX.1-2008 */
{
	int fd = luaL_checkinteger(L, 1);
	DIR *d = fdopendir(fd);
	size_t bufz;
	luxio_readdir_state *s;

	if (d == NULL) {
		lua_pushnil(L);
		lua_pushinteger(L, errno);
		return 2;
	}

	s = lua_newuserdata(L, sizeof(*s));

	s->dirp = d;
	/* + 256 because it'd always be +1 if it weren't for the horrors
	 * of Solaris.  If we were using autoconf, we could use Ben
	 * Hutchings' function mentioned in his article "readdir_r considered
	 * harmful".
	 */
	bufz = sizeof(struct dirent) + fpathconf(fd, _PC_NAME_MAX) + 256;
	s->buf = malloc(bufz);

	luxio__bless_readdir(L);

	return 1;
}

/**% closedir
 * closedir(handle);
 * closedir(handle)
 */
static int
luxio_closedir(lua_State *L) /* 5.1.2 */
{
	luxio_readdir_state *s = luaL_checkudata(L, 1, LUXIO_READDIR_METATABLE);

	if (s->dirp != NULL) {
		closedir(s->dirp);
		s->dirp = NULL;
	}

	free(s->buf);
	s->buf = NULL;

	return 0;
}

/**% readdir
 * retval = readdir_r(handle, buf, ent);
 * nil, errno = readdir(handle) -- error _or_ end of directory
 * dirent, errno = readdir(handle) -- table with d_ino, d_name, and d_type
 */
static int
luxio_readdir(lua_State *L) /* 5.1.2 */
{
	luxio_readdir_state *s = luaL_checkudata(L, 1, LUXIO_READDIR_METATABLE);
	int err;

	err = readdir_r(s->dirp, s->buf, &s->ent);

	if (err == 0 && s->ent != NULL) {
		lua_pushinteger(L, 0);
		lua_createtable(L, 0, 3);
		lua_pushinteger(L, s->ent->d_ino);
		lua_setfield(L, -2, "d_ino");
		lua_pushstring(L, s->ent->d_name);
		lua_setfield(L, -2, "d_name");
#ifdef HAVE_D_TYPE
		lua_pushinteger(L, s->ent->d_type);
		lua_setfield(L, -2, "d_type");
#endif
		return 2;
	}

	if (s->ent == NULL) {
		/* end of directory */
		lua_pushnil(L);

		return 1;
	}

	lua_pushinteger(L, err);

	return 1;
}

/**% rewinddir
 * rewinddir(handle);
 * rewinddir(handle)
 */
static int
luxio_rewinddir(lua_State *L) /* 5.1.2 */
{
	luxio_readdir_state *s = luaL_checkudata(L, 1, LUXIO_READDIR_METATABLE);

	rewinddir(s->dirp);

	return 0;
}

/**# Working directory *******************************************************/

/**% chdir
 * retval = chdir(path);
 * retval, errno = chdir(path)
 */
static int
luxio_chdir(lua_State *L) /* 5.2.1 */
{
	const char *path = luaL_checkstring(L, 1);

	lua_pushinteger(L, chdir(path));
	lua_pushinteger(L, errno);

	return 2;
}

/**% getcwd
 * retval = getcwd(buf, buflen);
 * path, errno = getcwd()
 * nil, errno = getcwd()
 */
static int
luxio_getcwd(lua_State *L) /* 5.2.2 */
{
	char buf[PATH_MAX];

	if (getcwd(buf, PATH_MAX) == NULL) {
		lua_pushnil(L);
	} else {
		lua_pushstring(L, buf);
	}

	lua_pushinteger(L, errno);

	return 2;
}

/**# General file creation ***************************************************/

/**% open
 * fd = open(path, flags[, mode]);
 * fd, errno = open(path, flags[, mode])
 */
static int
luxio_open(lua_State *L) /* 5.3.1 */
{
	const char *pathname = luaL_checkstring(L, 1);
	int flags = luaL_checkint(L, 2);
	mode_t mode = luaL_optinteger(L, 3, INVALID_MODE);
	int result;

	if ((flags & O_CREAT) && mode == INVALID_MODE) {
		lua_pushstring(L, "open with O_CREAT called with no mode");
		lua_error(L);
	}

	if (mode == INVALID_MODE)
		result = open(pathname, flags);
	else
		result = open(pathname, flags, mode);

	lua_pushinteger(L, result);
	lua_pushinteger(L, errno);

	return 2;
}

/* TODO: creat() 5.3.2 */

/**% umask
 * retval = umask(mask);
 * retval = umask(mask)
 */
static int
luxio_umask(lua_State *L) /* 5.3.3 */
{
	mode_t mask = luaL_checkinteger(L, 1);

	lua_pushinteger(L, umask(mask));

	return 1;
}

/**% link
 * retval = link(existing, new);
 * retval, errno = link(existing, new)
 */
static int
luxio_link(lua_State *L) /* 5.3.4 */
{
	const char *existing = luaL_checkstring(L, 1);
	const char *new = luaL_checkstring(L, 2);

	lua_pushinteger(L, link(existing, new));
	lua_pushinteger(L, errno);

	return 2;
}

/**% symlink
 * retval = symlink(oldpath, newpath);
 * retval, errno = symlink(oldpath, newpath)
 */
static int
luxio_symlink(lua_State *L) /* POSIX.1-2001, Unknown location */
{
	const char *oldpath = luaL_checkstring(L, 1);
	const char *newpath = luaL_checkstring(L, 2);

	lua_pushinteger(L, symlink(oldpath, newpath));
	lua_pushinteger(L, errno);
	
	return 2;
}

/**% readlink
 * retval, target = readlink(path)
 * retval, errno = readlink(path)
 */
static int
luxio_readlink(lua_State *L) /* POSIX.1-2001, Unknown location */
{
	char buffer[PATH_MAX];
	ssize_t ret;
	const char *path = luaL_checkstring(L, 1);

	lua_pushinteger(L, (ret = readlink(path, buffer, PATH_MAX)));
	if (ret > 0) {
		lua_pushinteger(L, errno);
	} else {
		lua_pushstring(L, buffer);
	}

	return 2;
}

/**# Special file creation ***************************************************/

/**% mkdir
 * retval = mkdir(pathname, mode);
 * retval, errno = mkdir(pathname, mode)
 */
static int
luxio_mkdir(lua_State *L) /* 5.4.1 */
{
	const char *pathname = luaL_checkstring(L, 1);
	mode_t mode = luaL_checkinteger(L, 2);

	lua_pushinteger(L, mkdir(pathname, mode));
	lua_pushinteger(L, errno);

	return 2;
}

/**% mkfifo
 * retval = mkfifo(pathname, mode);
 * retval, errno = mkfifo(pathname, mode)
 */
static int
luxio_mkfifo(lua_State *L) /* 5.4.2 */
{
	const char *pathname = luaL_checkstring(L, 1);
	mode_t mode = luaL_checkinteger(L, 2);

	lua_pushinteger(L, mkfifo(pathname, mode));
	lua_pushinteger(L, errno);

	return 2;
}

/**# File removal ************************************************************/

/**% unlink
 * retval = unlink(pathname);
 * retval, errno = unlink(pathname)
 */
static int
luxio_unlink(lua_State *L) /* 5.5.1 */
{
	const char *s = luaL_checkstring(L, 1);

	lua_pushinteger(L, unlink(s));
	lua_pushinteger(L, errno);

	return 2;
}

/**% rmdir
 * retval = rmdir(pathname);
 * retval, errno = rmdir(pathname)
 */
static int
luxio_rmdir(lua_State *L) /* 5.5.2 */
{
	const char *pathname = luaL_checkstring(L, 1);

	lua_pushinteger(L, rmdir(pathname));
	lua_pushinteger(L, errno);

	return 2;
}

/**% rename
 * retval = rename(old, new);
 * retval, errno = rename(old, new)
 */
static int
luxio_rename(lua_State *L) /* 5.5.3 */
{
	const char *old = luaL_checkstring(L, 1);
	const char *new = luaL_checkstring(L, 2);

	lua_pushinteger(L, rename(old, new));
	lua_pushinteger(L, errno);

	return 2;
}

/**# File characteristics ****************************************************/

static void
luxio_push_stat_table(lua_State *L, struct stat *s)
{
	lua_createtable(L, 0, 13);

#define PUSH_ENTRY(e) do { lua_pushstring(L, #e);	\
	lua_pushinteger(L, s->st_##e);			\
	lua_settable(L, 3); } while (0)

	PUSH_ENTRY(dev);
	PUSH_ENTRY(ino);
	PUSH_ENTRY(mode);
	PUSH_ENTRY(nlink);
	PUSH_ENTRY(uid);
	PUSH_ENTRY(gid);
	PUSH_ENTRY(rdev);
	PUSH_ENTRY(size);
	PUSH_ENTRY(blksize);
	PUSH_ENTRY(blocks);
	PUSH_ENTRY(atime);
	PUSH_ENTRY(mtime);
	PUSH_ENTRY(ctime);

#undef PUSH_ENTRY
}

/**% stat
 * retval = stat(pathname, statbuf);
 * retval, errno = stat(pathname) -- 'errno' can be  table containing dev, ino, uid, gid, etc
 */
static int
luxio_stat(lua_State *L) /* 5.6.2 */
{
	const char *pathname = luaL_checkstring(L, 1);
	struct stat s;
	int r = stat(pathname, &s);

	lua_pushinteger(L, r);

	if (r < 0) {
		lua_pushinteger(L, errno);
	} else {
		luxio_push_stat_table(L, &s);
	}

	return 2;
}

/**% fstat
 * retval = fstat(fd, statbuf);
 * retval, errno = fstat(fd) -- 'errno' can be a table containing dev, ino, uid, gid, etc
 */
static int
luxio_fstat(lua_State *L) /* 5.6.2 */
{
	int fd = luaL_checkinteger(L, 1);
	struct stat s;
	int r = fstat(fd, &s);

	lua_pushinteger(L, r);

	if (r < 0) {
		lua_pushinteger(L, errno);
	} else {
		luxio_push_stat_table(L, &s);
	}

	return 2;
}

/**% lstat
 * retval = lstat(pathname, statbuf);
 * retval, errno = lstat(pathname) -- 'errno' can be a table containing dev, ino, uid, gid, etc
 */
static int
luxio_lstat(lua_State *L) /* POSIX.1-2001 */
{
	const char *pathname = luaL_checkstring(L, 1);
	struct stat s;
	int r = lstat(pathname, &s);

	lua_pushinteger(L, r);

	if (r < 0) {
		lua_pushinteger(L, errno);
	} else {
		luxio_push_stat_table(L, &s);
	}

	return 2;
}

#define STAT_IS(x) static int luxio_S_IS##x(lua_State *L) { \
	int mode = luaL_checkinteger(L, 1);               \
	lua_pushinteger(L, S_IS##x(mode));                 \
	return 1;                                         \
	}

/**% S_ISREG
 * retval = S_ISREG(m);
 * retval = S_ISREG(m)
 */
STAT_IS(REG)

/**% S_ISDIR
 * retval = S_ISDIR(m);
 * retval = S_ISDIR(m)
 */
STAT_IS(DIR)

/**% S_ISCHR
 * retval = S_ISCHR(m);
 * retval = S_ISCHR(m)
 */
STAT_IS(CHR)

/**% S_ISBLK
 * retval = S_ISBLK(m);
 * retval = S_ISBLK(m)
 */
STAT_IS(BLK)

/**% S_ISFIFO
 * retval = S_ISFIFO(m);
 * retval = S_ISFIFO(m)
 */
STAT_IS(FIFO)

#ifdef S_ISLNK
/**% S_ISLNK
 * retval = S_ISLNK(m);
 * retval = S_ISLNK(m)
 */
STAT_IS(LNK)
#endif

#ifdef S_ISSOCK
/**% S_ISSOCK
 * retval = S_ISSOCK(m);
 * retval = S_ISSOCK(m)
 */
STAT_IS(SOCK)
#endif

#undef STAT_IS

/* TODO: access() 5.6.3 */

/**% chmod
 * retval = chmod(path, mode);
 * retval, errno = chmod(path, mode)
 */
static int
luxio_chmod(lua_State *L) /* 5.6.4 */
{
	const char *path = luaL_checkstring(L, 1);
	mode_t mode = luaL_checkinteger(L, 2);

	lua_pushinteger(L, chmod(path, mode));
	lua_pushinteger(L, errno);

	return 2;
}

/**% fchmod
 * retval = fchmod(fd, mode);
 * retval, errno = fchmod(fd, mode)
 */
static int
luxio_fchmod(lua_State *L) /* 5.6.4 */
{
	int fd = luaL_checkinteger(L, 1);
	mode_t mode = luaL_checkinteger(L, 2);

	lua_pushinteger(L, fchmod(fd, mode));
	lua_pushinteger(L, errno);

	return 2;
}

/**% chown
 * retval = chown(path, owner, group);
 * retval, errno = chown(path, owner, group)
 */
static int
luxio_chown(lua_State *L) /* 5.6.5 */
{
	const char *path = luaL_checkstring(L, 1);
	uid_t owner = luaL_checkinteger(L, 2);
	gid_t group = luaL_checkinteger(L, 3);

	lua_pushinteger(L, chown(path, owner, group));
	lua_pushinteger(L, errno);

	return 2;
}

/* TODO: utime() 5.6.6 */

/**% ftruncate
 * retval = ftruncate(fd, length);
 * retval, errno = ftruncate(fd, length)
 */
static int
luxio_ftruncate(lua_State *L) /* 5.6.7 */
{
	int fildes = luaL_checkinteger(L, 1);
        off_t length = luaL_checkinteger(L, 2);

	lua_pushinteger(L, ftruncate(fildes, length));
	lua_pushinteger(L, errno);

	return 2;
}

/* 5.7 Configurable pathname variables ***************************************/

/* TODO: pathconf(), fpathconf() 5.7.1 */

/**# Pipes *******************************************************************/

/**% pipe
 * retval = pipe(pipearray);
 * retval, errno = pipe(pipetable) -- fills in [1] and [2] in pipetable
 */
static int
luxio_pipe(lua_State *L) /* 6.1.1 */
{
	int res, pipefd[2];
	luaL_checktype(L, 1, LUA_TTABLE);

	res = pipe(pipefd);
	if (res == 0) {
		lua_pushinteger(L, pipefd[0]);
		lua_rawseti(L, 1, 1);
		lua_pushinteger(L, pipefd[1]);
		lua_rawseti(L, 1, 2);
	}

	lua_pushinteger(L, res);
	lua_pushinteger(L, errno);

	return 2;
}

#ifdef _GNU_SOURCE

/**% pipe2
 * retval = pipe2(pipearray, flags);
 * retval, errno = pipe2(pipetable, flags) -- fills in [1] and [2] in pipetable
 */
static int
luxio_pipe2(lua_State *L) /* GNU extension */
{
	int res, pipefd[2];
	int flags;

	luaL_checktype(L, 1, LUA_TTABLE);
	flags = luaL_checkinteger(L, 2);

	res = pipe2(pipefd, flags);
	if (res == 0) {
		lua_pushinteger(L, pipefd[0]);
		lua_rawseti(L, 1, 1);
		lua_pushinteger(L, pipefd[1]);
		lua_rawseti(L, 1, 2);
	}

	lua_pushinteger(L, res);
	lua_pushinteger(L, errno);

	return 2;
}
#endif

/**% socketpair
 * retval = socketpair(domain, type, protocol, fdarray)
 * retval, errno = socketpair(domain, type, protocol, fdtable)
 */
static int
luxio_socketpair(lua_State *L) /* POSIX.1-2001 */
{
	int domain = luaL_checkinteger(L, 1);
	int type = luaL_checkinteger(L, 2);
	int protocol = luaL_checkinteger(L, 3);
	int sv[2];
	int res;
	luaL_checktype(L, 4, LUA_TTABLE);

	res = socketpair(domain, type, protocol, sv);
	if (res == 0) {
		lua_pushinteger(L, sv[0]);
		lua_rawseti(L, 4, 1);
		lua_pushinteger(L, sv[1]);
		lua_rawseti(L, 4, 2);
	}

	lua_pushinteger(L, res);
	lua_pushinteger(L, errno);

	return 2;
}

/**# File descriptor manipulation ********************************************/

/**% dup
 * retval = dup(oldfd);
 * retval, errno = dup(oldfd)
 */
static int
luxio_dup(lua_State *L) /* 6.2.1 */
{
	int oldfd = luaL_checkint(L, 1);

	lua_pushinteger(L, dup(oldfd));
	lua_pushinteger(L, errno);

	return 2;
}

/**% dup2
 * retval = dup2(oldfd, newfd);
 * retval, errno = dup2(oldfd, newfd)
 */
static int
luxio_dup2(lua_State *L) /* 6.2.1 */
{
	int oldfd = luaL_checkint(L, 1);
	int newfd = luaL_checkint(L, 2);

	lua_pushinteger(L, dup2(oldfd, newfd));
	lua_pushinteger(L, errno);

	return 2;
}

#ifdef _GNU_SOURCE
/**% dup3
 * retval = dup3(oldfd, newfd, flags);
 * retval, errno = dup2(oldfd, newfd, flags)
 */
static int
luxio_dup3(lua_State *L) /* GNU extension */
{
	int oldfd = luaL_checkint(L, 1);
	int newfd = luaL_checkint(L, 2);
	int flags = luaL_checkint(L, 3);

	lua_pushinteger(L, dup3(oldfd, newfd, flags));
	lua_pushinteger(L, errno);

	return 2;
}
#endif

/**# File descriptor deassignment ********************************************/

/**% close
 * retval = close(fd);
 * retval, errno = close(fd)
 */
static int
luxio_close(lua_State *L) /* 6.3.1 */
{
	lua_pushinteger(L, close(luaL_checkint(L, 1)));
	lua_pushinteger(L, errno);

	return 2;
}

/**# Input and output ********************************************************/

/**% read
 * retval = read(fd, buf, count);
 * retval, errno = read(fd, count)
 * string, errno = read(fd, count)
 */
static int
luxio_read(lua_State *L) /* 6.4.1 */
{
    int fd = luaL_checkint(L, 1);
    int count = luaL_checkint(L, 2);
    ssize_t result;
    char *buf = malloc(count);

    if (buf == NULL) {
        lua_pushstring(L, "unable to allocate read buffer: memory exhausted");
        lua_error(L);
    }

    result = read(fd, buf, count);

    if (result == -1) {
        lua_pushinteger(L, result);
        lua_pushinteger(L, errno);
    } else {
        /* sadly there appears to be no way to avoid this copy.
         * luaL_Buffer actually builds things on the C stack bytes at a time,
         * and there is no way to pre-allocate a Lua type other than a
         * userdatum.  Additionally, should Lua call its panic function because
         * it can't allocate memory to copy this into, our buf will be leaked.
         * We could perhaps fix this with a lot of faff, involving creating
         * a userdatum for our buffer, and setting a __gc metamethod.
         */
        lua_pushlstring(L, buf, result);
        lua_pushinteger(L, errno);
    }

    free(buf);

    return 2;
}

/**% write
 * retval = write(fd, buf, count);
 * retval, errno = write(fd, string[, start_offset/0])
 */
static int
luxio_write(lua_State *L) /* 6.4.2 */
{
	int fd = luaL_checkint(L, 1);
	size_t count;
	const char *buf = luaL_checklstring(L, 2, &count);
	size_t offset = luaL_optinteger(L, 3, 0);

	if (offset > count) offset = count;

	lua_pushinteger(L, write(fd, buf + offset, count - offset));
	lua_pushinteger(L, errno);

	return 2;
}

/**% writev
 * retval = writev(fd, iov, blks);
 * retval, errno = writev(fd, string[, ...])
 */
static int
luxio_writev(lua_State *L) /* POSIX.1-2001 */
{
	int fd = luaL_checkint(L, 1);
	int blks = lua_gettop(L) - 1;
	int c;
	struct iovec *iov;

	/* check there is at least one string to write */
	luaL_checkstring(L, 2);

	iov = malloc(blks * sizeof(*iov));

	for (c = 0; c < blks; c++) {
		iov[c].iov_base = (void *)luaL_checkstring(L, c + 2);
		iov[c].iov_len = lua_rawlen(L, c + 2);
	}

	lua_pushinteger(L, writev(fd, iov, blks));
	lua_pushinteger(L, errno);

	free(iov);

	return 2;
}

#ifdef HAVE_SENDFILE
/**% sendfile
 * retval = sendfile(out_fd, in_fd, offset/NULL, count);
 * retval, errno = sendfile(out_fd, in_fd, offset/nil, count)
 */
static int
luxio_sendfile(lua_State *L) /* Linux-specific */
{
	int out_fd = luaL_checkint(L, 1);
	int in_fd = luaL_checkint(L, 2);
	off_t offset;
	size_t count = luaL_checkint(L, 4);
	ssize_t r;

	if (lua_isnil(L, 3)) {
		r = sendfile(out_fd, in_fd, NULL, count);
		lua_pushinteger(L, r);
		lua_pushinteger(L, errno);
		return 2;
	}

	offset = luaL_checkint(L, 3);
	r = sendfile(out_fd, in_fd, &offset, count);
	lua_pushinteger(L, r);
	lua_pushinteger(L, errno);
	lua_pushinteger(L, offset);

	return 3;
}
#endif /* HAVE_SENDFILE */

#ifdef HAVE_SPLICE
/**% splice
 * retval = splice(fd_in, off_in, fd_out, off_out, len, flags);
 * retval, errno = splice(fd_in, off_in, fd_out, off_out, len, flags)
 */
static int
luxio_splice(lua_State *L) /* Linux-specific */
{
	int fd_in = luaL_checkinteger(L, 1);
	loff_t off_in = luaL_optlong(L, 2, -1);
	int fd_out = luaL_checkinteger(L, 3);
	loff_t off_out = luaL_optlong(L, 4, -1);
	size_t len = luaL_checkinteger(L, 5);
	unsigned int flags = luaL_checkinteger(L, 6);

	loff_t *poff_in = &off_in;
	loff_t *poff_out = &off_out;

	if (off_in == -1) poff_in = NULL;
	if (off_out == -1) poff_out = NULL;

	lua_pushinteger(L, splice(fd_in, poff_in, fd_out, poff_out,
				  len, flags));
	lua_pushinteger(L, errno);

	return 2;
}
#endif

/**# Control operations on files *********************************************/

/**> fcntl
 * Supported commands:
 *  F_GETFD/F_SETFD, F_GETFL/F_SETFL, F_GETPIPE_SZ/F_SETPIPE_SZ, F_DUPFD,
 *  F_DUPFD_CLOEXEC, F_SETLK, F_SETLKW, F_GETLK
 *
 * C-Style: retval = fcntl(fd, cmd[, argument]);
 *
 * Lua-Style: retval, errno = fcntl(fd, cmd[, argument])
 */
static int
luxio_fcntl(lua_State *L) /* 6.5.2 */
{
	int fd = luaL_checkint(L, 1);
	int cmd = luaL_checkint(L, 2);
	long arg_long;
	struct flock flock;

	switch (cmd) {
	/* commands that take no argument */
	case F_GETFD:
	case F_GETFL:
#ifdef F_GETPIPE_SZ
	case F_GETPIPE_SZ:
#endif
		lua_pushinteger(L, fcntl(fd, cmd));
		lua_pushinteger(L, errno);
		return 2;

	/* commands that take a long */
	case F_DUPFD:
#ifdef F_DUPFD_CLOEXEC
	case F_DUPFD_CLOEXEC:
#endif
	case F_SETFD:
	case F_SETFL:
#ifdef F_SETPIPE_SZ
	case F_SETPIPE_SZ:
#endif
		arg_long = luaL_checkinteger(L, 3);
		lua_pushinteger(L, fcntl(fd, cmd, arg_long));
		lua_pushinteger(L, errno);

		return 2;

	/* commands that take exciting things */
	case F_SETLK:
	case F_SETLKW:
	case F_GETLK:
		luaL_checktype(L, 3, LUA_TTABLE);

		lua_getfield(L, 3, "l_type");
		lua_getfield(L, 3, "l_whence");
		lua_getfield(L, 3, "l_start");
		lua_getfield(L, 3, "l_len");
		flock.l_type = lua_tonumber(L, -4);
		flock.l_whence = lua_tonumber(L, -3);
		flock.l_start = lua_tonumber(L, -2);
		flock.l_len = lua_tonumber(L, -1);
		flock.l_pid = 0;

		lua_pushinteger(L, fcntl(fd, cmd, &flock));
		lua_pushinteger(L, errno);

		if (cmd == F_GETLK) {
			lua_pushnumber(L, flock.l_type);
			lua_pushnumber(L, flock.l_whence);
			lua_pushnumber(L, flock.l_start);
			lua_pushnumber(L, flock.l_len);
			lua_pushnumber(L, flock.l_pid);
			lua_setfield(L, 3, "l_pid");
			lua_setfield(L, 3, "l_len");
			lua_setfield(L, 3, "l_start");
			lua_setfield(L, 3, "l_whence");
			lua_setfield(L, 3, "l_type");
		}

		return 2;

	default:
		lua_pushstring(L, "unhandled fcntl() command");
		lua_error(L);
	}

	return 0; /* never get here, but keep compiler happy */
}

#ifdef _LARGEFILE64_SOURCE
static int
luxio_lseek(lua_State *L) /* 6.5.3 */
{
	int fd = luaL_checkint(L, 1);
	off64_t offset = (off64_t)luaL_checknumber(L, 2); /* 56b is enough! */
	int whence = luaL_checkint(L, 3);

	lua_pushinteger(L, (lua_Number)lseek64(fd, offset, whence));
	lua_pushinteger(L, errno);

	return 2;
}
#else
/**% lseek
 * retval = lseek(fd, offset, whence);
 * retval, errno = lseek(fd, offset, whence)
 */
static int
luxio_lseek(lua_State *L) /* 6.5.3 */
{
	int fd = luaL_checkint(L, 1);
	off_t offset = luaL_checkinteger(L, 2);
	int whence = luaL_checkint(L, 3);

	lua_pushinteger(L, lseek(fd, offset, whence));
	lua_pushinteger(L, errno);

	return 2;
}
#endif

/**# File synchronisation ****************************************************/

/**% fsync
 * retval = fsync(filedes);
 * retval, errno = fsync(filedes)
 */
static int
luxio_fsync(lua_State *L) /* 6.6.1 */
{
	int fildes = luaL_checkinteger(L, 1);

	lua_pushinteger(L, fsync(fildes));
	lua_pushinteger(L, errno);

	return 2;
}

#ifdef HAVE_FDATASYNC
/**% fdatasync
 * retval = fdatasync(filedes);
 * retval = fdatasync(filedes)
 */
static int
luxio_fdatasync(lua_State *L) /* 6.6.2 */
{
	int fildes = luaL_checkinteger(L, 1);

	lua_pushinteger(L, fdatasync(fildes));
	lua_pushinteger(L, errno);

	return 2;
}
#endif

/* 6.7 Asynchronous input and output */

/* TODO: aio_read() 6.7.2 */
/* TODO: aio_write() 6.7.3 */
/* TODO: lio_listio() 6.7.4 */
/* TODO: aio_error() 6.7.5 */
/* TODO: aio_return() 6.7.6 */
/* TODO: aio_cancel() 6.7.7 */
/* TODO: aio_suspend() 6.7.8 */
/* TODO: aio_fsync() 6.7.9 */

/**# General Terminal Interface **********************************************/

/* TODO: cfgetispeed(), cfgetospeed(), cfsetispeed(), cfsetospeed() 7.1.3 */
/* TODO: tcgetattr(), tcsetattr() 7.2.1 */
/* TODO: tcsendbreak(), tcdrain(), tcflush(), tcflow() 7.2.2 */

/**% tcgetpgrp
 * retval = tcgetpgrp(filedes);
 * retval, errno = tcgetpgrp(filedes)
 */
static int
luxio_tcgetpgrp(lua_State *L) /* 7.2.3 */
{
	lua_pushinteger(L, tcgetpgrp(luaL_checkinteger(L, 1)));
	lua_pushinteger(L, errno);

	return 2;
}

/**% tcsetpgrp
 * retval = tcsetpgrp(filedes, pgrp_id);
 * retval = tcsetpgrp(filedes, pgrp_id)
 */
static int
luxio_tcsetpgrp(lua_State *L) /* 7.2.4 */
{
	int fildes = luaL_checkinteger(L, 1);
	pid_t pgrp_id = luaL_checkinteger(L, 2);

	lua_pushinteger(L, tcsetpgrp(fildes, pgrp_id));
	lua_pushinteger(L, errno);

	return 2;
}

/* 8.1 Referenced C Language Routines ****************************************/

/* These are ANSI C functions POSIX imports.  TODO any Lua doesn't already */

/* 9.1 Database access *******************************************************/

/* TODO: getgrgid(), getgrgid_r(), getgrnam(), getgrnam_r() 9.2.1 */
/* TODO: getpwuid(), getpwuid_r(), getpwnam(), getpwnam_r() 9.2.2 */

/* 10 Data interchange format ************************************************/

/* This is just related to data structures and file formats, not functions */

/* 11 Synchronisation ********************************************************/

/* Semaphores, mutexes, etc should be handled by a dedicated threading lib */

/* 12 Memory management ******************************************************/

/* While it might be interesting to bind mmap and mlock etc, it's difficult to
 * see how this might work in Lua.  Perhaps emulate a string?
 */

/* 13 Execution scheduling ***************************************************/

/* TODO: all of this. */

/**# Clock and timer functions ***********************************************/

/* TODO: clock_settime(), clock_gettime(), clock_getres() 14.2.1 */
/* Timer functions excluded, based on signals */

/**% nanosleep
 * retval = nanosleep(req_timespec, rem_timespec);
 * retval, errno, rem_sec, rem_nsec = nanosleep(req_sec, req_nsec)
 */
static int
luxio_nanosleep(lua_State *L) /* 14.2.5 */
{
	struct timespec req, rem = { 0, 0 };

	req.tv_sec = luaL_checkinteger(L, 1);
	req.tv_nsec = luaL_checkinteger(L, 2);

	lua_pushinteger(L, nanosleep(&req, &rem));
	lua_pushinteger(L, errno);
	lua_pushinteger(L, rem.tv_sec);
	lua_pushinteger(L, rem.tv_nsec);

	return 4;
}

/**# Message passing *********************************************************/

#if defined(_POSIX_MESSAGE_PASSING) && defined(__linux__)

/* TODO: This code assumes mqd_t is an integer.  On BSD it is not, so
 * this needs to be rewritten using userdata.
 */

#include <mqueue.h>

/**% mq_open
 * retval = mq_open(name, oflag[, mode, attr]);
 * retval, errno = mq_open(name, oflag[, mode])
 */
static int
luxio_mq_open(lua_State *L) /* 15.2.1 */
{
	const char *name = luaL_checkstring(L, 1);
	int oflag = luaL_checkinteger(L, 2);
	mode_t mode = luaL_optinteger(L, 3, INVALID_MODE);
	mqd_t mq;

	if ((oflag & O_CREAT) && mode == INVALID_MODE) {
		lua_pushstring(L, "mq_open with O_CREATE called with no mode");
		lua_error(L);
	}

	if (oflag & O_CREAT) {
		mq = mq_open(name, oflag, mode, NULL);
	} else {
		mq = mq_open(name, oflag);
	}

	lua_pushinteger(L, mq);
	lua_pushinteger(L, errno);

	return 2;
}

/**% mq_close
 * retval = mq_close(mq);
 * retval, errno = mq_close(mq)
 */
static int
luxio_mq_close(lua_State *L) /* 15.2.2 */
{
	mqd_t mq = luaL_checkinteger(L, 1);

	lua_pushinteger(L, mq_close(mq));
	lua_pushinteger(L, errno);
	return 2;
}

/**% mq_unlink
 * retval = mq_unlink(name);
 * retval, errno = mq_unlink(name)
 */
static int
luxio_mq_unlink(lua_State *L) /* 15.2.3 */
{
	const char *name = luaL_checkstring(L, 1);

	lua_pushinteger(L, mq_unlink(name));
	lua_pushinteger(L, errno);
	return 2;
}

/**% mq_send
 * retval = mq_send(mq, msg_ptr, msg_len, msg_prio);
 * retval, errno = mq_send(mq, string, prio)
 */
static int
luxio_mq_send(lua_State *L) /* 15.2.4 */
{
	mqd_t mq = luaL_checkinteger(L, 1);
	size_t msg_len;
	const char *msg_ptr = luaL_checklstring(L, 2, &msg_len);
	unsigned int msg_prio = luaL_checkinteger(L, 3);

	lua_pushinteger(L, mq_send(mq, msg_ptr, msg_len, msg_prio));
	lua_pushinteger(L, errno);
	return 2;
}

/**% mq_receive
 * retval = mq_receive(mq, msg_ptr, msg_ptr_sz, msg_prio);
 * retval, errno, msg, prio = mq_receive(mq)
 */
static int
luxio_mq_receive(lua_State *L) /* 15.2.5 */
{
	mqd_t mq = luaL_checkinteger(L, 1);
	unsigned int msg_prio;
	struct mq_attr attr;

	/* Find out the maximum size of a message */
	if (mq_getattr(mq, &attr) == -1) {
		lua_pushinteger(L, -1);
		lua_pushinteger(L, errno);
		return 2;
	} else {
		char msg_ptr[attr.mq_msgsize];
		int r = mq_receive(mq, msg_ptr, sizeof(msg_ptr), &msg_prio);
		lua_pushinteger(L, r);
		lua_pushinteger(L, errno);
		if (r == -1) {
			return 2;
		}
		lua_pushlstring(L, msg_ptr, r);
		lua_pushinteger(L, msg_prio);
		return 4;
	}
}

/* TODO: mq_notify() 15.2.6 */

static int
luxio_make_attr_table(lua_State *L, struct mq_attr *attr)
{
	int top = lua_gettop(L) + 1;

	lua_createtable(L, 0, 4);

#define PUSH_ENTRY(e) do { lua_pushstring(L, "mq_" #e); \
	lua_pushinteger(L, attr->mq_##e); \
	lua_settable(L, top); } while (0)

	PUSH_ENTRY(flags);
	PUSH_ENTRY(maxmsg);
	PUSH_ENTRY(msgsize);
	PUSH_ENTRY(curmsgs);

#undef PUSH_ENTRY

	return 1;
}

/**% mq_setattr
 * retval = mq_setattr(mq, new, old);
 * retval, errno, old = mq_setattr(mq, flags)
 */
static int
luxio_mq_setattr(lua_State *L) /* 15.2.7 */
{
	mqd_t mq = luaL_checkinteger(L, 1);
	struct mq_attr mqstat = { luaL_checkinteger(L, 2), 0, 0, 0 };
	struct mq_attr omqstat = { 0, 0, 0, 0 };

	lua_pushinteger(L, mq_setattr(mq, &mqstat, &omqstat));
	lua_pushinteger(L, errno);
	luxio_make_attr_table(L, &omqstat);

	return 3;
}

/**% mq_getattr
 * retval = mq_getattr(mq, old);
 * retval, errno, old = mq_getattr(mq)
 */
static int
luxio_mq_getattr(lua_State *L) /* 15.2.8 */
{
	mqd_t mq = luaL_checkinteger(L, 1);
	struct mq_attr mqstat;

	lua_pushinteger(L, mq_getattr(mq, &mqstat));
	lua_pushinteger(L, errno);
	luxio_make_attr_table(L, &mqstat);

	return 3;
}

#if _XOPEN_SOURCE >= 600 || _POSIX_C_SOURCE >= 200112L

/**% mq_timedsend
 * retval = mq_timedsend(mq, msg, len, prio, timeout)
 * retval, errno = mq_timedsend(mq, msg, prio, tv_secs, tv_nsec)
 */
static int
luxio_mq_timedsend(lua_State *L) /* POSIX.1-2001 */
{
	mqd_t mq = luaL_checkinteger(L, 1);
	size_t msg_len;
	const char *msg_ptr = luaL_checklstring(L, 2, &msg_len);
	unsigned int msg_prio = luaL_checkinteger(L, 3);
	time_t tv_secs = luaL_checkinteger(L, 4);
	long tv_nsec = luaL_checkinteger(L, 5);
	struct timespec abs_timeout = { tv_secs, tv_nsec };

	lua_pushinteger(L, mq_timedsend(mq, msg_ptr, msg_len, msg_prio,
				&abs_timeout));
	lua_pushinteger(L, errno);
	return 2;
}

/**% mq_timedreceive
 * retval = mq_timedsend(mq, bug, len, prio, timeout)
 * retval, errno, msg, prio = mq_timedreceive(mq, tv_secs, tv_nsec)
 */
static int
luxio_mq_timedreceive(lua_State *L) /* POSIX.1-2001 */
{
	mqd_t mq = luaL_checkinteger(L, 1);
	unsigned int msg_prio;
	struct mq_attr attr;
	time_t tv_secs = luaL_checkinteger(L, 2);
	long tv_nsec = luaL_checkinteger(L, 3);
	struct timespec abs_timeout = { tv_secs, tv_nsec };

	/* Find out the maximum size of a message */
	if (mq_getattr(mq, &attr) == -1) {
		lua_pushinteger(L, -1);
		lua_pushinteger(L, errno);
		return 2;
	} else {
		char msg_ptr[attr.mq_msgsize];
		int r = mq_timedreceive(mq, msg_ptr, sizeof(msg_ptr), &msg_prio,
			     &abs_timeout);
		lua_pushinteger(L, r);
		lua_pushinteger(L, errno);
		if (r == -1) {
			return 2;
		}
		lua_pushlstring(L, msg_ptr, r);
		lua_pushinteger(L, msg_prio);
		return 4;
	}
}
#endif /* POSIX.1-2001 check */
#endif /* _POSIX_MESSAGE_PASSING */

/* 16 Thread management ******************************************************/

/* Nope: use a threading library. */

/**# Socket handling *********************************************************/

/**>
 * This interface is slightly cooked.  We provide userdata encapsulations for
 * sockaddr and addrinfo types.  Use `getaddrinfo()` to obtain an addrinfo
 * object, then use `ipairs()` over it to get each entry to try.
 *
 * \t r, addrinfo = getaddrinfo("www.rjek.com", "80", 0, l.AF_UNSPEC, l.SOCK_STREAM)
 * \t for _, ai in ipairs(addrinfo) do
 * \t    sock = socket(ai.ai_family, ai.ai_socktype, ai.ai_protocol)
 * \t    if connect(sock, ai.ai_addr) >= 0 then break end
 * \t end
 */

#define LUXIO_SOCKADDR_METATABLE_NAME "luxio.sockaddr"
#ifndef UNIX_PATH_MAX
/* From man 7 unix */
#define UNIX_PATH_MAX   108
#endif

static int
luxio__sockaddr_index(lua_State *L)
{
	struct sockaddr *sa = luaL_checkudata(L, 1, LUXIO_SOCKADDR_METATABLE_NAME);
	const char *var = luaL_checkstring(L, 2);
	if (strcmp(var, "family") == 0) {
		lua_pushnumber(L, sa->sa_family);
		return 1;
	}
	if (sa->sa_family == AF_INET) {
		struct sockaddr_in *sa_in = (struct sockaddr_in *)sa;
		if (strcmp(var, "port") == 0) {
			lua_pushnumber(L, ntohs(sa_in->sin_port));
			return 1;
		} else if (strcmp(var, "address") == 0) {
			char tmp_buf[INET_ADDRSTRLEN];
			lua_pushstring(L, inet_ntop(AF_INET, &sa_in->sin_addr, tmp_buf, INET_ADDRSTRLEN));
			return 1;
		}
	}
	if (sa->sa_family == AF_INET6) {
		struct sockaddr_in6 *sa_in = (struct sockaddr_in6 *)sa;
		if (strcmp(var, "port") == 0) {
			lua_pushnumber(L, ntohs(sa_in->sin6_port));
			return 1;
		} else if (strcmp(var, "flowinfo") == 0) {
			lua_pushnumber(L, sa_in->sin6_flowinfo);
			return 1;
		} else if (strcmp(var, "scope_id") == 0) {
			lua_pushnumber(L, sa_in->sin6_scope_id);
			return 1;
		} else if (strcmp(var, "address") == 0) {
			char tmp_buf[INET6_ADDRSTRLEN];
			lua_pushstring(L, inet_ntop(AF_INET6, &sa_in->sin6_addr, tmp_buf, INET6_ADDRSTRLEN));
			return 1;
		}
	}
	if (sa->sa_family == AF_UNIX) {
		struct sockaddr_un *sa_un = (struct sockaddr_un *)sa;
		if (strcmp(var, "path") == 0) {
			lua_pushstring(L, sa_un->sun_path);
			return 1;
		}
	}
	return luaL_error(L, "unknown field %s in sockaddr", var);
}

static int
luxio__sockaddr_tostring(lua_State *L)
{
	struct sockaddr *sa = luaL_checkudata(L, 1, LUXIO_SOCKADDR_METATABLE_NAME);
	if (sa->sa_family == AF_INET) {
		struct sockaddr_in *sa_in = (struct sockaddr_in *)sa;
		char tmp_buf[INET_ADDRSTRLEN];
		inet_ntop(AF_INET, &sa_in->sin_addr, tmp_buf, INET_ADDRSTRLEN);
		lua_pushfstring(L, "sockaddr: AF_INET %d %s", ntohs(sa_in->sin_port), tmp_buf);
	} else if (sa->sa_family == AF_INET6) {
		struct sockaddr_in6 *sa_in = (struct sockaddr_in6 *)sa;
		char tmp_buf[INET6_ADDRSTRLEN];
		inet_ntop(AF_INET6, &sa_in->sin6_addr, tmp_buf, INET6_ADDRSTRLEN);
		lua_pushfstring(L, "sockaddr: AF_INET6 %d %s", ntohs(sa_in->sin6_port), tmp_buf);
	} else if (sa->sa_family == AF_UNIX) {
		struct sockaddr_un *sa_un = (struct sockaddr_un *)sa;
		lua_pushfstring(L, "sockaddr: AF_UNIX %s", sa_un->sun_path);
	} else {
		lua_pushfstring(L, "sockaddr: unknown family %d", sa->sa_family);
	}
	return 1;
}

/* Label the userdata on the top of the stack as a sockaddr */
static int
luxio___makesockaddr(lua_State *L)
{
	int create_mt = luaL_newmetatable(L, LUXIO_SOCKADDR_METATABLE_NAME);

	if (create_mt) {
		lua_pushcfunction(L, luxio__sockaddr_index);
		lua_setfield(L, -2, "__index");
		lua_pushcfunction(L, luxio__sockaddr_tostring);
		lua_setfield(L, -2, "__tostring");
	}

	lua_setmetatable(L, -2);

	return 1;
}

/* Copy the given sockaddr and push a userdata onto the stack labelled as a sockaddr */
static int
luxio__makesockaddr(lua_State *L, const struct sockaddr *sa, socklen_t len)
{
	struct sockaddr *sa_copy = lua_newuserdata(L, len);

	memcpy(sa_copy, sa, len);

	return luxio___makesockaddr(L);
}

/* Utility for sizing supported socket addresses */
static socklen_t
luxio___sockaddr_len(lua_State *L, const struct sockaddr *sa)
{
	switch (sa->sa_family) {
	case AF_INET:
		return sizeof(struct sockaddr_in);
	case AF_INET6:
		return sizeof(struct sockaddr_in6);
	case AF_UNIX:
		return SUN_LEN((struct sockaddr_un *)sa);
	}

	return luaL_error(L, "unknown address family %d", sa->sa_family);
}

static int
luxio_makesockaddr(lua_State *L)
{
	int family = luaL_checkinteger(L, 1);
	if (family == AF_INET) {
		struct sockaddr_in *sa_in;
		int port = luaL_checkinteger(L, 2);
		const char *address = luaL_checkstring(L, 3);
		sa_in = lua_newuserdata(L, sizeof(*sa_in));
		if (inet_pton(AF_INET, address, &sa_in->sin_addr) != 1) {
			return luaL_error(L, "unable to parse address: %s", address);
		}
		if (port < 0 || port > 65535) {
			return luaL_error(L, "port %d out of range", port);
		}
		sa_in->sin_family = AF_INET;
		sa_in->sin_port = htons(port);
		return luxio___makesockaddr(L);
	}
	if (family == AF_INET6) {
		struct sockaddr_in6 *sa_in;
		int port = luaL_checkinteger(L, 2);
		const char *address = luaL_checkstring(L, 3);
		sa_in = lua_newuserdata(L, sizeof(*sa_in));
		if (inet_pton(AF_INET6, address, &sa_in->sin6_addr) != 1) {
			return luaL_error(L, "unable to parse address: %s", address);
		}
		if (port < 0 || port > 65535) {
			return luaL_error(L, "port %d out of range", port);
		}
		/* TODO: Support flow and scope */
		sa_in->sin6_family = AF_INET6;
		sa_in->sin6_port = htons(port);
		sa_in->sin6_flowinfo = 0;
		sa_in->sin6_scope_id = 0;
		return luxio___makesockaddr(L);
	}
	if (family == AF_UNIX) {
		struct sockaddr_un *sa_un;
		size_t pathlen;
		const char *path = luaL_checklstring(L, 2, &pathlen);
		if (pathlen >= UNIX_PATH_MAX) {
			return luaL_error(L, "unable to create local socket, path too long (%d chars)",
					  pathlen);
		}
		sa_un = lua_newuserdata(L, sizeof(sa_un->sun_family) + pathlen + 1);
		sa_un->sun_family = AF_UNIX;
		strcpy(&sa_un->sun_path[0], path);
		return luxio___makesockaddr(L);
	}
	return luaL_error(L, "unknown socket family %d", family);
}

/**% socket
 * retval = socket(domain, type, protocol);
 * retval, errno = socket(domain, type, protocol)
 */
static int
luxio_socket(lua_State *L)
{
	int domain = luaL_checkint(L, 1);
	int type = luaL_checkint(L, 2);
	int protocol = luaL_checkint(L, 3);

	lua_pushinteger(L, socket(domain, type, protocol));
	lua_pushinteger(L, errno);

	return 2;
}

/**% listen
 * retval = listen(fd, backlog);
 * retval, errno = listen(fd, backlog)
 */
static int
luxio_listen(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	int backlog = luaL_checkint(L, 2);

	lua_pushinteger(L, listen(sockfd, backlog));
	lua_pushinteger(L, errno);

	return 2;
}

/**% shutdown
 * retval = shutdown(fd, how);
 * retval, errno = shutdown(fd, how)
 */
static int
luxio_shutdown(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	int how = luaL_checkint(L, 2);

	lua_pushinteger(L, shutdown(sockfd, how));
	lua_pushinteger(L, errno);

	return 2;
}

/**% connect
 * retval = connect(fd, sockaddr, sockaddrlen);
 * retval, errno = connect(fd, sockaddr)
 */
static int
luxio_connect(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	struct sockaddr *addr = luaL_checkudata(L, 2, LUXIO_SOCKADDR_METATABLE_NAME);
	socklen_t l = luxio___sockaddr_len(L, addr);

	lua_pushinteger(L, connect(sockfd, addr, l));
	lua_pushinteger(L, errno);

	return 2;
}

/**% bind
 * retval = bind(fd, sockaddr, sockaddrlen);
 * retval, errno = bind(fd, sockaddr)
 */
static int
luxio_bind(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	struct sockaddr *addr = luaL_checkudata(L, 2, LUXIO_SOCKADDR_METATABLE_NAME);
	socklen_t l = luxio___sockaddr_len(L, addr);

	lua_pushinteger(L, bind(sockfd, addr, l));
	lua_pushinteger(L, errno);

	return 2;
}

/**% accept
 * retval = accept(fd, sockaddrinfo, sockaddrlen);
 * retval, sockaddr = accept(fd)
 * retval, errno = accept(fd)
 */
static int
luxio_accept(lua_State *L)
{
	int listenfd = luaL_checkint(L, 1);
	int clientfd;
	struct sockaddr_storage addr;
	socklen_t l = sizeof(addr);

	clientfd = accept(listenfd, (struct sockaddr *)&addr, &l);

	lua_pushinteger(L, clientfd);

	if (clientfd > -1) {
		luxio__makesockaddr(L, (const struct sockaddr *)&addr, l);
	} else {
		lua_pushinteger(L, errno);
	}

	return 2;
}

/**% getsockopt
 * retval = getsockopt(fd, level, optname, optval, optlen);
 * retval, errno = getsockopt(fd, level, optname)
 * retval, opt = getsockopet(fd, level, optname)
 */
static int
luxio_getsockopt(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	int level = luaL_checkint(L, 2);
	int optname = luaL_checkint(L, 3);
	int res, r_int;
	char r_ifname[IFNAMSIZ];
	socklen_t l;

	switch (level) {
	case SOL_SOCKET:
		switch (optname) {
		/* options that involve strings */
#ifdef SO_BINDTODEVICE
		case SO_BINDTODEVICE:
#endif
			l = IFNAMSIZ;
			res = getsockopt(sockfd, level, optname, r_ifname, &l);
			lua_pushinteger(L, res);
			if (res == -1) {
				lua_pushinteger(L, errno);
			} else {
				lua_pushstring(L, r_ifname);
			}

			return 2;

                /* other options probably expect integers */
		default:
			l = sizeof(r_int);
			res = getsockopt(sockfd, level, optname, &r_int, &l);
			lua_pushinteger(L, res);
			if (res == -1) {
				lua_pushinteger(L, errno);
			} else {
				lua_pushinteger(L, r_int);
			}

			return 2;
		}
	case IPPROTO_IP:
		switch (optname) {
#ifdef TCP_CORK
		case TCP_CORK:
#endif
#ifdef TCP_DEFER_ACCEPT
		case TCP_DEFER_ACCEPT:
#endif
#ifdef TCP_NODELAY
		case TCP_NODELAY:
#endif
			l = sizeof(r_int);
			res = getsockopt(sockfd, level, optname, &r_int, &l);
			lua_pushinteger(L, res);
			if (res == -1) {
				lua_pushinteger(L, errno);
			} else {
				lua_pushinteger(L, r_int);
			}

			return 2;

		default:
			return luaL_error(L, "unhandled IPPROTO_IP option %d",
					  optname);
		}
	}

	return luaL_error(L, "unhandled socket level %d", level);
}

/**% setsockopt
 * retval = setsockopt(fd, level, optname, optval, optlen);
 * retval, errno = setsockopt(fd, level, optname, optval)
 */
static int
luxio_setsockopt(lua_State *L)
{
	int sockfd = luaL_checkint(L, 1);
	int level = luaL_checkint(L, 2);
	int optname = luaL_checkint(L, 3);
	int res, r_int;
	const char *r_ifname;
	socklen_t l;
	size_t sz;

	switch (level) {
	case SOL_SOCKET:
		switch (optname) {
		/* options that involve strings */
#ifdef SO_BINDTODEVICE
		case SO_BINDTODEVICE:
#endif
			r_ifname = luaL_checklstring(L, 4, &sz);
			res = setsockopt(sockfd, level, optname,
					 (void *)r_ifname, (socklen_t)sz);
			lua_pushinteger(L, res);
			lua_pushinteger(L, errno);

			return 2;
		/* other options are probably integers */
		default:
			l = sizeof(r_int);
			r_int = luaL_checkint(L, 4);
			res = setsockopt(sockfd, level, optname, &r_int, l);
			lua_pushinteger(L, res);
			lua_pushinteger(L, errno);

			return 2;
		}
	case IPPROTO_IP:
		switch (optname) {
#ifdef TCP_CORK
		case TCP_CORK:
#endif
#ifdef TCP_DEFER_ACCEPT
		case TCP_DEFER_ACCEPT:
#endif
#ifdef TCP_NODELAY
		case TCP_NODELAY:
#endif
			l = sizeof(r_int);
			r_int = luaL_checkint(L, 4);
			res = setsockopt(sockfd, level, optname, &r_int, l);
			lua_pushinteger(L, res);
			lua_pushinteger(L, errno);

			return 2;

		default:
			return luaL_error(L, "unhandled IPPROTO_IP option %d",
					  optname);
		}
	}

	return luaL_error(L, "unhandled socket level %d", level);
}

/**% gai_strerror
 * retval = gai_strerror(errnum);
 * retval = gai_strerror(errno)
 */
static int
luxio_gai_strerror(lua_State *L)
{
	lua_pushstring(L, gai_strerror(luaL_checkint(L, 1)));

	return 1;
}

/**% getaddrinfo
 * retval = getaddrinfo(node, service, hints, res);
 * errcode = getaddrinfo(node, service[, ai_flags/0, ai_family/AF_UNSPEC, ai_socktype/0, ai_protocol/0])
 * addrinfo = getaddrinfo(node, service[, ai_flags/0, ai_family/AF_UNSPEC, ai_socktype/0, ai_protocol/0])
 */
static int
luxio_getaddrinfo(lua_State *L)
{
	const char *node = luaL_checkstring(L, 1);
	const char *serv = luaL_checkstring(L, 2);
	struct addrinfo hints, *results, *rp;
	int r, c;

	memset(&hints, '\0', sizeof(hints));
	hints.ai_flags = luaL_optint(L, 3, 0);
	hints.ai_family = luaL_optint(L, 4, AF_UNSPEC);
	hints.ai_socktype = luaL_optint(L, 5, 0);
	hints.ai_protocol = luaL_optint(L, 6, 0);

	if (node[0] == '\0')
		node = NULL;

	if (serv[0] == '\0')
		serv = NULL;

	r = getaddrinfo(node, serv, &hints, &results);

	lua_pushinteger(L, r);

	if (r < 0)
		return 1;

	lua_newtable(L); /* table we return with entries */

	for (rp = results, c = 1; rp != NULL; rp = rp->ai_next, c++) {
		lua_createtable(L, 0, 6); /* entry table */

		lua_pushliteral(L, "ai_flags");
		lua_pushinteger(L, rp->ai_flags);
		lua_rawset(L, -3);

		lua_pushliteral(L, "ai_family");
		lua_pushinteger(L, rp->ai_family);
		lua_rawset(L, -3);

		lua_pushliteral(L, "ai_socktype");
		lua_pushinteger(L, rp->ai_socktype);
		lua_rawset(L, -3);

		lua_pushliteral(L, "ai_protocol");
		lua_pushinteger(L, rp->ai_protocol);
		lua_rawset(L, -3);

		lua_pushliteral(L, "ai_canonname");
		lua_pushstring(L, rp->ai_canonname);
		lua_rawset(L, -3);

		lua_pushliteral(L, "ai_addr");
		luxio__makesockaddr(L, rp->ai_addr, rp->ai_addrlen);
		lua_rawset(L, -3);

		lua_rawseti(L, -2, c);
	}

	freeaddrinfo(results);

	return 2;
}

/* Poll-binding functions ****************************************************/

#define LUXIO_POLLFD_METATABLE "luxio.pollfdarray"

typedef struct {
	struct pollfd *pollfds;
	int allocated;
} luxio_pollfds;

static int
luxio__pollfds_gc(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	free(pfds->pollfds);
	pfds->pollfds = NULL;
	pfds->allocated = 0;
	return 0;
}

static int
luxio__pollfds_tostring(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	lua_pushfstring(L, "pollfds: %d slot%s", pfds->allocated, pfds->allocated == 1 ? "" : "s");
	return 1;
}

static int
luxio__pollfds_len(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	lua_pushnumber(L, pfds->allocated);
	return 1;
}

static int
luxio_pollfds_new(lua_State *L)
{
	luxio_pollfds *pfds = lua_newuserdata(L, sizeof(*pfds));
	int create_table = luaL_newmetatable(L, LUXIO_POLLFD_METATABLE);
	pfds->pollfds = NULL;
	pfds->allocated = 0;

	if (create_table) {
		lua_pushcfunction(L, luxio__pollfds_gc);
		lua_setfield(L, -2, "__gc");
		lua_pushcfunction(L, luxio__pollfds_tostring);
		lua_setfield(L, -2, "__tostring");
		lua_pushcfunction(L, luxio__pollfds_len);
		lua_setfield(L, -2, "__len");
	}

	lua_setmetatable(L, -2);

	return 1;
}

static int
luxio_pollfds_resize(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	int desired_size = luaL_checkint(L, 2);
	int idx;
	struct pollfd *newfds = realloc(pfds->pollfds, sizeof(struct pollfd) * desired_size);
	if (newfds != NULL) {
		if (desired_size > pfds->allocated) {
			for (idx = pfds->allocated; idx < desired_size; ++idx) {
				newfds[idx].fd = -1;
				newfds[idx].events = newfds[idx].revents = 0;
			}
		}
		pfds->pollfds = newfds;
		pfds->allocated = desired_size;
	} else {
		return luaL_error(L, "Unable to resize pollfds array");
	}
	/* Return the pollfds array for neatness */
	return 1;
}

static int
luxio_pollfds_set_slot(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	int slot = luaL_checkint(L, 2);
	int fd = luaL_checkint(L, 3);
	short events = luaL_checkint(L, 4);
	short revents;
	if (slot == 0 || slot > pfds->allocated || slot < -pfds->allocated) {
		return luaL_error(L, "slot out of range 1 .. %d", pfds->allocated);
	}

	revents = luaL_optint(L, 4, pfds->pollfds[slot - 1].revents);

	pfds->pollfds[slot - 1].fd = fd;
	pfds->pollfds[slot - 1].events = events;
	pfds->pollfds[slot - 1].revents = revents;

	return 0;
}

static int
luxio_pollfds_get_slot(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	int slot = luaL_checkint(L, 2);

	if (slot == 0 || slot > pfds->allocated || slot < -pfds->allocated) {
		return luaL_error(L, "slot out of range 1 .. %d", pfds->allocated);
	}

	lua_pushnumber(L, pfds->pollfds[slot - 1].fd);
	lua_pushnumber(L, pfds->pollfds[slot - 1].events);
	lua_pushnumber(L, pfds->pollfds[slot - 1].revents);
	return 3;
}

static int
luxio_poll(lua_State *L)
{
	luxio_pollfds *pfds = luaL_checkudata(L, 1, LUXIO_POLLFD_METATABLE);
	int timeout = luaL_checkint(L, 2);

	lua_pushinteger(L, poll(pfds->pollfds, pfds->allocated, timeout));
	lua_pushinteger(L, errno);

	return 2;
}

/* Bit/flag operation functions **********************************************/

static int
luxio_bitop_or(lua_State *L)
{
	int value = luaL_checkint(L, 1);
	int n = lua_gettop(L);

	while (n > 1) {
		value |= luaL_checkint(L, n--);
	}

	lua_pushnumber(L, value);

	return 1;
}

static int
luxio_bitop_and(lua_State *L)
{
	int value = luaL_checkint(L, 1);
	int n = lua_gettop(L);

	while (n > 1) {
		value &= luaL_checkint(L, n--);
	}

	lua_pushnumber(L, value);

	return 1;
}

static int
luxio_bitop_clear(lua_State *L)
{
	int value = luaL_checkint(L, 1);
	int n = lua_gettop(L);

	while (n > 1) {
		value &= ~luaL_checkint(L, n--);
	}

	lua_pushnumber(L, value);

	return 1;
}

static int
luxio_bitop_invert(lua_State *L)
{
	int value = luaL_checkint(L, 1);
	int n = lua_gettop(L);

	/* Special case, passed 1 value, we invert that rather than
	 * inverting the other bits supplied
	 */
	if (n == 1) {
		value = ~value;
	} else {
		while (n > 1) {
		     value ^= luaL_checkint(L, n--);
		}
	}

	lua_pushnumber(L, value);

	return 1;
}

static int
luxio_bitop_test(lua_State *L)
{
	int value = luaL_checkint(L, 1);
	int goal = 0;
	int n = lua_gettop(L);

	while (n > 1) {
		goal |= luaL_checkint(L, n--);
	}

	lua_pushboolean(L, (value & goal) == goal);

	return 1;
}

/* Time-related functions ****************************************************/

#define LUXIO_TIMEVAL_METATABLE "luxio.timeval"

static int
luxio_timeval_lt(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	struct timeval *b = luaL_checkudata(L, 2, LUXIO_TIMEVAL_METATABLE);

	lua_pushboolean(L, timercmp(a, b, <));

	return 1;
}

static int
luxio_timeval_le(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	struct timeval *b = luaL_checkudata(L, 2, LUXIO_TIMEVAL_METATABLE);

	/* <= is not portable, so use ! > */
	lua_pushboolean(L, !timercmp(a, b, >));

	return 1;
}

static int
luxio_timeval_eq(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	struct timeval *b = luaL_checkudata(L, 2, LUXIO_TIMEVAL_METATABLE);

	/* == is not portable, so use ! != */
	lua_pushboolean(L, !timercmp(a, b, !=));

	return 1;
}

#define LUXIO_TIME_BUFLEN (1024)
static int
luxio_timeval_tostring(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	char buffer[LUXIO_TIME_BUFLEN];

	snprintf(buffer, LUXIO_TIME_BUFLEN, "timeval: %ld.%06ld",
		 (long)a->tv_sec, (long)a->tv_usec);
	
	lua_pushstring(L, buffer);

	return 1;
}

static int
luxio_timeval_index(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	const char *s = luaL_checkstring(L, 2);

	if (strcmp(s, "tv_sec") == 0)
		lua_pushinteger(L, a->tv_sec);
	else if (strcmp(s, "tv_usec") == 0)
		lua_pushinteger(L, a->tv_usec);
	else if (strcmp(s, "seconds") == 0)
		lua_pushnumber(L, (lua_Number)(a->tv_sec) +
			       ((lua_Number)(a->tv_usec) / 1000000));
	else if (strcmp(s, "useconds") == 0)
		lua_pushinteger(L, a->tv_usec + (a->tv_sec * 1000000));
	else
		luaL_error(L, "Unknown field %s in timeval", s);

	return 1;
}

static int
luxio_timeval_newindex(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	const char *s = luaL_checkstring(L, 2);

	if (strcmp(s, "tv_sec") == 0)
		a->tv_sec = luaL_checkinteger(L, 3);
	else if (strcmp(s, "tv_usec") == 0)
		a->tv_usec = luaL_checkinteger(L, 3);
	else if (strcmp(s, "seconds") == 0) {
		lua_Number v = luaL_checknumber(L, 3);
		a->tv_sec = (time_t)v;
		a->tv_usec = (suseconds_t)((v - a->tv_sec) * 1000000);
	} else if (strcmp(s, "useconds") == 0) {
		lua_Number v = luaL_checknumber(L, 3);
		a->tv_sec = (time_t)(v / 1000000);
		a->tv_usec = (suseconds_t)v % 1000000;
	} else
		luaL_error(L, "Unknown field %s in timeval", s);

	return 0;
}

static void luxio__bless_timeval(lua_State *L);

static int
luxio_timeval_add(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	struct timeval *b = luaL_checkudata(L, 2, LUXIO_TIMEVAL_METATABLE);
	struct timeval *ret = lua_newuserdata(L, sizeof(*ret));

	timeradd(a, b, ret);

	luxio__bless_timeval(L);

	return 1;
}

static int
luxio_timeval_sub(lua_State *L)
{
	struct timeval *a = luaL_checkudata(L, 1, LUXIO_TIMEVAL_METATABLE);
	struct timeval *b = luaL_checkudata(L, 2, LUXIO_TIMEVAL_METATABLE);
	struct timeval *ret = lua_newuserdata(L, sizeof(*ret));

	timersub(a, b, ret);

	luxio__bless_timeval(L);

	return 1;
}

static void
luxio__bless_timeval(lua_State *L)
{
	int create = luaL_newmetatable(L, LUXIO_TIMEVAL_METATABLE);

	if (create) {
		lua_pushcfunction(L, luxio_timeval_le);
		lua_setfield(L, -2, "__le");
		lua_pushcfunction(L, luxio_timeval_lt);
		lua_setfield(L, -2, "__lt");
		lua_pushcfunction(L, luxio_timeval_eq);
		lua_setfield(L, -2, "__eq");
		lua_pushcfunction(L, luxio_timeval_tostring);
		lua_setfield(L, -2, "__tostring");
		lua_pushcfunction(L, luxio_timeval_index);
		lua_setfield(L, -2, "__index");
		lua_pushcfunction(L, luxio_timeval_newindex);
		lua_setfield(L, -2, "__newindex");
		lua_pushcfunction(L, luxio_timeval_add);
		lua_setfield(L, -2, "__add");
		lua_pushcfunction(L, luxio_timeval_sub);
		lua_setfield(L, -2, "__sub");
	}

	lua_setmetatable(L, -2);
}

static int
luxio_timeval_zero(lua_State *L)
{
	struct timeval *r = lua_newuserdata(L, sizeof(*r));

	timerclear(r);

	luxio__bless_timeval(L);

	return 1;
}

static int
luxio_gettimeofday(lua_State *L)
{
	struct timeval *r = lua_newuserdata(L, sizeof(*r));

	int ret = gettimeofday(r, NULL);

	if (ret == -1) {
		lua_pushinteger(L, -1);
		lua_pushinteger(L, errno);
		return 2;
	}

	luxio__bless_timeval(L);

	return 1;
}

/**# Misc utility functions **************************************************/

/**% strerror
 * retval = strerror(errno);
 * retval = strerror(errno)
 */
static int
luxio_strerror(lua_State *L)
{
	lua_pushstring(L, strerror(luaL_checkint(L, 1)));

	return 1;
}

static const struct luaL_Reg
luxio_functions[] = {
	{ "open", luxio_open },
	{ "close", luxio_close },
	{ "read", luxio_read },
	{ "write", luxio_write },
	{ "writev", luxio_writev },
	{ "lseek", luxio_lseek },
	{ "ftruncate", luxio_ftruncate },
	{ "fsync", luxio_fsync },
#ifdef HAVE_FDATASYNC
	{ "fdatasync", luxio_fdatasync },
#endif
	{ "rename", luxio_rename },
	{ "link", luxio_link },
	{ "unlink", luxio_unlink },
	{ "symlink", luxio_symlink },
	{ "readlink", luxio_readlink },
#ifdef HAVE_SENDFILE
	{ "sendfile", luxio_sendfile },
#endif
#ifdef HAVE_SPLICE
	{ "splice", luxio_splice },
#endif
	{ "dup", luxio_dup },
	{ "dup2", luxio_dup2 },
#ifdef _GNU_SOURCE
	{ "dup3", luxio_dup3 },
#endif
        { "pipe", luxio_pipe },
#ifdef _GNU_SOURCE
        { "pipe2", luxio_pipe2 },
#endif
	{ "socketpair", luxio_socketpair },
	{ "fcntl", luxio_fcntl },
	{ "umask", luxio_umask },
	{ "chmod", luxio_chmod },
	{ "fchmod", luxio_fchmod },
	{ "chown", luxio_chown },
	{ "mkfifo", luxio_mkfifo },
	{ "mkdir", luxio_mkdir },
	{ "rmdir", luxio_rmdir },

#define STAT_IS_ENTRY(x) { "S_IS" #x, luxio_S_IS##x }
	STAT_IS_ENTRY(REG),
	STAT_IS_ENTRY(DIR),
	STAT_IS_ENTRY(CHR),
	STAT_IS_ENTRY(BLK),
	STAT_IS_ENTRY(FIFO),
#ifdef S_ISLNK
	STAT_IS_ENTRY(LNK),
#endif
#ifdef S_ISSOCK
	STAT_IS_ENTRY(SOCK),
#endif
#undef STAT_IS_ENTRY

	{ "stat", luxio_stat },
	{ "lstat", luxio_lstat },
	{ "fstat", luxio_fstat },

	{ "socket", luxio_socket },
	{ "listen", luxio_listen },
	{ "shutdown", luxio_shutdown },
	{ "connect", luxio_connect },
	{ "bind", luxio_bind },
	{ "accept", luxio_accept },
	{ "getsockopt", luxio_getsockopt },
	{ "setsockopt", luxio_setsockopt },

	{ "getaddrinfo", luxio_getaddrinfo },
	{ "gai_strerror", luxio_gai_strerror },

	{ "make_sockaddr", luxio_makesockaddr },

	{ "pollfds_new", luxio_pollfds_new },
	{ "pollfds_resize", luxio_pollfds_resize },
	{ "pollfds_setslot", luxio_pollfds_set_slot },
	{ "pollfds_getslot", luxio_pollfds_get_slot },
	{ "poll", luxio_poll },

	{ "zero_timeval", luxio_timeval_zero },
	{ "gettimeofday", luxio_gettimeofday },

	{ "fork", luxio_fork },
	{ "exec", luxio_exec },
	{ "execp", luxio_execp },
	{ "waitpid", luxio_waitpid },

#define WAITPID_STATUS_ENTRY(x) { #x, luxio_##x }
	WAITPID_STATUS_ENTRY(WIFEXITED),
	WAITPID_STATUS_ENTRY(WEXITSTATUS),
	WAITPID_STATUS_ENTRY(WIFSIGNALED),
	WAITPID_STATUS_ENTRY(WTERMSIG),
#ifdef WCOREDUMP
	WAITPID_STATUS_ENTRY(WCOREDUMP),
#endif
	WAITPID_STATUS_ENTRY(WIFSTOPPED),
	WAITPID_STATUS_ENTRY(WSTOPSIG),
#ifdef WIFCONTINUED
	WAITPID_STATUS_ENTRY(WIFCONTINUED),
#endif
#undef WAITPID_STATUS_ENTRY

	{ "kill", luxio_kill },

	{ "strerror",  luxio_strerror },
	{ "_exit", luxio__exit },

	{ "setenv", luxio_setenv },
	{ "unsetenv", luxio_unsetenv },
	{ "getenv", luxio_getenv },

	{ "opendir", luxio_opendir },
	{ "fdopendir", luxio_fdopendir },
	{ "closedir", luxio_closedir },
	{ "readdir", luxio_readdir },
	{ "rewinddir", luxio_rewinddir },

	{ "chdir", luxio_chdir },
	{ "getcwd", luxio_getcwd },

	{ "alarm", luxio_alarm },
	{ "pause", luxio_pause },
	{ "sleep", luxio_sleep },

	{ "getpid", luxio_getpid },
	{ "getppid", luxio_getppid },

	{ "getuid", luxio_getuid },
	{ "geteuid", luxio_geteuid },
	{ "getgid", luxio_getgid },
	{ "getegid", luxio_getegid },
	{ "setuid", luxio_setuid },
	{ "setgid", luxio_setgid },
	{ "getlogin", luxio_getlogin },

	{ "uname", luxio_uname },

	{ "time", luxio_time },
	{ "times", luxio_times },

	{ "tcgetpgrp", luxio_tcgetpgrp},
	{ "tcsetpgrp", luxio_tcsetpgrp},

	{ "nanosleep", luxio_nanosleep },

#if defined(_POSIX_MESSAGE_PASSING) && defined(__linux__)

#if _XOPEN_SOURCE >= 600 || _POSIX_C_SOURCE >= 200112L
	{ "mq_timedsend", luxio_mq_timedsend },
	{ "mq_timedreceive", luxio_mq_timedreceive },
#endif
	{ "mq_open", luxio_mq_open },
	{ "mq_close", luxio_mq_close },
	{ "mq_unlink", luxio_mq_unlink },
	{ "mq_send", luxio_mq_send },
	{ "mq_receive", luxio_mq_receive },
	{ "mq_setattr", luxio_mq_setattr },
	{ "mq_getattr", luxio_mq_getattr }, 
#endif

	{ NULL, NULL }
};

static const struct luaL_Reg
luxio_bitop_functions[] = {
	{ "bor", luxio_bitop_or },
	{ "band", luxio_bitop_and },
	{ "binvert", luxio_bitop_invert },
	{ "btest", luxio_bitop_test },
	{ "bclear", luxio_bitop_clear },

	{ NULL, NULL }
};

#include "luxio_constants.h"

#define NUMERIC_CONSTANT(x) do { lua_pushstring(L, #x);  \
				 lua_pushinteger(L, x); \
				 lua_settable(L, -3); } while (0)

int
luaopen_luxio(lua_State *L)
{
	int e;
	const char *n;
	lua_Number v;

#if (LUA_VERSION_NUM > 501)
	luaL_newlib(L, luxio_functions);
	luaL_newlib(L, luxio_bitop_functions);
#else
	luaL_register(L, "luxio", luxio_functions);
	lua_createtable(L, 0, (sizeof(luxio_bitop_functions) /
			       sizeof(struct luaL_Reg)) - 1);
	luaL_register(L, NULL, luxio_bitop_functions);
#endif
	lua_setfield(L, -2, "bit");

	for (e = 0;; e++) {
		n = luxio_numeric_constants[e].name;
		v = luxio_numeric_constants[e].number;

		if (n == NULL) break;
		lua_pushstring(L, n);
		lua_pushnumber(L, v);
		lua_settable(L, -3);
	}

	lua_pushstring(L, "_VERSION");
	lua_pushfstring(L, "Luxio %d", LUXIO_RELEASE);
	lua_settable(L, -3);

	lua_pushstring(L, "_COPYRIGHT");
	lua_pushstring(L, LUXIO_COPYRIGHT);
	lua_settable(L, -3);

	lua_pushstring(L, "_RELEASE");
	lua_pushnumber(L, LUXIO_RELEASE);
	lua_settable(L, -3);

	lua_pushstring(L, "_ABI");
	lua_pushnumber(L, LUXIO_ABI);
	lua_settable(L, -3);

	/* push values that are not compile-time known */
#ifdef SIGRTMIN
	NUMERIC_CONSTANT(SIGRTMIN);
	NUMERIC_CONSTANT(SIGRTMAX);
#endif
#ifdef HAVE_D_TYPE
	NUMERIC_CONSTANT(DT_UNKNOWN);
	NUMERIC_CONSTANT(DT_FIFO);
	NUMERIC_CONSTANT(DT_CHR);
	NUMERIC_CONSTANT(DT_DIR);
	NUMERIC_CONSTANT(DT_BLK);
	NUMERIC_CONSTANT(DT_REG);
	NUMERIC_CONSTANT(DT_LNK);
	NUMERIC_CONSTANT(DT_SOCK);
#endif
	return 1;
}

#undef NUMERIC_CONSTANT
