/***************************************************************************
* Copyright (c) 2013 Abdurrahman AVCI <abdurrahmanavci@gmail.com
* Copyright (c) 2015-2024 Lubuntu Artwork Team
*
* Permission is hereby granted, free of charge, to any person
* obtaining a copy of this software and associated documentation
* files (the "Software"), to deal in the Software without restriction,
* including without limitation the rights to use, copy, modify, merge,
* publish, distribute, sublicense, and/or sell copies of the Software,
* and to permit persons to whom the Software is furnished to do so,
* subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
* THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
* OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
* OR OTHER DEALINGS IN THE SOFTWARE.
*
***************************************************************************/

import QtQuick 2.0
import SddmComponents 2.0

Rectangle {
    id: container
    width: 1024
    height: 768

    property int sessionIndex: session.index

    TextConstants { id: textConstants }

    Background {
        anchors.fill: parent
        source: Qt.resolvedUrl(config.background)
        onStatusChanged: {
            var defaultBackground = Qt.resolvedUrl(config.defaultBackground)
            if (status == Image.Error && source != defaultBackground) {
                source = defaultBackground
            }
        }
    }

    Connections {
        target: sddm
        onLoginSucceeded: {
        }

        onLoginFailed: {
            txtMessage.text = textConstants.loginFailed
            listView.currentItem.password.text = ""
        }
    }

    Repeater {
        model: screenModel
        Background {
            x: geometry.x; y: geometry.y; width: geometry.width; height: geometry.height
            source: config.background
            fillMode: Image.PreserveAspectCrop
            onStatusChanged: {
                if (status == Image.Error && source != config.defaultBackground) {
                    source = config.defaultBackground
                }
            }
        }
    }

    Rectangle {
        property variant geometry: screenModel.geometry(screenModel.primary)
        x: geometry.x; y: geometry.y; width: geometry.width; height: geometry.height
        color: "transparent"

        Component {
            id: userDelegate

            PictureBox {
                anchors.verticalCenter: parent.verticalCenter
                name: (model.realName === "") ? model.name : model.realName
                icon: model.icon

                focus: (listView.currentIndex === index) ? true : false
                state: (listView.currentIndex === index) ? "active" : ""

                onLogin: sddm.login(model.name, password, sessionIndex);

                MouseArea {
                    anchors.fill: parent
                    hoverEnabled: true
                    onEntered: listView.currentIndex = index
                    onClicked: listView.focus = true
                }
            }
        }

        Row {
            anchors.fill: parent

            Text {
                id: txtMessage
                anchors.top: usersContainer.bottom
                anchors.margins: 20
                anchors.horizontalCenter: parent.horizontalCenter
                color: "white"
                text: textConstants.promptSelectUser
                font.pixelSize: 16
                font.family: "Ubuntu"
            }

            Item {
                id: usersContainer
                width: parent.width
                height: 300
                anchors.verticalCenter: parent.verticalCenter

                ImageButton {
                    id: prevUser
                    anchors.left: parent.left
                    anchors.verticalCenter: parent.verticalCenter
                    anchors.margins: 10
                    source: Qt.resolvedUrl("angle-left.png")
                    onClicked: listView.decrementCurrentIndex()

                    KeyNavigation.backtab: btnShutdown
                    KeyNavigation.tab: listView
                }

                ListView {
                    id: listView
                    height: parent.height
                    anchors.left: prevUser.right
                    anchors.right: nextUser.left
                    anchors.verticalCenter: parent.verticalCenter
                    anchors.margins: 10

                    clip: true
                    focus: true

                    spacing: 5

                    model: userModel
                    delegate: userDelegate
                    orientation: ListView.Horizontal
                    currentIndex: userModel.lastIndex

                    KeyNavigation.backtab: prevUser
                    KeyNavigation.tab: nextUser
                }

                ImageButton {
                    id: nextUser
                    anchors.right: parent.right
                    anchors.verticalCenter: parent.verticalCenter
                    anchors.margins: 10
                    source: Qt.resolvedUrl("angle-right.png")
                    onClicked: listView.incrementCurrentIndex()
                    KeyNavigation.backtab: listView
                    KeyNavigation.tab: session
                }
            }
        }
    }

    Rectangle {
        id: actionBar
        anchors.top: parent.top
        anchors.horizontalCenter: parent.horizontalCenter
        width: parent.width
        height: 64
        color: "#44ffffff"

        Row {
            anchors.left: parent.left
            anchors.margins: 5
            height: parent.height
            spacing: 5

            Text {
                height: parent.height
                anchors.verticalCenter: parent.verticalCenter
                text: textConstants.session
                font.pixelSize: 14
                color: "white"
                verticalAlignment: Text.AlignVCenter
            }

            ComboBox {
                id: session
                width: 245
                anchors.verticalCenter: parent.verticalCenter
                arrowIcon: Qt.resolvedUrl("angle-down.png")
                model: sessionModel
                index: sessionModel.lastIndex
                font.pixelSize: 14
                color: "#44ffffff"
                KeyNavigation.backtab: nextUser
                KeyNavigation.tab: layoutBox
            }

            Text {
                height: parent.height
                anchors.verticalCenter: parent.verticalCenter
                text: textConstants.layout
                font.pixelSize: 14
                color: "white"
                verticalAlignment: Text.AlignVCenter
            }

            LayoutBox {
                id: layoutBox
                width: 90
                anchors.verticalCenter: parent.verticalCenter
                font.pixelSize: 14
                arrowIcon: Qt.resolvedUrl("angle-down.png")
                KeyNavigation.backtab: session
                KeyNavigation.tab: btnShutdown
            }
        }

        Row {
            height: parent.height
            anchors.right: parent.right
            anchors.margins: 5
            spacing: 5

            Clock {
                id: clock
                color: "white"
                timeFont.family: "Ubuntu"
                timeFont.bold: true
                timeFont.pixelSize: 28
                dateFont.pixelSize: 12
            }

            ImageButton {
                id: btnSuspend
                height: parent.height
                source: Qt.resolvedUrl("suspend.png")
                visible: sddm.canSuspend
                onClicked: sddm.suspend()
                KeyNavigation.backtab: layoutBox
                KeyNavigation.tab: btnReboot
            }

            ImageButton {
                id: btnReboot
                height: parent.height
                source: Qt.resolvedUrl("reboot.png")
                visible: sddm.canReboot
                onClicked: sddm.reboot()
                KeyNavigation.backtab: btnSuspend
                KeyNavigation.tab: btnShutdown
            }

            ImageButton {
                id: btnShutdown
                height: parent.height
                source: Qt.resolvedUrl("shutdown.png")
                visible: sddm.canPowerOff
                onClicked: sddm.powerOff()
                KeyNavigation.backtab: btnReboot
                KeyNavigation.tab: prevUser
            }
        }
    }
}
