--[[

 Unit tests for the URI parsing module of the Lua/APR binding.

 Author: Peter Odding <peter@peterodding.com>
 Last Change: February 11, 2011
 Homepage: http://peterodding.com/code/lua/apr/
 License: MIT

--]]

local apr = require 'apr'
local hostinfo = 'scheme://user:pass@host:80'
local pathinfo = '/path/file?query-param=value#fragment'
local input = hostinfo .. pathinfo

-- Parse a URL into a table of components.
local parsed = assert(apr.uri_parse(input))

-- Validate the parsed URL fields.
assert(parsed.scheme   == 'scheme')
assert(parsed.user     == 'user')
assert(parsed.password == 'pass')
assert(parsed.hostname == 'host')
assert(parsed.port     == '80')
assert(parsed.hostinfo == 'user:pass@host:80')
assert(parsed.path     == '/path/file')
assert(parsed.query    == 'query-param=value')
assert(parsed.fragment == 'fragment')

-- Check that complete and partial URL `unparsing' works.
assert(apr.uri_unparse(parsed) == input)
assert(apr.uri_unparse(parsed, 'hostinfo') == hostinfo)
assert(apr.uri_unparse(parsed, 'pathinfo') == pathinfo)

-- Make sure uri_port_of_scheme() works.
assert(apr.uri_port_of_scheme 'ssh' == 22)
assert(apr.uri_port_of_scheme 'http' == 80)
assert(apr.uri_port_of_scheme 'https' == 443)

-- The following string constant was generated using this PHP code:
--   $t=array();
--   for ($i=0; $i <= 255; $i++) $t[] = chr($i);
--   echo urlencode(join($t));
local uri_encoded = '%00%01%02%03%04%05%06%07%08%09%0A%0B%0C%0D%0E%0F%10%11%12%13%14%15%16%17%18%19%1A%1B%1C%1D%1E%1F+%21%22%23%24%25%26%27%28%29%2A%2B%2C-.%2F0123456789%3A%3B%3C%3D%3E%3F%40ABCDEFGHIJKLMNOPQRSTUVWXYZ%5B%5C%5D%5E_%60abcdefghijklmnopqrstuvwxyz%7B%7C%7D%7E%7F%80%81%82%83%84%85%86%87%88%89%8A%8B%8C%8D%8E%8F%90%91%92%93%94%95%96%97%98%99%9A%9B%9C%9D%9E%9F%A0%A1%A2%A3%A4%A5%A6%A7%A8%A9%AA%AB%AC%AD%AE%AF%B0%B1%B2%B3%B4%B5%B6%B7%B8%B9%BA%BB%BC%BD%BE%BF%C0%C1%C2%C3%C4%C5%C6%C7%C8%C9%CA%CB%CC%CD%CE%CF%D0%D1%D2%D3%D4%D5%D6%D7%D8%D9%DA%DB%DC%DD%DE%DF%E0%E1%E2%E3%E4%E5%E6%E7%E8%E9%EA%EB%EC%ED%EE%EF%F0%F1%F2%F3%F4%F5%F6%F7%F8%F9%FA%FB%FC%FD%FE%FF'
local chars = {}
for i = 0, 255 do chars[#chars+1] = string.char(i) end
local plain_chars = table.concat(chars)

-- Check that apr.uri_encode() works
assert(apr.uri_encode(plain_chars):lower() == uri_encoded:lower())

-- Check that apr.uri_decode() works
assert(apr.uri_decode(uri_encoded):lower() == plain_chars:lower())
