/*
 * Copyright (c) Wipro Technologies Ltd, 2002.  All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it would be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write the Free Software Foundation, Inc., 59
 * Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 */
/**************************************************************************
 * 
 *    TEST IDENTIFIER	: reboot01 
 * 
 *    EXECUTED BY	: root / superuser
 * 
 *    TEST TITLE	: Basic test for reboot(2)
 * 
 *    TEST CASE TOTAL	: 2
 * 
 *    AUTHOR		: Aniruddha Marathe <aniruddha.marathe@wipro.com>
 * 
 *    SIGNALS
 * 	Uses SIGUSR1 to pause before test if option set.
 * 	(See the parse_opts(3) man page).
 *
 *    DESCRIPTION
 *    This is a Phase I test for the reboot(2) system call.
 *    It is intended to provide a limited exposure of the system call.
 *     
 * 
 * 	Setup:
 *	  Setup signal handling.
 *	  Pause for SIGUSR1 if option specified.
 *	  setting the flag value for two tests.
 * 
 * 	Test:
 *	 Loop if the proper options are given.
 *	 for two test cases for two flag values 
 *	  Execute system call
 *	  Check return code, if system call failed (return=-1)
 *		Log the errno and Issue a FAIL message.
 *	  Otherwise, Issue a PASS message.
 * 
 * 	Cleanup:
 * 	  Print errno log and/or timing stats if options given
 * 
 * USAGE:  <for command-line>
 * reboot01 [-c n] [-e] [-i n] [-I x] [-p x] [-t] [-h] [-f] [-p]
 * where:
 * 	-c n : run the test for n number of times. 
 *	-e   : Turn on errno logging.
 *	-i n : Execute test n times.
 *	-I x : Execute test for x seconds.
 *	-p   : Pause for SIGUSR1 before starting
 *	-P x : Pause for x seconds between iterations.
 *	-t   : Turn on syscall timing.
 *
 *RESTRICTIONS:
 *for lib4 and lib5 reboot(2) system call is implemented as 
 *int reboot(int magic, int magic2, int flag, void *arg); This test case
 *is written for int reboot(int flag); which is implemented under glibc 
 *Therefore this testcase may not work under libc4 and libc5 libraries
 *****************************************************************************/

#include <unistd.h>
#include <sys/reboot.h>
#include "test.h"
#include "usctest.h"
#include <errno.h>
#include <linux/reboot.h>

static void setup();
static void cleanup();

char *TCID = "reboot01"; /* Test program identifier.    */
int TST_TOTAL = 2;       /* Total number of test cases. */
extern int Tst_count;    /* Test Case counter for tst_* routines */

static int flag[2] = {LINUX_REBOOT_CMD_CAD_ON, LINUX_REBOOT_CMD_CAD_OFF};
static unsigned char *option_message[] = {"LINUX_REBOOT_CMD_CAD_ON", 
					"LINUX_REBOOT_CMD_CAD_OFF"};

int
main(int ac, char **av)
{

	int lc, i;	/* loop counter */
	char *msg;	/* message returned from parse_opts */

	/* parse standard options */
	if ((msg = parse_opts(ac, av, (option_t *)NULL, NULL))
		!= (char *)NULL) {
		tst_brkm(TBROK, tst_exit, "OPTION PARSING ERROR - %s", msg);
	}

	/* perform global setup for test */
	setup();

	/* check looping state if -i option given */
	for (lc = 0; TEST_LOOPING(lc); lc++) {

		/* reset Tst_count in case we are looping. */
		Tst_count = 0;

		for(i = 0; i < TST_TOTAL; i++) {

			TEST(reboot(flag[i]));
			/* check return code */
			if (TEST_RETURN == -1) {
				 TEST_ERROR_LOG(TEST_ERRNO);
				 tst_resm(TFAIL, "reboot(2) Failed for "
						 "option %s",
						  option_message[i]);
			} else {
				tst_resm(TPASS, "reboot(2) Passed for "
						"option %s", option_message[i]);
			}
		}	/*End of TEST CASE LOOPING*/
	}	/*End for TEST_LOOPING*/

	/*Clean up and exit*/
	cleanup();

	/*NOTREACHED*/
	return 0;
}

/* setup() - performs all ONE TIME setup for this test */
void
setup()
{
	/* capture signals */
	tst_sig(NOFORK, DEF_HANDLER, cleanup);

	/* Check whether we are root*/
	if (geteuid() != 0) {
		tst_brkm(TBROK, tst_exit, "Test must be run as root");
	}

	/* Pause if that option was specified */
	TEST_PAUSE;

}	/* End setup() */

/*
 * cleanup() - Performs one time cleanup for this test at
 * completion or premature exit
 */

void
cleanup()
{
	/*
	* print timing stats if that option was specified.
	* print errno log if that option was specified.
	*/
	TEST_CLEANUP;

	/* exit with return code appropriate for results */
	tst_exit();
}	/* End cleanup() */

